/*******************************************************************
 *
 *  ttexec.h                                                     1.0
 *
 *    Execution context routines.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#ifndef TTEXEC_H
#define TTEXEC_H

  struct _TExecution_Context;

  typedef struct _TExecution_Context *PExecution_Context;

#ifdef TT_INTERPRETER_INDIRECT  /* indirect implementation */

#define EXEC_OPS   PExecution_Context exc,
#define EXEC_OP    PExecution_Context exc
#define EXEC_ARGS  exc,
#define EXEC_ARG   exc

#else                          /* static implementation */

#define EXEC_OPS   /* void */
#define EXEC_OP    /* void */
#define EXEC_ARGS  /* void */
#define EXEC_ARG   /* void */

#endif

 typedef  TT_F26Dot6 (*TRound_Function)( EXEC_OPS TT_F26Dot6 distance,
                                                  TT_F26Dot6 compensation );
 /* Rounding function, as used by the interpreter */
 
 typedef  void (*TMove_Function)( EXEC_OPS TT_VecRecord* zone,
                                           Int           point,
                                           TT_F26Dot6    distance );
 /* Point displacement along the freedom vector routine, as */
 /* used by the interpreter                                 */
 
 typedef  TT_F26Dot6 (*TProject_Function)( EXEC_OPS TT_F26Dot6 Vx,
                                                    TT_F26Dot6 Vy );
 /* Distance projection along one of the proj. vectors, as used */
 /* by the interpreter                                          */
 
 typedef struct _TExecution_Context 
 {
   void*       owner;     /* pointer to owner instance record */
                          /* (void*) to avoid circular deps.  */
   /* instructions state */

   Int         error;     /* last execution error */
 
   Int         curRange;  /* current code range number   */
   PByte       code;      /* current code range          */
   Int         IP;        /* current instruction pointer */
   Int         codeSize;  /* size of current range       */
 
   Byte        opcode;    /* current opcode              */
   Int         length;    /* length of current opcode    */
 
   Bool        step_ins;  /* true if the interpreter must */
                          /* increment IP after ins. exec */
 
   Int         numFDefs;  /* number of function defs */
   TDefRecord  *FDefs;    /* table of FDefs entries  */
 
   Int         numIDefs;  /* number of instruction defs */
   TDefRecord  *IDefs;    /* table of IDefs entries     */
 
   Int         callTop,    /* top of call stack during execution */
               callSize;   /* size of call stack */
   TCallRecord *callStack; /* call stack */
 
   TCodeRangeTable codeRangeTable;  /* table of valid coderanges */
                                    /* useful for the debugger   */
 
   Int         storeSize;  /* size of current storage */
   PStorage    storage;    /* storage area            */
 
   Int         stackSize;  /* size of exec. stack */
   Int         top;        /* top of exec. stack  */
   PStorage    stack;      /* current exec. stack */
 
   Int         args,
               new_top;    /* new top after exec.    */
 
   TT_F26Dot6  period;     /* values used for the */
   TT_F26Dot6  phase;      /* 'SuperRounding'     */
   TT_F26Dot6  threshold;
 
   Long        scale1;     /* values used to scale between */
   Long        scale2;     /* FUnits and Pixel coordinates */
 
   TT_F26Dot6  pointSize;  /* current pointSize  */
   Int         ppem;       /* current PPEM       */
   Int         resolution; /* current resolution */
 
   TT_VecRecord zp0,  /* zone records */
                zp1,
                zp2,
                pts,
                twilight;
 
   Int          numContours; /* number of contours in current glyph */
   PUShort      endContours; /* array of contours' end-points       */
 
   Bool         instruction_trap;  /* If True, the interpreter will */
                                   /* exit after each instruction   */

   TGraphicsState  GS;          /* current graphics state */

   TGraphicsState  default_GS;  /* graphics state resulting from */
                                /* the prep program              */

   Int         cvtSize;
   PShort      cvt;
 
   /* latest interpreter additions */
 
   Long         F_dot_P; /* dot product of freedom and projection vectors */
 
   Long         compensations[4];  /* engine compensations. Default to 0 */
 
   TRound_Function   func_round;
   
   TProject_Function func_project, 
                     func_dualproj, 
                     func_freeProj;

   TMove_Function    func_move;
   
 } TExecution_Context;


 Bool  Goto_CodeRange( PExecution_Context exec, Int range, Int IP );
 /* Goto a specified coderange */

 PCodeRange  Get_CodeRange( PExecution_Context exec, Int range );
 /* Return a pointer to a given coderange record */
 /* Used only by the debugger                    */

 Bool  Set_CodeRange( PExecution_Context exec,
                      Int                range,
                      void*              base,
                      Int                length );
 /* Set a given code range properties */

 Bool Clear_CodeRange( PExecution_Context exec, Int range );
 /* Clear a given coderange */


 Bool Create_Context( PInstance_Record  ins );
 /* Create an execution context for this instance */
 /* This call executes the font program to set    */
 /* the Function & Instruction Definitions tables */

 /* This function must be called only once */

 Bool Reset_Context( PInstance_Record  ins,
                     Int               pointSize,
                     Int               resolution  );
 /* Prepare instance's execution context for a new point size    */
 /* This call executes the prep program if the ppem or transform */
 /* has changed.                                                 */

 Bool Run_Context( PInstance_Record  ins );
 /* Run an instance's bytecode stream          */
 /* This call executes the glyph instructions  */
 /* pointed by the instance's exec. context    */

#endif /* TTEXEC_H */

