#include "ttdebug.h"
#include <stdio.h>
#include <malloc.h>

const char *OpStr[256]
        = {
            "SVTCA y",       /* Set vectors to coordinate axis y    */
            "SVTCA x",       /* Set vectors to coordinate axis x    */
            "SPvTCA y",      /* Set Proj. vec. to coord. axis y     */
            "SPvTCA x",      /* Set Proj. vec. to coord. axis x     */
            "SFvTCA y",      /* Set Free. vec. to coord. axis y     */
            "SFvTCA x",      /* Set Free. vec. to coord. axis x     */
            "SPvTL //",      /* Set Proj. vec. parallel to segment  */
            "SPvTL +",       /* Set Proj. vec. normal to segment    */
            "SFvTL //",      /* Set Free. vec. parallel to segment  */
            "SFvTL +",       /* Set Free. vec. normal to segment    */
            "SPvFS",         /* Set Proj. vec. from stack           */
            "SFvFS",         /* Set Free. vec. from stack           */
            "GPV",           /* Get projection vector               */
            "GFV",           /* Get freedom vector                  */
            "SFvTPv",        /* Set free. vec. to proj. vec.        */
            "ISECT",         /* compute intersection                */

            "SRP0",          /* Set reference point 0               */
            "SRP1",          /* Set reference point 1               */
            "SRP2",          /* Set reference point 2               */
            "SZP0",          /* Set Zone Pointer 0                  */
            "SZP1",          /* Set Zone Pointer 1                  */
            "SZP2",          /* Set Zone Pointer 2                  */
            "SZPS",          /* Set all zone pointers               */
            "SLOOP",         /* Set loop counter                    */
            "RTG",           /* Round to Grid                       */
            "RTHG",          /* Round to Half-Grid                  */
            "SMD",           /* Set Minimum Distance                */
            "ELSE",          /* Else                                */
            "JMPR",          /* Jump Relative                       */
            "SCvTCi",        /* Set CVT                             */
            "SSwCi",         /*                                     */
            "SSW",           /*                                     */

            "DUP",
            "POP",
            "CLEAR",
            "SWAP",
            "DEPTH",
            "CINDEX",
            "MINDEX",
            "AlignPTS",
            "INS_$28",
            "UTP",
            "LOOPCALL",
            "CALL",
            "FDEF",
            "ENDF",
            "MDAP[-]",
            "MDAP[r]",

            "IUP[y]",
            "IUP[x]",
            "SHP[0]",
            "SHP[1]",
            "SHC[0]",
            "SHC[1]",
            "SHZ[0]",
            "SHZ[1]",
            "SHPIX",
            "IP",
            "MSIRP[0]",
            "MSIRP[1]",
            "AlignRP",
            "RTDG",
            "MIAP[-]",
            "MIAP[r]",

            "NPushB",
            "NPushW",
            "WS",
            "RS",
            "WCvtP",
            "RCvt",
            "GC[0]",
            "GC[1]",
            "SCFS",
            "MD[0]",
            "MD[1]",
            "MPPEM",
            "MPS",
            "FlipON",
            "FlipOFF",
            "DEBUG",

            "LT",
            "LTEQ",
            "GT",
            "GTEQ",
            "EQ",
            "NEQ",
            "ODD",
            "EVEN",
            "IF",
            "EIF",
            "AND",
            "OR",
            "NOT",
            "DeltaP1",
            "SDB",
            "SDS",

            "ADD",
            "SUB",
            "DIV",
            "MUL",
            "ABS",
            "NEG",
            "FLOOR",
            "CEILING",
            "ROUND[G]",
            "ROUND[B]",
            "ROUND[W]",
            "ROUND[?]",
            "NROUND[G]",
            "NROUND[B]",
            "NROUND[W]",
            "NROUND[?]",

            "WCvtF",
            "DeltaP2",
            "DeltaP3",
            "DeltaC1",
            "DeltaC2",
            "DeltaC3",
            "SROUND",
            "S45Round",
            "JROT",
            "JROF",
            "ROFF",
            "INS_$7B",
            "RUTG",
            "RDTG",
            "SANGW",
            "AA",

            "FlipPT",
            "FlipRgON",
            "FlipRgOFF",
            "INS_$83",
            "INS_$84",
            "ScanCTRL",
            "SDPVTL[0]",
            "SDPVTL[1]",
            "GetINFO",
            "IDEF",
            "ROLL",
            "MAX",
            "MIN",
            "ScanTYPE",
            "IntCTRL",
            "INS_$8F",

            "INS_$90",
            "INS_$91",
            "INS_$92",
            "INS_$93",
            "INS_$94",
            "INS_$95",
            "INS_$96",
            "INS_$97",
            "INS_$98",
            "INS_$99",
            "INS_$9A",
            "INS_$9B",
            "INS_$9C",
            "INS_$9D",
            "INS_$9E",
            "INS_$9F",

            "INS_$A0",
            "INS_$A1",
            "INS_$A2",
            "INS_$A3",
            "INS_$A4",
            "INS_$A5",
            "INS_$A6",
            "INS_$A7",
            "INS_$A8",
            "INS_$A9",
            "INS_$AA",
            "INS_$AB",
            "INS_$AC",
            "INS_$AD",
            "INS_$AE",
            "INS_$AF",

            "PushB[0]",
            "PushB[1]",
            "PushB[2]",
            "PushB[3]",
            "PushB[4]",
            "PushB[5]",
            "PushB[6]",
            "PushB[7]",
            "PushW[0]",
            "PushW[1]",
            "PushW[2]",
            "PushW[3]",
            "PushW[4]",
            "PushW[5]",
            "PushW[6]",
            "PushW[7]",

            "MDRP[G]",
            "MDRP[B]",
            "MDRP[W]",
            "MDRP[?]",
            "MDRP[rG]",
            "MDRP[rB]",
            "MDRP[rW]",
            "MDRP[r?]",
            "MDRP[mG]",
            "MDRP[mB]",
            "MDRP[mW]",
            "MDRP[m?]",
            "MDRP[mrG]",
            "MDRP[mrB]",
            "MDRP[mrW]",
            "MDRP[mr?]",
            "MDRP[pG]",
            "MDRP[pB]",

            "MDRP[pW]",
            "MDRP[p?]",
            "MDRP[prG]",
            "MDRP[prB]",
            "MDRP[prW]",
            "MDRP[pr?]",
            "MDRP[pmG]",
            "MDRP[pmB]",
            "MDRP[pmW]",
            "MDRP[pm?]",
            "MDRP[pmrG]",
            "MDRP[pmrB]",
            "MDRP[pmrW]",
            "MDRP[pmr?]",

            "MIRP[G]",
            "MIRP[B]",
            "MIRP[W]",
            "MIRP[?]",
            "MIRP[rG]",
            "MIRP[rB]",
            "MIRP[rW]",
            "MIRP[r?]",
            "MIRP[mG]",
            "MIRP[mB]",
            "MIRP[mW]",
            "MIRP[m?]",
            "MIRP[mrG]",
            "MIRP[mrB]",
            "MIRP[mrW]",
            "MIRP[mr?]",
            "MIRP[pG]",
            "MIRP[pB]",

            "MIRP[pW]",
            "MIRP[p?]",
            "MIRP[prG]",
            "MIRP[prB]",
            "MIRP[prW]",
            "MIRP[pr?]",
            "MIRP[pmG]",
            "MIRP[pmB]",
            "MIRP[pmW]",
            "MIRP[pm?]",
            "MIRP[pmrG]",
            "MIRP[pmrB]",
            "MIRP[pmrW]",
            "MIRP[pmr?]"
         };

/*
 * Hex8, Hex16, and Hex32 deleted
 */

/*******************************************************************
 *
 *  Function    :  Cur_U_Line
 *
 *  Description :  Returns a string of the current unassembled
 *                 line at Code^[IP].
 *
 *  Input  :  Code    base code range
 *            IP      current instruction pointer
 *
 *  Output :  line string
 *
 *****************************************************************/

char *Cur_U_Line( PByte Code, Int IP )
{
  Byte Op;
  Int n, i;
  static char S[2000];
  char *pS;

  Op = Code[IP];
  pS = S + sprintf( S, "%04x: %02x  %s", IP, Op, OpStr[Op] );

  switch (Op)
  {
  case 0x40 :
    n = Code[IP+1];
    pS += sprintf( pS, "(%02x)", n );
    for (i = 1; pS - S > 20 && i <= n; ++i)
      pS += sprintf( pS, " $%02x", Code[IP+i+1] );
    break;
    
  case 0x41 :
    n = Code[IP+1];
    pS += sprintf( pS, "(%02x)", n );
    for (i = 1; pS - 2 > 20 && i <= n; ++i)
      pS += sprintf( pS, " $%02x%02x", Code[IP+i*2+1], Code[IP+i*2+2] );
    break;

  case 0xB0 : case 0xB1 : case 0xB2 : case 0xB3 :
  case 0xB4 : case 0xB5 : case 0xB6 : case 0xB7 :
    n = Op-0xB0;
    for (i = 0; pS - 2 > 20 && i <= n; ++i)
      pS += sprintf( pS, " $%02x", Code[IP+i+1] );
    break;

  case 0xB8 : case 0xB9 : case 0xBA : case 0xBB :
  case 0xBC : case 0xBD : case 0xBE : case 0xBF :
    n = Op-0xB8;
    for (i = 0; pS - 2 > 20 && i <= n; ++i)
      pS += sprintf( pS, " $%02x%02x", Code[IP+i*2+1], Code[IP+i*2+2] );
    break;
  }

  return S;
}

/*******************************************************************
 *
 *  Function    :  Get_Length
 *
 *  Description :  Returns the length in bytes of the instruction at
 *                 current instruction pointer.
 *
 *  Input  :  Code  base code range
 *            IP    current instruction pointer
 *
 *  Output :  Length in bytes
 *
 *****************************************************************/

Int Get_Length( PByte Code, Int IP )
{
  Byte Op;
  Int N;

  Op = Code[IP];

  switch (Op)
  {
  case 0x40 : N = 2 + Code[IP+1]; break;
  case 0x41 : N = 2 + Code[IP+1]*2; break;

  case 0xB0 : case 0xB1 : case 0xB2 : case 0xB3 :
  case 0xB4 : case 0xB5 : case 0xB6 : case 0xB7 :
    N = 2 + ( Op-0xB0 );
    break;
  
  case 0xB8 : case 0xB9 : case 0xBA : case 0xBB :
  case 0xBC : case 0xBD : case 0xBE : case 0xBF :
    N = 3 + ( Op-0xB8 )*2;
    break;

  default:
    N = 1;
    break;
  }

  return N;

}

/*******************************************************************
 *
 *  Function    :  Generate_Range
 *
 *  Description :  Create a list of unassembled lines for a
 *                 given code range
 *
 *  Input  :
 *
 *  Output :
 *
 *****************************************************************/

void Generate_Range( PCodeRange CR, Int index, TRangeRec *RR )
{
  Int Adr, Line, N;

  N    = CR->Size;

  RR->Code = (PByte) CR->Base;
  RR->Size = N;

  Line = 0;

  if (N > 0)
  {
    Adr  = 0;
    RR->Disassembled = (PShort) malloc( sizeof(Short) * N );

    while (Adr < N)
    {
      RR->Disassembled[Line] = Adr;
      ++Line;
      Adr += Get_Length( RR->Code, Adr );
    }
  }

  RR->NLines = Line;
  RR->Index  = index;
  RR->Breaks = NULL;
}

/*******************************************************************
 *
 *  Function    :  Get_Dis_Line
 *
 *  Description :  Returns the line index of address 'addr'
 *                 in the coderange 'cr'
 *
 *****************************************************************/

Int Get_Dis_Line( TRangeRec *cr, Int addr )
{
  Int  l, r, m;

  if (cr->NLines == 0 || addr > cr->Disassembled[cr->NLines-1] )
    return -1;

  l = 0;
  r = cr->NLines-1;

  while ( r-l > 1 )
  {
    if (cr->Disassembled[l] == addr)
      return l;

    if (cr->Disassembled[r] == addr)
      return r;

    m = (l+r) >> 1;
    if (cr->Disassembled[m] == addr)
      return m;
    else
      if (cr->Disassembled[m] < addr)
	l = m;
      else
        r = m;
  }

  if (cr->Disassembled[r] == addr)
    return r;

  return l;
}

/*******************************************************************
 *
 *  Function    :  Throw_Range
 *
 *  Description :  Destroys a list of unassembled lines for a
 *                 given code range
 *
 *  Input  :
 *
 *  Output :
 *
 *****************************************************************/

void Throw_Range( TRangeRec *RR )
{
  PBreakPoint  B, Bnext;

  if (RR->Size > 0)
    free( RR->Disassembled );

  RR->Disassembled = NULL;
  RR->Size         = 0;
  RR->Code         = NULL;
  RR->NLines       = 0;

  B = RR->Breaks;
  RR->Breaks = NULL;

  while (B)
  {
    Bnext = B->Next;
    free( B );
    /* XXXX B := B^.Next; */
    B = Bnext;
  }
}

/*******************************************************************
 *
 *  Function    :  Set_Break
 *
 *  Description :  Sets a Breakpoint ON
 *
 *  Input  :
 *
 *  Output :
 *
 *****************************************************************/

void Set_Break( PBreakPoint *Head, Int Range, Int Adr )
{
  PBreakPoint  BP, Old, Cur;

  Old = NULL;
  Cur = *Head;

  while (Cur && Cur->Address < Adr)
  {
    Old = Cur;
    Cur = Cur->Next;
  }

  /* No duplicates */
  if (Cur)
    if (Cur->Address == Adr && Cur->Range == Range)
      return;

  BP = (PBreakPoint) malloc( sizeof(TBreakPoint) );
  BP->Address = Adr;
  BP->Range   = Range;
  BP->Next    = Cur;

  if (!Old)
    *Head = BP;
  else
    Old->Next = BP;
}

/*******************************************************************
 *
 *  Function    :  Clear_Break
 *
 *  Description :  Clears a breakpoint OFF
 *
 *  Input  :
 *
 *  Output :
 *
 *****************************************************************/

void Clear_Break( PBreakPoint *Head, PBreakPoint Bp )
{
  PBreakPoint  Old, Cur;

  Old = NULL;
  Cur = *Head;

  while (Cur && Cur != Bp)
  {
    Old = Cur;
    Cur = Cur->Next;
  }

  if (!Cur) return;

  if (!Old)
    *Head = Cur->Next;
  else
    Old->Next = Cur->Next;
}

void Toggle_Break( PBreakPoint *Head, Int Range, Int Adr )
{
  PBreakPoint  Bp;
  Bp = Find_BreakPoint( *Head, Range, Adr );
  if (Bp)
    Clear_Break( Head, Bp );
  else
    Set_Break( Head, Range, Adr );
}

/*******************************************************************
 *
 *  Function    :  Clear_All_Breaks
 *
 *  Description :  Clears all breakpoints
 *
 *  Input  :
 *
 *  Output :
 *
 *****************************************************************/

void Clear_All_Breaks( PBreakPoint *Head )
{
  PBreakPoint  Old, Cur;

  Cur   = *Head;
  *Head = NULL;

  while (Cur)
  {
    Old = Cur;
    Cur = Cur->Next;
    free( Old );
  }
}

/*******************************************************************
 *
 *  Function    :  Find_BreakPoint
 *
 *  Description :  Find a breakpoint at address IP
 *
 *  Input  :   Head     break points sorted linked list
 *             IP       address of expected breakpoint
 *
 *  Output :   pointer to breakpoint if found
 *             nil otherwise.
 *
 *****************************************************************/

PBreakPoint Find_BreakPoint( PBreakPoint Head, Int Range, Int IP )
{
  PBreakPoint  Cur, Res;

  Cur = Head;
  Res = NULL;

  while (Cur)
  {
    if (Cur->Address == IP && Cur->Range == Range)
      Res = Cur;

    if (Cur->Address >= IP)
      Cur = NULL;
    else
      Cur = Cur->Next;
  }

  return Res;
}

