#include "slick.sh"

//////////////////////////////////////////////////////////////////////////////
// global variables
//
#define CONTEXT_TB_FORM_NAME_STRING '_tbcontext_combo_etab'
#define CONTEXT_TOOLTIP_DELAYINC    100

#define PIC_LSPACE_Y   60    // Extra line spacing for list box.
#define PIC_LINDENT_X  150    // Indent before for list box bitmap.
#define PIC_RINDENT_X  40    // Indent after list box bitmap (hard-coded)

//////////////////////////////////////////////////////////////////////////////
// ordered access levels
//
#define CLASS_ACCESS_PRIVATE   0
#define CLASS_ACCESS_PROTECTED 1
#define CLASS_ACCESS_PACKAGE   2
#define CLASS_ACCESS_PUBLIC    3

//////////////////////////////////////////////////////////////////////////////
// built-in limits
//
#define MAX_SYMBOL_MATCHES    512
#define MAX_RECURSIVE_SEARCH   32
#define MAX_FUNCTION_NESTING    8


//############################################################################
//////////////////////////////////////////////////////////////////////////////
// Globals needed for tool tips for context control
//
static int gi_ContextTimerID = -1;
static int gi_ContextWindowId = -1;
static boolean gContextToolTipShown = false;
static boolean gi_haveContextWindow = true;

static _str gcontext_window_filename = '';
static int  gcontext_window_seekpos  = 0;

//////////////////////////////////////////////////////////////////////////////
// Called when this module is loaded (before defload).  Used to
// initialize the timer variable and window IDs.
// 
definit()
{
   // IF editor is initalizing from invocation
   if (arg(1)!='L') {
      gi_ContextTimerID = -1;
      gi_ContextWindowId = -1;
      gContextToolTipShown = false;
      gi_haveContextWindow = true;
   }
}

//////////////////////////////////////////////////////////////////////////////
// Called when this module is loaded (after definit).  Used to
// correctly initialize the window IDs (if those forms are available),
// and loads the array of pictures used for different tag types.
//
defload()         
{
}


//############################################################################
//////////////////////////////////////////////////////////////////////////////
// kill the context combo box tool tip
//
static void killContextCBTimer()
{
   //say("killContextCBTimer("gi_ContextTimerID")");
   if (gi_ContextTimerID>=0) _kill_timer( gi_ContextTimerID );
   gi_ContextTimerID = -1;
   if (gContextToolTipShown) _bbhelp('C');
   gContextToolTipShown = false;
}

// check mouse position against context combo box, maybe display tool tip
static void contextTimerCB()
{
   //say("contextTimerCB()");
   // verify that gi_ContextWindowId is a valid combo box
   if (!_iswindow_valid(gi_ContextWindowId) ||
       gi_ContextWindowId.p_object!=OI_COMBO_BOX) {
      killContextCBTimer();
      return;
   }

   // Get our x and y mouse coordinates, and map to list box
   int orig_wid = p_window_id;
   p_window_id = gi_ContextWindowId;
   int mx, my;
   mou_get_xy(mx,my);
   int mou_y = my;
   _map_xy(0,p_cb_list_box,mx,my);
   int pic_width = p_cb_picture.p_width;

   // needed below for common code that pops up button help
   int parent_wid = p_window_id;
   _str caption = '';
   int x, y;

   if (p_cb_text_box.mou_in_window()) {

      // mouse is over combo box text box window?
      caption = p_text;
      x = p_x;
      y = p_y+p_height;
      _lxy2dxy(SM_TWIP,x,y);
      _map_xy(p_parent,0,x,y);

      // caption is not obscured, no bubble help
      int caption_width = _text_width(caption);
      if (caption_width < p_cb_text_box.p_width - pic_width) {
         killContextCBTimer();
         return;
      }

   } else if (p_cb_list_box.p_visible && p_cb_list_box.mou_in_window() &&
              mx > 0 && mx < p_cb_list_box.p_client_width) {

      // combo is dropped down and mouse is over list box
      p_window_id = p_cb_list_box;
      parent_wid = p_window_id;
      if (p_scroll_left_edge>=0) {
         parse _scroll_page() with line_pos down_count;
         goto_point(line_pos);
         down(down_count);
         set_scroll_pos(p_scroll_left_edge,0);
         p_scroll_left_edge = -1;
      }
      //if (_on_line0()) return;
      if (mou_last_y() > p_font_height*p_char_height) {
         scroll_down();
      }
      p_cursor_y=mou_last_y();
      p_cursor_x=p_left_edge+p_windent_x;
      caption = _lbget_text();
      p_window_id = gi_ContextWindowId;

      // show tool tip for the context combo box
      x = p_x + PIC_LINDENT_X + pic_width + PIC_RINDENT_X;
      y = p_y + p_height;
      _lxy2dxy(SM_TWIP,x,y);
      _map_xy(p_parent,0,x,y);
      mou_y -= y;
      mou_y -= (mou_y % p_cb_list_box.p_font_height);
      y += mou_y;

      // If tooltip still in same tab but tab no longer has partial caption:
      int caption_width = _text_width(caption);
      int client_width  = _dx2lx(p_xyscale_mode,p_cb_list_box.p_client_width);
      if (caption_width < client_width - pic_width - PIC_LINDENT_X - PIC_RINDENT_X) {
         killContextCBTimer();
         return;
      }

   } else {
      // mouse has moved outside of combo box or drop-down list
      killContextCBTimer();
      p_window_id = orig_wid;
      return;
   }

   // restore window ID
   p_window_id = orig_wid;

   // Same as before, then just return
   static int giOldContextWindowId;
   static _str gzOldContextCaption;
   if (gi_ContextWindowId == giOldContextWindowId &&
       caption :== gzOldContextCaption && gContextToolTipShown) {
      return;
   }
   giOldContextWindowId = gi_ContextWindowId;
   gzOldContextCaption = caption;
   //if (gContextToolTipShown) {
   //   _bbhelp('C');
   //}

   // show tool tip for the context combo box
   if (x < 0) x = 0;
   _bbhelp('M',parent_wid,x,y,caption);
   gContextToolTipShown=true;

   if (gi_ContextTimerID>=0) _kill_timer( gi_ContextTimerID );
   gi_ContextTimerID = _set_timer( CONTEXT_TOOLTIP_DELAYINC, contextTimerCB, 0 );
}


//////////////////////////////////////////////////////////////////////////////
// Constants for drawing list box
//

defeventtab _tbcontext_combo_etab;
void _tbcontext_combo_etab.on_drop_down(int reason)
{
   gi_haveContextWindow = true;
   killContextCBTimer();
   if (_no_child_windows()) {
      return;
   }

   // set caption and bitmaps for current context
   if (reason==DROP_DOWN) {
      _mdi.p_child._UpdateContext(true);
      _mdi.p_child._UpdateLocals(true,true);
      // set caption and bitmaps for current context
      cb_prepare_expand(p_window_id, 0, 0);
      p_cb_list_box._lbclear();
      p_cb_list_box.p_picture=_pic_file;

      p_cb_list_box._lbadd_item("---Locals---",0,_pic_project);
      _str lcl_start_point = p_cb_list_box.point();
      int num_locals = tag_get_num_of_locals();
      for (i=1; i<=num_locals; i++) {
         tag_get_local2(i, proc_name, type_name, file_name, 
                         line_no, start_seekpos, 
                         scope_line_no, scope_seekpos,
                         end_line_no, end_seekpos,
                         class_name, tag_flags, arguments, return_type);
         tag_list_insert_tag(p_cb_list_box, 0, PIC_LINDENT_X, proc_name, type_name, file_name, line_no, class_name, tag_flags, return_type:+VS_TAGSEPARATOR_args:+arguments);
      }
      _str lcl_end_point = p_cb_list_box.point();
      p_cb_list_box._lbadd_item("---Buffer---",0,_pic_project);
      _str ctx_start_point = p_cb_list_box.point();
      int num_context = tag_get_num_of_context();
      for (i=1; i<=num_context; i++) {
         tag_get_context(i, proc_name, type_name, file_name, 
                         line_no, start_seekpos, 
                         scope_line_no, scope_seekpos,
                         end_line_no, end_seekpos,
                         class_name, tag_flags, arguments, return_type);
         tag_list_insert_tag(p_cb_list_box, 0, PIC_LINDENT_X, proc_name, type_name, file_name, line_no, class_name, tag_flags, return_type:+VS_TAGSEPARATOR_args:+arguments);
      }
      _str ctx_end_point = p_cb_list_box.point();
      p_cb_list_box._lbsort('i',lcl_start_point,lcl_end_point);
      p_cb_list_box._lbsort('i',ctx_start_point,ctx_end_point);
      //p_cb_list_box._lbsort('i');
      //p_cb_list_box._remove_duplicates();
      int h = p_pic_space_y/2 + _dy2ly(p_cb_list_box.p_xyscale_mode,p_cb_list_box.p_font_height);//cb_list_box._text_height();
      h *= (p_cb_list_box.p_Noflines>4)? p_cb_list_box.p_Noflines : 4; 
      int sh = (_screen_height()*_twips_per_pixel_y()) / 2;
      p_cb_list_box.p_height = (h>sh)? sh:h;
      p_cb_list_box._lbtop();
   } else if (reason==DROP_UP_SELECTED) {
      _str selected_caption = p_cb_list_box._lbget_text();
      if (pos("---",selected_caption)==1) {
         p_picture = _pic_project;
         return;
      }
      //say("SELECTED: "selected_caption);
      _mdi.p_child._UpdateContext(true);
      _mdi.p_child._UpdateLocals(true,true);
      int num_locals = tag_get_num_of_locals();
      for (i=1; i<=num_locals; i++) {
         tag_get_local(i, proc_name, type_name, file_name, 
                       line_no, class_name, tag_flags, arguments, return_type);
         caption = tag_tree_make_caption(proc_name, type_name, class_name, tag_flags, arguments, false);
         if (selected_caption :== caption) {
            push_tag_in_file(proc_name, file_name, class_name, type_name, line_no);
            break;
         }
      }
      int num_context = tag_get_num_of_context();
      for (i=1; i<=num_context; i++) {
         tag_get_context(i, proc_name, type_name, file_name, 
                         line_no, start_seekpos, 
                         scope_line_no, scope_seekpos,
                         end_line_no, end_seekpos,
                         class_name, tag_flags, arguments, return_type);
         caption = tag_tree_make_caption(proc_name, type_name, class_name, tag_flags, arguments, false);
         if (selected_caption :== caption) {
            push_tag_in_file(proc_name, file_name, class_name, type_name, line_no);
            break;
         }
      }
   }
}
// callback for when a context combo box is created
void _tbcontext_combo_etab.on_create()
{
   p_width = 3600;
   gi_haveContextWindow = true;
   gcontext_window_filename = '';
   gcontext_window_seekpos = 0;
   p_pic_point_scale=8;
   p_pic_space_y=PIC_LSPACE_Y;
   p_style=PSCBO_NOEDIT;
   if (_no_child_windows()) {
      p_cb_list_box._lbclear();
      p_picture = 0;
      ContextMessage('');
   } else {
      _UpdateContextWindow(true);
   }
}

// monitor mouse move events when over the context window
void _tbcontext_combo_etab.mouse_move()
{
   //say("CONTEXT: mouse_move");
   gi_ContextWindowId = p_window_id;
   //gContextToolTipShown=false;
   if (gi_ContextTimerID >= 0 || gContextToolTipShown) {
      //say("shown="gContextToolTipShown);
      return;
   }
   gi_ContextTimerID = _set_timer(CONTEXT_TOOLTIP_DELAYINC, contextTimerCB, 0);
}

// leave message in context box
//
static void ContextMessage(_str msg, int pic_index=0)
{
   if (msg=='') {
      msg = "no current context";
   }
   if (_mdi.p_button_bar && gi_haveContextWindow) {
      gi_haveContextWindow = false;
      eventtab=defeventtab _tbcontext_combo_etab;
      for (i=1;i<=_last_window_id();++i) {
         if (_iswindow_valid(i) && !i.p_edit && i.p_eventtab==eventtab) {
            gi_haveContextWindow = true;
            i.p_picture = pic_index;
            i.p_text = msg;
         }
      }
   }
}
// update the current context combo box window(s)
// the current object must be the active buffer
//
void _UpdateContextWindow(boolean AlwaysUpdate=false)
{
   //say("_UpdateContext()");
   // make sure timer has waited long enough
   if (!AlwaysUpdate && _idle_time_elapsed()<300) {
      //say("don't always update");
      return;
   }

   // do not update if no context window to update
   if (!gi_haveContextWindow) {
      return;
   }

   // if there is no current buffer, blow out of here
   if (_no_child_windows()) {
      //say("no child windows");
      ContextMessage('');
      return;
   }

   // blow out of here if cursor hasn't moved and file not modified
   int curr_seekpos = _mdi.p_child._nrseek();
   if ((_mdi.p_child.p_ModifyFlags&MODIFYFLAG_CONTEXTWIN_UPDATED) &&
       gcontext_window_seekpos==curr_seekpos && 
       gcontext_window_filename:==_mdi.p_child.p_buf_name) {
      //say("no cursor movement");
      return;
   }
   gcontext_window_filename = _mdi.p_child.p_buf_name;
   gcontext_window_seekpos  = curr_seekpos;
   _mdi.p_child.p_ModifyFlags |= MODIFYFLAG_CONTEXTWIN_UPDATED;

   // Update the current context
   _mdi.p_child._UpdateContext(true);
   _mdi.p_child._UpdateLocals(true);
   
   // Update the context message, if current context is local variable
   int local_id = _mdi.p_child.tag_current_local();
   if (local_id > 0) {
      tag_get_local(local_id, proc_name, type_name, file_name, line_no, 
                    class_name, flags, arguments, return_type);
      caption = tag_tree_make_caption(proc_name, type_name, class_name, flags, arguments, false);
      tag_tree_filter_member(0, type_name, 0, flags, i_access, i_type);
      tag_tree_select_bitmap(i_access, i_type, leaf_flag, pic_index);
      ContextMessage(caption, pic_index);
      _mdi.p_child.p_ModifyFlags |= MODIFYFLAG_CONTEXTWIN_UPDATED;
      return;
   }

   // Update the context message
   int context_id = _mdi.p_child.tag_current_context();
   if (context_id <= 0) {
      ContextMessage('');
      //say("no context");
      return;
   }
   tag_get_context(context_id, proc_name, type_name, file_name, 
                   line_no, start_seekpos, 
                   scope_line_no, scope_seekpos,
                   end_line_no, end_seekpos,
                   class_name, flags, arguments, return_type);

   caption = tag_tree_make_caption(proc_name, type_name, class_name, flags, arguments, false);
   tag_tree_filter_member(0, type_name, 0, flags, i_access, i_type);
   tag_tree_select_bitmap(i_access, i_type, leaf_flag, pic_index);
   ContextMessage(caption, pic_index);
   _mdi.p_child.p_ModifyFlags |= MODIFYFLAG_CONTEXTWIN_UPDATED;
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
// update the current context and context tree message
// the current object must be the active buffer
//
void _UpdateContext(boolean AlwaysUpdate=false)
{
   //say("_UpdateContext()");
   // make sure timer has waited long enough
   if (!AlwaysUpdate && _idle_time_elapsed()<1500) {
      return;
   }

   // if there is no current buffer, blow out of here
   //if (_no_child_windows()) {
   //   tag_clear_context();
   //   return;
   //}

   // if this buffer is not taggable, blow out of here
   PSindex=find_index(p_extension'-proc-search',PROC_TYPE)
   if (_isEditorCtl() && !index_callable(PSindex)) {
      tag_clear_context();
      return;
   }

   // current context names
   _str proc_name;
   _str signature;
   _str return_type;
   int outer_context, dummy_context;
   int start_line_no, start_seekpos;
   int scope_line_no, scope_seekpos;
   int end_line_no, end_seekpos;
   outer_context = dummy_context = 0;
   start_line_no = start_seekpos = 0;
   scope_line_no = scope_seekpos = 0;
   end_line_no = end_seekpos = 0;
   proc_name = signature = return_type = '';

   // check current buffer position against context
   int have_context = tag_check_context();

   // outside of current context
   if (!have_context || !(p_ModifyFlags&MODIFYFLAG_CONTEXT_UPDATED)) {

      //say("Recalculating context="have_context);

      // if context is being recalculated, then invalidate
      // locals, context window, and proctree
      p_ModifyFlags &= ~MODIFYFLAG_LOCALS_UPDATED;
      p_ModifyFlags &= ~MODIFYFLAG_CONTEXTWIN_UPDATED;
      p_ModifyFlags &= ~MODIFYFLAG_PROCTREE_UPDATED;
      p_ModifyFlags &= ~MODIFYFLAG_FCTHELP_UPDATED;

      // find extension specific tagging functions
      LTindex=find_index('vs'p_extension'-list-tags',PROC_TYPE)
      if (index_callable(LTindex)) {
         // call list-tags with LTF_SET_TAG_CONTEXT flags
         // might want to update proctree at same time
         tag_clear_context();
         save_pos(p);
         status = call_index(0, '', p_extension, VSLTF_SET_TAG_CONTEXT, LTindex);
         restore_pos(p);

      } else { // PSindex was already verified above
         // use slower timer if calling proc_search
         //if (!AlwaysUpdate && _idle_time_elapsed()<300) {
         //if (!AlwaysUpdate && _idle_time_elapsed()<1000) {
         //   return;
         //}
         // call proc-search to find current context
         tag_clear_context();
         save_pos(p);
         save_search(p1,p2,p3,p4);
         top();
         proc_name='';
         findfirst=1;
         context_id=0;
         for (;;) {
            //say("_UpdateContext");
            proc_name='';
            status=1;
            status=call_index(proc_name,findfirst,p_extension,PSindex);
            start_line_no = p_RLine;
            start_seekpos = _nrseek();
            //say("proc_name="proc_name" line_no="start_line_no);
            if (context_id > 0) {
               if (status) {
                  bottom();
               } else {
                  _nrseek(start_seekpos-1);
               }
               _clex_find(~COMMENT_CLEXFLAG,'-O');
               right();
               end_line_no = p_RLine;
               end_seekpos = _nrseek();
               _nrseek(start_seekpos);
               tag_end_context(context_id, end_line_no, end_seekpos);
            }
            if (status) {
               break;
            }
            if (proc_name != '') {
               // search backward for non-blank character
               tag_tree_decompose_tag(proc_name, tag_name, class_name, type_name, tag_flags, signature, return_type);
               context_id=tag_insert_context(0, tag_name, type_name, p_buf_name, start_line_no, start_seekpos, start_line_no, start_seekpos, end_line_no, end_seekpos, class_name, tag_flags, return_type"\1"signature);
            }
            findfirst=0;
         }
         restore_search(p1,p2,p3,p4);
         restore_pos(p);
      }

      // call post-processing function for update context
      int AUindex=find_index('_'p_extension'_after_UpdateContext',PROC_TYPE);
      if (index_callable(AUindex)) {
         save_search(p1,p2,p3,p4);
         call_index(AUindex);
         restore_search(p1,p2,p3,p4);
      }

      // set the modify flags, showing that the context is up to date
      p_ModifyFlags |= MODIFYFLAG_CONTEXT_UPDATED;
   }
}

// current object must be the current buffer
void _UpdateLocals(boolean AlwaysUpdate=false, boolean list_all=false)
{
   // if there is no current buffer, blow out of here
   //if (_no_child_windows()) {
   //   //say("NO CHILD WINDOWS");
   //   tag_clear_locals(0);
   //   return;
   //}

   // if this buffer is not taggable, blow out of here
   if (!_isEditorCtl()) {
      //say("NOT AN EDITOR CONTROL!");
      return;
   }

   //say("_UpdateLocals()");
   // update the current context
   _UpdateContext(true);
   //say("_UpdateLocals");
   int context_id = tag_current_context();

   static _str last_file_name;
   static int  last_start_seekpos;
   static int  last_end_seekpos;
   static int  last_context_id;
   static boolean last_list_all;

   // is the current context defined?
   if (context_id > 0) {

      tag_get_detail2(VS_TAGDETAIL_context_file,          context_id, cur_file_name);
      tag_get_detail2(VS_TAGDETAIL_context_start_linenum, context_id, cur_start_line_no);
      tag_get_detail2(VS_TAGDETAIL_context_start_seekpos, context_id, cur_start_seekpos);
      tag_get_detail2(VS_TAGDETAIL_context_end_seekpos,   context_id, cur_end_seekpos);

      // end seekpos is position of cursor or end of function
      int end_seekpos = cur_end_seekpos;
      if (!list_all) {
         end_seekpos=_nrseek();
      }

      // locals already up to date?
      if (last_file_name     :== cur_file_name && 
          last_context_id    :== context_id &&
          last_start_seekpos :== cur_start_seekpos &&
          last_end_seekpos   :== end_seekpos &&
          last_list_all      :== list_all &&
          (p_ModifyFlags & MODIFYFLAG_LOCALS_UPDATED)) {
         return;
      }
      last_file_name     = cur_file_name;
      last_start_seekpos = cur_start_seekpos;
      last_end_seekpos   = end_seekpos;
      last_context_id    = context_id;
      last_list_all      = list_all;

      // set list-tags flags appropriately
      int ltf_flags = VSLTF_SET_TAG_CONTEXT;
      if (!list_all) {
         ltf_flags |= VSLTF_SKIP_OUT_OF_SCOPE;
      }

      // check if we have support for local variable search
      VS_TAG_BROWSE_INFO outer_locals[];
      outer_locals._makeempty();
      tag_clear_locals(0);
      int i,k,n;
      int LLindex=find_index(p_extension'-list-locals',PROC_TYPE);
      if (index_callable(LLindex)) {

         // list locals for each level of context
         boolean local_hash:[]; local_hash._makeempty();
         int func_nesting=0;
         context_id = tag_current_context();
         while (context_id>0 && func_nesting<MAX_FUNCTION_NESTING) {

            // check that the current context is a function
            tag_get_detail2(VS_TAGDETAIL_context_type, context_id, cur_type_name);
            if (tag_tree_type_is_func(cur_type_name)) {

               // get the start line, seekpos, end seekpos information for parsing
               tag_get_detail2(VS_TAGDETAIL_context_start_linenum, context_id, cur_start_line_no);
               tag_get_detail2(VS_TAGDETAIL_context_start_seekpos, context_id, cur_start_seekpos);
               tag_get_detail2(VS_TAGDETAIL_context_end_seekpos, context_id, end_seekpos);
               if (context_id==tag_current_context() && !list_all) {
                  end_seekpos=_nrseek();
               }

               // call list-tags with LTF_SET_TAG_CONTEXT flags
               local_hash:[cur_start_seekpos]=true;
               save_pos(p);
               p_line = cur_start_line_no;
               //say("CCUR_START_SEEKPOS="cur_start_seekpos" CEND_SEEKPOS="end_seekpos);
               _nrseek(cur_start_seekpos);
               call_index(0,'',p_extension,ltf_flags,0,0,cur_start_seekpos,end_seekpos,LLindex);
               restore_pos(p);

               // save/append the current locals to the array
               n=tag_get_num_of_locals();
               for (i=1; i<=n; i++) {
                  VS_TAG_BROWSE_INFO cm;
                  tag_get_local2(i, cm.member_name, cm.type_name, cm.file_name,
                                 cm.line_no, cm.seekpos,
                                 cm.scope_line_no, cm.scope_seekpos,
                                 cm.end_line_no, cm.end_seekpos,
                                 cm.class_name, cm.flags, cm.arguments, cm.return_type);
                  tag_get_detail2(VS_TAGDETAIL_local_parents, i, cm.category);
                  outer_locals[outer_locals._length()] = cm;
                  //say("Cgot "cm.member_name);
               }
            }

            // get the next level higher of function nesting
            tag_get_detail2(VS_TAGDETAIL_context_outer, context_id, context_id);
            func_nesting++;
         }

         // list locals for each level of local
         int local_id = tag_current_local();
         while (local_id>0) {

            // get the type, start line, seekpos, end seekpos information for parsing
            tag_get_detail2(VS_TAGDETAIL_local_type, local_id, cur_type_name);
            tag_get_detail2(VS_TAGDETAIL_local_start_linenum, local_id, cur_start_line_no);
            tag_get_detail2(VS_TAGDETAIL_local_start_seekpos, local_id, cur_start_seekpos);
            tag_get_detail2(VS_TAGDETAIL_local_end_seekpos,   local_id, cur_end_seekpos);

            // check that the current local is a function
            if (tag_tree_type_is_func(cur_type_name) &&
                !local_hash._indexin(cur_start_seekpos)) {

               // call list-tags with LTF_SET_TAG_CONTEXT flags
               local_hash:[cur_start_seekpos]=true;
               save_pos(p);
               p_line = cur_start_line_no;
               //say("LCUR_START_SEEKPOS="cur_start_seekpos" LEND_SEEKPOS="cur_end_seekpos);
               _nrseek(cur_start_seekpos);
               call_index(0,'',p_extension,ltf_flags,0,0,cur_start_seekpos,cur_end_seekpos,LLindex);
               restore_pos(p);

               // save/append the current locals to the array
               n=tag_get_num_of_locals();
               for (i=1; i<=n; i++) {
                  VS_TAG_BROWSE_INFO cm;
                  tag_get_local2(i, cm.member_name, cm.type_name, cm.file_name,
                                 cm.line_no, cm.seekpos,
                                 cm.scope_line_no, cm.scope_seekpos,
                                 cm.end_line_no, cm.end_seekpos,
                                 cm.class_name, cm.flags, cm.arguments, cm.return_type);
                  tag_get_detail2(VS_TAGDETAIL_local_parents, i, cm.category);
                  outer_locals[outer_locals._length()] = cm;
                  //say("Lgot "cm.member_name);
               }

               // append locals from inner functions
               tag_clear_locals(0)
               n=outer_locals._length();
               for (i=0; i<n; i++) {
                  VS_TAG_BROWSE_INFO cm = outer_locals[i];
                  _str local_sig = cm.return_type;
                  if (cm.arguments!='') {
                     strappend(local_sig, VS_TAGSEPARATOR_args:+cm.arguments);
                  }
                  k=tag_insert_local2(cm.member_name, cm.type_name, cm.file_name,
                                      cm.line_no, cm.seekpos,
                                      cm.scope_line_no, cm.scope_seekpos,
                                      cm.end_line_no, cm.end_seekpos,
                                      cm.class_name, cm.flags, local_sig);
                  if (cm.category!='') {
                     tag_set_local_parents(k,cm.category);
                  }
               }

               local_id = tag_current_local();
               continue;
            }

            // get the next level higher of function nesting
            tag_get_detail2(VS_TAGDETAIL_local_outer, local_id, local_id);
         }

         // append locals from inner functions
         tag_clear_locals(0)
         n=outer_locals._length();
         for (i=0; i<n; i++) {
            VS_TAG_BROWSE_INFO cm = outer_locals[i];
            _str local_sig = cm.return_type;
            if (cm.arguments!='') {
               strappend(local_sig, VS_TAGSEPARATOR_args:+cm.arguments);
            }
            k=tag_insert_local2(cm.member_name, cm.type_name, cm.file_name,
                                cm.line_no, cm.seekpos,
                                cm.scope_line_no, cm.scope_seekpos,
                                cm.end_line_no, cm.end_seekpos,
                                cm.class_name, cm.flags, local_sig);
            if (cm.category!='') {
               tag_set_local_parents(k,cm.category);
            }
         }
      }

      //say("_UpdateLocals: "tag_get_num_of_locals()" locals");
      //for (j=1; j<=tag_get_num_of_locals(); j++) {
      //   tag_get_detail2(VS_TAGDETAIL_local_parents, j, class_parents);
      //   tag_get_detail2(VS_TAGDETAIL_local_name, j, proc_name);
      //   say("   LCL: j="j" name="proc_name" parents="class_parents);
      //}

      // call post-processing function for update locals
      int AUindex=find_index('_'p_extension'_after_UpdateLocals',PROC_TYPE);
      if (index_callable(AUindex)) {
         call_index(AUindex);
      }

      // locals are updated, all done
      p_ModifyFlags |= MODIFYFLAG_LOCALS_UPDATED;
      return;
   }
   tag_clear_locals(0);
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
// Check if the given class is in the same package as the other class.
// Returns true of so, false otherwise.
//
static boolean _IsSamePackageAs(_str class_one, _str class_two, boolean case_sensitive)
{
   int pos_one = pos(VS_TAGSEPARATOR_package, class_one);
   int pos_two = pos(VS_TAGSEPARATOR_package, class_two);
   if (pos_one && pos_two) {
      // both have packages, that's encouraging
      _str package_one = substr(class_one, 1, pos_one-1);
      _str package_two = substr(class_two, 1, pos_two-1);
      if (case_sensitive && package_one :== package_two) {
         return true;
      }
      if (!case_sensitive && !stricmp(package_one,package_two)) {
         return true;
      }
   } else if (!pos_one && !pos_two) {
      return true;
   }
   // packages don't match, or one class in package, other is not...
   return false;
}

//////////////////////////////////////////////////////////////////////////////
// Match the given symbol based on the current context information
//    symbol        -- current symbol to look for
//    pushtag_flags -- VS_TAGFILTER_*, -1 if not applicable
//    strict        -- use strict "language" rules for finding match
//    class_name    -- look for identifier in the specified class
//    max_matches   -- maximum number of items to stuff in match list
//
// Order of searching is:
//   1) local variables in current function
//   2) members of current class, including inherited members
//   3) globals found in the current file
//   4) globals, (not static variables), found in other files
//   5) any symbol found in this file
//   6) any symbol found in any tag file
// Failing that, it repeats steps (1-6) with pushtag_flags set to -1,
// thus disabling any filtering.
// The current object must be an editor control or current buffer.
//
void _MatchSymbolInContext(_str symbol_prefix, _str search_class_name,
                           int &num_matches,int max_matches=MAX_SYMBOL_MATCHES,
                           int pushtag_flags=VS_TAGFILTER_ANYTHING,
                           boolean strict=true, boolean allow_locals=true,
                           boolean find_parents=false, boolean find_all=false,
                           boolean exact_match=false, boolean case_sensitive=true)
{
   //say("_MatchSymbolInContext("symbol_prefix","search_class_name","pushtag_flags","case_sensitive")");
   // list of matching tags
   typeless tag_files = tags_filenamea(p_extension);
   _str match_tag_db = '';

   // update the current context
   _UpdateContext(true);
   _UpdateLocals(true);
   int context_id = tag_current_context();

   // gulp up the critical information about current context
   _str cur_proc_name = '', cur_class_name = '', cur_type_name = '';
   if (context_id > 0) {
      tag_get_detail2(VS_TAGDETAIL_context_name, context_id, cur_proc_name);
      tag_get_detail2(VS_TAGDETAIL_context_class, context_id, cur_class_name);
      tag_get_detail2(VS_TAGDETAIL_context_type, context_id, cur_type_name);
      if (tag_tree_type_is_class(cur_type_name)) {
         cur_class_name = tag_join_class_name(cur_proc_name, cur_class_name, tag_files, case_sensitive);
      }
      if (tag_tree_type_is_package(cur_type_name)) {
         cur_class_name = cur_proc_name;
      }
   }

   // for function context, first try to find a local variable
   boolean found_definition = false;
   if (allow_locals) {
      found_definition = tag_list_class_locals(0, 0, tag_files, 
                                               symbol_prefix, search_class_name,
                                               pushtag_flags, VS_TAGCONTEXT_ANYTHING,
                                               num_matches, max_matches,
                                               exact_match, case_sensitive)? true:false;
      // found some matches?  then quit looking
      if (found_definition && !find_all && exact_match) {
         //say("got a local");
         return;
      }
   }

   // set up context flags
   int  context_flags = VS_TAGCONTEXT_ACCESS_package; // generous
   _str effective_class_name = search_class_name;
   if (search_class_name == '') {
      // if the current context is a class, then the effective
      // current class is it, not cur_class_name
      context_flags = VS_TAGCONTEXT_ACCESS_private;
      effective_class_name = cur_class_name;
      //if (context_id > 0 && tag_tree_type_is_class(cur_type_name)) {
      //   effective_class_name = tag_join_class_name(cur_proc_name, cur_class_name, tag_files, case_sensitive);
      //}
      //if (context_id > 0 && tag_tree_type_is_package(cur_type_name)) {
      //   effective_class_name = cur_proc_name;
      //}
   } else if (search_class_name :== '::') {
      effective_class_name = '';
   } else if (search_class_name :== cur_class_name) {
      context_flags = VS_TAGCONTEXT_ACCESS_private;
   } else if (tag_is_parent_class(search_class_name, cur_class_name,
                                  tag_files, case_sensitive, true)) {
      context_flags = VS_TAGCONTEXT_ACCESS_protected;
   } else if (_IsSamePackageAs(search_class_name, cur_class_name,
                               case_sensitive)) {
      context_flags = VS_TAGCONTEXT_ACCESS_package;
   }
   if (allow_locals) {
      context_flags |= VS_TAGCONTEXT_ALLOW_locals;
   }
   if (find_parents) {
      context_flags |= VS_TAGCONTEXT_FIND_derived;
   }
   if (pushtag_flags & VS_TAGFILTER_ANYSTRUCT) {
      context_flags |= VS_TAGCONTEXT_ALLOW_anonymous;
   }

   // look up class members for each level of scope
   while (effective_class_name != '') {

      //say("effective_class_name="effective_class_name);
      // look up class members for current class
      found_definition = _ListSymbolsInClass(symbol_prefix, effective_class_name, 
                                             0, 0, 0, num_matches, max_matches, 
                                             pushtag_flags, context_flags, 
                                             exact_match, case_sensitive);

      // if were given a search class, then stop when out of scope
      if (strict && search_class_name != '' && search_class_name :!= cur_class_name) {
         //say("got a context match, search_class_name="search_class_name);
         return;
      }
      // found some matches?  then quit looking
      if (num_matches > 0 && found_definition && !find_all && exact_match) {
         return;
      }

      // strip last part of class name off
      //effective_class_name = _GetOuterClassName(effective_class_name);
      tag_split_class_name(effective_class_name,junk,effective_class_name);
   }

   // found some matches?  then quit looking
   if (num_matches > 0 && found_definition && !find_all && exact_match) {
      return;
   }

   // look up globals found in this file
   if (search_class_name == '' || search_class_name=='::') {
      _str no_tag_files[]; no_tag_files._makeempty();
      tag_list_context_globals(0, 0, symbol_prefix, true, no_tag_files,
                               pushtag_flags, VS_TAGCONTEXT_ONLY_static,
                               num_matches, max_matches, exact_match, case_sensitive);
      if (!exact_match || num_matches==0 || find_all || !found_definition) {
         tag_list_context_globals(0, 0, symbol_prefix, true, tag_files,
                                  pushtag_flags, VS_TAGCONTEXT_ONLY_non_static,
                                  num_matches, max_matches, exact_match, case_sensitive);
      }
   }

   // look up symbols imported from packages
   i = tag_find_context('', false, false, false, '');
   while (i > 0) {
      tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
      if (type_name :== 'import') {
         tag_get_detail2(VS_TAGDETAIL_context_name, i, import_name);
         tag_get_detail2(VS_TAGDETAIL_context_return, i, aliased_to);
         if (aliased_to != '') {
            import_name = aliased_to;
         }
         //say("_MatchSymbolInContext: importing "import_name);
         boolean found_package = false;
         if (tag_check_for_package(import_name, tag_files, true, case_sensitive)) {
            tag_get_detail(VS_TAGDETAIL_return, aliased_to);
            if (aliased_to != '' && tag_check_for_package(aliased_to, tag_files, true, case_sensitive)) {
               import_name = aliased_to;
            }
            found_package = true;
         }
         if (!found_package && pos('[.]([*]|:v)',import_name,1,'r')) {
            import_name = substr(import_name, 1, pos('.')-1);
            if (tag_check_for_package(import_name, tag_files, true, case_sensitive)) {
               found_package = true;
            }
         }
         if (found_package) {
            found_definition = _ListSymbolsInClass(symbol_prefix, import_name, 
                                                   0, 0, 0, num_matches, max_matches, 
                                                   pushtag_flags, context_flags, 
                                                   exact_match, case_sensitive);
            // found some matches?  then quit looking
            if (num_matches > 0 && found_definition && exact_match) {
               return;
            }
            if (num_matches >= max_matches) {
               break;
            }
         }
      }
      i = tag_next_context('', false, false, false, '');
   }

   // strict search or found some matches?   then quit searching
   if (num_matches > 0 || strict) {
      //say("no matches");
      return;
   }

   // look up any matching local symbol
   i = tag_find_local(symbol_prefix, exact_match, case_sensitive);
   while (i > 0) {
      tag_get_detail2(VS_TAGDETAIL_local_type, i, type_name);
      tag_get_detail2(VS_TAGDETAIL_local_flags, i, tag_flags);
      if (tag_filter_type(0,pushtag_flags, type_name, tag_flags)) {
         ++num_matches;
         tag_insert_match_fast(VS_TAGMATCH_local, i);
         if (num_matches >= max_matches) {
            //say("hit match limit (locals)");
            break;
         }
      }
      i = tag_next_local(symbol_prefix, exact_match, case_sensitive);
   }

   // look up any symbol_prefix found in this file
   i = tag_find_context(symbol_prefix, exact_match, case_sensitive);
   while (i > 0) {
      tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
      tag_get_detail2(VS_TAGDETAIL_context_flags, i, tag_flags);
      if (tag_filter_type(0,pushtag_flags, type_name, tag_flags) &&
          type_name:!='import' && type_name:!='friend' && type_name:!='include') {
         ++num_matches;
         tag_insert_match_fast(VS_TAGMATCH_context, i);
         if (num_matches >= max_matches) {
            //say("hit match limit (locals)");
            break;
         }
      }
      i = tag_next_context(symbol_prefix, exact_match, case_sensitive);
   }

   // still nothing, just look up any matching symbol
   i=0;
   _str tag_filename = next_tag_filea(tag_files, i, false, true);
   while (tag_filename != '') {
      if (exact_match) {
         status=tag_find_equal(symbol_prefix, case_sensitive);
      } else {
         status=tag_find_prefix(symbol_prefix, case_sensitive);
      }
      for (;;) {
         //say("_MatchSymbolInContext 3");
         if (status) {
            break;
         }
         tag_get_detail(VS_TAGDETAIL_type, type_name);
         tag_get_detail(VS_TAGDETAIL_flags, tag_flags);
         if (tag_filter_type(0,pushtag_flags, type_name, tag_flags)) {
            ++num_matches;
            tag_insert_match_fast(VS_TAGMATCH_tag, 0);
            if (num_matches >= max_matches) {
               //say("hit match limit (locals)");
               break;
            }
         }
         if (exact_match) {
            status=tag_next_equal(case_sensitive);
         } else {
            status=tag_next_prefix(symbol_prefix, case_sensitive);
         }
      }
      // next tag file, please
      tag_filename = next_tag_filea(tag_files, i, false, true);
   }

   // still nothing, try to match all, not just tagwin stuff
   if (!num_matches && (pushtag_flags&VS_TAGFILTER_ANYTHING)!=VS_TAGFILTER_ANYTHING) {
      _MatchSymbolInContext(symbol_prefix, search_class_name, num_matches, max_matches,
                            VS_TAGFILTER_ANYTHING, false, allow_locals, 
                            find_parents, find_all, exact_match, case_sensitive);
   }
}

//////////////////////////////////////////////////////////////////////////////
// Attempt to locate the given symbol in the given class by searching
// first locals, then the current file, then tag files, looking strictly
// for the class definition, not just class members.  Recursively
// looks for symbols in inherited classes and resolves items in
// enumerated types to the correct class scope (since enums do not form
// a namespace).  The order of searching parent classes is depth-first,
// preorder (root node searched before children).
//    symbol_prefix -- symbol or symbol prefix to match
//    search_class  -- name of class to look for symbols in
//    treewid       -- window of tree to insert into, 0 implies
//                     insert matches (tag_insert_match)
//    tree_index    -- tree index to insert items under, ignored
//                     if (treewid == 0)
//    depth         -- Recursive call depth, bails out at 32
//    num_matches   -- (reference) number of matches found so far
//    max_matches   -- maximum number of matches to be considered
//    pushtag_flags -- VS_TAGFILTER_* flags, -1 to ignore these
//    context_flags -- VS_TAGCONTEXT_* flags, -1 to ignore these
// Look at num_matches to see if any matches were found.  Generally
// if (num_matches >= max_matches) there may be more matches, but
// the search terminated early.
// The current object must be an editor control or the current buffer.
//
void _ListSymbolsInClass(_str symbol_prefix, _str search_class_name,
                         int treewid, int tree_index, int depth,
                         int &num_matches, int max_matches,
                         int pushtag_flags, int context_flags,
                         boolean exact_match, boolean case_sensitive)
{
   //say("_ListSymbolsInClass("symbol_prefix","search_class_name","case_sensitive")");

   // This function is NOT designed to list globals, bad caller, bad.
   if (search_class_name == '') {
      return;
   }

   // update the current context and locals
   _UpdateContext(true);
   _UpdateLocals(true);

   // try to find symbols in this specific class first
   boolean found_definition=false;
   _str outer_class_name, inner_class_name;
   tag_split_class_name(search_class_name, inner_class_name, outer_class_name);
   _str class_parents = '';
   typeless tag_files = tags_filenamea(p_extension);
   _str tag_filename;
   int i,k,status;

   // for function context, first try to find a local variable
   if (context_flags & VS_TAGCONTEXT_ALLOW_locals) {
      found_definition = tag_list_class_locals(treewid, tree_index, tag_files,
                                               symbol_prefix, search_class_name,
                                               pushtag_flags, context_flags,
                                               num_matches, max_matches,
                                               exact_match, case_sensitive)? true:false;

      // if we found the item in local variables, get inheritance information
      //say("got here, inner="inner_class_name);
      i = tag_find_local(inner_class_name, true, case_sensitive, false, outer_class_name);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_local_type, i, type_name);
         tag_get_detail2(VS_TAGDETAIL_local_name, i, proc_name);
         tag_get_detail2(VS_TAGDETAIL_local_class, i, class_name);
         //say("LCL: proc_name="proc_name" class_name="class_name);
         if (tag_tree_type_is_class(type_name)) {
            found_definition = true;
            tag_get_detail2(VS_TAGDETAIL_local_parents, i, class_parents);
            //say("LCL: parents="class_parents" class="class_name" inner="inner_class_name" outer="outer_class_name);
            break;
         }
         i = tag_next_local(inner_class_name, true, case_sensitive, false, outer_class_name);
      }
   }

   // found members of a local structure, search no further
   if (!(context_flags & VS_TAGCONTEXT_ONLY_inclass) || !found_definition ||
       (context_flags & VS_TAGCONTEXT_FIND_derived)) {
      context_flags &= ~VS_TAGCONTEXT_ALLOW_locals;
      boolean found_context = tag_list_class_context(treewid, tree_index, tag_files,
                                                     symbol_prefix, search_class_name,
                                                     pushtag_flags, context_flags,
                                                     num_matches, max_matches,
                                                     exact_match, case_sensitive)? true:false;

      // if we found the item in current context, get inheritance information
      if (!found_definition) {
         i = tag_find_context(inner_class_name, true, case_sensitive, false, outer_class_name);
         while (i > 0) {
            tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
            //tag_get_detail2(VS_TAGDETAIL_context_class, i, class_name);
            //tag_get_detail2(VS_TAGDETAIL_context_name, i, proc_name);
            //say("CTX: proc_name="proc_name" class_name="class_name);
            if (tag_tree_type_is_class(type_name)) {
               found_definition = true;
               tag_get_detail2(VS_TAGDETAIL_context_parents, i, class_parents);
               //say("CTX: parents="class_parents" class="class_name" inner="inner_class_name" outer="outer_class_name);
               break;
            }
            i = tag_next_context(inner_class_name, true, case_sensitive, false, outer_class_name);
         }
      }
   }

   // found members of a structure within current file, search no further
   if (!(context_flags & VS_TAGCONTEXT_ONLY_inclass) || !found_definition ||
       (context_flags & VS_TAGCONTEXT_FIND_derived)) {
      //say("_ListTagsInClass to be called");
      // look up class members for current class
      int i=0;
      tag_filename = next_tag_filea(tag_files, i, false, true);
      while (tag_filename != '') {
         //say("**exact_match="exact_match" case="case_sensitive" num_matches="num_matches" tag_file="tag_filename);
         //say("**prefix="symbol_prefix" class="search_class_name" pushtag_flags="pushtag_flags" context_flags="context_flags);
         boolean found_tag = tag_list_class_tags(treewid, tree_index, tag_files,
                                                 symbol_prefix, search_class_name,
                                                 pushtag_flags, context_flags,
                                                 num_matches, max_matches,
                                                 exact_match,case_sensitive)? true : false;
         //say("num_matches="num_matches);
         // get the class parents, and stop here
         if (!found_definition) {
            status = tag_get_inheritance(search_class_name, class_parents);
            if (!status && class_parents != '') {
               //say("tag file: class_parents="class_parents);
               found_definition = true;
            }
         }
         tag_filename = next_tag_filea(tag_files, i, false, true);
      }
   }
   //say("num_matches="num_matches);
   if (num_matches > max_matches) {
      return;
   }

   // normalize the list of parents from the database
   _str normalized_parents='';
   _str normalized_types='';
   boolean allow_locals = (context_flags & VS_TAGCONTEXT_ALLOW_locals)? true:false;
   _NormalizeClassTypeList(class_parents, tag_files, 
                           allow_locals, case_sensitive,
                           normalized_parents,normalized_types);

   // demote access level before doing inherited classes
   if (context_flags & VS_TAGCONTEXT_ALLOW_private) {
      context_flags &= ~VS_TAGCONTEXT_ALLOW_private;
      context_flags |= VS_TAGCONTEXT_ALLOW_protected;
   }

   // add each of them to the list also
   if (!(context_flags & VS_TAGCONTEXT_ONLY_this_class)) {
      if (context_flags & VS_TAGCONTEXT_ONLY_parents) {
         context_flags |= VS_TAGCONTEXT_ONLY_this_class;
         context_flags &= ~VS_TAGCONTEXT_ONLY_parents;
      }
      while (normalized_parents != '') {
         // add transitively inherited class members
         parse normalized_parents with p1 ';' normalized_parents;
         //say("p1="p1);
         if (depth < MAX_RECURSIVE_SEARCH) {
            context_flags &= ~VS_TAGCONTEXT_FIND_derived;
            _ListSymbolsInClass(symbol_prefix, p1, treewid, tree_index, depth+1,
                                num_matches, max_matches, pushtag_flags,
                                context_flags, exact_match, case_sensitive);
         }
      }
   }
}

//////////////////////////////////////////////////////////////////////////////
// Returns true if the given function is already found in the set of matches.
// This is used by _ListVirtualMethods() to filter out abstract methods
// that already have been overloaded.
//    proc_name      -- proc name to search for
//    signature      -- signature of proc name, to allow overloading
//    case_sensitive -- case sensitive comparison?
//
static boolean _IsAlreadyInMatchSet(_str proc_name, _str signature,
                                    boolean case_sensitive)
{
   int i, n = tag_get_num_of_matches();
   for (i=1; i<=n; i++) {
      _str match_name;
      _str match_args;
      tag_get_detail2(VS_TAGDETAIL_match_name,i,match_name);
      tag_get_detail2(VS_TAGDETAIL_match_args,i,match_args);
      if ((case_sensitive && match_name :== proc_name) || 
          (!case_sensitive && !stricmp(match_name,proc_name))) {
         if (tag_tree_compare_args(signature, proc_name, true)) {
            return true;
         }
      }
   }
   return false;
}


//////////////////////////////////////////////////////////////////////////////
// For each item in 'class_list', normalize the class and place it in
// 'normal_list', along with the tag type, placed in 'type_list'.
//    class_list   -- list of class names, seperated by semicolons
//    allow_locals -- allow local classes in list
//    normal_list  -- list of normalized class names
//    type_list    -- list of tag types found for normalized class names
// Returns 0 on success, <0 on error.
//
int _NormalizeClassTypeList(_str class_parents, typeless tag_files,
                            boolean allow_locals,
                            boolean case_sensitive,
                            _str &normalized_parents,
                            _str &normalized_types)
{
   //say("_NormalizeClassTypeList: parents="class_parents);
   // normalize the list of parents from the database
   _str case_opts=(case_sensitive)? '':'i';
   normalized_parents='';
   normalized_types='';
   while (class_parents != '') {
      parse class_parents with parent ';' class_parents;
      if (parent != '') {

         _str normalized  = '';
         _str normal_type = '';
         _str inner_parent, outer_parent;
         tag_split_class_name(parent, inner_parent, outer_parent);
         //say("parent="parent" inner="inner_parent" outer="outer_parent);

         // attempt to normalize the class name
         if (allow_locals) {
            i = tag_find_local(inner_parent, true, case_sensitive);
            while (i > 0) {
               tag_get_detail2(VS_TAGDETAIL_local_type, i, type_name);
               tag_get_detail2(VS_TAGDETAIL_local_class, i, class_name);
               if (tag_tree_type_is_class(type_name) &&
                  (outer_parent=='' || pos(outer_parent,class_name,1,case_opts))) {
                  tag_get_detail2(VS_TAGDETAIL_local_name, i, proc_name);
                  normal_type = type_name;
                  normalized  = tag_join_class_name(proc_name, class_name, tag_files, case_sensitive);
                  break;
               }
               i = tag_next_local(inner_parent, true, case_sensitive);
            }
         }
         // try to find class in current context
         if (normalized == '') {
            i = tag_find_context(inner_parent, true, case_sensitive);
            while (i > 0) {
               tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
               tag_get_detail2(VS_TAGDETAIL_context_class, i, class_name);
               if (tag_tree_type_is_class(type_name) &&
                   (outer_parent=='' || pos(outer_parent,class_name,1,case_opts))) {
                  tag_get_detail2(VS_TAGDETAIL_context_name, i, proc_name);
                  normal_type = type_name;
                  normalized  = tag_join_class_name(proc_name, class_name, tag_files, case_sensitive);
                  break;
               }
               i = tag_next_context(inner_parent, true, case_sensitive);
            }
         }
         // try to find in some tag file
         if (normalized == '') {
            // didn't find it in our tag file, search others
            i=0;
            tag_filename = next_tag_filea(tag_files,i,false,true);
            while (tag_filename != '') {
               status = tag_find_class(normalized, parent, 1, case_sensitive);
               if (!status) {
                  tag_split_class_name(normalized, inner_parent, outer_parent);
                  //say("tag_find_tag("inner_parent","outer_parent")");
                  status = tag_find_tag(inner_parent,'interface',outer_parent);
                  normal_type = (!status)? 'interface':'class';
                  break;
               }
               tag_filename = next_tag_filea(tag_files,i,false,true);
            }
         }
         // didn't find it, bummer
         if (normalized == '') {
            normalized = parent;
         }
         //say("PARENT="parent" NORMAL="normalized);
         // append to the new parent list
         if (normalized_parents == '') {
            normalized_parents = normalized;
            normalized_types   = normal_type;
         } else {
            normalized_parents = normalized_parents ';' normalized;
            normalized_types   = normalized_types ';' normal_type;
         }
      }
   }

   // that's all folks
   return 0;
}

//////////////////////////////////////////////////////////////////////////////
// Compress the given signature to a very short string
//
static _str compress_signature(_str signature)
{
   if (signature=='void') {
      return '';
   }
   _str new_signature = '';
   while (signature != '') {
      int p = pos('(:v|[;,*&^]|\[|\])', signature, 1, 'r');
      if (!p) {
         break;
      }
      _str ch = substr(signature, p, pos(''));
      signature = substr(signature, p+pos('')+1);
      switch (ch) {
      case ';':
      case ',':
      case '*':
      case '&':
      case '^':
      case '[':
      case ']':
      case 'int':
      case 'integer':
      case 'real':
      case 'double':
      case 'float':
      case 'char':
      case 'bool':
      case 'boolean':
      case 'const':
      case 'void':
         strappend(new_signature,ch);
      default:
      }
   }
   return new_signature;
}
//////////////////////////////////////////////////////////////////////////////
// Recursive helper function for _ListVirtualMethods (documented below)
//
static void _ListVirtualMethodsR(_str search_class_name, _str search_type,
                                 _str &abstract_indexes, boolean only_abstract,
                                 int treewid, int tree_index,
                                 typeless tag_files, int depth,
                                 int &num_matches, int max_matches,
                                 boolean allow_locals, boolean case_sensitive,
                                 _str (&found_concrete):[])
{
   //say("_ListVirtualMethodsR("search_class_name")");

   // try to find symbols in this specific class first
   boolean found_definition=false;
   _str outer_class_name, inner_class_name;
   tag_split_class_name(search_class_name, inner_class_name, outer_class_name);
   _str class_parents = '';
   _str tag_filename;
   int i,k,status;
   int flag_mask=VS_TAGFLAG_const|VS_TAGFLAG_volatile|VS_TAGFLAG_mutable;

   // are all methods abstract (is this an interface?)
   boolean all_abstract = false;
   if (search_type=='interface' && depth==0) {
      all_abstract = true;
   }

   // for function context, first try to find a local variable
   if (allow_locals) {
      // find abstract functions
      i = tag_find_local('', false, case_sensitive, false, search_class_name);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_local_type, i, type_name);
         tag_get_detail2(VS_TAGDETAIL_local_flags, i, tag_flags);
         if (tag_tree_type_is_func(type_name) && (tag_flags & VS_TAGFLAG_inclass) &&
             (tag_flags & VS_TAGFLAG_access) != VS_TAGFLAG_private &&
             !(tag_flags & VS_TAGFLAG_final) && !(tag_flags & VS_TAGFLAG_destructor)) {
            tag_get_detail2(VS_TAGDETAIL_local_name, i, tag_name);
            tag_get_detail2(VS_TAGDETAIL_local_args, i, signature);
            if (tag_flags&flag_mask) strappend(tag_name,'/'(tag_flags&flag_mask));
            if (signature != '') strappend(tag_name,'/'compress_signature(signature));
            if (!found_concrete._indexin(tag_name)) {
               if (all_abstract || (tag_flags & VS_TAGFLAG_abstract) ||
                   (!only_abstract && (tag_flags & VS_TAGFLAG_virtual))) {
                  if (treewid) {
                     k=tag_tree_insert_fast(treewid,tree_index,VS_TAGMATCH_local,i,0,1,0,0,0);
                  } else {
                     k=tag_insert_match_fast(VS_TAGMATCH_local,i);
                  }
                  if (all_abstract || (tag_flags & VS_TAGFLAG_abstract)) {
                     strappend(abstract_indexes,k' ');
                  }
                  found_concrete:[tag_name] = type_name;
                  if (k < 0 || ++num_matches >= max_matches) {
                     break;
                  }
               } else {
                  found_concrete:[tag_name] = type_name;
               }
            }
         }
         i = tag_next_local('', false, case_sensitive, false, search_class_name);
      }

      // if we found the item in local variables, get inheritance information
      i = tag_find_local(inner_class_name, true, case_sensitive, false, outer_class_name);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_local_type, i, type_name);
         if (tag_tree_type_is_class(type_name)) {
            found_definition = true;
            tag_get_detail2(VS_TAGDETAIL_local_parents, i, class_parents);
            break;
         }
         i = tag_next_local(inner_class_name, true, case_sensitive, false, outer_class_name);
      }
   }

   // found members of a local structure, search no further
   if (!found_definition) {

      // find abstract functions
      i = tag_find_context('', false, case_sensitive, false, search_class_name);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
         tag_get_detail2(VS_TAGDETAIL_context_flags, i, tag_flags);
         if (tag_tree_type_is_func(type_name) && (tag_flags & VS_TAGFLAG_inclass) &&
             (tag_flags & VS_TAGFLAG_access) != VS_TAGFLAG_private &&
             !(tag_flags & VS_TAGFLAG_final) && !(tag_flags & VS_TAGFLAG_destructor)) {
            tag_get_detail2(VS_TAGDETAIL_context_name, i, tag_name);
            tag_get_detail2(VS_TAGDETAIL_context_args, i, signature);
            if (tag_flags&flag_mask) strappend(tag_name,'/'(tag_flags&flag_mask));
            if (signature != '') strappend(tag_name,'/'compress_signature(signature));
            if (!found_concrete._indexin(tag_name)) {
               if (all_abstract || (tag_flags & VS_TAGFLAG_abstract) ||
                   (!only_abstract && (tag_flags & VS_TAGFLAG_virtual))) {
                  if (treewid) {
                     k=tag_tree_insert_fast(treewid,tree_index,VS_TAGMATCH_context,i,0,1,0,0,0);
                  } else {
                     k=tag_insert_match_fast(VS_TAGMATCH_context,i);
                  }
                  if (all_abstract || (tag_flags & VS_TAGFLAG_abstract)) {
                     strappend(abstract_indexes,k' ');
                  }
                  found_concrete:[tag_name] = type_name;
                  if (k < 0 || ++num_matches >= max_matches) {
                     break;
                  }
               } else {
                  found_concrete:[tag_name] = type_name;
               }
            }
         }
         i = tag_next_context('', false, case_sensitive, false, search_class_name);
      }

      // if we found the item in current context, get inheritance information
      i = tag_find_context(inner_class_name, true, case_sensitive, false, outer_class_name);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
         if (tag_tree_type_is_class(type_name)) {
            found_definition = true;
            tag_get_detail2(VS_TAGDETAIL_context_parents, i, class_parents);
            break;
         }
         i = tag_next_context(inner_class_name, true, case_sensitive, false, outer_class_name);
      }
   }

   // found members of a structure within current file, search no further
   if (!found_definition) {
      // look up class members for current class
      int i=0;
      tag_filename = next_tag_filea(tag_files, i, false, true);
      while (tag_filename != '') {
         status = tag_find_in_class(search_class_name);
         while (!status) {
            tag_get_detail(VS_TAGDETAIL_type, type_name);
            tag_get_detail(VS_TAGDETAIL_flags, tag_flags);
            //say("CANDIDATE: "tag_name" flags="(tag_flags & VS_TAGFLAG_abstract));
            if (tag_tree_type_is_func(type_name) && (tag_flags & VS_TAGFLAG_inclass) &&
                (tag_flags & VS_TAGFLAG_access) != VS_TAGFLAG_private &&
                !(tag_flags & VS_TAGFLAG_final) && !(tag_flags & VS_TAGFLAG_destructor)) {
               tag_get_detail(VS_TAGDETAIL_name, tag_name);
               tag_get_detail(VS_TAGDETAIL_arguments, signature);
               if (tag_flags&flag_mask) strappend(tag_name,'/'(tag_flags&flag_mask));
               if (signature != '') strappend(tag_name,'/'compress_signature(signature));
               if (!found_concrete._indexin(tag_name)) {
                  if (all_abstract || (tag_flags & VS_TAGFLAG_abstract) ||
                      (!only_abstract && (tag_flags & VS_TAGFLAG_virtual))) {
                     //say("  MATCH");
                     if (treewid) {
                        k=tag_tree_insert_fast(treewid,tree_index,VS_TAGMATCH_tag,0,0,1,0,0,0);
                     } else {
                        k=tag_insert_match_fast(VS_TAGMATCH_tag,0);
                     }
                     if (all_abstract || (tag_flags & VS_TAGFLAG_abstract)) {
                        strappend(abstract_indexes,k' ');
                     }
                     found_concrete:[tag_name] = type_name;
                     if (k < 0 || ++num_matches >= max_matches) {
                        break;
                     }
                  } else {
                     found_concrete:[tag_name] = type_name;
                  }
               }
            }
            status = tag_next_in_class();
         }

         // get the class parents, and stop here
         if (!found_definition) {
            status = tag_get_inheritance(search_class_name, class_parents);
            if (!status && class_parents != '') {
               found_definition = true;
            }
         }
         tag_filename = next_tag_filea(tag_files, i, false, true);
      }
   }
   //say("num_matches="num_matches);

   // normalize the list of parents from the database
   _str normalized_parents='';
   _str normalized_types='';
   _NormalizeClassTypeList(class_parents, tag_files, 
                           allow_locals, case_sensitive,
                           normalized_parents, normalized_types);

   // demote access level before doing inherited classes
   //if (context_flags & VS_TAGCONTEXT_ALLOW_private) {
   //   context_flags &= ~VS_TAGCONTEXT_ALLOW_private;
   //   context_flags |= VS_TAGCONTEXT_ALLOW_protected;
   //}

   // add each of them to the list also
   while (normalized_parents != '') {
      // add transitively inherited class members
      parse normalized_parents with p1 ';' normalized_parents;
      parse normalized_types   with n1 ';' normalized_types;
      //say("p1="p1" n1="n1);
      if (depth < MAX_RECURSIVE_SEARCH) {
         _ListVirtualMethodsR(p1, n1, abstract_indexes, 
                              only_abstract, treewid, tree_index,
                              tag_files, depth+1, num_matches, max_matches,
                              allow_locals, case_sensitive, found_concrete);
      }
   }
}
//////////////////////////////////////////////////////////////////////////////
// List the abstract functions in the given class, struct, or interface.
// by searching first locals, then the current file, then tag files,
// looking strictly for the class definition, not just class members.
// Recursively looks for symbols in inherited classes.  The order of
// searching parent classes is depth-first, preorder (root node searched
// before children).
//    search_class   -- name of class to look for symbols in
//    search_type    -- class/interface/struct
//    only_abstract  -- list only abstract methods or unimplemented functions
//    treewid        -- window of tree to insert into, 0 implies
//                      insert matches (tag_insert_match)
//    tree_index     -- tree index to insert items under, ignored
//                      if (treewid == 0)
//    num_matches    -- (reference) number of matches found so far
//    max_matches    -- maximum number of matches to be considered
//    allow_locals   -- allow local structs?
//    case_sensitive -- case sensitive tag searching?
//    depth         -- Recursive call depth, bails out at 32
// Look at num_matches to see if any matches were found.  Generally
// if (num_matches >= max_matches) there may be more matches, but
// the search terminated early.
// The current object must be an editor control or the current buffer.
//
void _ListVirtualMethods(_str search_class_name, _str search_type,
                         _str &abstract_indexes, boolean only_abstract,
                         int treewid, int tree_index,
                         int &num_matches, int max_matches,
                         boolean allow_locals, boolean case_sensitive)
{
   //say("_ListVirtualMethods("search_class_name")");

   // This function is NOT designed to list globals, bad caller, bad.
   if (search_class_name == '') {
      return;
   }

   // update the current context and locals
   _UpdateContext(true);
   _UpdateLocals(true);

   // try to find symbols in this specific class first
   typeless tag_files = tags_filenamea(p_extension);
   _str found_concrete:[];found_concrete._makeempty();
   _ListVirtualMethodsR(search_class_name, search_type, abstract_indexes, only_abstract,
                        treewid, tree_index, tag_files, 0, num_matches, max_matches,
                        allow_locals, case_sensitive, found_concrete);
}

//////////////////////////////////////////////////////////////////////////////
// List the symbols visible in the different contexts, including
//   -- Locals
//   -- Class Members
//   -- Module Variables (static)
// The current object must be an editor control or current buffer.
//
_str _MatchThisOrSelf()
{
   //say("_ListThisOrSelf()");
   // update the current context
   _UpdateContext(true);
   int context_id = tag_current_context();
   if (context_id <= 0) {
      return '';
   }

   // not a function or proc, not a static method, in a class method
   tag_get_detail2(VS_TAGDETAIL_context_class, context_id, cur_class_name);
   tag_get_detail2(VS_TAGDETAIL_context_type,  context_id, cur_type_name);
   tag_get_detail2(VS_TAGDETAIL_context_flags, context_id, cur_tag_flags);
   if (cur_class_name == '' || 
       cur_tag_flags & VS_TAGFLAG_static || 
       !tag_tree_type_is_func(cur_type_name) || cur_type_name :== 'proto') {
      return '';
   }

   // not really a class method, just a member of a package
   typeless tag_files = tags_filenamea(p_extension);
   tag_split_class_name(cur_class_name, inner_name, outer_name);
   if (tag_check_for_package(inner_name, tag_files, true, true)) {
      return '';
   }

   // add the item to the tree
   return cur_class_name;
}

//////////////////////////////////////////////////////////////////////////////
// List the symbols visible in the different contexts, including
//   -- Class Members
// The current object must be an editor control or current buffer.
//
void _ListMembersInContext(int treewid, int tree_index,
                           boolean data_only, boolean funcs_only,
                           boolean this_class_only, boolean constructors_only,
                           boolean exact_match, boolean case_sensitive,
                           int &num_matches, int max_matches=MAX_SYMBOL_MATCHES)
{
   //say("_ListMembersInContext()");
   // update the current context
   _UpdateContext(true);
   _UpdateLocals(true);
   //say("_ListMembersInContext");
   int context_id = tag_current_context();

   // gulp up the current context information
   int context_flags = VS_TAGCONTEXT_ONLY_inclass|VS_TAGCONTEXT_ALLOW_private|VS_TAGCONTEXT_ALLOW_protected|VS_TAGCONTEXT_ALLOW_package;
   if (data_only) {
      context_flags |= VS_TAGCONTEXT_ONLY_data;
   }
   if (funcs_only) {
      context_flags |= VS_TAGCONTEXT_ONLY_funcs;
   }
   if (constructors_only) {
      context_flags |= VS_TAGCONTEXT_ONLY_constructors;
      context_flags |= VS_TAGCONTEXT_ONLY_parents; /* hard-coded, kind of */
   }
   if (this_class_only) {
      context_flags |= VS_TAGCONTEXT_ONLY_this_class;
   }
   boolean const_only    = false;
   boolean volatile_only = false;
   boolean static_only   = false;
   _str cur_class_name = '';
   _str cur_proc_name  = '';
   _str cur_type_name  = '';
   if (context_id > 0) {
      tag_get_context(context_id, 
                      cur_proc_name, cur_type_name, cur_file_name, 
                      cur_start_line_no, cur_start_seekpos, 
                      cur_scope_line_no, cur_scope_seekpos,
                      cur_end_line_no, cur_end_seekpos,
                      cur_class_name, cur_tag_flags, 
                      cur_signature, cur_return_type);
      if (tag_tree_type_is_func(cur_type_name)) {
         if (cur_tag_flags & VS_TAGFLAG_volatile) {
            context_flags |= VS_TAGCONTEXT_ONLY_volatile;
         }
         if (cur_tag_flags & VS_TAGFLAG_const) {
            context_flags |= VS_TAGCONTEXT_ONLY_const;
         }
         if (cur_class_name != '' && (cur_tag_flags & VS_TAGFLAG_static)) {
            context_flags |= VS_TAGCONTEXT_ONLY_static;
         }
         if (cur_class_name != '') {
            tag_split_class_name(cur_class_name,inner_name,outer_name);
            _str qualified_name=_QualifySymbolName(inner_name,outer_name,p_buf_name,true);
            if (qualified_name!='' && qualified_name!=inner_name) {
               cur_class_name=qualified_name;
            }
         }
      }
   }

   typeless tag_files = tags_filenamea(p_extension);
   _str effective_class_name = cur_class_name;
   _str low_effective_name = '';
   // if the current context is a class, then the effective
   // current class is it, not cur_class_name
   if (context_id > 0 && tag_tree_type_is_class(cur_type_name)) {
      effective_class_name = tag_join_class_name(cur_proc_name, cur_class_name, tag_files, case_sensitive);
   }
   if (context_id > 0 && tag_tree_type_is_package(cur_type_name)) {
      effective_class_name = cur_proc_name;
   }

   // blow away this category if there is no current class
   if (treewid > 0 && effective_class_name == '') {
      treewid._TreeDelete(tree_index);
      return;
   }

   // look up class members for each level of scope
   while (effective_class_name != '') {

      _ListSymbolsInClass('',effective_class_name, treewid, tree_index,
                          0, num_matches, max_matches, VS_TAGFILTER_ANYTHING,
                          context_flags, exact_match, case_sensitive);

      // strip last part of class name off
      tag_split_class_name(effective_class_name, junk, effective_class_name);
   }
}

//////////////////////////////////////////////////////////////////////////////
// List the parameters of the current symbol if it is a #define
// The current object must be an editor control or current buffer.
//
void _ListParametersOfDefine(int treewid, int tree_index,
                             int &num_matches,
                             int max_matches=MAX_SYMBOL_MATCHES)
{
   //say("_ListParametersOfDefine()");
   // update the current context
   _UpdateContext(true);
   int context_id = tag_current_context();
   if (context_id <= 0) {
      treewid._TreeDelete(tree_index);
      return;
   }

   // is this a #define?
   tag_get_detail2(VS_TAGDETAIL_context_type, context_id, type_name);
   if (type_name :!= 'define') {
      treewid._TreeDelete(tree_index);
      return;
   }
      
   // get the #define signature
   tag_get_detail2(VS_TAGDETAIL_context_args, context_id, signature);

   // insert each argument
   while (signature != '') {
      parse signature with a ',' signature;
      a = strip(a);
      if (a != '') {
         if (treewid) {
            k=tag_tree_insert_tag(treewid, tree_index, 0, 1, 0, a, 'param', p_buf_name, p_line, '', 0, '');
         } else {
            k=tag_insert_match('', a, 'param', p_buf_name, p_line, '', 0, '');
         }
         if (k < 0 || ++num_matches >= max_matches) {
            break;
         }
      }
   }
}

// List the parameters of the given class symbol
// NOTE: this method is very C++ specific, whatcha gonna do...
//
static void _ListParametersOfClass(int treewid, int tree_index,
                                   _str type_name, int tag_flags, _str signature,
                                   int &num_matches, int max_matches)
{
   // is this a template class?
   if (type_name :== 'class' && (tag_flags & VS_TAGFLAG_template)) {
      // insert each argument
      while (signature != '') {
         _str a;
         parse signature with a ',' signature;
         a = strip(a);
         if (a != '') {
            _str type_name = 'var';
            if (pos('class', a)==1) {
               type_name = 'class';
               parse a with 'class' a;
            } else if (pos('typename', a)==1) {
               type_name = 'typedef';
               parse a with 'typename' a;
            } else {
            }
            parse a with a '=' .;
            a = strip(a);
            if (pos('{:v}:b@$',a,1,'r')) {
               a = substr(a, pos('S0'), pos('0'));
            }

            int k;
            if (treewid) {
               k=tag_tree_insert_tag(treewid, tree_index, 0, 1, 0, a, type_name, p_buf_name, p_line, '', 0, '');
            } else {
               k=tag_insert_match('', a, type_name, p_buf_name, p_line, '', 0, '');
            }
            if (k < 0 || ++num_matches >= max_matches) {
               break;
            }
         }
      }
   }
}

//////////////////////////////////////////////////////////////////////////////
// List the parameters of the current symbol if it is a #define
// The current object must be an editor control or current buffer.
//
void _ListParametersOfTemplate(int treewid, int tree_index,
                               boolean case_sensitive, int &num_matches,
                               int max_matches=MAX_SYMBOL_MATCHES)
{
   //say("_ListParametersOfTemplate()");
   // update the current context
   _UpdateContext(true);
   int context_id = tag_current_context();
   if (context_id <= 0) {
      treewid._TreeDelete(tree_index);
      return;
   }

   // is this a template class?
   tag_get_detail2(VS_TAGDETAIL_context_type,  context_id, type_name);
   tag_get_detail2(VS_TAGDETAIL_context_flags, context_id, tag_flags);
   tag_get_detail2(VS_TAGDETAIL_context_args,  context_id, signature);
   tag_get_detail2(VS_TAGDETAIL_context_class, context_id, cur_class_name);
   _ListParametersOfClass(treewid, tree_index, 
                          type_name, tag_flags, signature, 
                          num_matches, max_matches);

   // for each level of class nesting
   typeless tag_files = tags_filenamea(p_extension);
   while (cur_class_name != '') {
      //_str symbol_name = _GetClassNameOnly(cur_class_name);
      //cur_class_name = _GetOuterClassName(cur_class_name);
      tag_split_class_name(cur_class_name, symbol_name, cur_class_name);

      // Look for template parameters within the current context
      int context_id = tag_find_context(symbol_name, true, case_sensitive, false, cur_class_name);
      while (context_id > 0) {
         tag_get_detail2(VS_TAGDETAIL_context_type,  context_id, type_name);
         tag_get_detail2(VS_TAGDETAIL_context_flags, context_id, tag_flags);
         tag_get_detail2(VS_TAGDETAIL_context_args,  context_id, signature);
         //tag_get_detail2(VS_TAGDETAIL_context_class, context_id, class_name);
         //if (class_name :== cur_class_name || 
         //   (!case_sensitive && lowcase(class_name) :== lowcase(cur_class_name))) {
            // got a match, list the parameters
            _ListParametersOfClass(treewid, tree_index, 
                                   type_name, tag_flags, signature, 
                                   num_matches, max_matches);
         //}
         context_id = tag_next_context(symbol_name, true, case_sensitive, false, cur_class_name);
      }

      if (num_matches == 0) {
         int i=0;
         _str tag_filename = next_tag_filea(tag_files, i, false, true);
         while (tag_filename != '') {
            int status = tag_find_equal(symbol_name, case_sensitive, cur_class_name);
            while (!status) {
               tag_get_detail(VS_TAGDETAIL_type, type_name);
               tag_get_detail(VS_TAGDETAIL_flags, tag_flags);
               tag_get_detail(VS_TAGDETAIL_arguments, signature);

               _ListParametersOfClass(treewid, tree_index, 
                                      type_name, tag_flags, signature, 
                                      num_matches, max_matches);
               status = tag_next_equal(case_sensitive, cur_class_name);
            }
            tag_filename = next_tag_filea(tag_files, i, false, true);
         }
      }
   }

   // no template parameters found?
   if (num_matches == 0) {
      treewid._TreeDelete(tree_index);
   }
}

// Find the given tag in the given tag database and populate the 'cm'
// datastructure for the class browser.
//
boolean _GetContextTagInfo(struct VS_TAG_BROWSE_INFO &cm,
                           _str match_tag_database, _str tag_name,
                           _str file_name, int line_no)
{
   //say("_GetContextTagInfo("tag_name")");
   cm.tag_database   = match_tag_database;
   cm.category       = '';
   cm.qualified_name = '';
   if (match_tag_database != '') {
      // find in the given tag database
      status = tag_read_db(match_tag_database);
      if (status == 0) {
         status = tag_find_closest(tag_name, file_name, line_no, 1);
         if (status == 0) {
            tag_get_info(cm.member_name, cm.type_name, cm.file_name, cm.line_no, cm.class_name, cm.flags);
            tag_get_detail(VS_TAGDETAIL_return, cm.return_type);
            tag_get_detail(VS_TAGDETAIL_arguments, cm.arguments);
            return true;
         }
      }
   } else {
      // maybe it was a local variable?
      int i = tag_find_local(tag_name, true, true);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_local_line, i, context_line);
         if (context_line == line_no) {
            tag_get_local(i, cm.member_name, cm.type_name, cm.file_name, cm.line_no, cm.class_name, cm.flags, cm.arguments, cm.return_type);
            if (cm.file_name :== file_name) {
               return true;
            }
         }
         i = tag_next_local(tag_name, true, true);
      }
      // maybe it was from the current file?
      i = tag_find_context(tag_name, true, true);
      while (i > 0) {
         tag_get_detail2(VS_TAGDETAIL_context_line, i, context_line);
         if (context_line == line_no) {
            tag_get_context(i, cm.member_name, cm.type_name, cm.file_name, cm.line_no, d1, d2, d3, d4, d5, cm.class_name, cm.flags, cm.arguments, cm.return_type);
            if (cm.file_name :== file_name) {
               return true;
            }
         }
         i = tag_next_context(tag_name, true, true);
      }
   }
   // did not find a match, really quite depressing, use what we know
   cm.member_name = tag_name;
   cm.type_name   = '';
   cm.file_name   = file_name;
   cm.line_no     = line_no;
   cm.class_name  = '';
   cm.flags       = 0;
   cm.arguments   = '';
   cm.return_type = '';
   return false;
}


//////////////////////////////////////////////////////////////////////////////
// List the global symbols of the given type
// The current object must be an editor control or current buffer.
//
_str _QualifySymbolName(_str search_name, _str context_class, _str context_file, boolean case_sensitive) 
{
   //say("_QualifySymbolName("search_name","context_class","context_file","case_sensitive")");
   // clear out the match array
   tag_clear_matches();

   // context to find 'search_name' in is current buffer?
   typeless tag_files = tags_filenamea(p_extension);
   _str orig_context_class = context_class;
   if (file_eq(p_buf_name, context_file)) {
      // gulp up the current context information
      _UpdateContext(true);
      int context_id = tag_current_context();
      if (context_id > 0) {
         tag_get_detail2(VS_TAGDETAIL_context_type, context_id, cur_type_name);
         tag_get_detail2(VS_TAGDETAIL_context_name, context_id, cur_proc_name);
         tag_get_detail2(VS_TAGDETAIL_context_class, context_id, cur_class_name);

         if (tag_tree_type_is_func(cur_type_name)) {
            // update the current list of locals
            //say("QualifySymbolName1.1: _UpdateLocals");
            _UpdateLocals(true);
            // for function context, first try to find a local variable
            int i = tag_find_local(search_name, 1, case_sensitive, false, ''/*context_class*/);
            while (i > 0) {
               tag_get_detail2(VS_TAGDETAIL_local_class, i, class_name);
               //tag_get_detail2(VS_TAGDETAIL_local_name, i, proc_name);
               //say("_QualifySymbolName1.2: LOCAL proc_name="proc_name" class_name="class_name" context_class="context_class);
               //if (class_name :== context_class || 
               //    (!case_sensitive && !stricmp(class_name,context_class))) {
               _str candidate = tag_join_class_name(search_name,class_name,tag_files,case_sensitive);
               //say("_QualifySymbolName1.3: CANDIDATE="candidate);
               if (candidate :!= '') {
                  return candidate;
               }
               //}
               i = tag_next_local(search_name, 1, case_sensitive, false, context_class);
            }
         }
         if (context_class == '') {
            //say("_QualifySymbolName2: context_class=''");
            if (tag_tree_type_is_class(cur_class_name)) {
               //say("_QualifySymbolName3: _JoinClassWithOuter");
               //context_class = _JoinClassWithOuter(cur_proc_name, cur_class_name,case_sensitive);
               context_class = tag_join_class_name(cur_proc_name, cur_class_name, tag_files, case_sensitive);
            } else {
               //say("_QualifySymbolName4: using "cur_class_name);
               context_class = cur_class_name;
            }
         }
      }

      // try to find items matching 'search_name' in current file
      i = tag_find_context(search_name, 1, case_sensitive);
      while (i > 0) {
         //say("_QUalifySYmbolName5: found a context match");
         tag_insert_match_fast(VS_TAGMATCH_context, i);
         i = tag_next_context(search_name, 1, case_sensitive);
      }
   }

   // try to find items matching 'search_name' in tag files
   i=0;
   _str tag_filename=next_tag_filea(tag_files, i, false, true);
   while (tag_filename != '') {
      status=tag_find_equal(search_name, case_sensitive);
      while (!status) {
         //say("_QualifySymbolName1.6: found global");
         tag_insert_match_fast(VS_TAGMATCH_tag, 0);
         status=tag_next_equal(case_sensitive);
      }
      tag_filename=next_tag_filea(tag_files, i, false, true);
   }

   // we have already searched for 'search_name' as a local
   // next step is to try to find it in the current class or outer class
   _str effective_class_name = context_class;
   while (effective_class_name != '') {

      // look up class members found in this file
      int num_matches = tag_get_num_of_matches();
      for (i=1; i<=num_matches; ++i) {
         tag_get_match(i, tag_file, proc_name, type_name, file_name, line_no, 
                          class_name, tag_flags, signature, return_type);
         //say("_QualifySymbolName7: proc_name="proc_name" class="class_name" effect="effective_class_name" type="type_name);
         if ((case_sensitive && class_name :== effective_class_name) || 
             (!case_sensitive && !stricmp(class_name, effective_class_name)) || 
             (((tag_flags & VS_TAGFLAG_access) != VS_TAGFLAG_private) && 
              tag_is_parent_class(class_name, effective_class_name,
                                  tag_files, case_sensitive, true))) {

            _str candidate = tag_join_class_name(search_name, effective_class_name, tag_files, case_sensitive);
            //say("_QualifySymbolName8: CANDIDATE="candidate);
            if (candidate != '') {
               return candidate;
            }
         }
      }

      // strip last part of class name off
      //say("_QualifySymbolName9: effect="effective_class_name);
      tag_split_class_name(effective_class_name, junk, effective_class_name);
   }

   // look up the item in the global scope
   num_matches = tag_get_num_of_matches();
   for (i=1; i<=num_matches; ++i) {
      tag_get_detail2(VS_TAGDETAIL_match_class, i, class_name);
      //tag_get_detail2(VS_TAGDETAIL_match_name, i, proc_name);
      //say("_QualifySymbolNameE: found match="proc_name" class="class_name);
      if (class_name :== '') {
         tag_get_detail2(VS_TAGDETAIL_match_name, i, proc_name);
         return proc_name;
      }
   }

   // look up imports in the context file
   //tag_clear_matches();
   if (p_extension:=='java') {
      // special case for Java, 'java.lang' is imported implicitely
      tag_insert_match('','java.lang.*','import',context_file,0,'',0,'');
   }
   if (p_extension:=='pas' || p_extension:=='mod' || p_extension:=='m3') {
      // special case for Delphi Pascal and Modula, import "System" module
      tag_insert_match('','System','import',context_file,0,'',0,'');
   }
   _str context_ext = _file_case(get_extension(context_file));
   if (file_eq(p_buf_name, context_file)) {
      // look up import statements found in this file
      int num_context = tag_get_num_of_context();
      for (i=1; i<=num_context ;++i) {
         tag_get_detail2(VS_TAGDETAIL_context_type, i, type_name);
         if (type_name :== 'import' || tag_tree_type_is_package(type_name)) {
            tag_insert_match_fast(VS_TAGMATCH_context, i);
         }
      }
   } else if (context_ext :!= 'jar' && context_ext :!= 'zip') {
      // look up import statements found in given file, from tag files
      i=0;
      tag_filename = next_tag_filea(tag_files, i, false, true);
      while (tag_filename != '') {
         status=tag_find_in_file(context_file);
         while (!status) {
            tag_get_detail(VS_TAGDETAIL_type, type_name);
            if (type_name :== 'import' || type_name :== 'package') {
               tag_insert_match_fast(VS_TAGMATCH_tag, 0);
            }
            status=tag_next_in_file();
         }
         tag_filename = next_tag_filea(tag_files, i, false, true);
      }
   }
   // Go through our list of matches, and see if any work
   _str candidate = '';
   num_matches = tag_get_num_of_matches();
   for (i=1; i<=num_matches ;++i) {
      tag_get_detail2(VS_TAGDETAIL_match_type, i, type_name);
      tag_get_detail2(VS_TAGDETAIL_match_name, i, proc_name);
      //say("_QualifySymbolName: type="type_name);
      if (type_name :== 'import') {
         if (p_extension:=='java') {
            _str case_opt = (case_sensitive)? 'i':'';
            int start_pos = pos("[.:]([*]|"search_name")$", proc_name, 1, 'r'case_opt);
            if (start_pos) {
               proc_name = substr(proc_name, 1, pos('S')-1);
            }
         }
         //say("_QualifySymbolName: import="proc_name);
         candidate = tag_join_class_name(search_name, proc_name, tag_files, case_sensitive);
      } else if (tag_tree_type_is_package(type_name)) {
         //say("_QualifySymbolName: package="proc_name);
         candidate = tag_join_class_name(search_name, proc_name, tag_files, case_sensitive);
      } else if (tag_tree_type_is_class(type_name)) {
         tag_get_detail2(VS_TAGDETAIL_match_class, i, class_name);
         //say("_QualifySymbolName: class="class_name" name="proc_name);
         candidate = tag_join_class_name(proc_name, class_name, tag_files, case_sensitive);
      } else {
         candidate = '';
      }
      //say("_QualifySymbolNameA: candidate="candidate);
      if (candidate != '') {
         //say("_QualifiedSymbolNameA.1: returning "candidate);
         return candidate;
      }
   }

   //say("_QualifySymbolName: ");
   return '';
}

// this function removes the duplicate functions from the
// current match set.  The technique used is to first compute
// the set partition across the set of matches using the match
// tag type name and signature / argument comparison as an
// equivelence relation.  Then for each partition, we select the
// first item or the best item in the set.
//
// The result is returned through the reference parameter
// 'unique_indexes' is an array of integers represent match
// indexes of the unique selected tags.
//
void removeDuplicateFunctions(int (&unique_indexes)[])
{
   int num_matches = tag_get_num_of_matches();
   int num_left = num_matches;
   int num_sets = 0;
   int i, first_item = 1;
   boolean all_indexes[]; all_indexes._makeempty();
   _str equiv_indexes[]; equiv_indexes._makeempty();

   // initialize the set of match indexes
   for (i=1; i<=num_matches; i++) {
      all_indexes[i] = true;
   }

   // until the set of indexes is empty
   while (num_left > 0) {
      // find first remaining item
      for (i=first_item; i<=num_matches; i++) {
         if (all_indexes[i]) {
            all_indexes[i] = false;
            equiv_indexes[num_sets] = first_item = i;
            num_left--;
            break;
         }
      }

      // get all the information about the current item
      tag_get_detail2(VS_TAGDETAIL_match_type,first_item,cur_type_name);
      tag_get_detail2(VS_TAGDETAIL_match_args,first_item,cur_signature);
      tag_get_detail2(VS_TAGDETAIL_match_name,first_item,cur_proc_name);
      int cur_is_func = tag_tree_type_is_func(cur_type_name);
      //say("removeDuplicateFunctions: proc_name="cur_proc_name" sig="cur_signature" type="cur_type_name);

      // find items in its equivelance class
      for (i=first_item+1; i<=num_matches; i++) {
         if (all_indexes[i]) {
            // check if types match adequately
            tag_get_detail2(VS_TAGDETAIL_match_type,i,i_type_name);
            if (i_type_name == cur_type_name ||
                tag_tree_type_is_func(i_type_name) == cur_is_func) {
               // OK, now check signatures
               tag_get_detail2(VS_TAGDETAIL_match_args,i,i_signature);
               tag_get_detail2(VS_TAGDETAIL_match_name,i,i_proc_name);
               //say("removeDuplicateFunctions: proc_name[i]="i_proc_name" sig="i_signature" type="i_type_name);
               if (!tag_tree_compare_args(i_signature, cur_signature, false)) {
                  all_indexes[i] = false;
                  strappend(equiv_indexes[num_sets], ' 'i);
                  num_left--;
               }
            }
         }
      }

      // increment the number of sets processed
      num_sets++;
   }

   // remove all but one item from each equivelence class
   for (i=0; i<num_sets; i++) {
      int best_match = 0;
      int best_score = 0;
      while (equiv_indexes[i] != '') {
         parse equiv_indexes[i] with k equiv_indexes[i];
         // calculate weighted score for this match
         int i_score = 1;
         tag_get_detail2(VS_TAGDETAIL_match_type,k,k_type_name);
         tag_get_detail2(VS_TAGDETAIL_match_flags,k,k_tag_flags);
         // good to be a proc or proto (might have default values)
         if (k_type_name=='proto' || k_type_name=='procproto') {
            i_score++;
         }
         // not good to be external (can omit prototype in Pascal)
         if (!(k_tag_flags & VS_TAGFLAG_extern)) {
            i_score++;
         }
         // pick the best one...
         if (i_score > best_score) {
            best_match = k;
            best_score = i_score;
         }
      }
      unique_indexes[i] = best_match;
   }
}

