//
// bedisext.h - Interface needed by implementors of BE disassemblers
//
// bedis_init will be called before any other function.
// If initialisation fails, FALSE is returned and err says why.
//
// bedis_term is called after all the other functions.
// bedis_term can be omitted, if there is no module termination.
//
// bedis_disassemble is fed an address, and generates a text version.
// How many bytes were needed is returned also.
// It uses read (set up earlier) to fetch data.
// It uses symbolic (set up earlier) to get symbolic addresses.
// If couldn't fetch data for disassembly, returns FALSE.
// BE displays addresses of instructions, and opcodes, but bedis_dissasemble
// displays addresses in operands.
//
// bedis_code returns TRUE if the instruction at a given address contains
// a branch, branch and link, jump, call etc. to an address, and returns
// that address, if it is known.
// If it is a computed branch (eg: via a register), then FALSE is returned.
// BE can bring up a disassembly listing for numbers annotated 'code', and
// values returned from this function.
// A simple disassembler not supporting this feature could omit this function.
//
// bedis_data returns TRUE if instruction references a data item.
// The '*' keypress can be used to jump to data.
// A simple disassembler not supporting this feature could omit this function.
//
// bedis_options is a way to request disassembly to be done in a given way,
// based on a user typed option string.
// This function is optional.
//
// If 64 bit address space entrypoints are implemented, then BE will use
// these, otherwise it will call the older 32 bit entrypoints.
//

#ifndef BEDISEXT_H
#define	BEDISEXT_H

#ifndef Boolean_DEFINED
#define	Boolean_DEFINED
typedef int Boolean;
#define	TRUE  1
#define FALSE 0
#endif

#if   defined(OS2)
#define	BEDISEXPORT
#define	BEDISENTRY _System
#define	BEDISADDR32 unsigned
#define	BEDISADDR64 unsigned_long_long
#elif defined(WIN32)
#define	BEDISEXPORT __declspec(dllexport)
#define	BEDISENTRY __stdcall
#define	BEDISADDR32 unsigned
#define	BEDISADDR64 unsigned __int64
#elif defined(DOS32)
#define	BEDISEXPORT
#define	BEDISENTRY __export _cdecl
#define	BEDISADDR32 unsigned
#define	BEDISADDR64 unsigned __int64
#elif defined(NW)
#define	BEDISEXPORT
#define	BEDISENTRY
#define	BEDISADDR32 unsigned
#define	BEDISADDR64 unsigned __int64
#else
#define	BEDISEXPORT
#define	BEDISENTRY
#define	BEDISADDR32 unsigned
#define	BEDISADDR64 unsigned long long
#endif

#ifdef AIX
typedef void (*BEDIS_EP)(void);
typedef struct { BEDIS_EP ep; const char *name; } BEDIS_EXPORT;
#endif

typedef Boolean (*BEDISREAD32)(BEDISADDR32 addr, unsigned char & b);
typedef Boolean (*BEDISSYMBOLIC32)(BEDISADDR32 addr, char *id, BEDISADDR32 & offset);

#ifdef BE64
typedef Boolean (*BEDISREAD64)(BEDISADDR64 addr, unsigned char & b);
typedef Boolean (*BEDISSYMBOLIC64)(BEDISADDR64 addr, char *id, BEDISADDR64 & offset);
#endif

extern "C" {

BEDISEXPORT Boolean BEDISENTRY bedis_init(
	BEDISREAD32 read,
	BEDISSYMBOLIC32 symbolic,
	const char *(&err)
	);

BEDISEXPORT Boolean BEDISENTRY bedis_disassemble(
	BEDISADDR32 addr,
	BEDISADDR32 & bytes,
	char *display
	);

BEDISEXPORT Boolean BEDISENTRY bedis_ref_code(
	BEDISADDR32 addr,
	BEDISADDR32 & ref_addr
	);

BEDISEXPORT Boolean BEDISENTRY bedis_ref_data(
	BEDISADDR32 addr,
	BEDISADDR32 & ref_addr
	);

#ifdef BE64

BEDISEXPORT Boolean BEDISENTRY bedis_init_64(
	BEDISREAD64 read,
	BEDISSYMBOLIC64 symbolic,
	const char *(&err)
	);

BEDISEXPORT Boolean BEDISENTRY bedis_disassemble_64(
	BEDISADDR64 addr,
	BEDISADDR64 & bytes,
	char *display
	);

BEDISEXPORT Boolean BEDISENTRY bedis_ref_code_64(
	BEDISADDR64 addr,
	BEDISADDR64 & ref_addr
	);

BEDISEXPORT Boolean BEDISENTRY bedis_ref_data_64(
	BEDISADDR64 addr,
	BEDISADDR64 & ref_addr
	);

#endif

BEDISEXPORT Boolean BEDISENTRY bedis_options(
	const char *options,
	const char *(&err)
	);

BEDISEXPORT void BEDISENTRY bedis_term(void);

}

#endif
