/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) FontConverter.java 1.9 - last change made 07/31/96
 */

package sunsoft.jws.visual.rt.type;

import java.awt.Font;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * Converts Font objects to strings and back again.  An example of the
 * string representation: "name=Helvetica;style=bold;size=14".  The
 * styles that can currently be used are "plain", "bold", and
 * "italic".
 *
 * @version 1.9, 07/31/96
 */
public class FontConverter extends Converter {

  private static final int styleDefault = Font.PLAIN;
  private static final int sizeDefault = 12;

  /**
   * Converts a Font instance to a string.
   */
  public String convertToString(Object obj) {
    if (obj == null)
      return "";

    Font font = (Font) obj;
    String style;

    switch (font.getStyle()) {
    case Font.PLAIN:
      style = "plain";
      break;
    case Font.BOLD:
      style = "bold";
      break;
    case Font.ITALIC:
      style = "italic";
      break;
    default:
      System.out.println("Warning: unknown font style: " + font.getStyle());
      style = "plain";
      break;
    }

    return("name=" + font.getName() + ";style=" + style
	   + ";size=" + font.getSize());
  }

  /**
   * Converts a string to a new instance of Font.
   *
   * @exception ParseException when there is a format problem with the string
   */
  public Object convertFromString(String s) {
    if (s == null || s.length() == 0)
      return null;

    SubFieldTokenizer sft = new SubFieldTokenizer(s);
    Hashtable table = sft.getHashtable();

    String name = (String) table.get("name");
    if (name == null || name.length() <= 0)
      throw new ParseException("Missing font name: " + s);

    int style;
    String styleString = (String) table.get("style");
    if (styleString != null) {
      if (styleString.equals("italic"))
	style = Font.ITALIC;
      else if (styleString.equals("bold"))
	style = Font.BOLD;
      else if (styleString.equals("plain"))
	style = Font.PLAIN;
      else {
	throw new ParseException("Invalid font style: " + s);
      }
    }
    else {
      style = styleDefault;
    }
    
    int size;
    String sizeString = (String) table.get("size");
    if (sizeString != null) {
      try {
	size = Integer.valueOf(sizeString).intValue();
      } catch (NumberFormatException e) {
	throw new ParseException("Invalid font size: " + s);
      }
      if (size <= 0) {
	throw new ParseException("Negative font size: " + s);
      }
    }
    else {
      size = sizeDefault;
    }
    
    return(new Font(name, style, size));
  }
}
