/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) Converter.java 1.52 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.*;

import java.util.*;

/**
 * Base class for all converters.  Converts a type of object to a string
 * and back again.
 *
 * @version 1.52, 07/30/96
 */
public abstract class Converter {
  /**
   * Table of names for each registered converter.
   */
  private static Hashtable converterNameTable = new Hashtable();

  /**
   * Table of instances for each converter that has been instantiated.
   */
  private static Hashtable converterInstanceTable = new Hashtable();

  /**
   * Adds a new type converter to the global table of converters.  A
   * converter must be listed for this table in order for the search
   * for a converter for that particular type to be successful.
   *
   * @param typeName the name of the type (what is returned by a call to getClass().getType() for an instance of that type)
   * @param converterClassName the full name of the converter class
   */
  public static void addConverter(String typeName,
				  String converterClassName) {
    converterNameTable.put(typeName, converterClassName);
  }

  /**
   * Initialize the type converters for the types we know about.
   */
  static {
    addConverter("[I", "sunsoft.jws.visual.rt.type.IntArrayConverter");
    addConverter("[D", "sunsoft.jws.visual.rt.type.DoubleArrayConverter");
    addConverter("java.lang.String",
		 "sunsoft.jws.visual.rt.type.StringConverter");
    addConverter("[Ljava.lang.String;",
		 "sunsoft.jws.visual.rt.type.StringArrayConverter");
    addConverter("java.lang.Boolean",
		 "sunsoft.jws.visual.rt.type.BooleanConverter");
    addConverter("java.lang.Character",
		 "sunsoft.jws.visual.rt.type.CharacterConverter");
    addConverter("java.lang.Integer",
		 "sunsoft.jws.visual.rt.type.IntegerConverter");
    addConverter("java.awt.Color",
		 "sunsoft.jws.visual.rt.type.ColorConverter");
    addConverter("java.awt.Font",
		 "sunsoft.jws.visual.rt.type.FontConverter");
    addConverter("java.awt.Point",
		 "sunsoft.jws.visual.rt.type.PointConverter");
    addConverter("java.awt.Dimension",
		 "sunsoft.jws.visual.rt.type.DimensionConverter");
    addConverter("java.awt.Insets",
		 "sunsoft.jws.visual.rt.type.InsetsConverter");
    addConverter("sunsoft.jws.visual.rt.awt.GBConstraints",
		 "sunsoft.jws.visual.rt.type.GBConstraintsConverter");
    addConverter("sunsoft.jws.visual.rt.base.AttributeManager",
		 "sunsoft.jws.visual.rt.type.AMConverter");
    addConverter("sunsoft.jws.visual.rt.type.AMRef",
		 "sunsoft.jws.visual.rt.type.AMRefConverter");
    addConverter("sunsoft.jws.visual.rt.base.Attribute",
		 "sunsoft.jws.visual.rt.type.AttributeConverter");
    addConverter("sunsoft.jws.visual.rt.base.AttributeList",
		 "sunsoft.jws.visual.rt.type.AttributeListConverter");
    addConverter("sunsoft.jws.visual.rt.type.ImageRef",
		 "sunsoft.jws.visual.rt.type.ImageRefConverter");
    addConverter("sunsoft.jws.visual.rt.type.AlignmentEnum",
		 "sunsoft.jws.visual.rt.type.BaseEnumConverter");
    addConverter("sunsoft.jws.visual.rt.type.AnchorEnum",
		 "sunsoft.jws.visual.rt.type.BaseEnumConverter");
    addConverter("sunsoft.jws.visual.rt.type.OrientationEnum",
		 "sunsoft.jws.visual.rt.type.BaseEnumConverter");
    addConverter("sunsoft.jws.visual.rt.type.ReliefEnum",
		 "sunsoft.jws.visual.rt.type.BaseEnumConverter");
    addConverter("sunsoft.jws.visual.rt.type.ModeEnum",
		 "sunsoft.jws.visual.rt.type.BaseEnumConverter");
    addConverter("unknown", "sunsoft.jws.visual.rt.type.UnknownTypeConverter");
  }

  /**
   * Returns an existing converter for the given type.  Creates a new
   * converter only if necessary (typically the first time one is asked for.)
   */
  public static Converter getConverter(String typeName) {
    Converter converter;

    converter = (Converter)converterInstanceTable.get(typeName);
    if (converter != null)
      return converter;

    String converterType = (String) converterNameTable.get(typeName);
    if (converterType == null) {
      // Load the class for the type and try again.  Some types have
      // static initializers that register their converters.
      loadType(typeName);
      converterType = (String) converterNameTable.get(typeName);
    }

    if (converterType == null) {
      converterType = (String) converterNameTable.get("unknown");
      if (converterType == null)
	throw new Error("No converter defined for the \"unknown\" type.");
    }
    try {
      Class c = Class.forName(converterType);
      converter = (Converter) c.newInstance();
      converter.setConverterType(typeName);
      converterInstanceTable.put(typeName, converter);
      return converter;
    }
    catch (Exception e) {
      throw new Error(e.getMessage());
    }
  }

  private static void loadType(String typeName) {
    // For arrays, use the array type
    if (typeName.charAt(0) == '[') {
      int i;
      int len = typeName.length();
      for (i=0; i<len; i++) {
	if (typeName.charAt(i) != '[')
	  break;
      }
      i++;
      if (i < len)
	typeName = typeName.substring(i, len-1);
    }

    try {
      Class.forName(typeName);
    }
    catch (ClassNotFoundException ex) {
    }
  }

  /**
   * Returns true if there is a converter for the given type.
   */
  public static boolean hasConverter(String typeName) {
    return(converterNameTable.containsKey(typeName));
  }

  /**
   * The type editors (for more complex types.)
   */
  private static Hashtable typeEditorNameTable = new Hashtable();

  /**
   * Registers a type editor for a type.  At run-time (in generated
   * applications) there will typically be no editors, but they are
   * needed for the attribute editor in the designer.  The designer
   * will set up all the standard ones.
   *
   * @see TypeEditor
   */
  public static void addTypeEditor(String typeName,
				   String editorClassName) {
    typeEditorNameTable.put(typeName, editorClassName);
  }

  /**
   * Returns true if there is an editor for the given type.
   *
   * @see TypeEditor
   */
  public static boolean hasTypeEditor(String typeName) {
    return(typeEditorNameTable.containsKey(typeName));
  }

  /**
   * Returns a new instance of a type editor.  The caller (typically the
   * Designer) gets a new one of these every time, one for each
   * attribute being edited, even if they are the same type.  Caching
   * instances of these type editors is up to the caller.
   */
  public static TypeEditor newTypeEditor(String typeName) {
    String editorType = (String) typeEditorNameTable.get(typeName);

    if (editorType != null) {
      try {
	// instances of type editors are NOT cached
	Class c = Class.forName(editorType);
	return((TypeEditor) c.newInstance());
      }
      catch (Exception ex) {
	throw new VJException(Global.newline() + "    " + ex.toString());
      }
    }

    return null;
  }
  
  /**
   * Returns whether a converter instance has an associated type editor.
   *
   * @see TypeEditor
   */
  public boolean hasTypeEditor() {
    return(hasTypeEditor(getConverterType()));
  }

  /**
   * Returns a new instance of the type editor associated with this
   * converter.
   */
  public TypeEditor newTypeEditor() {
    return(newTypeEditor(getConverterType()));
  }

  // ------ Interfaces for Sub-Classers -----------------------------------

  /**
   * The name of the type being edited.
   */
  protected String converterType;

  /**
   * An interface that can be overridden in sub-classes to whom the type
   * converted is important.
   *
   * @see BaseEnumConverter
   */
  protected void setConverterType(String type) {
    converterType = type;
  }

  /**
   * Returns the type of object converted by this converter.
   */
  public String getConverterType() {
    return(converterType);
  }

  /**
   * Returns the string representation for an instance of the type this
   * converter converts.  Must be declared in subclasses to convert an
   * object of the type specific to that subclass of Converter.
   * <p>
   * One of the two "convertToString" methods must be overridden in
   * the converter sub-class.  The overridden "convertToString" method
   * should NOT call "super.convertToString".  It is preferrable to
   * override the StringBuffer version (the other one) because this
   * will result in better performance.
   */
  public String convertToString(Object obj) {
    enterConvert(TOSTRING, false);
    StringBuffer buf = new StringBuffer();
    convertToString(obj, buf);
    exitConvert(TOSTRING, false);

    return buf.toString();
  }

  /**
   * Places a string representation of an instance of the type this
   * converter converts into a string buffer.
   */
  public void convertToString(Object obj, StringBuffer buf) {
    enterConvert(TOSTRING, true);
    buf.append(convertToString(obj));
    exitConvert(TOSTRING, true);
  }

  /**
   * Returns a new instance of the type this converter converts, as
   * specified by the string given.  Must be declared in subclasses of
   * Converter to convert a string representation into an object of
   * the type converted by the subclass.
   */
  public abstract Object convertFromString(String s);

  /**
   * Converts an instance of the type into a block of code.
   */
  public void convertToCodeBlock(String amName,
				 Attribute a, int indent, StringBuffer buf) {

    Converter c = getConverter(a.getType());

    indent(buf, indent);
    buf.append(amName);
    buf.append(".set(\"");
    buf.append(a.getName());
    buf.append("\", ");
    buf.append(c.convertToCode(a.getValue()));
    buf.append(");");
    newline(buf);
  }

  /**
   * Converts an instance of the type converted into a line of code.
   * This method provides a default way for any type to get a
   * convertToCode method into it.  It generates code that will feed
   * the string representation of the object into the appropriate type
   * converter.  The performance isn't as good as customized
   * convertToCode functions in subclasses since more classes have to
   * be loaded at runtime.
   */
  public String convertToCode(Object obj) {
    if (obj != null)
      return("convert(\"" + obj.getClass().getName() + "\", \""
	     + convertToString(obj) + "\")");
    else
      return("null");
  }

  /**
   * Returns the string that should be displayed in the attribute
   * editor.  Subclassers that want something displayed other than
   * what is returned from convertToString should override this
   * method to return that.
   */
  public String displayString(Object obj) {
    return(convertToString(obj));
  }

  /**
   * Returns true if this type should be displayed in an editor.
   *
   * For the attribute editor, a return value of false means that the
   * the textfield will be hidden.
   *
   * @return true
   */
  public boolean viewableAsString() {
    return true;
  }

  /**
   * Returns true if this type is simple enough to be edited as a string
   * in an editor.
   *
   * Sub-classers that represent type too complex for this should override
   * this function to return false.  For the attribute editor, this means
   * that the textfield will be read-only.
   *
   * @see #viewableAsString
   * @return same as viewableAsString
   */
  public boolean editableAsString() {
    return viewableAsString();
  }

  /**
   * These weird looking enter/exit methods ensure that the converter
   * sub-class is overriding at least one of the "convertToString"
   * methods, and at least one of the "convertToCode" methods.
   * An error will be thrown at runtime if this in not the case.
   * If this check wasn't done here , then the failure to override one
   * of the methods would result in an infinite loop.
   */
  private static final int TOSTRING = 0;
  private static final int TOCODE = 1;

  private boolean converting[] = {false, false};
  private boolean isBuffered[] = {false, false};
  private int convertRecurse[] = {0, 0};

  private void enterConvert(int c, boolean isBuffered) {
    if (converting[c] && this.isBuffered[c] != isBuffered)
      throw new Error("Sub-classes of Converter MUST override at least one " +
		      "of the \"convertToString\" methods, and at least one " +
		      "of the \"convertToCode\" methods.");

    this.isBuffered[c] = isBuffered;
    converting[c] = true;
    convertRecurse[c]++;
  }

  private void exitConvert(int c, boolean isBuffered) {
    if (!converting[c])
      throw new Error("Convert exit without enter");

    if (this.isBuffered[c] != isBuffered)
      throw new Error("isBuffered mismatch in exitConvert");

    convertRecurse[c]--;
    if (convertRecurse[c] == 0)
      converting[c] = false;
  }

  // ------ Utility Functions ----------------------------------------------

  /**
   * Returns a string that can be used as a newline.  This string includes
   * a carriage return if we are running on Windows.
   */
  public static String newline() {
    return Global.newline();
  }

  /**
   * Appends a newline to buf.  This also appends a carriage return
   * if we are running on Windows.
   */
  public static void newline(StringBuffer buf) {
    Global.newline(buf);
  }

  private static final String indentString = "  ";
  private static int indentLevel = 0;

  /**
   * Appends spaces to "buf" based on the current indent level.
   */
  protected static void indent(StringBuffer buf) {
    for (int i=0; i<indentLevel; i++)
      buf.append(indentString);
  }

  /**
   * Appends spaces to "buf" based on the given indent level.
   */
  protected static void indent(StringBuffer buf, int indentLevel) {
    for (int i=0; i<indentLevel; i++)
      buf.append(' ');
  }

  /**
   * Increments the indent level.
   */
  protected static void incrIndent() {
    indentLevel++;
  }

  /**
   * Decrements the indent level.
   */
  protected static void decrIndent() {
    indentLevel--;
  }

  /**
   * Returns the current indent level.
   */
  protected static int indentLevel() {
    return indentLevel;
  }

  /**
   * Returns the last token in a class name.  i.e. the name that you
   * can use for a class when you've imported the class already.
   */
  public static String shortClassName(String className) {
    int index = className.lastIndexOf('.');
    if (index == -1)
      return(className);
    else
      return(className.substring(index + 1));
  }
}
