/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) AMRef.java 1.14 - last change made 07/18/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.AttributeManager;
import sunsoft.jws.visual.rt.base.Root;
import java.util.*;

/**
 * A reference to an attribute manager object.  The reference can
 * merely be the expected name of the object, if necessary.  That way
 * there will be no error until the reference is actually needed.
 * This reference can be made to serve in forward-referencing
 * situations (like file loads) when the object referred to hasn't
 * been loaded yet.
 *
 * @see AttributeManager
 * @version 1.14, 07/18/96
 */
public class AMRef {
  private String name;
  private AttributeManager mgr;

  // AMRef Table: a record of unresolved AMRefs created
  private static Vector refRecord = null;

  /**
   * Starts recording the AMRefs that are created using a name only.
   * Later a call to the stopRecording method can be used in order to
   * force the resolution of all the AMRefs created since this call
   * was made.  The Designer uses this in order to insure that all
   * AMRefs created during the loading of a file are force to resolve
   * and bind to the real AttributeManager objects to which they
   * refer.
   *
   * @see #stopRecording
   */
  public static void startRecording() {
    refRecord = new Vector();
  }

  /**
   * Stops recording the AMRefs that are created and resolves them.
   * Resolution occurrs within the scope of the tree given.  If scope
   * is null then stops recording, clears the list, and doesn't
   * resolve anything.
   *
   * @param scope the attribute manager tree in which to resolve names
   * @see #startRecording
   */
  public static void stopRecording(AttributeManager scope) {
    if (refRecord != null) {
      if (scope != null) {
	for (Enumeration e = refRecord.elements(); e.hasMoreElements(); ) {
	  AMRef ref = (AMRef) e.nextElement();
	  ref.getRef(scope);
	}
      }
      refRecord = null;
    }
  }

  /**
   * Creates a shadow reference from the name only.
   *
   * @see #getRef
   * @param name name of object to which this reference refers
   */
  public AMRef(String name) {
    this.name = name;
    if (refRecord != null)
      refRecord.addElement(this);
  }

  /**
   * Creates an already-resolved reference.
   *
   * @param mgr the object referred to
   */
  public AMRef(AttributeManager mgr) {
    this.mgr = mgr;
  }

  /**
   * Returns the object referred to.  Resolves it from the name if necessary.
   *
   * @param scope the attribute manager tree in which to resolve the name
   */
  public AttributeManager getRef(AttributeManager scope) {
    if (mgr == null) {
      Root root = scope.getRoot();
      if (root != null)
	mgr = root.resolve(name);
    }

    return(mgr);
  }

  /**
   * Returns the name of the object referred to.
   */
  public String getName() {
    if (mgr != null)
      name = mgr.getName();
    return(name);
  }
}
