/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) GenericWindowShadow.java 1.11 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.shadow;

import sunsoft.jws.visual.rt.shadow.java.awt.WindowShadow;
import sunsoft.jws.visual.rt.awt.RootFrame;
import sunsoft.jws.visual.rt.awt.RootDialog;
import sunsoft.jws.visual.rt.base.Global;
import sunsoft.jws.visual.rt.base.VJException;

import java.awt.*;

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    class           java.lang.String          rt.awt.RootFrame 
    title           java.lang.String          "Generic Window" 
 * </pre>
 *
 * class: the java class (that must be a sub-class of
 * sunsoft.jws.visual.rt.awt.RootFrame and have a null constructor) of
 * a user-written window class.  The GenericWindowShadow class can be
 * used for quickly incorporating user's existing windows into a
 * Visual Java GUI.
 * <p>
 * Check the super class for additional attributes.
 *
 * @see RootFrame
 * @see RootDialog
 * @see GenericComponentShadow
 * @version 1.11, 08/02/96
 */
public class GenericWindowShadow extends WindowShadow {
  private String className;
  private Class genericClass;

  public GenericWindowShadow() {
    attributes.add("class", "java.lang.String",
		   "sunsoft.jws.visual.rt.awt.RootFrame", NOEDITOR);
    attributes.add("title", "java.lang.String", "Generic Window", NOEDITOR);
    attributes.alias("text", "title");

    if (Global.isIrix())
      attributes.add("font", "java.awt.Font",
		     new Font("Helvetica", Font.PLAIN, 12), DONTFETCH);

    if (Global.isWindows()) {
      attributes.add("background", "java.awt.Color",
		     Color.lightGray, DONTFETCH);
      attributes.add("font", "java.awt.Font",
		     new Font("Dialog", Font.PLAIN, 12), DONTFETCH);
    }
  }

  protected boolean useLayoutSize() {
    return false;
  }

  protected Object getOnBody(String key) {
    if (key.equals("class"))
      return getFromTable("class");
    else if (key.equals("title")) {
      if (body instanceof RootFrame)
	return ((RootFrame)body).getTitle();
      else
	return ((RootDialog)body).getTitle();
    }
    else
      return super.getOnBody(key);
  }

  protected void setOnBody(String key, Object value) {
    if (key.equals("class")) {
      // Don't create a new instance unless the class name has changed
      if (className.equals((String)value))
	return;

      Object obj = loadClass((String)value);
      destroy();
      body = obj;
      create();
    }
    else if (key.equals("title")) {
      if (body instanceof RootFrame)
	((RootFrame)body).setTitle((String)value);
      else
	((RootDialog)body).setTitle((String)value);
    }
    else
      super.setOnBody(key, value);
  }

  public void createBody() {
    body = loadClass((String)get("class"));
  }

  protected void registerBody() {
    super.registerBody();

    Window win = (Window)body;
    if (win.countComponents() == 0)
      win.add(new Label("Generic Window"));
  }

  private Object loadClass(String name) {
    Class c;
    Object obj;

    // Load the class if the name doesn't match the previous name
    if (!name.equals(className)) {
      try {
	c = Class.forName(name);
      }
      catch (ClassNotFoundException ex) {
	throw new VJException("Class not found: " + name);
      }
    }
    else {
      c = genericClass;
    }

    // Create a new instance from the class
    try {
      obj = c.newInstance();
      if (!(obj instanceof RootFrame) && !(obj instanceof RootDialog)) {
	throw new VJException("\"" + name + "\" is not a subclass of " +
				   "RootFrame or RootDialog");
      }
    }
    catch (IllegalAccessException ex) {
      throw new VJException("Illegal access: \"" + name + "\"");
    }
    catch (InstantiationException ex) {
      throw new VJException("\"" + name +
				 "\" could not be instantiated");
    }
    catch (NoSuchMethodError ex) {
      throw new VJException("\"" + name +
				 "\" does not have a null constructor");
    }

    // No errors occurred, so update the name and class variables.
    genericClass = c;
    className = name;

    return obj;
  }
}
