/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) GenericComponentShadow.java 1.9 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.shadow;

import sunsoft.jws.visual.rt.shadow.java.awt.CanvasShadow;
import sunsoft.jws.visual.rt.awt.GBPanel;
import sunsoft.jws.visual.rt.base.VJException;

import java.awt.Button;
import java.awt.Component;

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    class           java.lang.String          java.awt.Button 
 * </pre>
 *
 * class: the java class (that must be a sub-class of
 * java.awt.Component and have a null constructor) of a user-written
 * AWT class that there is no wrapper shadow class for yet.
 * GenericComponentShadow is useful for quickly incorporating a user's
 * existing custom AWT components in a Visual Java GUI.
 * <p>
 * Check the super class for additional attributes.
 *
 * @see Component
 * @see GenericWindowShadow
 * @version 1.9, 08/02/96
 */
public class GenericComponentShadow extends CanvasShadow {
  private String className;
  private Class genericClass;

  public GenericComponentShadow() {
    attributes.add("class", "java.lang.String",
		   "java.awt.Button", NOEDITOR);
  }

  protected Object getOnBody(String key) {
    if (key.equals("class"))
      return getFromTable("class");
    else
      return super.getOnBody(key);
  }

  protected void setOnBody(String key, Object value) {
    if (key.equals("class")) {
      // Don't create a new instance unless the class name has changed
      if (className.equals((String)value))
	return;

      Object obj = loadClass((String)value);
      destroy();
      body = obj;
      create();
    }
    else
      super.setOnBody(key, value);
  }

  public void createBody() {
    body = loadClass((String)get("class"));
  }

  private Object loadClass(String name) {
    Class c;
    Object obj;

    // Load the class if the name doesn't match the previous name
    if (!name.equals(className)) {
      try {
	c = Class.forName(name);
      }
      catch (ClassNotFoundException ex) {
	throw new VJException("Class not found: " + name);
      }
    }
    else {
      c = genericClass;
    }

    // Create a new instance from the class
    try {
      obj = c.newInstance();
      if (!(obj instanceof Component)) {
	throw new VJException("\"" + name +
				   "\" is not a subclass of Component");
      }
    }
    catch (IllegalAccessException ex) {
      throw new VJException("Illegal access: \"" + name + "\"");
    }
    catch (InstantiationException ex) {
      throw new VJException("\"" + name +
				 "\" could not be instantiated");
    }
    catch (NoSuchMethodError ex) {
      throw new VJException("\"" + name +
				 "\" does not have a null constructor");
    }

    // No errors occurred, so update the name and class variables.
    genericClass = c;
    className = name;

    // Set the runtime flag for GBPanel instances
    if ((obj instanceof GBPanel) && inDesignerRoot())
      ((GBPanel)obj).setRuntime(false);

    return obj;
  }
}
