/**
 * MultiLineLabel.java
 *
 * This example is from the book _Java in a Nutshell_ by David Flanagan.
 * Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
 * You may study, use, modify, and distribute this example for any purpose.
 * This example is provided WITHOUT WARRANTY either expressed or implied.
 *
 * Tilman 05/07/96: added a maxChars parameter to specify a maximum line
 *                  length. If any of the lines is longer, it will be cut
 *                  at the last space character before maxChars, until it fits
 *                  into maxChars. Set maxChars to -1 if you don't care.
 *
 * Van 07/30/96: added the removal of backslash-r from the multi-line string
 * 		 that might have been added by strings in Windows (which
 * 		 expect newlines to be indicated by backslash-r backslash-n)
 *
 * @version @(#)MultiLineLabel.java 1.3 96/07/31
 */


package sunsoft.jws.visual.rt.awt;

import java.awt.*;
import java.util.*;

public class MultiLineLabel extends Canvas {
    public static final int LEFT = Label.LEFT; // Alignment constants
    public static final int CENTER = Label.CENTER;
    public static final int RIGHT = Label.RIGHT;

    protected String label;
    protected Vector lines;         // The lines of text to display
    protected int num_lines;          // The number of lines
    protected int maxChars;           // maximum width of lines
    protected int margin_width;       // Left and right margins
    protected int margin_height;      // Top and bottom margins
    protected int line_height;        // Total height of the font
    protected int line_ascent;        // Font height above baseline
    protected int[] line_widths;      // How wide each line is
    protected int max_width;          // The width of the widest line
    protected int alignment = LEFT;   // The alignment of the text.
    
    /**
     * This method breaks a specified label up into an array of lines.
     * It uses the StringTokenizer utility class.
     */
    protected void newLabel(String label) {
        this.label = label;

        StringTokenizer t = new StringTokenizer(label, "\r\n");
	lines = new Vector();
	while (t.hasMoreElements()) {
	    String line = t.nextToken();
	    if ((maxChars == -1) || (line.length() <= maxChars))
		lines.addElement(line);
	    else {
		while (line.length() > maxChars) {
		    int offset = line.lastIndexOf(' ', maxChars);
		    if (offset == -1) {
			// didn't get one within maxChars!
			offset = line.indexOf(' ');
			if (offset == -1)
			    break;
		    }
		    lines.addElement(line.substring(0, offset));
		    line = line.substring(offset + 1);
		}
		lines.addElement(line);
	    }
	}
	num_lines = lines.size();
        line_widths = new int[num_lines];
    }
    
    /**
     * This method figures out how the font is, and how wide each
     * line of the label is, and how wide the widest line is.
     */
    protected void measure() {
        if (this.getFont() == null) return;
        FontMetrics fm = this.getFontMetrics(this.getFont());
        // If we don't have font metrics yet, just return.
        if (fm == null) return;
        
        line_height = fm.getHeight();
        line_ascent = fm.getAscent();
        max_width = 0;
        for(int i = 0; i < num_lines; i++) {
            line_widths[i] = fm.stringWidth((String) lines.elementAt(i));
            if (line_widths[i] > max_width) max_width = line_widths[i];
        }
    }
    
    // Here are four versions of the constructor.

    /**
     * Break the label up into separate lines, and save the other info.
     */
    public MultiLineLabel(String label, int maxChars,
			  int margin_width, int margin_height,
                  int alignment) {
        this.maxChars = maxChars;
	newLabel(label);
        this.margin_width = margin_width;
        this.margin_height = margin_height;
        this.alignment = alignment;
    }
    public MultiLineLabel(String label, int maxChars,
			  int margin_width, int margin_height) {
        this(label, maxChars, margin_width, margin_height, LEFT);
    }
    public MultiLineLabel(String label, int maxChars,
			  int alignment) {
        this(label, maxChars, 10, 10, alignment);
    }
    public MultiLineLabel(String label, int maxChars) {
        this(label, maxChars, 10, 10, LEFT);
    }

    public MultiLineLabel(String label) {
        this(label, -1, 10, 10, LEFT);
    }

    public MultiLineLabel() {
	this("MultiLineLabel");
    }

    // Methods to set the various attributes of the component
    public void setLabel(String label) {
        newLabel(label);
        measure();
        repaint();
    }

    public String getLabel() {
        return label;
    }

    public void setFont(Font f) {
        super.setFont(f);
        measure();
        repaint();
    }

    public void setForeground(Color c) { 
        super.setForeground(c); 
        repaint(); 
    }

    public void setAlignment(int a) { alignment = a; repaint(); }
    public void setMarginWidth(int mw) { margin_width = mw; repaint(); }
    public void setMarginHeight(int mh) { margin_height = mh; repaint(); }
    public int getAlignment() { return alignment; }
    public int getMarginWidth() { return margin_width; }
    public int getMarginHeight() { return margin_height; }
    
    /**
     * This method is invoked after our Canvas is first created
     * but before it can actually be displayed.  After we've
     * invoked our superclass's addNotify() method, we have font
     * metrics and can successfully call measure() to figure out
     * how big the label is.
     */
    public void addNotify() { super.addNotify(); measure(); }
    
    /**
     * This method is called by a layout manager when it wants to
     * know how big we'd like to be.
     */
    public Dimension preferredSize() {
        return new Dimension(max_width + 2*margin_width, 
                     num_lines * line_height + 2*margin_height);
    }

    /**
     * This method is called when the layout manager wants to know
     * the bare minimum amount of space we need to get by.
     */
    public Dimension minimumSize() {
        return new Dimension(max_width, num_lines * line_height);
    }
    
    /**
     * This method draws the label (applets use the same method).
     * Note that it handles the margins and the alignment, but that
     * it doesn't have to worry about the color or font--the superclass
     * takes care of setting those in the Graphics object we're passed.
     */
    public void paint(Graphics g) {
        int x, y;
        Dimension d = this.size();
        y = line_ascent + (d.height - num_lines * line_height)/2;
        for(int i = 0; i < num_lines; i++, y += line_height) {
            switch(alignment) {
            case LEFT:
                x = margin_width; break;
            case CENTER:
            default:
                x = (d.width - line_widths[i])/2; break;
            case RIGHT:
                x = d.width - margin_width - line_widths[i]; break;
            }
            g.drawString((String) lines.elementAt(i), x, y);
        }
    }
}
