/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.io.IOException;
import java.io.InputStream;

/**
 * A input stream that replaces JFormDesigner 1.0 class names on the fly to
 * 2.0 class names.
 *
 * @author Karl Tauber
 */
class V1FilterInputStream
	extends InputStream
{
	// Note: all search strings must start with different characters
	private static final byte[][] SEARCH_REPLACE  = {
		"class=\"com.ktauber.form.model.".getBytes(),
		"class=\"com.jformdesigner.model.".getBytes(),

		"<class>com.ktauber.form.runtime.".getBytes(),
		"<class>com.jformdesigner.runtime.".getBytes(),

		"string>com.ktauber.form.wrapper.JGoodiesForms".getBytes(),
		"string>com.jformdesigner.designer.wrapper.JGoodiesForms".getBytes()
	};

	private final InputStream in;

	private final int[] firstSearchBytes;

	private final byte[] buf;
	private int bufCount;			// number of valid bytes in buf
	private int bufPos = -1;		// current position in buf

	private byte[] searchBuf;
	private byte[] replaceBuf;
	private int replacePos = -1;	// index into replaceBuf

	V1FilterInputStream( InputStream in ) {
		this.in = in;

		firstSearchBytes = new int[SEARCH_REPLACE.length / 2];
		int maxLen = 0;
		for( int i = 0; i < SEARCH_REPLACE.length; i += 2 ) {
			firstSearchBytes[i/2] = SEARCH_REPLACE[i][0];
			maxLen = Math.max( maxLen, SEARCH_REPLACE[i].length );
		}

		buf = new byte[maxLen];
	}

	@Override
	public int read() throws IOException {
		if( replacePos >= 0 ) {
			// return from replaceBuf
			int ch = replaceBuf[replacePos++];
			if( replacePos >= replaceBuf.length )
				replacePos = -1;
			return ch;
		}

		// get next byte
		int ch;
		if( bufPos < 0 ) {
			ch = in.read();
			if( ch == -1 )
				return -1;
		} else {
			// return from buffer
			ch = buf[bufPos++];
			if( bufPos >= bufCount ) {
				// reached the end of the buffer
				bufPos = -1;
				bufCount = 0;
			}
		}

		int index = -1;
		for( int i = 0; i < firstSearchBytes.length; i++ ) {
			if( ch == firstSearchBytes[i] ) {
				index = i * 2;
				break;
			}
		}
		if( index < 0 )
			return ch;

		searchBuf = SEARCH_REPLACE[index];
		replaceBuf = SEARCH_REPLACE[index + 1];

		// current byte is first byte in searchBuf
		// --> read searchBuf.length bytes into buffer and compare it to searchBuf
		int searchPos = 1;
		if( bufCount <= 0 )
			buf[bufCount++] = (byte) ch;
		else {
			// the buffer still contains valid data --> move it to index 0
			buf[0] = (byte) ch;
			System.arraycopy( buf, bufPos, buf, 1, bufCount - bufPos );
			bufCount = bufCount - bufPos + 1;
			bufPos = 1;

			// compare in buffer
			for( int i = 1; i < searchBuf.length && i < bufCount; i++ ) {
				if( buf[i] != searchBuf[i] )
					break;
				searchPos++;
			}
		}

		// get next byte, put it into buffer and compare it
		for( int i = searchPos; i < searchBuf.length; i++ ) {
			int ch2 = in.read();
			if( ch2 == -1 )
				break;
			buf[bufCount++] = (byte) ch2;
			if( ch2 != searchBuf[i] )
				break;
			searchPos++;
		}

		if( searchPos == searchBuf.length ) {
			// found searchBuf
			// --> clear buffer and return replaceBuf
			if( bufCount == searchBuf.length ) {
				// clear buffer
				bufCount = 0;
				bufPos = -1;
			} else {
				// buffer contains valid data at the end --> set bufPos
				bufPos = searchBuf.length;
			}

			replacePos = 0;
			return replaceBuf[replacePos++];
		}

		// only one byte in buffer
		if( bufCount == 1 ) {
			bufCount = 0;
			bufPos = -1;
			return ch;
		}

		// return bytes from buffer
		bufPos = 0;
		return buf[bufPos++];
	}

	@Override
	public int available() throws IOException {
		return in.available();
	}

	@Override
	public void close() throws IOException {
		in.close();
	}
}
