/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.util.*;

/**
 * @author Karl Tauber
 */
class LayoutCreatorRegistry
{
	/** Maps layout class name to layout creator class name or class
	 *  <String, String or Class>. */
	private static HashMap<String, Object> layoutsMap = new HashMap<String, Object>();

	/** Maps layout class to layout creator class <Class, Class>. */
	private static HashMap<Class<?>, Class<?>> layoutsCacheMap = new HashMap<Class<?>, Class<?>>();

	static {
		String groupLayoutCreatorClass;
		try {
			// use Java 6 GroupLayout if available
			Class.forName( "javax.swing.GroupLayout" );
			groupLayoutCreatorClass = "com.jformdesigner.runtime.GroupLayout6Creator";

			Class.forName( groupLayoutCreatorClass ); // check whether is is available
		} catch( ClassNotFoundException ex ) {
			groupLayoutCreatorClass = "com.jformdesigner.runtime.GroupLayoutCreator";
		}

		layoutsMap.put( "java.awt.BorderLayout",  "com.jformdesigner.runtime.BorderLayoutCreator" );
		layoutsMap.put( "java.awt.CardLayout",    "com.jformdesigner.runtime.CardLayoutCreator" );
		layoutsMap.put( "java.awt.FlowLayout",    "com.jformdesigner.runtime.FlowLayoutCreator" );
		layoutsMap.put( "java.awt.GridLayout",    "com.jformdesigner.runtime.GridLayoutCreator" );
		layoutsMap.put( "java.awt.GridBagLayout", "com.jformdesigner.runtime.GridBagLayoutCreator" );
		layoutsMap.put( "javax.swing.BoxLayout",  "com.jformdesigner.runtime.BoxLayoutCreator" );
		layoutsMap.put( "com.jformdesigner.runtime.NullLayout", "com.jformdesigner.runtime.NullLayoutCreator" );
		layoutsMap.put( "org.jdesktop.layout.GroupLayout",      groupLayoutCreatorClass );
		layoutsMap.put( "com.jgoodies.forms.layout.FormLayout", "com.jformdesigner.runtime.FormLayoutCreator" );
		layoutsMap.put( "info.clearthought.layout.TableLayout", "com.jformdesigner.runtime.TableLayoutCreator" );
		layoutsMap.put( "com.intellij.uiDesigner.core.GridLayoutManager", "com.jformdesigner.runtime.IntelliJGridLayoutCreator" );
		layoutsMap.put( "org.jdesktop.swingx.HorizontalLayout", "com.jformdesigner.runtime.HorizontalLayoutCreator" );
		layoutsMap.put( "org.jdesktop.swingx.VerticalLayout", "com.jformdesigner.runtime.VerticalLayoutCreator" );

		layoutsMap.put( "javax.swing.JLayeredPane", "com.jformdesigner.runtime.JLayeredPaneCreator" );
		layoutsMap.put( "javax.swing.JMenu",        "com.jformdesigner.runtime.JMenuCreator" );
		layoutsMap.put( "javax.swing.JMenuBar",     "com.jformdesigner.runtime.JMenuBarCreator" );
		layoutsMap.put( "javax.swing.JPopupMenu",   "com.jformdesigner.runtime.JPopupMenuCreator" );
		layoutsMap.put( "javax.swing.JScrollPane",  "com.jformdesigner.runtime.JScrollPaneCreator" );
		layoutsMap.put( "javax.swing.JSplitPane",   "com.jformdesigner.runtime.JSplitPaneCreator" );
		layoutsMap.put( "javax.swing.JTabbedPane",  "com.jformdesigner.runtime.JTabbedPaneCreator" );
		layoutsMap.put( "javax.swing.JToolBar",     "com.jformdesigner.runtime.JToolBarCreator" );
		layoutsMap.put( "javax.swing.JViewport",    "com.jformdesigner.runtime.JViewportCreator" );

		layoutsMap.put( "com.jformdesigner.runtime.GenericIndexLayout",  "com.jformdesigner.runtime.GenericIndexLayoutCreator" );

		// deprecated (for compatibility)
		layoutsMap.put( "javax.swing.ScrollPaneLayout", "com.jformdesigner.runtime.JScrollPaneCreator" );
		layoutsMap.put( "javax.swing.plaf.basic.BasicTabbedPaneUI$TabbedPaneLayout",
				"com.jformdesigner.runtime.JTabbedPaneCreator" );
	}

	static LayoutCreator createLayoutCreator( Class<?> layoutClass )
		throws InstantiationException, IllegalAccessException
	{
		Class<?> creatorClass = getLayoutCreatorClass( layoutClass );
		if( creatorClass == null )
			throw new IllegalArgumentException( "layout class \"" + layoutClass + "\" not supported" );

		return (LayoutCreator) creatorClass.newInstance();
	}

	private static Class<?> getLayoutCreatorClass( Class<?> layoutClass ) {
		if( layoutClass == null )
			throw new IllegalArgumentException( "layoutClass is null" );

		Class<?> creatorClass = layoutsCacheMap.get( layoutClass );
		if( creatorClass != null )
			return creatorClass;

		Object nameOrClass = layoutsMap.get( layoutClass.getName() );
		if( nameOrClass instanceof Class<?> ) {
			creatorClass = (Class<?>) nameOrClass;
			layoutsCacheMap.put( layoutClass, creatorClass );
			return creatorClass;
		}

		String name = (String) nameOrClass;
		if( name == null ) {
			Class<?> cls = layoutClass.getSuperclass();
			return (cls != null) ? getLayoutCreatorClass( cls ) : null;
		}

		try {
			creatorClass = Class.forName( name );
			layoutsCacheMap.put( layoutClass, creatorClass );
			return creatorClass;
		} catch( Exception ex ) {
			return null;
		}
	}
}
