/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import com.intellij.uiDesigner.core.GridConstraints;
import com.intellij.uiDesigner.core.GridLayoutManager;
import com.jformdesigner.model.FormLayoutConstraints;
import com.jformdesigner.model.FormLayoutManager;

/**
 * @author Karl Tauber
 */
public class IntelliJGridLayoutCreator
	extends AbstractLayoutCreator
{
	private static final Insets DEFAULT_MARGIN = new Insets( 0, 0, 0, 0 );
	private static final int DEFAULT_SIZE_POLICY =
		GridConstraints.SIZEPOLICY_CAN_GROW | GridConstraints.SIZEPOLICY_CAN_SHRINK;
	private static final int DEFAULT_SPACER_SIZE_POLICY =
		GridConstraints.SIZEPOLICY_CAN_GROW | GridConstraints.SIZEPOLICY_WANT_GROW;
	private static final int DEFAULT_SPACER_SIZE_POLICY2 =
		GridConstraints.SIZEPOLICY_CAN_SHRINK;

	public LayoutManager createLayoutManager( Container container, FormLayoutManager formLayout )
		throws InstantiationException, IllegalAccessException
	{
		int columnCount = formLayout.getPropertyInt( "$columnCount", 0 );
		int rowCount = formLayout.getPropertyInt( "$rowCount", 0 );
		Insets margin = (Insets) formLayout.getProperty( "margin", DEFAULT_MARGIN );
		int hGap = formLayout.getPropertyInt( "hGap", -1 );
		int vGap = formLayout.getPropertyInt( "vGap", -1 );
		boolean sameSizeHorizontally = formLayout.getPropertyBoolean( "sameSizeHorizontally", false );
		boolean sameSizeVertically = formLayout.getPropertyBoolean( "sameSizeVertically", false );

		return new GridLayoutManager( Math.max( rowCount, 1 ), Math.max( columnCount, 1 ),
			margin, hGap, vGap, sameSizeHorizontally, sameSizeVertically );
	}

	@Override
	public Object createLayoutConstraints( FormLayoutConstraints formConstraints ) {
		return createConstraints( formConstraints );
	}

	public static GridConstraints createConstraints( FormLayoutConstraints formConstraints ) {
		int defaultFill = GridConstraints.FILL_NONE;
		int defaultHSizePolicy = DEFAULT_SIZE_POLICY;
		int defaultVSizePolicy = DEFAULT_SIZE_POLICY;

		String beanClassName = formConstraints.getComponent().getClassName();
		if( beanClassName.equals( "com.jformdesigner.designer.wrapper.HSpacer" ) ) {
			defaultFill = GridConstraints.FILL_HORIZONTAL;
			defaultHSizePolicy = DEFAULT_SPACER_SIZE_POLICY;
			defaultVSizePolicy = DEFAULT_SPACER_SIZE_POLICY2;
		} else if( beanClassName.equals( "com.jformdesigner.designer.wrapper.VSpacer" ) ) {
			defaultFill = GridConstraints.FILL_VERTICAL;
			defaultHSizePolicy = DEFAULT_SPACER_SIZE_POLICY2;
			defaultVSizePolicy = DEFAULT_SPACER_SIZE_POLICY;
		}

		int row = formConstraints.getPropertyInt( "row", 0 );
		int column = formConstraints.getPropertyInt( "column", 0 );
		int rowSpan = formConstraints.getPropertyInt( "rowSpan", 1 );
		int colSpan = formConstraints.getPropertyInt( "colSpan", 1 );
		int anchor = formConstraints.getPropertyInt( "anchor", GridConstraints.ANCHOR_CENTER );
		int fill = formConstraints.getPropertyInt( "fill", defaultFill );
		int indent = formConstraints.getPropertyInt( "indent", 0 );
		boolean useParentLayout = formConstraints.getPropertyBoolean( "useParentLayout", false );
		int hSizePolicy = formConstraints.getPropertyInt( "hSizePolicy", defaultHSizePolicy );
		int vSizePolicy = formConstraints.getPropertyInt( "vSizePolicy", defaultVSizePolicy );
		Dimension minimumSize = (Dimension) formConstraints.getProperty( "minimumSize" );
		Dimension preferredSize = (Dimension) formConstraints.getProperty( "preferredSize" );
		Dimension maximumSize = (Dimension) formConstraints.getProperty( "maximumSize" );

		return new GridConstraints( row, column, rowSpan, colSpan, anchor, fill,
			hSizePolicy, vSizePolicy, minimumSize, preferredSize, maximumSize, indent, useParentLayout );
	}
}
