/*
 * Copyright (C) 2009-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.LayoutManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import javax.swing.GroupLayout;
import javax.swing.LayoutStyle;
import javax.swing.SwingConstants;
import com.jformdesigner.model.FormLayoutConstraints;
import com.jformdesigner.model.FormLayoutManager;

/**
 * @author Karl Tauber
 * @since 5.0
 */
public class GroupLayout6Creator
	extends AbstractLayoutCreator
{
	private enum MyComponentPlacement { RELATED, UNRELATED, INDENT, PADDING_SEPARATE }
	private static final int PADDING_SEPARATE_VALUE = 18;

	public LayoutManager createLayoutManager( Container container, FormLayoutManager formLayout )
		throws InstantiationException, IllegalAccessException
	{
		boolean autocreateContainerGaps = formLayout.getPropertyBoolean( "autocreateContainerGaps", false );
		boolean autocreateGaps = formLayout.getPropertyBoolean( "autocreateGaps", false );
		boolean honorsVisibility = formLayout.getPropertyBoolean( "honorsVisibility", true );

		GroupLayout layout = new GroupLayout( container );
		layout.setAutoCreateContainerGaps( autocreateContainerGaps );
		layout.setAutoCreateGaps( autocreateGaps );
		layout.setHonorsVisibility( honorsVisibility );

		return layout;
	}

	@Override
	public void addComponentToContainer( Container container, Component component,
										 Object constraints, int index,
										 FormLayoutConstraints formConstraints )
	{
		// GroupLayout adds the components to the container
	}

	@Override
	public void finishLayoutInitialization( Container container, FormLayoutManager formLayout ) {
		String horizontalGroup = formLayout.getPropertyString( "$horizontalGroup", "" );
		String verticalGroup = formLayout.getPropertyString( "$verticalGroup", "" );

		ComponentProvider componentProvider = getFormCreator();
		Map<String, ArrayList<String>> hLinkSizeGroups = new HashMap<String, ArrayList<String>>();
		Map<String, ArrayList<String>> vLinkSizeGroups = new HashMap<String, ArrayList<String>>();

		GroupLayout layout = (GroupLayout) container.getLayout();
		layout.setHorizontalGroup( composeHorizontalGroup( layout, horizontalGroup, componentProvider, hLinkSizeGroups ) );
		layout.setVerticalGroup( composeVerticalGroup( layout, verticalGroup, componentProvider, vLinkSizeGroups ) );

		composeLinks( layout, hLinkSizeGroups, SwingConstants.HORIZONTAL, componentProvider );
		composeLinks( layout, vLinkSizeGroups, SwingConstants.VERTICAL, componentProvider );
	}

	public static GroupLayout.Group composeHorizontalGroup( GroupLayout layout,
		String str, ComponentProvider componentProvider, Map<String, ArrayList<String>> linkSizeGroups )
	{
		return composeGroups( layout, str, componentProvider, true, linkSizeGroups );
	}

	public static GroupLayout.Group composeVerticalGroup( GroupLayout layout,
		String str, ComponentProvider componentProvider, Map<String, ArrayList<String>> linkSizeGroups )
	{
		return composeGroups( layout, str, componentProvider, false, linkSizeGroups );
	}

	private static GroupLayout.Group composeGroups( GroupLayout layout,
		String str, ComponentProvider componentProvider,
		boolean isHorizontal, Map<String, ArrayList<String>> linkSizeGroups )
	{
		GroupLayout.Group rootGroup = null;
		String[] rootTokens = GroupLayout6Creator.toTokens( str, ';' );
		for( int i = rootTokens.length - 1; i >= 0; i-- ) {
			GroupLayout.Group group = composeGroup( layout, null, rootTokens[i],
					componentProvider, isHorizontal, true, true, linkSizeGroups );
			if( rootGroup == null )
				rootGroup = group;
			else {
				if( !(rootGroup instanceof GroupLayout.ParallelGroup) ) {
					// add multiple roots into a parallel group
					GroupLayout.ParallelGroup parallelGroup = layout.createParallelGroup();
					parallelGroup.addGroup( rootGroup );
					rootGroup = parallelGroup;
				}
				((GroupLayout.ParallelGroup)rootGroup).addGroup( group );
			}
		}
		return rootGroup;
	}

	private static GroupLayout.Group composeGroup( GroupLayout layout,
		GroupLayout.Group parentGroup, String str, ComponentProvider componentProvider,
		boolean isHorizontal, boolean first, boolean last, Map<String, ArrayList<String>> linkSizeGroups )
	{
		GroupLayout.Group group = null;

		String[] tokens = toTokens( str, ' ' );
		if( tokens.length == 0 )
			return layout.createSequentialGroup();

		String token0 = tokens[0];

		boolean isParallel = "par".equals( token0 );
		boolean isSequential = "seq".equals( token0 );
		if( isParallel || isSequential ) {
			GroupLayout.Alignment groupAlignment = GroupLayout.Alignment.LEADING;
			GroupLayout.Alignment alignment = GroupLayout.Alignment.LEADING;
			boolean resizable = true;

			// initialize parameters
			int tokenIndex = 1;
			if( tokenIndex < tokens.length && !tokens[tokenIndex].startsWith( "{" ) ) {
				String[] attrs = toTokens( tokens[tokenIndex++], ':' );
				int attrIndex = 0;
				if( attrIndex < attrs.length ) {
					if( isParallel )
						groupAlignment = decodeAlignment( attrs[attrIndex++] );

					if( attrIndex < attrs.length ) {
						alignment = decodeAlignment( attrs[attrIndex++] );

						attrIndex++; // skip minimum size
						if( attrIndex < attrs.length )
							resizable = (decodeSize( attrs[attrIndex++] ) != GroupLayout.PREFERRED_SIZE);
					}
				}
			}

			// create parallel or sequential group
			group = isParallel
				? (GroupLayout.Group) layout.createParallelGroup( groupAlignment, resizable )
				: (GroupLayout.Group) layout.createSequentialGroup();

			// create sub groups
			if( tokenIndex < tokens.length ) {
				String token = tokens[tokenIndex++];
				assert token.startsWith( "{" );
				assert token.endsWith( "}" );

				String[] subGroups = toTokens( token.substring( 1, token.length() - 1 ), ',' );
				for( int i = 0; i < subGroups.length; i++ ) {
					boolean isLast = last && (!isSequential || i == (subGroups.length - 1));
					composeGroup( layout, group, subGroups[i].trim(), componentProvider,
							isHorizontal, first, isLast, linkSizeGroups );
					if( first && isSequential )
						first = false;
				}
			}

			// add group to parent
			if( parentGroup != null ) {
				if( parentGroup instanceof GroupLayout.SequentialGroup )
					((GroupLayout.SequentialGroup)parentGroup).addGroup( group );
				else
					((GroupLayout.ParallelGroup)parentGroup).addGroup( alignment, group );
			}
		} else {
			String compId = null;
			GroupLayout.Alignment alignment = GroupLayout.Alignment.LEADING;
			MyComponentPlacement paddingType = MyComponentPlacement.RELATED;
			int minimumSize = GroupLayout.DEFAULT_SIZE;
			int preferredSize = GroupLayout.DEFAULT_SIZE;
			int maximumSize = GroupLayout.DEFAULT_SIZE;

			// initialize parameters
			String[] attrs = (tokens.length > 1) ? toTokens( tokens[1], ':' ) : null;
			int attrIndex = 0;
			if( "comp".equals( token0 ) ) {
				compId = attrs[attrIndex++];
				if( attrIndex < attrs.length ) {
					String linkSizeId = attrs[attrIndex++];
					if( linkSizeId != null ) {
						ArrayList<String> compIDs = linkSizeGroups.get( linkSizeId );
						if( compIDs == null ) {
							compIDs = new ArrayList<String>();
							linkSizeGroups.put( linkSizeId, compIDs );
						}
						compIDs.add( compId );
					}

					if( attrIndex < attrs.length )
						alignment = decodeAlignment( attrs[attrIndex++] );
				}

			} else if( "space".equals( token0 ) ) {
				if( attrIndex < attrs.length )
					paddingType = decodePaddingType( attrs[attrIndex++] );
			} else
				throw new IllegalArgumentException();

			// initialize sizes
			boolean explicitMinimumSize = false;
			boolean explicitPreferredSize = false;
			if( attrIndex < attrs.length ) {
				explicitMinimumSize = (attrs[attrIndex] != null);
				minimumSize = decodeSize( attrs[attrIndex++] );
				if( attrIndex < attrs.length ) {
					explicitPreferredSize = (attrs[attrIndex] != null);
					preferredSize = decodeSize( attrs[attrIndex++] );
					if( attrIndex < attrs.length )
						maximumSize = decodeSize( attrs[attrIndex++] );

					if( preferredSize != GroupLayout.DEFAULT_SIZE && preferredSize < 0 ) {
						// auto-correct invalid preferred size
						preferredSize = GroupLayout.DEFAULT_SIZE;
					}
				}
			}

			if( parentGroup == null )
				group = parentGroup = layout.createSequentialGroup();

			if( compId != null ) {
				// get component
				Component comp = componentProvider.getComponent( compId );

				// fix minimum size
				if( !explicitMinimumSize ) {
					if( isHorizontal && comp.getClass().getName().equals( "javax.swing.JComboBox" ) ) {
						minimumSize = 0;
					} else if( preferredSize >= 0 ) {
						Dimension dim = comp.getMinimumSize();
						int dimMin = isHorizontal ? dim.width : dim.height;
						if( dimMin > preferredSize )
							minimumSize = GroupLayout.PREFERRED_SIZE;
					}
				}

				// add component to group
				if( parentGroup instanceof GroupLayout.SequentialGroup )
					((GroupLayout.SequentialGroup)parentGroup).addComponent( comp, minimumSize, preferredSize, maximumSize );
				else
					((GroupLayout.ParallelGroup)parentGroup).addComponent( comp, alignment, minimumSize, preferredSize, maximumSize );
			} else { // space
				if( !explicitMinimumSize || !explicitPreferredSize ) {
					GroupLayout.SequentialGroup seqGroup = (GroupLayout.SequentialGroup) parentGroup;
					if( first || last ) {
						// add container gap to group
						seqGroup.addContainerGap( preferredSize, maximumSize );
					} else {
						if( paddingType == MyComponentPlacement.RELATED ) {
							// add related gap to group
							seqGroup.addPreferredGap( LayoutStyle.ComponentPlacement.RELATED, preferredSize, maximumSize );
						} else if( paddingType == MyComponentPlacement.UNRELATED ) {
							// add unrelated gap to group
							seqGroup.addPreferredGap( LayoutStyle.ComponentPlacement.UNRELATED, preferredSize, maximumSize );
						} else if( paddingType == MyComponentPlacement.PADDING_SEPARATE ) {
							// add "separate" gap to group
							if( preferredSize == GroupLayout.DEFAULT_SIZE )
								preferredSize = PADDING_SEPARATE_VALUE;
							if( maximumSize == GroupLayout.DEFAULT_SIZE )
								maximumSize = PADDING_SEPARATE_VALUE;
							seqGroup.addGap( PADDING_SEPARATE_VALUE, preferredSize, maximumSize );
						} else {
							//TODO INDENT ?
						}
					}
				} else {
					// fix minimum and maximum sizes
					if( minimumSize < 0 )
						minimumSize = preferredSize;
					minimumSize = Math.min( minimumSize, preferredSize );
					maximumSize = Math.max( maximumSize, preferredSize );

					// add space to group
					if( parentGroup instanceof GroupLayout.SequentialGroup )
						((GroupLayout.SequentialGroup)parentGroup).addGap( minimumSize, preferredSize, maximumSize );
					else
						((GroupLayout.ParallelGroup)parentGroup).addGap( minimumSize, preferredSize, maximumSize );
				}
			}
		}
		return group;
	}

	private static GroupLayout.Alignment decodeAlignment( String str ) {
		if( str == null )
			return GroupLayout.Alignment.LEADING;

		if( str.length() == 1 ) {
			switch( str.charAt( 0 ) ) {
				case 'l': return GroupLayout.Alignment.LEADING;
				case 't': return GroupLayout.Alignment.TRAILING;
				case 'c': return GroupLayout.Alignment.CENTER;
				case 'b': return GroupLayout.Alignment.BASELINE;
			}
		}

		return GroupLayout.Alignment.LEADING;
	}

	private static int decodeSize( String str ) {
		if( str == null )
			return GroupLayout.DEFAULT_SIZE;

		if( str.length() == 1 ) {
			switch( str.charAt( 0 ) ) {
				case 'd': return GroupLayout.DEFAULT_SIZE;
				case 'p': return GroupLayout.PREFERRED_SIZE;
				case 'x': return Short.MAX_VALUE;
			}
		}
		return Integer.parseInt( str );
	}

	private static MyComponentPlacement decodePaddingType( String str ) {
		if( str == null )
			return MyComponentPlacement.RELATED;

		if( str.length() == 1 ) {
			switch( str.charAt( 0 ) ) {
				case 'u': return MyComponentPlacement.UNRELATED;
				case 's': return MyComponentPlacement.PADDING_SEPARATE;
				case 'i': return MyComponentPlacement.INDENT;
			}
		}
		return MyComponentPlacement.RELATED;
	}

	private static void composeLinks( GroupLayout layout, Map<String, ArrayList<String>> linkSizeGroups,
		int axis, ComponentProvider componentProvider )
	{
		if( linkSizeGroups.isEmpty() )
			return;

		for( ArrayList<String> compIDs : linkSizeGroups.values() ) {
			Component[] comps = new Component[compIDs.size()];
			for( int i = 0; i < comps.length; i++ )
				comps[i] = componentProvider.getComponent( compIDs.get( i ) );

			layout.linkSize( axis, comps );
		}
	}

	public static String[] toTokens( String str, char sep ) {
		if( str == null || str.length() == 0 )
			return new String[0];

		int tokenCount = 1;
		int depth = 0;
		int length = str.length();
		for( int i = 0; i < length; i++ ) {
			char ch = str.charAt( i );
			if( ch == sep && depth == 0 )
				tokenCount++;
			else if( ch == '{' )
				depth++;
			else if( ch == '}' )
				depth--;
		}

		if( tokenCount == 1 )
			return new String[] { str };

		int tokenIndex = 0;
		int beginIndex = 0;
		String[] tokens = new String[tokenCount];
		for( int i = 0; i < length; i++ ) {
			char ch = str.charAt( i );
			if( ch == sep && depth == 0 ) {
				if( i > beginIndex ) {
					tokens[tokenIndex] = str.substring( beginIndex, i );
				}
				tokenIndex++;
				beginIndex = i + 1;
			} else if( ch == '{' )
				depth++;
			else if( ch == '}' )
				depth--;
		}
		if( beginIndex < length )
			tokens[tokenIndex] = str.substring( beginIndex );

		return tokens;
	}
}
