/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.util.*;
import com.jgoodies.forms.factories.FormFactory;
import com.jgoodies.forms.layout.BoundedSize;
import com.jgoodies.forms.layout.ColumnSpec;
import com.jgoodies.forms.layout.ConstantSize;
import com.jgoodies.forms.layout.FormSpec;
import com.jgoodies.forms.layout.RowSpec;
import com.jgoodies.forms.layout.Size;
import com.jgoodies.forms.layout.Sizes;

/**
 * @author Karl Tauber
 */
public class FormSpecCoder
{
	public static Boolean useLayoutMap;

	private static final Object[] columnSpecs = {
		"min",			FormFactory.MIN_COLSPEC,
		"m",			FormFactory.MIN_COLSPEC,
		"pref",			FormFactory.PREF_COLSPEC,
		"p",			FormFactory.PREF_COLSPEC,
		"default",		FormFactory.DEFAULT_COLSPEC,
		"d",			FormFactory.DEFAULT_COLSPEC,
		"glue",			FormFactory.GLUE_COLSPEC,
		"relgap",		FormFactory.RELATED_GAP_COLSPEC,
		"rgap",			FormFactory.RELATED_GAP_COLSPEC,
		"gap",			FormFactory.RELATED_GAP_COLSPEC,
		"unrelgap",		FormFactory.UNRELATED_GAP_COLSPEC,
		"ugap",			FormFactory.UNRELATED_GAP_COLSPEC,
		"labelcompgap",	FormFactory.LABEL_COMPONENT_GAP_COLSPEC,
		"lcgap",		FormFactory.LABEL_COMPONENT_GAP_COLSPEC,
		"button",		FormFactory.BUTTON_COLSPEC,
		"growbutton",	FormFactory.GROWING_BUTTON_COLSPEC,
		"gbutton",		FormFactory.GROWING_BUTTON_COLSPEC,
	};

	private static final Object[] rowSpecs = {
		"min",			FormFactory.MIN_ROWSPEC,
		"m",			FormFactory.MIN_ROWSPEC,
		"pref",			FormFactory.PREF_ROWSPEC,
		"p",			FormFactory.PREF_ROWSPEC,
		"default",		FormFactory.DEFAULT_ROWSPEC,
		"d",			FormFactory.DEFAULT_ROWSPEC,
		"glue",			FormFactory.GLUE_ROWSPEC,
		"relgap",		FormFactory.RELATED_GAP_ROWSPEC,
		"rgap",			FormFactory.RELATED_GAP_ROWSPEC,
		"gap",			FormFactory.RELATED_GAP_ROWSPEC,
		"unrelgap",		FormFactory.UNRELATED_GAP_ROWSPEC,
		"ugap",			FormFactory.UNRELATED_GAP_ROWSPEC,
		"narrowlinegap",FormFactory.NARROW_LINE_GAP_ROWSPEC,
		"nlinegap",		FormFactory.NARROW_LINE_GAP_ROWSPEC,
		"linegap",		FormFactory.LINE_GAP_ROWSPEC,
		"pargap",		FormFactory.PARAGRAPH_GAP_ROWSPEC,
	};

	private static HashMap<String, ColumnSpec> columnSpecMap;
	private static HashMap<String, RowSpec> rowSpecMap;

	static {
		columnSpecMap = new HashMap<String, ColumnSpec>( columnSpecs.length );
		for( int i = 0; i < columnSpecs.length; i += 2 )
			columnSpecMap.put( (String) columnSpecs[i], (ColumnSpec) columnSpecs[i+1] );

		rowSpecMap = new HashMap<String, RowSpec>( rowSpecs.length );
		for( int i = 0; i < rowSpecs.length; i += 2 )
			rowSpecMap.put( (String) rowSpecs[i], (RowSpec) rowSpecs[i+1] );
	}

	//---- decode -------------------------------------------------------------

	public static ColumnSpec[] decodeColumnSpecs( String encodedColumnSpecs ) {
        if (encodedColumnSpecs == null)
            throw new NullPointerException("The column description must not be null.");

        StringTokenizer tokenizer = new StringTokenizer(encodedColumnSpecs, ", ");
        int columnCount = tokenizer.countTokens();
        ColumnSpec[] columnSpecs = new ColumnSpec[columnCount];
        for (int i = 0; i < columnCount; i++) {
            columnSpecs[i] = decodeColumnSpec(tokenizer.nextToken());
        }
        return columnSpecs;
	}

	public static RowSpec[] decodeRowSpecs( String encodedRowSpecs ) {
        if (encodedRowSpecs == null)
            throw new NullPointerException("The row description must not be null.");

        StringTokenizer tokenizer = new StringTokenizer(encodedRowSpecs, ", ");
        int rowCount = tokenizer.countTokens();
        RowSpec[] rowSpecs = new RowSpec[rowCount];
        for (int i = 0; i < rowCount; i++) {
            rowSpecs[i] = decodeRowSpec(tokenizer.nextToken());
        }
        return rowSpecs;
	}

	public static ColumnSpec decodeColumnSpec( String encodedColumnSpec ) {
		String spec = encodedColumnSpec.toLowerCase();

		// ignore custom specs
		if( spec.startsWith( "id." ) ) {
			int sepIndex = spec.indexOf( '=' );
			if( isUseLayoutMap() ) {
				String id = spec.substring( "id.".length(), sepIndex );
		    	// use reflection for compatibility with JGoodies Forms 1.1 or 1.0
				try {
					return (ColumnSpec) ColumnSpec.class.getMethod( "decode", new Class[] { String.class } )
						.invoke( null, new Object[] { "${" + id + "}" } );
				} catch( Exception ex ) {
					// ignore
				}
			}

			spec = spec.substring( sepIndex + 1 );
		}

		// handle predefined specs
		ColumnSpec columnSpec = columnSpecMap.get( spec );
		if( columnSpec != null )
			return columnSpec;

		return newColumnSpec( spec );
	}

	@SuppressWarnings("deprecation")
	public static ColumnSpec newColumnSpec( String spec ) {
		if( spec.indexOf( '<' ) >= 0 )
			return (ColumnSpec) parseAndInitValues( spec, true );

		return new ColumnSpec( spec );
	}

	public static RowSpec decodeRowSpec( String encodedRowSpec ) {
		String spec = encodedRowSpec.toLowerCase();

		// ignore custom specs
		if( spec.startsWith( "id." ) ) {
			int sepIndex = spec.indexOf( '=' );
			if( isUseLayoutMap() ) {
				String id = spec.substring( "id.".length(), sepIndex );
		    	// use reflection for compatibility with JGoodies Forms 1.1 or 1.0
				try {
					return (RowSpec) RowSpec.class.getMethod( "decode", new Class[] { String.class } )
						.invoke( null, new Object[] { "${" + id + "}" } );
				} catch( Exception ex ) {
					// ignore
				}
			}

			spec = spec.substring( sepIndex + 1 );
		}

		// handle predefined specs
		RowSpec rowSpec = rowSpecMap.get( spec );
		if( rowSpec != null )
			return rowSpec;

		return newRowSpec( spec );
	}

	@SuppressWarnings("deprecation")
	public static RowSpec newRowSpec( String spec ) {
		if( spec.indexOf( '<' ) >= 0 )
			return (RowSpec) parseAndInitValues( spec, false );

		return new RowSpec( spec );
	}

	//---- encode -------------------------------------------------------------

	public static String encodeColumnSpecs( ColumnSpec[] columnSpecs ) {
        StringBuffer buffer = new StringBuffer();
        for( int i = 0; i < columnSpecs.length; i++ ) {
			if( i > 0 )
				buffer.append( ", " );
			buffer.append( encodeColumnSpec( columnSpecs[i] ) );
		}
        return buffer.toString();
	}

	public static String encodeRowSpecs( RowSpec[] rowSpecs ) {
        StringBuffer buffer = new StringBuffer();
        for( int i = 0; i < rowSpecs.length; i++ ) {
			if( i > 0 )
				buffer.append( ", " );
			buffer.append( encodeRowSpec( rowSpecs[i] ) );
		}
        return buffer.toString();
	}

	public static String encodeColumnSpec( ColumnSpec columnSpec ) {
		return encodeColumnSpec( columnSpec, false );
	}

	public static String encodeColumnSpec( ColumnSpec columnSpec, boolean newBoundedStyle ) {
		for( int i = 0; i < columnSpecs.length; i += 2 ) {
			if( columnSpec == columnSpecs[i+1] )
				return (String) columnSpecs[i];
		}

		return encodeFormSpec( columnSpec, ColumnSpec.DEFAULT, newBoundedStyle );
	}

	public static String encodeRowSpec( RowSpec rowSpec ) {
		return encodeRowSpec( rowSpec, false );
	}

	public static String encodeRowSpec( RowSpec rowSpec, boolean newBoundedStyle ) {
		for( int i = 0; i < rowSpecs.length; i += 2 ) {
			if( rowSpec == rowSpecs[i+1] )
				return (String) rowSpecs[i];
		}

		return encodeFormSpec( rowSpec, RowSpec.DEFAULT, newBoundedStyle );
	}

	public static String encodeFormSpec( FormSpec spec, FormSpec.DefaultAlignment def ) {
		return encodeFormSpec( spec, def, false );
	}

	public static String encodeFormSpec( FormSpec spec, FormSpec.DefaultAlignment def, boolean newBoundedStyle ) {
        StringBuffer buffer = new StringBuffer();

        FormSpec.DefaultAlignment defaultAlignment = spec.getDefaultAlignment();
		if( defaultAlignment != def ) {
			buffer.append( defaultAlignment );
			buffer.append( ':' );
		}

		buffer.append( encodeSize( spec.getSize(), newBoundedStyle ) );

		double resizeWeight = spec.getResizeWeight();
		if( resizeWeight != FormSpec.NO_GROW ) {
			buffer.append( ':' );
			if( resizeWeight == FormSpec.DEFAULT_GROW )
				buffer.append( "grow" );
			else {
				buffer.append( "grow(" );
				buffer.append( resizeWeight );
				buffer.append( ')' );
			}
		}
		return buffer.toString();
	}

	public static String encodeSize( Size size ) {
		return encodeSize( size, false );
	}

	public static String encodeSize( Size size, boolean newBoundedStyle ) {
		if( size == Sizes.MINIMUM )
			return "min";
		else if( size == Sizes.PREFERRED )
			return "pref";
		else if( size == Sizes.DEFAULT )
			return "default";
		else if( size instanceof ConstantSize ) {
			ConstantSize constSize = (ConstantSize) size;
			double value = constSize.getValue();
			ConstantSize.Unit unit = constSize.getUnit();

			String abbreviation = unit.abbreviation();
			if( unit == ConstantSize.DIALOG_UNITS_X || unit == ConstantSize.DIALOG_UNITS_Y )
				abbreviation = "dlu";

			if( unit == ConstantSize.MILLIMETER ||
				unit == ConstantSize.CENTIMETER ||
				unit == ConstantSize.INCH )
			{
				return Double.toString( value ) + abbreviation;
			} else
				return Integer.toString( (int) value ) + abbreviation;
		} else if( isPrototypeSize( size ) ) {
	    	// use reflection for compatibility with JGoodies Forms 1.1 or 1.0
			try {
				return (String) size.getClass().getMethod( "encode", (Class[]) null ).invoke( size, (Object[]) null );
			} catch( Exception ex ) {
				throw new IllegalArgumentException( "Prototype size specification requires JGoodies Forms 1.2 or later." );
			}
		} else if( size instanceof BoundedSize ) {
			BoundedSize boundedSize = (BoundedSize) size;
			Size basis = boundedSize.getBasis();
			Size lowerBound = boundedSize.getLowerBound();
			Size upperBound = boundedSize.getUpperBound();

			StringBuffer buffer = new StringBuffer();
			if( newBoundedStyle ) {
				buffer.append( '[' );
				if( lowerBound != null ) {
					buffer.append( encodeSize( lowerBound ) );
					buffer.append( ',' );
				}
				buffer.append( encodeSize( basis ) );
				if( upperBound != null ) {
					buffer.append( ',' );
					buffer.append( encodeSize( upperBound ) );
				}
				buffer.append( ']' );
			} else if( lowerBound != null && upperBound != null ) {
				buffer.append( '(' );
				if( lowerBound != null ) {
					buffer.append( encodeSize( lowerBound ) );
					buffer.append( '<' );
				}
				buffer.append( encodeSize( basis ) );
				if( upperBound != null ) {
					buffer.append( '<' );
					buffer.append( encodeSize( upperBound ) );
				}
				buffer.append( ')' );
			} else {
				if( lowerBound != null ) {
					buffer.append( "max(" );
					buffer.append( encodeSize( basis ) );
					buffer.append( ';' );
					buffer.append( encodeSize( lowerBound ) );
					buffer.append( ')' );
				} else if( upperBound != null ) {
					buffer.append( "min(" );
					buffer.append( encodeSize( basis ) );
					buffer.append( ';' );
					buffer.append( encodeSize( upperBound ) );
					buffer.append( ')' );
				} else
					buffer.append( encodeSize( basis ) );
			}
			return buffer.toString();
		} else {
			// fallback

			// use reflection for compatibility with JGoodies Forms 1.1 or 1.0
			try {
				return (String) size.getClass().getMethod( "encode", (Class[]) null ).invoke( size, (Object[]) null );
			} catch( Exception ex ) {
				return size.toString();
			}
		}
	}

	public static int[][] decodeGroupIndices( String encodedGroupIds ) {
		int[] groupIds = decodeIntArray( encodedGroupIds );
		return groupIds2Indices( groupIds );
	}

	public static int[][] groupIds2Indices( int[] groupIds ) {
		int[] ids = groupIds.clone();
		ArrayList<int[]> arr = null;
		for( int i = 0; i < ids.length; i++ ) {
			int groupId = groupIds[i];
			if( groupId == 0 )
				continue;

			// count the number of columns/rows in the group
			int count = 1;
			for( int j = i + 1; j < ids.length; j++ ) {
				if( ids[j] == groupId )
					count++;
			}
			if( count == 1 )
				continue;

			//
			int[] group = new int[count];
			int k = 0;
			for( int j = i; j < ids.length; j++ ) {
				if( ids[j] == groupId ) {
					group[k++] = j + 1;
					ids[j] = 0;
				}
			}

			if( arr == null )
				arr = new ArrayList<int[]>(); // lazy creation
			arr.add( group );
		}
		return (arr != null) ? arr.toArray( new int[arr.size()][] ) : null;
	}

	public static int[] decodeIntArray( String str ) {
		if( str == null )
			return new int[0];
		StringTokenizer st = new StringTokenizer( str, ", " );
		int count = st.countTokens();
		int[] array = new int[count];
		int i = 0;
		while( st.hasMoreTokens() )
			array[i++] = Integer.parseInt( st.nextToken() );
		return array;
	}

	//-------------------------------------------------------------------------
	// Copied from JGoodies FormSpec and Sizes to enable the usage of
	// the "(min<size<max)" syntax.

    // Parsing **************************************************************

    /**
     * Parses an encoded form spec and initializes all required fields.
     * The encoded description must be in lower case.
     *
     * @param encodedDescription   the FormSpec in an encoded format
     * @throws IllegalArgumentException if the string is empty, has no size,
     * or is otherwise invalid
     */
    private static FormSpec parseAndInitValues(String encodedDescription, boolean horizontal) {
        StringTokenizer tokenizer = new StringTokenizer(encodedDescription, ":");
        if (!tokenizer.hasMoreTokens()) {
            throw new IllegalArgumentException(
                                    "The form spec must not be empty.");
        }
        String token = tokenizer.nextToken();

        FormSpec.DefaultAlignment defaultAlignment = horizontal
										? ColumnSpec.DEFAULT : RowSpec.DEFAULT;
        Size size = Sizes.DEFAULT;
        double resizeWeight = FormSpec.NO_GROW;

        // Check if the first token is an orientation.
        FormSpec.DefaultAlignment alignment = valueOf(token, horizontal);
        if (alignment != null) {
            defaultAlignment = alignment;
            if (!tokenizer.hasMoreTokens()) {
                throw new IllegalArgumentException(
                                    "The form spec must provide a size.");
            }
            token = tokenizer.nextToken();
        }

        size = parseAndInitSize(token, horizontal);

        if (tokenizer.hasMoreTokens()) {
           resizeWeight = decodeResize(tokenizer.nextToken());
        }

        if( horizontal )
        	return new ColumnSpec( defaultAlignment, size, resizeWeight );
        else
        	return new RowSpec( defaultAlignment, size, resizeWeight );
    }


    /**
     * Parses an encoded size spec and initializes the size fields.
     *
     * @param token    a token that represents a size, either bounded or plain
     */
    private static Size parseAndInitSize(String token, boolean horizontal) {
        if (token.startsWith("max(") && token.endsWith(")")) {
        	return parseAndInitBoundedSize(token, false, horizontal);
        }
        if (token.startsWith("min(") && token.endsWith(")")) {
        	return parseAndInitBoundedSize(token, true, horizontal);
        }
        if (token.startsWith("(") && token.endsWith(")")) {
        	return parseAndInitBoundedSize(token.substring(1, token.length()-1), horizontal);
        }
        if (token.indexOf("<") >= 0) {
        	return parseAndInitBoundedSize(token, horizontal);
        }
        return decodeAtomicSize(token, horizontal);
    }

    /**
     * Parses an encoded compound size and sets the size fields.
     * The compound size has format:
     * ([lowerSize<]componentSize[<upperSize])
     *  componentSize must be a logical size. lowerSize and upperSize must
     * be a size constant.
     *
     * @param token  a token for a bounded size, e.g. "max(50dlu; pref)"
     * @param setMax  if true we set a maximum size, otherwise a minimum size
     * @return a Size that represents the parse result
     */
    private static Size parseAndInitBoundedSize(String token, boolean horizontal) {
        int delim1Index = token.indexOf('<');
        int delim2Index = token.indexOf('<', delim1Index+1);
        String sizeToken1 = (delim1Index > 0) ? token.substring(0, delim1Index) : "";
        String sizeToken2 = (delim2Index > 0) ? token.substring(delim1Index+1, delim2Index)
                                              : token.substring(delim1Index+1);
        String sizeToken3 = (delim2Index > 0) ? token.substring(delim2Index+1) : "";

        Size size1 = (sizeToken1.length() > 0) ? decodeAtomicSize(sizeToken1, horizontal) : null;
        Size size2 = decodeAtomicSize(sizeToken2, horizontal);
        Size size3 = (sizeToken3.length() > 0) ? decodeAtomicSize(sizeToken3, horizontal) : null;

        if (isComponentSize(size1) && size3 == null) {
        	// no lower bound specified
        	// move componentSize to size2 and upperSize to size3
            size3 = size2;
            size2 = size1;
            size1 = null;
        }

        if (size1 != null && !isConstant(size1))
            throw new IllegalArgumentException("Lower size must be a constant.");
        if (!isComponentSize(size2))
            throw new IllegalArgumentException("Basis size must be a component size.");
        if (size3 != null && !isConstant(size3))
            throw new IllegalArgumentException("Upper size must be a constant.");

        return Sizes.bounded(size2, size1, size3);
    }

    /**
     * Parses an encoded compound size and sets the size fields.
     * The compound size has format:
     * max(<atomic size>;<atomic size2>) | min(<atomic size1>;<atomic size2>)
     * One of the two atomic sizes must be a logical size, the other must
     * be a size constant.
     *
     * @param token  a token for a bounded size, e.g. "max(50dlu; pref)"
     * @param setMax  if true we set a maximum size, otherwise a minimum size
     * @return a Size that represents the parse result
     */
    private static Size parseAndInitBoundedSize(String token, boolean setMax, boolean horizontal) {
        int semicolonIndex = token.indexOf(';');
        String sizeToken1 = token.substring(4, semicolonIndex);
        String sizeToken2 = token.substring(semicolonIndex+1, token.length()-1);

        Size size1 = decodeAtomicSize(sizeToken1, horizontal);
        Size size2 = decodeAtomicSize(sizeToken2, horizontal);

        // Check valid combinations and set min or max.
        if (isConstant(size1)) {
            if (isComponentSize(size2)) {
                return Sizes.bounded(size2, setMax ? null : size1,
                                               setMax ? size1 : null);
            }
            throw new IllegalArgumentException(
                                "Bounded sizes must not be both constants.");
        } else {
            if (isConstant(size2)) {
                return Sizes.bounded(size1, setMax ? null : size2,
                                               setMax ? size2 : null);
            }
            throw new IllegalArgumentException(
                                "Bounded sizes must not be both logical.");
        }
    }


    /**
     * Decodes and returns an atomic size that is either a constant size or a
     * component size.
     *
     * @param token	the encoded size
     * @return the decoded size either a constant or component size
     */
    private static Size decodeAtomicSize(String token, boolean horizontal) {
        Size componentSize = valueOf(token);
        if (componentSize != null)
            return componentSize;
        else if( token.startsWith( "'" ) && token.endsWith( "'" ) ) {
        	if( token.length() < 2 )
        		throw new IllegalArgumentException( "Missing closing \"'\" for prototype." );
        	return newPrototypeSize( token.substring( 1, token.length() - 1 ) );
        } else
            return Sizes.constant(token, horizontal);
    }

    private static Size newPrototypeSize( String prototype ) {
    	// use reflection for compatibility with JGoodies Forms 1.1 or 1.0
    	try {
			Class<?> cls = Class.forName( "com.jgoodies.forms.layout.PrototypeSize" );
			return (Size) cls.getConstructor( new Class[] { String.class } )
				.newInstance( new Object[] { prototype } );
		} catch( Exception ex ) {
			throw new IllegalArgumentException( "Prototype size specification requires JGoodies Forms 1.2 or later." );
		}
    }


    /**
     * Decodes an encoded resize mode and resize weight and answers
     * the resize weight.
     *
     * @param token	the encoded resize weight
     * @return the decoded resize weight
     * @throws IllegalArgumentException if the string description is an
     *     invalid string representation
     */
    private static double decodeResize(String token) {
        if (token.equals("g") || token.equals("grow")) {
            return FormSpec.DEFAULT_GROW;
        }
        if (token.equals("n") || token.equals("nogrow") || token.equals("none")) {
            return FormSpec.NO_GROW;
        }
        // Must have format: grow(<double>)
        if ((token.startsWith("grow(") || token.startsWith("g("))
             && token.endsWith(")")) {
            int leftParen  = token.indexOf('(');
            int rightParen = token.indexOf(')');
            String substring = token.substring(leftParen + 1, rightParen);
            return Double.parseDouble(substring);
        }
        throw new IllegalArgumentException(
                    "The resize argument '" + token + "' is invalid. " +
                    " Must be one of: grow, g, none, n, grow(<double>), g(<double>)");
    }

    /**
     * Returns a DefaultAlignment that corresponds to the specified
     * string, null if no such alignment exists.
     *
     * @param str	the encoded alignment
     * @param isHorizontal   indicates the values orientation
     * @return the corresponding DefaultAlignment or null
     */
    private static FormSpec.DefaultAlignment valueOf(String str, boolean isHorizontal) {
        if (str.equals("f") || str.equals("fill"))
            return ColumnSpec.FILL;
        else if (str.equals("c") || str.equals("center"))
            return ColumnSpec.CENTER;
        else if (isHorizontal) {
            if (str.equals("r") || str.equals("right"))
                return ColumnSpec.RIGHT;
            else if (str.equals("l") || str.equals("left"))
                return ColumnSpec.LEFT;
            else
                return null;
        } else {
            if (str.equals("t") || str.equals("top"))
                return RowSpec.TOP;
            else if (str.equals("b") || str.equals("bottom"))
                return RowSpec.BOTTOM;
            else
                return null;
        }
    }

    /**
     * Returns an instance of <code>ComponentSize</code> that corresponds
     * to the specified string.
     * @param str   		the encoded component size
     * @return the corresponding ComponentSize or null if none matches
     */
    private static Size valueOf(String str) {
        if (str.equals("m") || str.equals("min"))
            return Sizes.MINIMUM;
        if (str.equals("p") || str.equals("pref"))
            return Sizes.PREFERRED;
        if (str.equals("d") || str.equals("default"))
            return Sizes.DEFAULT;
        else
            return null;
    }

    private static boolean isComponentSize( Size size ) {
    	return size == Sizes.DEFAULT ||
			   size == Sizes.PREFERRED ||
			   size == Sizes.MINIMUM;
    }

    private static boolean isConstant( Size size ) {
    	return size instanceof ConstantSize || isPrototypeSize( size );
    }

    private static boolean isPrototypeSize( Size size ) {
    	if( size == null )
    		return false;
		for( Class<?> cls = size.getClass(); cls != null; cls = cls.getSuperclass() ) {
			if( cls.getName().equals( "com.jgoodies.forms.layout.PrototypeSize" ) )
				return true;
		}
		return false;
    }

    private static boolean isUseLayoutMap() {
    	if( useLayoutMap == null ) {
        	try {
    			Class.forName( "com.jgoodies.forms.layout.PrototypeSize" );
    			useLayoutMap = Boolean.TRUE;
    		} catch( Exception ex ) {
    			useLayoutMap = Boolean.FALSE;
    		}
    	}
    	return useLayoutMap.booleanValue();
    }
}
