/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.beans.PersistenceDelegate;
import java.beans.XMLEncoder;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import com.jformdesigner.model.FormModel;

/**
 * Saves a form model to a JFormDesigner .jfd file.
 * <p>
 * This can be used to convert a proprietary form specification to
 * a JFormDesigner .jfd file. First create a <code>FormModel</code> from
 * your form specification, then save the model to a .jfd file.
 *
 * @author Karl Tauber
 * @since 1.0.1
 */
public class FormSaver
{
	private FormSaver() {
	}

	/**
	 * Saves a form model to the given output stream.
	 * Use this method if you want save a form e.g. to a database.
	 * <p>
	 * A <code>BufferedOutputStream</code> is used to improve performance.
	 *
	 * @param model The form model.
	 * @param out The output stream. Closed when this method returns.
	 * @throws IllegalArgumentException If the output stream is <code>null</code>.
	 * @throws MultiException If a problem occurred when encoding the form model to XML.
	 */
	public static void save( FormModel model, OutputStream out )
		throws MultiException
	{
		if( !(out instanceof ByteArrayOutputStream) &&
			!(out instanceof BufferedOutputStream) )
		  out = new BufferedOutputStream( out );

		if( "IBM J9 VM".equals( System.getProperty( "java.vm.name" ) ) )
			out = new FixCharOutputStream( out );

		if( model.fileHeader != null )
			out = new FileHeaderOutputStream( out, model.fileHeader );

		XMLExceptionListener exceptionListener = new XMLExceptionListener();
		final ClassLoader modelClassLoader = model.getClassLoader();

		Thread currentThread = Thread.currentThread();
		ClassLoader oldContextClassLoader = currentThread.getContextClassLoader();
		currentThread.setContextClassLoader( modelClassLoader );

		try {
			FormXMLPersistence.initialize();
			try {
				FormXMLPersistence.initializePropertyPersistenceDelegates( model.getRoot(), modelClassLoader );
			} catch( Throwable ex ) {
				Exception ex2 = (ex instanceof Exception)
					? (Exception) ex : new InvocationTargetException( ex );
				throw new MultiException( "Failed to initialize property persistence delegates for encoding.", new Exception[] { ex2 } );
			}

			XMLEncoder encoder = new XMLEncoder( out ) {
				@Override
				public PersistenceDelegate getPersistenceDelegate( Class<?> type ) {
					PersistenceDelegate persistenceDelegate = FormXMLPersistence.getPersistenceDelegate( type, modelClassLoader );
					if( persistenceDelegate == null )
						persistenceDelegate = super.getPersistenceDelegate( type );
					return persistenceDelegate;
				}
			};
			encoder.setExceptionListener( exceptionListener );
			encoder.writeObject( model );
			encoder.close();
		} finally {
			currentThread.setContextClassLoader( oldContextClassLoader );

			try {
				out.close();
			} catch( IOException ex ) {
				// ignore
			}
		}

		if( exceptionListener.getExceptionCount() > 0 )
			throw new MultiException( "Failed to encode.", exceptionListener.getExceptions() );
	}

	//---- class FileHeaderOutputStream ---------------------------------------

	private static class FileHeaderOutputStream
		extends FilterOutputStream
	{
		private String fileHeader;

		FileHeaderOutputStream( OutputStream out, String fileHeader ) {
			super( out );
			this.fileHeader = fileHeader;
		}

		@Override
		public void write( int b ) throws IOException {
			super.write( b );

			if( fileHeader != null && b == '\n' ) {
				// write header comment after first newline character
				out.write( "<!--\n".getBytes( "UTF-8" ) );
				out.write( fileHeader.getBytes( "UTF-8" ) );
				out.write( "\n-->\n".getBytes( "UTF-8" ) );
				fileHeader = null;
			}
		}

		@Override
		public void write( byte[] b, int off, int len ) throws IOException {
			if( fileHeader != null ) {
				// header not written --> use write(int)
				super.write( b, off, len );
			} else
				out.write( b, off, len );
		}
	}

	//---- class FileHeaderOutputStream ---------------------------------------

	/**
	 * The IBM J9 VM XMLEncoder does not encode some characters correctly and
	 * has some bugs. E.g. it writes invalid XML "<char><</char>" for '<'.
	 * This output stream fixes this problem on the fly.
	 */
	static class FixCharOutputStream
		extends ByteArrayOutputStream
	{
		private final OutputStream out;

		FixCharOutputStream( OutputStream out ) {
			super( 20*1024 );
			this.out = out;
		}

		@Override
		public void close() throws IOException {
			super.close();

			String s;
			try {
				s = toString( "UTF-8" );
			} catch( UnsupportedEncodingException e ) {
				s = toString();
			}

			// fix invalid character encodings
			s = s.replaceAll( "<char>&</char>", "<char>&amp;</char>" );
			s = s.replaceAll( "<char><</char>", "<char>&lt;</char>" );
			s = s.replaceAll( "<char>></char>", "<char>&gt;</char>" );
			s = s.replaceAll( "<char>\"</char>", "<char>&quot;</char>" );
			s = s.replaceAll( "<char>'</char>", "<char>&apos;</char>" );
			s = s.replaceAll( "<char>\r</char>", "<char>&#13;</char>" );

			// fix invalid access to static class fields
			s = s.replaceAll(
					"<object class=\"([^\"]+)\" method=\"getField\">[\\s]*<string>([^<]+)</string>[\\s]*</object>",
					"<object class=\"$1\" field=\"$2\"/>" );

			byte[] b;
			try {
				b = s.getBytes( "UTF-8" );
			} catch( UnsupportedEncodingException e ) {
				b = s.getBytes();
			}

			try {
				out.write( b );
			} finally {
				out.close();
			}
		}
	}
}
