/*
 * Copyright (C) 2009-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import javax.swing.LayoutFocusTraversalPolicy;

/**
 * A simple focus traversal policy that determines the traversal order based
 * on the given component array.
 * <p>
 * Usage:
 * <pre>
 *   panel.setFocusTraversalPolicy( new ArrayFocusTraversalPolicy( textField2, textField1, checkBox2 ) );
 *   panel.setFocusTraversalPolicyProvider( true );
 * </pre>
 *
 * @author Karl Tauber
 * @since 5.0
*/
public class ArrayFocusTraversalPolicy
	extends LayoutFocusTraversalPolicy
{
	private final Component[] components;

	public ArrayFocusTraversalPolicy( Component... components ) {
		this.components = components;
	}

	@Override
	public Component getComponentAfter( Container container, Component component ) {
		return nextComponent( component, 1 );
	}

	@Override
	public Component getComponentBefore( Container container, Component component ) {
		return nextComponent( component, -1 );
	}

	@Override
	public Component getFirstComponent( Container container ) {
		return nextComponent( components.length - 1, 1 );
	}

	@Override
	public Component getLastComponent( Container container ) {
		return nextComponent( 0, -1 );
	}

	@Override
	public Component getDefaultComponent( Container container ) {
		return getFirstComponent( container );
	}

	private Component nextComponent( Component component, int direction ) {
		int index = indexOf( component );
		return (index >= 0) ? nextComponent( index, direction ) : null;
	}

	private Component nextComponent( int index, int direction ) {
		if( components.length == 0 )
			return null;

		int nextIndex = nextIndex( index, direction );
		int initialIndex = nextIndex;
		do {
			if( accept( components[nextIndex] ) )
				return components[nextIndex];

			nextIndex = nextIndex( nextIndex, direction );
		} while( nextIndex != initialIndex );

		return null;
	}

	private int nextIndex( int index, int direction ) {
		index += direction;
		if( index < 0 )
			index = components.length - 1;
		else if( index >= components.length )
			index = 0;
		return index;
	}

	private int indexOf( Component component ) {
		for( int i = 0; i < components.length; i++ ) {
			if( components[i] == component )
				return i;
		}
		return -1;
	}
}
