/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

/**
 * An event provider allows you to receive events on form model modifications.
 *
 * @author Karl Tauber
 */
public interface FormModelEventProvider
{
	/**
	 * This event indicates that the whole form has been changed.
	 */
	public void fireFormChanged();

	/**
	 * This event indicates that the layout manager of a container has been changed.
	 *
	 * @param container the container where the layout manager has changed
	 * @param oldLayout the old layout manager
	 * @param newLayout the new layout manager
	 */
	public void fireLayoutChanged( FormContainer container,
		FormLayoutManager oldLayout, FormLayoutManager newLayout );

	/**
	 * This event indicates that the layout constraints of a component has been changed.
	 *
	 * @param layout the layout manager where the constraints has changed
	 * @param component the component where the layout constraints has changed
	 * @param oldConstraints the old constraints
	 * @param newCOnstraints the new constraints
	 */
	public void fireConstraintsChanged( FormLayoutManager layout, FormComponent component,
		FormLayoutConstraints oldConstraints, FormLayoutConstraints newCOnstraints );

	/**
	 * This event indicates that a component was added to the form.
	 *
	 * @param component the added component
	 * @param container the container that contains the added component
	 * @param componentIndex the index of the added component within the container
	 * @param constraints the layout constraints of the added component
	 */
	public void fireComponentAdded( FormComponent component, FormContainer container,
		int componentIndex, FormLayoutConstraints constraints );

	/**
	 * This event indicates that a component was removed from the form.
	 *
	 * @param component the removed component
	 * @param container the container that contained the removed component
	 * @param componentIndex the index of the removed component within the container
	 * @param constraints the layout constraints of the removed component
	 */
	public void fireComponentRemoved( FormComponent component, FormContainer container,
		int componentIndex, FormLayoutConstraints constraints );

	/**
	 * This event indicates that a model property value has been changed.
	 *
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 */
	public void fireModelPropertyChanged( String propertyName, int propertyIndex,
		Object oldValue, Object newValue );

	/**
	 * This event indicates that a component field value has been changed
	 * (e.g. {@link FormComponent#FIELD_NAME}).
	 *
	 * @param component the component where the field has changed
	 * @param fieldName the name of the changed field
	 * @param oldValue the old value of the field
	 * @param newValue the new value of the field
	 */
	public void fireComponentFieldChanged( FormComponent component, String fieldName,
		Object oldValue, Object newValue );

	/**
	 * This event indicates that a component property value has been changed.
	 *
	 * @param component the component where the property has changed
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 */
	public void fireComponentPropertyChanged( FormComponent component, String propertyName,
		int propertyIndex, Object oldValue, Object newValue );

	/**
	 * This event indicates that a component auxiliary property value has been changed.
	 *
	 * @param component the component where the auxiliary property has changed
	 * @param propertyName the name of the changed auxiliary property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the auxiliary property
	 * @param newValue the new value of the auxiliary property
	 */
	public void fireAuxiliaryPropertyChanged( FormComponent component, String propertyName,
		int propertyIndex, Object oldValue, Object newValue );

	/**
	 * This event indicates that a layout manager property value has been changed.
	 *
	 * @param layout the layout manager where the property has changed
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 */
	public void fireLayoutPropertyChanged( FormLayoutManager layout, String propertyName,
		int propertyIndex, Object oldValue, Object newValue );

	/**
	 * This event indicates that a layout constraints property value has been changed.
	 *
	 * @param constraints the layout constraints where the property has changed
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 */
	public void fireConstraintsPropertyChanged( FormLayoutConstraints constraints, String propertyName,
	   	int propertyIndex, Object oldValue, Object newValue );

	/**
	 * This event indicates that an event was added to a component.
	 *
	 * @param component the component where the event was added
	 * @param event the added event
	 * @param eventIndex the index of the added event within the component
	 */
	public void fireEventAdded( FormComponent component, FormEvent event, int eventIndex );

	/**
	 * This event indicates that an event was removed from a component.
	 *
	 * @param component the component where the event was removed
	 * @param event the removed event
	 * @param eventIndex the index of the removed event within the component
	 */
	public void fireEventRemoved( FormComponent component, FormEvent event, int eventIndex );

	/**
	 * This event indicates that an event field value has been changed
	 * (e.g. {@link FormEvent#FIELD_HANDLER}).
	 *
	 * @param event the event where the field has changed
	 * @param fieldName the name of the changed field
	 * @param oldValue the old value of the field
	 * @param newValue the new value of the field
	 */
	public void fireEventFieldChanged( FormEvent event, String fieldName, Object oldValue, Object newValue );

	/**
	 * This event indicates that a binding group was added to root.
	 *
	 * @param root the root where the binding was added
	 * @param bindingGroup the added binding group
	 * @param bindingGroupIndex the index of the added binding group within the root
	 * @since 5.0
	 */
	public void fireBindingGroupAdded( FormRoot root, FormBindingGroup bindingGroup, int bindingGroupIndex );

	/**
	 * This event indicates that a binding group was removed from root.
	 *
	 * @param root the root where the binding was removed
	 * @param bindingGroup the removed binding group
	 * @param bindingGroupIndex the index of the removed binding group within the root
	 * @since 5.0
	 */
	public void fireBindingGroupRemoved( FormRoot root, FormBindingGroup bindingGroup, int bindingGroupIndex );

	/**
	 * This event indicates that a binding group property value has been changed.
	 *
	 * @param bindingGroup the binding group where the property has changed
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 * @since 5.0
	 */
	public void fireBindingGroupPropertyChanged( FormBindingGroup bindingGroup,
		String propertyName, int propertyIndex, Object oldValue, Object newValue );

	/**
	 * This event indicates that a binding was added to a binding group.
	 *
	 * @param bindingGroup the binding group where the binding was added
	 * @param binding the added binding
	 * @param bindingIndex the index of the added binding within the binding group
	 * @since 5.0
	 */
	public void fireBindingAdded( FormBindingGroup bindingGroup, FormBinding binding, int bindingIndex );

	/**
	 * This event indicates that a binding was removed from a binding group.
	 *
	 * @param bindingGroup the binding group where the binding was removed
	 * @param binding the removed binding
	 * @param bindingIndex the index of the removed binding within the binding group
	 * @since 5.0
	 */
	public void fireBindingRemoved( FormBindingGroup bindingGroup, FormBinding binding, int bindingIndex );

	/**
	 * This event indicates that a binding property value has been changed.
	 *
	 * @param binding the binding where the property has changed
	 * @param propertyName the name of the changed property
	 * @param propertyIndex the index of the removed property or -1
	 * @param oldValue the old value of the property
	 * @param newValue the new value of the property
	 * @since 5.0
	 */
	public void fireBindingPropertyChanged( FormBinding binding,
		String propertyName, int propertyIndex, Object oldValue, Object newValue );
}
