/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * A form layout manager represents a <code>java.awt.LayoutManager</code> in the form model.
 * It has a layout manager class, layout properties and constraints of child components.
 * <p>
 * Take a look at the example source code (FormSaverExamples.java) for details.
 *
 * @author Karl Tauber
 */
public class FormLayoutManager
	extends FormObject
	implements Iterable<Map.Entry<FormComponent, FormLayoutConstraints>>
{
	private static final EmptyIterator<Map.Entry<FormComponent, FormLayoutConstraints>> EMPTY_ITERATOR
		= new EmptyIterator<Map.Entry<FormComponent, FormLayoutConstraints>>();

	private final Class<?> layoutClass;
	private FormContainer container;

	// Holding the constraints in the layout manager makes it easier to implement
	// Undo/Redo of layout manager switching.
	/** Maps form components to form layout constraints (FormComponent, FormLayoutConstraints). */
	private HashMap<FormComponent, FormLayoutConstraints> constraintsMap;

	/**
	 * Constructs a form layout manager object for the specified layout manager class.
	 */
	public FormLayoutManager( Class<?> managerClass ) {
		this.layoutClass = managerClass;
	}

	FormLayoutManager( FormLayoutManager obj, int dummy ) {
		super( obj, dummy );
		layoutClass = obj.layoutClass;

		// do not copy container and constraints
	}

	/**
	 * Clones this form component.
	 */
	@Override
	public Object clone() {
		return new FormLayoutManager( this, 0 );
	}

	/**
	 * Returns the layout manager class.
	 */
	public Class<?> getLayoutClass() {
		return layoutClass;
	}

	/**
	 * Returns the form layout constraints for the specified form component,
	 * or <code>null</code> the form component has no constraints.
	 */
	public FormLayoutConstraints getConstraints( FormComponent comp ) {
		return (constraintsMap != null) ? constraintsMap.get( comp ) : null;
	}

	/**
	 * Sets the form layout constraints for a form component.
	 */
	public FormLayoutConstraints setConstraints( FormComponent comp, FormLayoutConstraints constraints ) {
		return setConstraints( comp, constraints, true );
	}

	FormLayoutConstraints setConstraints( FormComponent comp, FormLayoutConstraints constraints,
										  boolean fireEvent )
	{
		if( constraintsMap == null ) {
			if( constraints == null )
				return null; // avoid creation of HashMap if all constraints are null

			constraintsMap = new HashMap<FormComponent, FormLayoutConstraints>();
		}

		FormLayoutConstraints oldConstraints;
		if( constraints != null ) {
			oldConstraints = constraintsMap.put( comp, constraints );
			constraints.setLayout( this );
			constraints.setComponent( comp );
		} else
			oldConstraints = constraintsMap.remove( comp );

		if( oldConstraints != null ) {
			oldConstraints.setLayout( null );
			oldConstraints.setComponent( null );
		}

		if( fireEvent ) {
			FormModel model = (container != null) ? container.getModel() : null;
			if( model != null && model.eventProvider != null )
				model.eventProvider.fireConstraintsChanged( this, comp, oldConstraints, constraints );
		}

		return oldConstraints;
	}

	/**
	 * Returns the form container for this object.
	 */
	public FormContainer getContainer() {
		return container;
	}

	void setContainer( FormContainer container ) {
		if( this.container != null && container != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.container + "\", new=\"" + container + "\")." );

		this.container = container;
	}

	/**
	 * Returns an iterator over the constraints in this object.
	 * <code>Iterator.next()</code> returns an <code>java.util.Map.Entry</code>
	 * where the key is the FormComponent and the value is the FormLayoutConstraints.
	 *
	 * @since 3.0
	 */
	public Iterator<Map.Entry<FormComponent, FormLayoutConstraints>> iterator() {
		return (constraintsMap != null && constraintsMap.size() > 0)
			? new MyIterator<Map.Entry<FormComponent, FormLayoutConstraints>>( constraintsMap.entrySet().iterator() )
			: EMPTY_ITERATOR;
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		FormModel model = (container != null) ? container.getModel() : null;
		if( model != null && model.eventProvider != null )
			model.eventProvider.fireLayoutPropertyChanged( this, name, index, oldValue, newValue );
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return unqualifiedClassName( getClass() )
			+ "[layoutClass=" + unqualifiedClassName( layoutClass )
			+ ",container=" + container + "] "
			+ super.toString();
	}
}
