/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

/**
 * A form layout constraints represents a constraints object
 * (e.g. java.awt.GridBagConstraints) in the form model.
 * It has a layout constraints class and constraints properties.
 * <p>
 * Take a look at the example source code (FormSaverExamples.java) for details.
 *
 * @author Karl Tauber
 */
public class FormLayoutConstraints
	extends FormObject
{
	public static final String PROP_VALUE = "value";

	private final Class<?> constraintsClass;

	private FormLayoutManager layout;
	private FormComponent component;

	/**
	 * Constructs a form layout constraints object for the specified class.
	 */
	public FormLayoutConstraints( Class<?> constraintsClass ) {
		this.constraintsClass = constraintsClass;
	}

	/**
	 * Constructs a form layout constraints object using the specified value.
	 * The constraints class is set to the given class and
	 * the value into the property {@link #PROP_VALUE}.
	 *
	 * @since 4.0.9
	 */
	public FormLayoutConstraints( Class<?> constraintsClass, Object value ) {
		this.constraintsClass = constraintsClass;
		setProperty( PROP_VALUE, value );
	}

	FormLayoutConstraints( FormLayoutConstraints obj, int dummy ) {
		super( obj, dummy );
		constraintsClass = obj.constraintsClass;

		// do not copy layout and component
	}

	/**
	 * Clones this form component.
	 */
	@Override
	public Object clone() {
		return new FormLayoutConstraints( this, 0 );
	}

	/**
	 * Returns the layout constraints class.
	 */
	public Class<?> getConstraintsClass() {
		return constraintsClass;
	}

	/**
	 * Returns the form layout manager for this object.
	 */
	public FormLayoutManager getLayout() {
		return layout;
	}

	void setLayout( FormLayoutManager layout ) {
		if( this.layout != null && layout != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.layout + "\", new=\"" + layout + "\")." );

		this.layout = layout;
	}

	/**
	 * Returns the form component for this object.
	 */
	public FormComponent getComponent() {
		return component;
	}

	void setComponent( FormComponent component ) {
		if( this.component != null && component != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.component + "\", new=\"" + component + "\")." );

		this.component = component;
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		FormContainer container = (layout != null) ? layout.getContainer() : null;
		FormModel model = (container != null) ? container.getModel() : null;
		if( model != null && model.eventProvider != null )
			model.eventProvider.fireConstraintsPropertyChanged( this, name, index, oldValue, newValue );
	}

	/**
	 * Compares this constraints with the given constraints.
	 */
	public boolean isEqual( FormLayoutConstraints constraints ) {
		if( constraints == this )
			return true;
		return constraints.constraintsClass == constraintsClass &&
			   super.isEqual( constraints );
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return unqualifiedClassName( getClass() )
			+ "[constraintsClass="
			+ ((constraintsClass != null) ? unqualifiedClassName( constraintsClass ) : "null")
			+ ",component=" + component + "] "
			+ super.toString();
	}
}
