/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.ArrayList;

/**
 * A form component represents a <code>java.awt.Component</code> in the form model.
 * It has a name, a class name and properties.
 * The name is used by the Java code generator as variable name or by the
 * FormLoader to access the Swing component.
 * <p>
 * {@link FormContainer} must be used for container components;
 * {@link FormWindow} for <code>java.awt.Window</code> and derived components;
 * {@link FormNonVisual} for non-visual JavaBeans (not derived from
 * <code>java.awt.Component</code>).
 * <p>
 * Example:
 * <pre>
 *   FormComponent label = new FormComponent("javax.swing.JLabel");
 *   label.setProperty("text", "hello");
 *   label.setProperty("foreground", Color.red);
 * </pre>
 * The same in Swing:
 * <pre>
 *   JLabel label = new JLabel();
 *   label.setText("hello");
 *   label.setForeground(Color.red);
 * </pre>
 *
 * @author Karl Tauber
 */
public class FormComponent
	extends FormObject
	implements FormSelectable
{
	public static final String FIELD_NAME = "name";

	private String name;
	private final String className;

	private FormContainer parent;
	FormModel model;

	private ArrayList<FormEvent> events;
	private FormAuxiliaryProperties auxiliaryProperties;

	/**
	 * Constructs a form component for the specified class.
	 */
	public FormComponent( String className ) {
		this.className = className;
	}

	FormComponent( FormComponent obj, int dummy ) {
		super( obj, dummy );
		name = obj.name;
		className = obj.className;
		if( obj.hasAuxiliary() )
			auxiliaryProperties = new FormAuxiliaryProperties( this, obj.auxiliaryProperties );

		// clone events
		int eventCount = obj.getEventCount();
		for( int i = 0; i < eventCount; i++ )
			addEvent( new FormEvent( obj.getEvent( i ) ) );

		// do not copy parent and model
	}

	/**
	 * Clones this form component.
	 */
	@Override
	public Object clone() {
		return new FormComponent( this, 0 );
	}

	/**
	 * Returns the name of this form component.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the name of this form component.
	 */
	public void setName( String name ) {
		String oldName = this.name;
		this.name = name;

		if( model != null ) {
			if( model.nameRegistry != null )
				model.nameRegistry.componentFieldChanged( this, FIELD_NAME, oldName, name );
			if( model.eventProvider != null )
				model.eventProvider.fireComponentFieldChanged( this, FIELD_NAME, oldName, name );

			if( oldName != null )
				model.updateReferences( oldName, name );
		} else {
			if( oldName != null ) {
				FormComponent root = this;
				while( root.parent != null )
					root = root.parent;
				root.updateReferences( oldName, name );
			}
		}
	}

	/**
	 * Returns the class name of this form component.
	 */
	public String getClassName() {
		return className;
	}

	/**
	 * Returns the parent of this form component.
	 * Or <code>null</code> if the form component has no parent.
	 */
	public FormContainer getParent() {
		return parent;
	}

	/**
	 * Sets the parent of this form component.
	 */
	void setParent( FormContainer parent ) {
		this.parent = parent;
	}

	/**
	 * Returns the form model of this form component.
	 */
	public FormModel getModel() {
		return model;
	}

	/**
	 * Sets the form model of this form component.
	 */
	void setModel( FormModel model ) {
		if( this.model != null && model != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.model + "\", new=\"" + model + "\")." );

		this.model = model;
	}

	/**
	 * Returns a object that contains auxiliary property.
	 * Auxiliary properties are used to store additional information for
	 * a form component (e.g. code generation options).
	 */
	public FormObject auxiliary() {
		if( auxiliaryProperties == null )
			auxiliaryProperties = new FormAuxiliaryProperties( this );

		return auxiliaryProperties;
	}

	/**
	 * Returns whether this form component has auxiliary properties or not.
	 */
	public boolean hasAuxiliary() {
		return auxiliaryProperties != null && auxiliaryProperties.getPropertyCount() > 0;
	}

	/**
	 * Convenience method to get an auxiliary String property value.
	 * Returns <code>def</code> if the auxiliary property is not a String or does not exist.
	 */
	public String getAuxiliaryPropertyString( String name, String def ) {
		if( auxiliaryProperties == null )
			return def;
		return auxiliaryProperties.getPropertyString( name, def );
	}

	/**
	 * Convenience method to get an auxiliary integer property value.
	 * Returns <code>def</code> if the auxiliary property is not an integer or does not exist.
	 */
	public int getAuxiliaryPropertyInt( String name, int def ) {
		if( auxiliaryProperties == null )
			return def;
		return auxiliaryProperties.getPropertyInt( name, def );
	}

	/**
	 * Convenience method to get an auxiliary boolean property value.
	 * Returns <code>def</code> if the auxiliary property is not a boolean or does not exist.
	 */
	public boolean getAuxiliaryPropertyBoolean( String name, boolean def ) {
		if( auxiliaryProperties == null )
			return def;
		return auxiliaryProperties.getPropertyBoolean( name, def );
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		if( model != null && model.eventProvider != null )
			model.eventProvider.fireComponentPropertyChanged( this, name, index, oldValue, newValue );
	}

	/**
	 * Returns the form layout constraints of this form component if the
	 * parent container has a form layout manager which uses constraints.
	 */
	public FormLayoutConstraints getConstraints() {
		if( parent != null ) {
			FormLayoutManager layout = parent.getLayout();
			if( layout != null )
				return layout.getConstraints( this );
		}
		return null;
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormComponentVisitor#visit} is called
	 * with this form component.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormComponentVisitor#visit}.
	 * @since 3.0
	 */
	public boolean accept( FormComponentVisitor visitor ) {
		return visitor.visit( this );
	}

	/**
	 * Returns the number of form events in this form component.
	 *
	 * @since 2.0
	 */
	public int getEventCount() {
		return (events != null) ? events.size() : 0;
	}

	/**
	 * Returns the form event at <code>index</code>.
	 *
	 * @since 2.0
	 */
	public FormEvent getEvent( int index ) {
		return events.get( index );
	}

	/**
	 * Returns all form events in this form component.
	 *
	 * @since 2.0
	 */
	public FormEvent[] getEvents() {
		return (events != null)
			? events.toArray( new FormEvent[events.size()] )
			: FormEvent.EMPTY_EVENTS;
	}

	/**
	 * Adds a form event to the end of this form component.
	 *
	 * @since 2.0
	 */
	public void addEvent( FormEvent event ) {
		addEvent( event, -1 );
	}

	/**
	 * Adds a form event to this form component at the specified position.
	 *
	 * @param event The form event to be added.
	 * @param index The position in the component's event list at which
	 * 		to insert the event; or -1 to insert at the end
	 * @since 2.0
	 */
	public void addEvent( FormEvent event, int index ) {
		if( events == null )
			events = new ArrayList<FormEvent>();

		if( index < 0 || index == events.size() ) {
			index = events.size();
			events.add( event );
		} else
			events.add( index, event );

		event.setComponent( this );

		if( model != null && model.eventProvider != null )
			model.eventProvider.fireEventAdded( this, event, index );
	}

	/**
	 * Removes the specified form event from this form component.
	 *
	 * @since 2.0
	 */
	public void removeEvent( FormEvent event ) {
		if( events == null )
			return;
		int index = events.indexOf( event );
		if( index >= 0 )
			removeEvent( index );
	}

	/**
	 * Removes the form event at the specified index from this form component.
	 *
	 * @since 2.0
	 */
	public void removeEvent( int index ) {
		if( events == null )
			return;

		FormEvent event = events.remove( index );
		event.setComponent( null );

		if( model != null && model.eventProvider != null )
			model.eventProvider.fireEventRemoved( this, event, index );
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return unqualifiedClassName( getClass() )
			+ "[name=" + name + ",className=" + className + "] "
			+ super.toString();
	}
}
