/*
 * Copyright (C) 2008-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.Map;

/**
 * A form binding represents a single binding in the form model.
 * Usually a <code>org.jdesktop.beansbinding.Binding</code>.
 *
 * @author Karl Tauber
 * @since 5.0
 */
public class FormBinding
	extends FormObject
{
	public static final String PROP_SOURCE = "source";
	public static final String PROP_SOURCE_PATH = "sourcePath";
	public static final String PROP_TARGET = "target";
	public static final String PROP_TARGET_PATH = "targetPath";

	private FormBindingGroup bindingGroup;

	public FormBinding() {
	}

	public FormBinding( String source, String sourcePath, String target, String targetPath ) {
		setProperty( PROP_SOURCE, source );
		setProperty( PROP_SOURCE_PATH, sourcePath );
		setProperty( PROP_TARGET, target );
		setProperty( PROP_TARGET_PATH, targetPath );
	}

	FormBinding( FormBinding obj ) {
		super( obj, 0 );
	}

	/**
	 * Clones this form binding.
	 */
	@Override
	public Object clone() {
		return new FormBinding( this );
	}

	public String getSource() {
		return getPropertyString( PROP_SOURCE );
	}

	public String getSourcePath() {
		return getPropertyString( PROP_SOURCE_PATH );
	}

	public String getTarget() {
		return getPropertyString( PROP_TARGET );
	}

	public String getTargetPath() {
		return getPropertyString( PROP_TARGET_PATH );
	}

	/**
	 * Returns the form binding group of this form binding.
	 */
	public FormBindingGroup getBindingGroup() {
		return bindingGroup;
	}

	/**
	 * Sets the form binding group of this form binding.
	 */
	void setBindingGroup( FormBindingGroup bindingGroup ) {
		if( this.bindingGroup != null && bindingGroup != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.bindingGroup + "\", new=\"" + bindingGroup + "\")." );

		this.bindingGroup = bindingGroup;
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormBindingVisitor#visit} is called
	 * with this form binding.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormBindingVisitor#visit}.
	 */
	public boolean accept( FormBindingVisitor visitor ) {
		return visitor.visit( this );
	}

	private FormModelEventProvider getEventProvider() {
		return (bindingGroup != null) ? bindingGroup.getEventProvider() : null;
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		FormModelEventProvider eventProvider = getEventProvider();
		if( eventProvider != null )
			eventProvider.fireBindingPropertyChanged( this, name, index, oldValue, newValue );
	}

	@Override
	void updateReferences( String oldName, String newName ) {
		super.updateReferences( oldName, newName );

		if( oldName.equals( getSource() ) )
			setProperty( PROP_SOURCE, newName );
		if( oldName.equals( getTarget() ) )
			setProperty( PROP_TARGET, newName );

		// update references in properties of type FormBinding[]
		for( Map.Entry<String, Object> entry : properties() ) {
			Object value = entry.getValue();
			if( !(value instanceof FormBinding[]) )
				continue;

			updateReferencesInSubBindings( entry.getKey(),
				(FormBinding[]) value, oldName, newName );
		}
	}

	private void updateReferencesInSubBindings( String name,
		FormBinding[] subBindings, String oldName, String newName )
	{
		FormBinding[] newSubBindings = null;
		for( int i = 0; i < subBindings.length; i++ ) {
			FormBinding subBinding = subBindings[i];
			int refCount = subBinding.getReferenceCount();
			if( refCount == 0 )
				continue;

			for( Map.Entry<String, Object> subEntry : subBinding.properties() ) {
				Object subValue = subEntry.getValue();

				if( subValue instanceof FormReference ) {
					if( oldName.equals( ((FormReference)subValue).getName() ) ) {
						if( newSubBindings == null )
							newSubBindings = clone( subBindings );
						newSubBindings[i].setProperty( subEntry.getKey(),
							(newName != null) ? new FormReference( newName ) : null );
					}

					if( --refCount <= 0 )
						break;
				}
			}
		}

		if( newSubBindings != null )
			setProperty( name, newSubBindings );
	}

	public static FormBinding[] clone( FormBinding[] bindings ) {
		FormBinding[] newBindings = new FormBinding[bindings.length];
		for( int i = 0; i < bindings.length; i++ )
			newBindings[i] = (FormBinding) bindings[i].clone();
		return newBindings;
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return unqualifiedClassName( getClass() )
			+ " "
			+ super.toString();
	}
}
