/* $Id: MsqlStatement.java,v 2.6 1999/07/06 05:50:58 borg Exp $ */
/* Copyright  1997-1998 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.IOException;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * The MsqlStatement class implements the JDBC Statement interface.
 * This class represents any SQL statement.  This class does minimal
 * locking designed simply around its asynchronous result set processing.
 * Applications should make sure either that multiple threads do not
 * share a Statement or that they do careful synchronization.  When
 * synchronizing on the statement, be careful not to grab any long-held
 * locks on it or you might prevent data loading.
 * <BR>
 * Last modified $Date: 1999/07/06 05:50:58 $
 * @version $Revision: 2.6 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlStatement implements Statement {
    // a list of batch statements to be executed
    private   ArrayList        batches             = new ArrayList();
    // the number of columns in the most recently retrieved result set
    private   int              columnCount         = -1;
    // JDBC 2.0 ResultSet concurrency
    private   int              concurrency         =ResultSet.CONCUR_READ_ONLY;
    // The connection that owns this Statement
    private   MsqlConnection   connection          = null;
    // The direction in which results may be fetched
    private   int              fetchDirection      = ResultSet.FETCH_FORWARD;
    // This statement has finished loading the data associated with its SQL
    private   boolean          loaded              = true;
    // The logging object
    protected MsqlLog          log                 = null;
    // The maximum field size for data of certain SQL types
    private   int              maxFieldSize        = 0;
    // The maximum number of rows to be fetched for each query
    private   int              maxRows             = 0;
    // A list of result sets returned by the last query
    private   ArrayList        resultSets          = new ArrayList();
    // The number of rows affected by the last SQL call
    private   int              updateCount         = -1;
    // JDBC 2.0 result set type
    private   int              type                = ResultSet.TYPE_FORWARD_ONLY;

    /**
     * Constructs a new MsqlStatement that is forward-only and read-only.
     * @param c the MsqlConnection that created this statement
     */
    MsqlStatement(MsqlConnection c) {
	this(c, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY, 0);
    }

    /**
     * Constructs a new MsqlStatement.
     * @param c the MsqlConnection that created this statement
     * @param t the result set type, for example, forward-only
     * @param concur the result set concurrency
     */
    MsqlStatement(MsqlConnection c, int t, int concur, int ll) {
	super();
	connection = c;
	type = t;
	concurrency = concur;
	log = new MsqlLog(ll, this);
    }

    /**
     * Adds a SQL string to the list of SQL statements that should be
     * executed as part of the next batch operation.  These statements
     * should only ever be updates.
     * @param sql the SQL statement to be added to the batch
     * @exception java.sql.SQLException this is never thrown
     */
    public void addBatch(String sql) throws SQLException {
	log.log("addBatch()", MsqlLog.JDBC, "Adding batch: \"" + sql + "\"");
	batches.add(sql);
    }
    
    /**
     * mSQL statement processing must run through to its natural conclusion.
     * This method thus simply waits until it is safe to use this Statement
     * for other things.  In other words, its behaviour is consistent with
     * JDBC compliant drivers but it is not exactly what the JDBC authors had
     * in mind for a cancel operation.
     * @exception java.sql.SQLException this is never thrown
     */
    public void cancel() throws SQLException {
	log.log("cancel()", MsqlLog.JDBC, "Cancelling.");
	closeAllResults();
    }

    /**
     * Clears the batch list.
     * @exception java.sql.SQLException this is never thrown
     */
    public void clearBatch() throws SQLException {
	log.log("clearBatch()", MsqlLog.JDBC, "Clearing batches.");
	batches.clear();
    }
  
    /**    
     * mSQL does not generated any warnings, so this method is always a
     * NO-OP.
     * @exception java.sql.SQLException this is never thrown
     */
    public void clearWarnings() throws SQLException {
	log.log("clearWarnings()", MsqlLog.JDBC, "Clearing warnings.");
    }

    /**
     * Closes the statement.  This method will wait until all results
     * have been loaded from the database in order to avoid screwing up
     * the mSQL protocol.
     * @exception java.sql.SQLException this should never actually be thrown
     */
    public void close() throws SQLException {
	log.log("close()", MsqlLog.JDBC, "Closing statement.");
	closeAllResults();
	log.close();
    }
  
    /**
     * Closes all the result sets associated with this statement.  This is
     * called by the close() method.
     */
    private synchronized void closeAllResults() {
	Iterator sets;

	while( !loaded ) {
	    log.log("closeAllResults()", MsqlLog.DRIVER,
		    "Waiting for load to complete before " +
		    "closing results.");
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	log.log("closeAllResults()", MsqlLog.DRIVER,
		"Closing all result sets.");
	sets = resultSets.iterator();	
	while( sets.hasNext() ) {
	    MsqlResultSet rslt = (MsqlResultSet)sets.next();

	    try {
		rslt.close();
	    }
	    catch( SQLException e ) {
		log.log("closeAllResults()", MsqlLog.ERROR,
			"Failed to closed result set: " + e.getMessage());
	    }
	}
    }

    synchronized void completeLoad() {
	log.log("completeLoad()", MsqlLog.DRIVER,"Load of results completed.");
	connection.release();
	loaded = true;
	notifyAll();
    }
	
    /**
     * Sends some random SQL to the database
     * @param sql the SQL to be sent to the database
     * @return true if the SQL produced a result set
     * @exception java.sql.SQLException an error occurred talking to the
     * database
     */
    public boolean execute(String sql) throws SQLException {
	ResultSet r = null;

	reset();
	log.log("execute()", MsqlLog.JDBC, "Executing: \"" + sql + "\"");
	connection.capture();
	synchronized( this ) {
	    loaded = false;
	}
	if( !sendSQL(sql) ) {
	    completeLoad();
	    return false;
	}
	else {
	    MsqlResultSet rslt;

	    // the result set is responsible for completing the load
	    rslt = new MsqlQueryData(this, columnCount, log.getLevel());
	    synchronized( resultSets ) {
		resultSets.add(rslt);
	    }
	    return true;
	}
    }

    /**
     * Executes the batch statements added to this Statement via
     * addBatch().
     * @return an array of update counts for each batch
     * @exception java.sql.SQLException one or more of the batch processes
     * failed
     */
    public int[] executeBatch() throws SQLException {
	int[] counts;
	
	log.log("executeBatch()", MsqlLog.JDBC, "Executing current batch.");
	counts = new int[batches.size()];
	for(int i=0; i<batches.size(); i++) {
	    String sql = (String)batches.get(i);

	    try {
		counts[i] = executeUpdate(sql);
	    }
	    catch( SQLException e ) {
		BatchUpdateException bue;

		bue = new BatchUpdateException(e.getMessage(), e.getSQLState(),
					       e.getErrorCode(), counts);
		bue.setNextException(e);
		throw bue;
	    }
	}
	return counts;
    }
    
    /**
     * Executes a query and returns a result set matching the type and
     * concurrency characteristics of this Statement.
     * @param sql the SQL to send to the database
     * @return a ResultSet matching the query
     * @exception java.sql.SQLException an erroroccurred talking to the
     * database
     */
    public synchronized ResultSet executeQuery(String sql)
    throws SQLException {
	reset();
	log.log("executeQuery()", MsqlLog.JDBC,
		"Executing query: \"" + sql + "\"");
	connection.capture();
	synchronized( this ) {
	    loaded = false;
	}
	if( sendSQL(sql) ) {
	    MsqlResultSet rslt;

	    rslt = new MsqlQueryData(this, columnCount, log.getLevel());
	    synchronized( resultSets ) {
		resultSets.add(rslt);
		return rslt;
	    }
	}
	else {
	    log.log("executeQuery()", MsqlLog.ERROR,
		    "Query returned no results.");
	    completeLoad();
	    throw new MsqlException("Non-query sent to executeQuery().");
	}
    }

    /**
     * Sends an UPDATE to the database.
     * @param sql the UPDATE SQL
     * @return the numberof affected rows
     * @exception java.sql.SQLException an error occurred talking to the
     * database
     */
    public int executeUpdate(String sql) throws SQLException {
	reset();
	log.log("executeUpdate()", MsqlLog.JDBC,
		"Executing update: \"" + sql + "\"");
	connection.capture();
	synchronized( this ) {
	    loaded = false;
	}
	if( !sendSQL(sql) ) {
	    return updateCount;
	}
	else {
	    log.log("executeUpdate()", MsqlLog.ERROR,
		    "Update returned result sets.");
	    // we have to go ahead and load this or the driver will be
	    // in an inconsistent state
	    new MsqlQueryData(this, columnCount, log.getLevel());
	    throw new MsqlException("Query sent to executeUpdate().");
	}
    }

    /**
     * This method has changed between mSQL-JDBC 1.x and 2.x to reflect
     * the fact that this is actually a JDBC method as of JDBC 2.0.  This
     * change should have no effect on even the most poorly designed
     * users of mSQL-JDBC.
     * @return the MsqlConnection that owns this Statement
     */
    public Connection getConnection() {
	return connection;
    }

    /**
     * @return the encoding for this application
     */
    public String getEncoding() {
	return connection.getEncoding();
    }
    
    /**
     * @return the direction in which results might be fetched
     * @exception java.sql.SQLException this is never thrown
     */
    public int getFetchDirection() throws SQLException {
	return fetchDirection;
    }

    /**
     * mSQL-JDBC always fetches all rows.  This is required by the mSQL
     * network protocol and cannot be modified.
     * @return 0
     * @exception java.sql.SQLException this is never thrown
     */
    public int getFetchSize() throws SQLException {
	return 0;
    }
    
    /**
     * @return the maximum field size allowed for BINARY, VARBINARY,
     * LONGVARBINARY, CHAR, VARCHAR, and LONGVARCHAR columns
     * @exception java.sql.SQLException this is never thrown
     */
    public int getMaxFieldSize() throws SQLException {
	return maxFieldSize;
    }

    /**
     * @return the maximum number of rows for a result set created by
     * this statement
     * @exception java.sql.SQLException this is never thrown
     */
    public int getMaxRows() throws SQLException {
	return maxRows;
    }

    /**
     * @return true if there are more results to process
     * @exception java.sql.SQLException this will not be thrown
     */
    public boolean getMoreResults() throws SQLException {
	return (resultSets.size() > 0);
    }

    /**
     * The query timeout is the number of seconds a statement should
     * wait before aborting the execution of a statement.  If the limit
     * is encountered, processing ends and an exception is thrown.
     * @return the query timeout
     * @exception java.sql.SQLException this is never thrown
     */
    public int getQueryTimeout() throws SQLException {
	return 0;
    }

    /**
     * Call this method after using the execute() method to send SQL
     * to the database and finding that a result set was created.
     * Watch your synchronization in this event.  If you call execute()
     * and then another thread uses this Statement to issue a new query
     * before you  call getResultSet(), you will find no results.
     * @return a result set matching the most recent query
     * @exception java.sql.SQLException no result set was waiting
     */
    public ResultSet getResultSet() throws SQLException {
	ResultSet rslt = (ResultSet)resultSets.get(0);
	
	resultSets.remove(0);
	return rslt;
    }

    /**
     * JDBC 2.0 result set concurrency.  This determines whether or
     * not a result set generated by this statement can be updated in
     * place.
     * @return the result set concurrency
     * @exception java.sql.SQLException this is never thrown
     */
    public int getResultSetConcurrency() throws SQLException {
	return concurrency;
    }
    
    /**
     * JDBC 2.0 result set type.  This is the kind of result set that will
     * be generated by this statement.
     * @return the result set type
     * @exception java.sql.SQLException this is never actually thrown
     */
    public int getResultSetType() throws SQLException {
	return type;
    }
    
    /**
     * Call this method after using the execute() method to send SQL
     * to the database and finding that rows were updated.  This method
     * will tell you how many rows were modified.
     * Watch your synchronization in this event.  If you call execute()
     * and then another thread uses this Statement to issue new SQL
     * before you  call getUpdateCount(), you will get a meaningless answer.
     * @return the number of rows affected by the last SQL sent to execute()
     */
    public int getUpdateCount() throws SQLException {
	int x = updateCount;

	updateCount = -1;
	return x;
    }

    /**
     * mSQL does not generate warnings, so this method always returns null.
     * @return null
     * @exception java.sql.SQLException this is never thrown
     */
    public final SQLWarning getWarnings() throws SQLException {
	return null;
    }

    /**
     * Resets the Statement in order to execute new SQL.
     */
    private synchronized void reset() {
	log.log("reset()", MsqlLog.DRIVER, "Resetting.");
	while( !loaded ) {
	    try { resultSets.wait(); }
	    catch( InterruptedException e ) { }
	}
	try { clearWarnings(); }
	catch( SQLException e ) { e.printStackTrace(); }
	resultSets.clear();
	updateCount = -1;
	columnCount = -1;
    }
    
    /**
     * Sends a SQL string to the database.  If this Statement is currently
     * processing some other SQL, it will not execute until that processing
     * is complete.
     * <P>
     * <I>Note:</I> Anyone calling this method is responsible for making
     * sure loaded = false and calling connection.capture().
     * @param sql the SQL to be passed to the database
     * @return true if the SQL resulted in a result set
     * @exception java.sql.SQLException some error occurred talking to the
     * database
     */
    private boolean sendSQL(String sql) throws SQLException {
	int i, rowcol;
	String tmp;

	log.log("sendSQL()", MsqlLog.MSQL, "Sending SQL to server.");
	try {
	    connection.getOutputStream().writeString("3 " +sql, getEncoding());
	}
	catch( IOException e ) {
	    log.log("sendSQL()", MsqlLog.ERROR, "Failed to send " +
		    "SQL to server: " + e.getMessage());
	    try {
		connection.close();
	    }
	    catch( Exception salt ) {
		salt.printStackTrace();
	    }
	    throw new MsqlException(e);
	}
	try {
	    tmp = connection.getInputStream().readString(getEncoding());
	}
	catch( IOException e ) {
	    log.log("sendSQL()", MsqlLog.ERROR, "Failed to receive " +
		    "SQL response: " + e.getMessage());
	    try {
		connection.close();
	    }
	    catch( Exception salt ) {
		salt.printStackTrace();
	    }
	    throw new MsqlException(e);
	}
	i = tmp.indexOf(':');
	if( i == -1 ) {
	    log.log("sendSQL()", MsqlLog.ERROR,
		    "Response from server did not make sense.");
	    completeLoad();
	    throw new MsqlException("Incorrect mSQL response.");
	}
	rowcol = Integer.parseInt(tmp.substring(0, i));
	if( rowcol == -1 ) {
	    log.log("sendSQL()", MsqlLog.ERROR,
		    "Server reported an error: " + tmp.substring(2));
	    completeLoad();
	    throw new MsqlException(tmp.substring(2));
	}
	i = tmp.indexOf(':', i+1);
	if( i == -1 ) {
	    log.log("sendSQL()", MsqlLog.DRIVER,
		    "Server modified " + rowcol + " rows.");
	    completeLoad();
	    updateCount = rowcol;
	    return false;
	}
	else {
	    try {
		rowcol = Integer.parseInt(tmp.substring(2, i));
	    }
	    catch( NumberFormatException e ) {
		log.log("sendSQL()", MsqlLog.DRIVER,
			"I forget what this means.");
		completeLoad();
		updateCount = -1;
		columnCount = -1;
		return false;
	    }
	    log.log("sendSQL()", MsqlLog.DRIVER,
		    "Statement required a result set with " +
		    rowcol + " columns.");
	    columnCount = rowcol;
	    return true;
	}
    }

    /**
     * This is a NO-OP since mSQL queries must run to their conclusion.
     * @param unused the number of seconds the Statement should wait
     * @exception java.sql.SQLException this is never thrown
     */
    public void setQueryTimeout(int unused) throws SQLException {
	log.log("setQueryTimeout()", MsqlLog.JDBC,
		"Setting query timeout to " + unused + ".");
    }

    /**
     * This method is required by the JDBC 2.0 specification but is not
     * a supported feature of mSQL.  It is therefore a NO-OP as specified
     * by the JDBC 2.0 specification.
     * @param unused the name of the cursor
     * @exception java.sql.SQLException this is never thrown
     */
    public void setCursorName(String unused) throws SQLException {
	log.log("setCursorName()", MsqlLog.JDBC,
		"Setting cursor name to " + unused + ".");
    }

    /**
     * Toggles on and off escape substitution before sending SQL to the
     * database.  The driver does not yet support this.
     * @param enable indicate whether to enable or disable escape processing
     * @exception java.sql.SQLException escape processing not supported
     */
    public void setEscapeProcessing(boolean enable) throws SQLException {
	log.log("setEscapeProcessing()", MsqlLog.JDBC,
		"Setting escape processing to " + enable + ".");
	if( enable ) {
	    log.log("setEscapeProcessing()", MsqlLog.ERROR,
		    "Escape processing is not yet supported.");
	    throw new SQLException("Escape processing is not yet supported.");
	}
    }

    /**
     * Provide a hint to mSQL-JDBC as to which direction results should
     * be fetched int.
     * @param dir the direction
     * @exception java.sql.SQLException this is never thrown
     */
    public void setFetchDirection(int dir) throws SQLException {
	log.log("setFetchDirection()", MsqlLog.JDBC,
		"Setting fetch direction to " + dir + ".");
	fetchDirection = dir;
    }

    /**
     * This is a NO-OP since no matter what you suggest, mSQL-JDBC
     * is forced by the mSQL network protocol to download all rows.
     * @param unused the suggested fetch size
     * @exception java.sql.SQLException this is never thrown
     */
    public void setFetchSize(int unused) throws SQLException {
	log.log("setFetchSize()", MsqlLog.JDBC,
		"Setting fetch size to " + unused + ".");
    }
    
    /**
     * Sets the maximum field size for certain data types.  If data
     * of that type is retrieved from the database and it exceeds this
     * value, the excess data will be silently discarded.
     * @param max the maximum field size
     * @exception java.sql.SQLException this is never thrown
     */
    public void setMaxFieldSize(int max) throws SQLException {
	log.log("setMaxFieldSize()", MsqlLog.JDBC,
		"Setting max field size to " + max + ".");
	maxFieldSize = max;
    }

    /**
     * Sets the maximum number of rows to be returned by results generated
     * by this statement.  It will silently discard excess rows.
     * @param max the maximum number of rows
     * @exception java.sql.SQLException this is never thrown
     */
    public void setMaxRows(int max) throws SQLException {
	log.log("setMaxRows()", MsqlLog.JDBC,
		"Setting max rows to " + max + ".");
	maxRows = max;
    }
}
