/* $Id: MsqlResultSetMetaData.java,v 2.3 1999/07/06 05:50:57 borg Exp $ */
/* Copyright (c) 1997-1998 George Reese */
package com.imaginary.sql.msql;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * The MsqlResultSetMetaData class is the mSQL implementation of the
 * JDBC ResultSetMetaData interface.  It should never be referenced directly.
 * You should instead use the JDBC API.<BR>
 * Last modified $Date: 1999/07/06 05:50:57 $
 * @version $Revision: 2.3 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlResultSetMetaData implements ResultSetMetaData {
    private String    catalog     = null;
    private int       columnCount = -1;
    private ArrayList metaData    = new ArrayList();

    /**
     * Constructs a new MsqlResultSetMetaData instance for the specified
     * result set.
     * @param f the supported result set
     */
    MsqlResultSetMetaData(String cat, ArrayList md) throws SQLException {
	super();
	catalog = cat;
	loadColumns(md.iterator());
    }

    /**
     * This operation is not supported.
     * @param column the number of the desired column
     * @return the catalog name for the requested column
     * @exception java.sql.SQLException this is always thrown
     */
    public String getCatalogName(int column) throws SQLException {
	return catalog;
    }

    /**
     * @return the column data for the specified column
     * @exception java.sql.SQLException an attempt was made to check an
     * invalid column
     */
    private MsqlColumn getColumn(int column) throws SQLException {
	return (MsqlColumn)metaData.get(column-1);
    }

    /**
     * UDTs are not supported by mSQL.  This will always return null.
     * @param column the number of the desired column
     * @return the fully qualified Java class name for the column map
     * @exception java.sql.SQLException this is never thrown.
     */
    public String getColumnClassName(int column) throws SQLException {
	return null;
    }
    
    /**
     * @return the number of columns in the result set
     * @exception java.sql.SQLException this is never thrown
     */
    public int getColumnCount() throws SQLException {
	return columnCount;
    }

    /**
     * @param column the column being queried
     * @return the display size of the column
     * @exception java.sql.SQLException an error occurred reading the column
     * attributes
     */
    public int getColumnDisplaySize(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	return c.getLength();
    }

    /**
     * mSQL-JDBC 2.0 behaviour change! This method now returns the column
     * label without the table name prefixed onto it.  In mSQL-JDBC 1.x
     * the table name was prepended.  You can retain the old behaviour
     * by passing the command line property -Dmsql.oldlabels=1.
     * For mSQL, this is the same as getColumnName().
     * @param column the column being queried
     * @return the column label
     * @exception java.sql.SQLException an error occurred reading the
     * column attributes
     */
    public String getColumnLabel(int column) throws SQLException {
	MsqlColumn c = getColumn(column);
	String old = System.getProperty("msql.oldlabels", "0");
	
	if( old.equals("1") ) { 
	    return (c.getTableName() + "." + c.getColumnName());
	}
	else {
	    return c.getColumnName();
	}
    }

    /**
     * @param column the desired column number
     * @return the name of the specified column
     * @exception java.sql.SQLException an error occurred reading the
     * column attributes
     */
    public String getColumnName(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	return c.getColumnName();
    }

    /**
     * @param column the number of the desired column
     * @return the type from java.sql.Types of this column
     * @exception java.sql.SQLException a error occurred reading the
     * column attributes
     */ 
   public int getColumnType(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	return c.getSQLType();
    }

    /**
     * @param column the column number of the desired column
     * @return the type name for the SQL type of the specified column
     * @exception java.sql.SQLException a error occurred reading the
     * column attributes
     */
    public String getColumnTypeName(int column) throws SQLException {
	int t = getColumn(column).getMsqlType();
	
	return MsqlTypes.getMsqlTypeNameForMsql(t);
    }

    /**
     * This is a total hack for mSQL, as it does not inherently have any
     * concept of precision or scale.
     * @param column the column whose scale is desired
     * @return the length of the column (a total hack)
     * @exception java.sql.SQLException an error occurred reading the
     * column attributes
     */
    public int getPrecision(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	return c.getLength();
    }

    /**
     * This is a total hack for mSQL, as it does not inherently have any
     * concept of precision or scale.
     * @param column the column whose scale is desired
     * @return 2 for REAL or DOUBLE, 0 for others
     * @exception java.sql.SQLException this is never thrown
     */
    public int getScale(int column) throws SQLException {
	switch( getColumnType(column) ) {
	case Types.REAL: case Types.DOUBLE:
	    return 2;
	default:
	    return 0;
	}
    }

    /**
     * mSQL does not have schemas.  This method returns "".
     * @param column the desired column
     * @return ""
     * @exception java.sql.SQLException this is never thrown
     */
    public String getSchemaName(int column) throws SQLException {
	return "";
    }

    /**
     * @param column the desired column number
     * @return the name of the table from which this column was drawn
     * @exception java.sql.SQLException an error occurred reading the column
     * attributes
     */
    public String getTableName(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	return c.getTableName();
    }

    /**
     * @param column the column being checked
     * @return true if the column will automatically increment on insert
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isAutoIncrement(int column) throws SQLException { 
	return false;
    }

    /**
     * @param column the column being queried
     * @return always true for mSQL
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isCaseSensitive(int column) throws SQLException {
	return true;
    }

    /**
     * @param column the column being queried
     * @return true if the field type is <CODE>MsqlTypes.MONEY</CODE>
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isCurrency(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	if( c.getMsqlType() == MsqlTypes.MONEY ) {
	    return true;
	}
	else {
	    return false;
	}
    }

    /**
     * This always returns true for the lack of anything better to do.
     * @param column the column being queried
     * @return true
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isDefinitelyWritable(int column) throws SQLException {
	return true;
    }

    /**
     * @param column the column being checked
     * @return true if the specified column can be nullified
     * @exception java.sql.SQLException this is never thrown
     */
    public int isNullable(int column) throws SQLException {
	MsqlColumn c = getColumn(column);
	
	if( c.isNullable() ) {
	    return ResultSetMetaData.columnNullable;
	}
	else {
	    return ResultSetMetaData.columnNoNulls;
	}
    }

    /**
     * mSQL does not support read-only mode.
     * @param column the column being queried
     * @return false
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isReadOnly(int column) throws SQLException {
	return false;
    }
    
    /**
     * @param column the column to be queried
     * @return true if the column can appear in a WHERE clause
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isSearchable(int column) throws SQLException {
	return true;
    }

    /**
     * @param column the column to be queried
     * @return true if the column data is a signed value
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isSigned(int column) throws SQLException {
	MsqlColumn c = getColumn(column);

	if( c.getMsqlType() != MsqlTypes.UINT ) {
	    return true;
	}
	else {
	    return false;
	}
    }

    /**
     * This is always true for mSQL columns.
     * @param column the column being queried
     * @return true
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isWritable(int column) throws SQLException {
	return true;
    }

    /**
     * Loads the columns for the meta data for the result set's columns
     * from an <CODE>Iterator</CODE> containing <CODE>ParsedRow</CODE>
     * objects that represent each column's data.
     * @param cols an <CODE>Iterator</CODE> of <CODE>ParsedRow</CODE>
     * objects, one element for each column
     * @throws java.sql.SQLException an error occurred parsing one of
     * the <CODE>ParsedRow</CODE> data.
     */
    private void loadColumns(Iterator cols) throws SQLException {
	columnCount = 0;
	while( cols.hasNext() ) {
	    ParsedRow row = (ParsedRow)cols.next();

	    columnCount++;
	    metaData.add(new MsqlColumn(catalog, row));
	}
    }
}

