/* $Id: ColumnResultsRow.java,v 2.1 1999/07/06 05:50:52 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Iterator;

/**
 * This class represents the odd unification of JDBC column meta-data
 * requirements and the format in which mSQL presents them via the
 * mSQL socket protocol.
 * <BR>
 * Last modified $Date: 1999/07/06 05:50:52 $ 
 * @version $Revision: 2.1 $
 * @author George Reese (borg@imaginary.com)
 */
public class ColumnResultsRow implements ParsedRow {
    // the JDBC catalog/mSQL database
    private String         catalog     = null;
    // the row of data from mSQL
    private ParsedRow      internalRow = null;

    /**
     * Constructs a new column results row using the specified
     * <CODE>ParsedRow</CODE> to represent the data coming from mSQL
     * about the specified table.
     * @param cat the name of the catalog for the meta data 
     * @param pr the row of data from mSQL
     */
    public ColumnResultsRow(String cat, ParsedRow pr) {
	super();
	catalog = cat;
	internalRow = pr;
    }

    /**
     * Provides an <CODE>Iterator</CODE> to help move through the columns.
     * @return the columns as an <CODE>Iterator</CODE>
     * @throws java.sql.SQLException an error occurred parsing data from mSQL
     */
    public Iterator columns() throws SQLException {
	return internalRow.columns();
    }

    /**
     * Retrieves the specified column as a <CODE>String</CODE>.
     * @param col the desired column
     * @return the value of the column as a <CODE>String</CODE>
     * @throws java.sql.SQLException an error occurred parsing data from mSQL
     */
    public String get(int col) throws SQLException {
	switch( col ) {
	case 0:
	    // catalog
	    {
		return catalog;
	    }
	    
	case 1:
	    // schema
	    {
		return null;
	    }
	    
	case 2:
	    // table name
	    {
		return internalRow.get(0);
	    }

	case 3:
	    // column name
	    {
		return internalRow.get(1);
	    }

	case 4:
	    // data type
	    {
		String ts = internalRow.get(2);
		int t = Integer.parseInt(ts);
		
		t = MsqlTypes.getSQLTypeForMsql(t);
		return (t + "");
	    }

	case 5:
	    // type name
	    {
		String ts = internalRow.get(2);
		int t = Integer.parseInt(ts);

		return MsqlTypes.getMsqlTypeNameForMsql(t);
	    }

	case 6:
	    // column size
	    {
		return internalRow.get(3);
	    }

	case 7:
	    // buffer length ---+++ UNUSED BY JDBC CURRENTLY +++---
	    {
		return null;
	    }

	case 8:
	    // decimal digits
	    {
		String ts = internalRow.get(2);
		int t = Integer.parseInt(ts);

		if( t == MsqlTypes.MONEY ) {
		    return "2";
		}
		else if( t != MsqlTypes.REAL ) {
		    return "0";
		}
		else {
		    // this is so arbitrary, but mSQL does not provide
		    // this information
		    return "5";
		}
	    }

	case 9:
	    // NUM_PREC_RADIX
	    {
		return "10";
	    }

	case 10:
	    // NULLABLE
	    {
		String nn = internalRow.get(4);

		if( nn.equalsIgnoreCase("N") ) {
		    return (DatabaseMetaData.columnNullable + "");
		}
		else if( nn.equalsIgnoreCase("Y") ) {
		    return (DatabaseMetaData.columnNoNulls + "");
		}
		else {
		    return (DatabaseMetaData.columnNullableUnknown + "");
		}
	    }

	case 11:
	    // remarks
	    {
		return "Eh?";
	    }

	case 12:
	    // column default
	    {
		return null;
	    }

	case 13:
	    // sql data type ---+++ UNUSED BY JDBC CURRENTLY +++---
	    {
		return null;
	    }

	case 14:
	    // sql datetime sub ---+++ UNUSED BY JDBC CURRENTLY +++---
	    {
		return null;
	    }

	case 15:
	    // char octet length
	    {
		// mSQL is single-byte only
		return internalRow.get(3);
	    }

	case 16:
	    // ordinal position
	    {
		//mSQL provides no way for finding this value
		// without doing a SELECT * FROM table
		return "0";
	    }

	case 17:
	    // is nullable
	    {
		String nn = internalRow.get(4);

		// mSQL answers the question "Is this field NOT nullable"?
		// JDBC is asking "Is this field nullable?"
		// Thus, "yes" from mSQL means "no" to JDBC
		if( nn.equalsIgnoreCase("N") ) {
		    return "YES";
		}
		else if( nn.equalsIgnoreCase("Y") ) {
		    return "NO";
		}
		else {
		    return "";
		}
	    }

	default:
	    {
		return null;
	    }
	}
    }

    /**
     * Provides a quick method for determining if this row contains
     * the specified column number.
     * @param col the desired column
     * @return true if the column exists, false otherwise
     */
    public boolean hasColumn(int col) {
	return (col < 18);
    }

    /**
     * This method always throws an exception since meta data result
     * sets are most definitely not updatable.
     * @param col unused
     * @param val unused
     * @throws java.sql.SQLException thrown any time this method is called
     */
    public void set(int col, String val) throws SQLException {
	throw new MsqlException("This result set may not be modified.");
    }

    /**
     * @return 18
     */
    public int size() {
	return 18;
    }
}
