/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glemainwindow.h"
#include	"gle.h"
#include	"gleprivate.h"
#include	"glewidget.h"
#include	"gleselector.h"
#include	"glemenus.h"
#include	"gleeditor.h"
#include	<gdk/gdk.h>



enum
{
  CLIST_COL_TYPE,
  CLIST_COL_STATE,
  CLIST_COL_REFCOUNT,
  CLIST_COL_GLENAME,
  CLIST_COL_NAME,
  CLIST_N_COLS
};


/* --- defines --- */
#define	GLE_MAIN_WINDOW_DATA(mw)	(gtk_object_get_data (GTK_OBJECT ((mw)), gle_key_main_window_data))
#define	INITIAL_CLIST_COLUMN_WIDTH	(50)


/* --- typedefs --- */
typedef	struct _GleMainWindowData	GleMainWindowData;
struct	_GleMainWindowData
{
  GleCustomerInfo	*customer_info;
  GList			*customer_info_list	/* of type GleCustomerInfo* */;

  /* widgets */
  GtkWidget		*label;
  GtkWidget		*clist;
  GtkWidget		*update;
};


/* --- prototypes --- */
extern GleMainWindow*	_gle_get_main_window		(void);
static void		gle_main_window_create		(void);
static void		gle_main_window_set_customer	(GtkWidget	*customer);
static gint		gle_main_window_delete_event	(GtkWidget	*widget);
static void		gle_main_window_destroy		(GtkWidget	*widget,
							 gpointer	 func_data);
static void		gle_main_window_update_tree	(GleMainWindow	*main);
static void		gle_clist_select_row	(GtkCList	*clist,
						 gint		 row,
						 gint		 column,
						 GdkEventButton	*event,
						 gpointer	 func_data);
static void	gle_main_window_customer_destroy	(GtkWidget	*customer,
							 GleMainWindow	*main_window);


/* --- variables --- */
static const gchar	*gle_key_main_window_data = "gle-main-window-data";
static GleMainWindow	*main_window = NULL;
static gchar *clist_indent = NULL;
static const gchar	*key_contents_alive_object= "gle-contents-alive-object";


/* --- functions --- */
static void
gle_select_customer (GtkWidget *widget,
		     gpointer   func_data)
{
  GtkWidget *selector;
  GtkWidget *customer;
  
  gle_main_window_set_customer (NULL);

  selector = gle_selector_setup (NULL);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (main_window),
					 "hide",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (selector));
  customer = gle_selector_get_widget (gle_key_main_window, NULL);
  gle_main_window_set_customer (customer);
}

GleMainWindow*
gle_main_window_popup (GtkWidget	 *customer)
{
  if (customer)
    g_return_val_if_fail (GTK_IS_WIDGET (customer), NULL);
  
  if (!main_window)
    gle_main_window_create ();
  
  if (!GTK_WIDGET_VISIBLE (main_window))
    gtk_widget_show (GTK_WIDGET (main_window));
  else
    gdk_window_raise (GTK_WIDGET (main_window)->window);

  if (gdk_pointer_is_grabbed ())
    gle_select_customer (NULL, NULL);
  else
    gle_main_window_set_customer (customer);

  return main_window;
}

GleMainWindow*
_gle_get_main_window (void)
{
  if (!main_window)
    gle_main_window_create ();

  return main_window;
}

void
gle_widget_set_life_time (GtkWidget      *widget)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_OBJECT (widget));
  
  gtk_signal_connect_object_while_alive (GTK_OBJECT (_gle_get_main_window ()),
					 "hide",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (widget));
}

static void
gle_main_window_set_customer (GtkWidget *customer)
{
  GleMainWindowData *data;

  data = GLE_MAIN_WINDOW_DATA (main_window);

  if (!main_window)
    gle_main_window_create ();

  if (customer)
    {
      g_return_if_fail (customer->parent == NULL);

      if (gle_object_has_gle_tag (GTK_OBJECT (customer)))
	customer = NULL;
    }
  
  if (customer)
    {
      GleCustomerInfo *info;
      GList *list;
      gchar *string;
      
      list = data->customer_info_list;
      while (list)
	{
	  info = list->data;
	  if (info->customer == customer)
	    break;
	  list = list->next;
	}
      if (!list)
	{
	  info = g_new (GleCustomerInfo, 1);
	  info->customer = GTK_WIDGET (customer);
	  data->customer_info_list = g_list_prepend (data->customer_info_list, info);
	  gtk_signal_connect_object_while_alive (GTK_OBJECT (customer),
						 "destroy",
						 GTK_SIGNAL_FUNC (gle_main_window_customer_destroy),
						 GTK_OBJECT (main_window));
	}
      data->customer_info = info;
      
      string = g_strconcat (gtk_type_name (GTK_OBJECT (customer)->klass->type),
			    "::",
			    GTK_IS_WINDOW (customer) && GTK_WINDOW (customer)->title ?
			    GTK_WINDOW (customer)->title :
			    "Unknown",
			    NULL);
      gtk_label_set (GTK_LABEL (data->label), string);
      g_free (string);
      gle_widget_make_sensitive (data->update);
    }
  else
    {
      gtk_label_set (GTK_LABEL (data->label), "<None>");
      data->customer_info = NULL;
      gle_widget_make_insensitive (data->update);
    }

  gle_main_window_update_tree (main_window);
}

static void
gle_clist_remove_contents_alive_object (GtkCList *clist)
{
  GtkObject *contents_alive_object;

  contents_alive_object = gtk_object_get_data (GTK_OBJECT (clist), key_contents_alive_object);
  gtk_object_remove_data (GTK_OBJECT (clist), key_contents_alive_object);

  gtk_clist_freeze (clist);
  gtk_object_sink (contents_alive_object);
  gtk_clist_thaw (clist);
}

static void
gle_main_window_create (void)
{
  GtkWidget *window;
  GtkWidget *window_vbox;
  GtkWidget *main_vbox;
  GtkWidget *status_hbox;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *any;
  GleMainWindowData *data;
  gchar *titles[CLIST_N_COLS];
  gchar *string;
  guint i;

  data = g_new (GleMainWindowData, 1);
  data->customer_info = NULL;
  data->customer_info_list = NULL;

  /* main window
   */
  string = g_strconcat ("GLE Engine", "  [", gle_prg_name (), "]", NULL);
  window	= gtk_widget_new (gtk_window_get_type (),
				  "GtkWidget::name", "GLE-Main-Window",
				  "GtkWindow::title", string,
				  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
				  "GtkObject::signal::delete_event", gle_main_window_delete_event, NULL,
				  "GtkObject::signal::destroy", gle_main_window_destroy, NULL,
				  "GtkWindow::allow_shrink", TRUE,
				  "GtkWindow::allow_grow", TRUE,
				  "GtkWindow::auto_shrink", FALSE,
				  NULL);
  main_window = GLE_MAIN_WINDOW (window);
  g_free (string);
  main_window = (GleMainWindow*) window;
  gle_object_set_gle_tag (GTK_OBJECT (main_window));
  window_vbox	= gtk_widget_new (gtk_vbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 0,
				  "GtkContainer::border_width", 0,
				  "GtkWidget::parent", window,
				  "GtkWidget::visible", TRUE,
				  NULL);
  main_vbox	= gtk_widget_new (gtk_vbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 5,
				  "GtkWidget::parent", window_vbox,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_set_child_packing (GTK_BOX (window_vbox), main_vbox, TRUE, TRUE, 0, GTK_PACK_START);
  status_hbox	= gtk_widget_new (gtk_hbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 5,
				  "GtkWidget::parent", main_vbox,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), status_hbox, FALSE, TRUE, 0, GTK_PACK_START);
  data->label	= gtk_widget_new (gtk_label_get_type (),
				  "GtkLabel::label", "<CustomWidget>",
				  "GtkWidget::parent", main_vbox,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), data->label, FALSE, TRUE, 0, GTK_PACK_START);
  titles[CLIST_COL_TYPE] = "Type";
  titles[CLIST_COL_STATE] = "State";
  titles[CLIST_COL_REFCOUNT] = "RefCount";
  titles[CLIST_COL_GLENAME] = "GleName";
  titles[CLIST_COL_NAME] = "Name";
  data->clist	= gtk_clist_new_with_titles (CLIST_N_COLS, titles);
  gtk_widget_set		(data->clist,
				 "GtkWidget::width", 4 * INITIAL_CLIST_COLUMN_WIDTH + 10,
				 "GtkWidget::height", 200,
				 "GtkObject::signal::select_row", gle_clist_select_row, window,
				 "GtkObject::signal::destroy", gle_clist_remove_contents_alive_object, NULL,
				 "GtkWidget::visible", TRUE,
				 NULL);
  gtk_object_set_data (GTK_OBJECT (data->clist),
		       key_contents_alive_object,
		       gtk_object_new (gtk_object_get_type (), NULL));
  gtk_clist_set_policy (GTK_CLIST (data->clist), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_selection_mode (GTK_CLIST (data->clist), GTK_SELECTION_BROWSE);
  gtk_clist_column_titles_passive (GTK_CLIST (data->clist));
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (data->clist), i, INITIAL_CLIST_COLUMN_WIDTH);
  gtk_box_pack_start (GTK_BOX (main_vbox), data->clist, TRUE, TRUE, 0);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), data->clist, TRUE, TRUE, 0, GTK_PACK_START);
  hbox		= gtk_widget_new (gtk_hbox_get_type (),
				  "GtkBox::homogeneous", TRUE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 0,
				  "GtkWidget::parent", main_vbox,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0, GTK_PACK_START);
  data->update	= gtk_widget_new (gtk_button_get_type (),
				  "GtkButton::label", "Update Tree",
				  "GtkObject::object_signal::clicked", gle_main_window_update_tree, window,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_pack_start (GTK_BOX (hbox), data->update, FALSE, TRUE, 0);
  button	= gtk_widget_new (gtk_button_get_type (),
				  "GtkButton::label", "Select Window",
				  "GtkObject::object_signal::clicked", gle_select_customer, window,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  any		= gtk_widget_new (gtk_hseparator_get_type (),
				  "GtkWidget::parent", window_vbox,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_set_child_packing (GTK_BOX (window_vbox), any, FALSE, TRUE, 0, GTK_PACK_START);
  button	= gtk_widget_new (gtk_button_get_type (),
				  "GtkButton::label", "Close",
				  "GtkWidget::visible", TRUE,
				  "GtkWidget::can_default", TRUE,
				  "GtkWidget::has_default", TRUE,
				  "GtkObject::object_signal::clicked", gtk_widget_destroy, window,
				  NULL);
  gtk_box_pack_start (GTK_BOX (window_vbox), button, FALSE, TRUE, 0);

  gtk_object_set_data (GTK_OBJECT (window), gle_key_main_window_data, data);
}

static void
gle_clist_remove_row (GtkWidget *widget,
		      GtkCList  *clist)
{
  gint row;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_WIDGET (widget));
  g_return_if_fail (clist != NULL);
  g_return_if_fail (GTK_IS_CLIST (clist));

  row = gtk_clist_find_row (clist, widget);
  if (row > 0)
    gtk_clist_remove (clist, row);
}

static void
gle_clist_append_widget (GtkWidget  *widget,
			 GtkCList   *clist)
{
  gint row;
  gchar *text[CLIST_N_COLS];
  const gchar *s;
  guint i;

  text[CLIST_COL_TYPE] = g_strconcat (clist_indent,
				      gtk_type_name (GTK_OBJECT (widget)->klass->type),
				      NULL);
  text[CLIST_COL_REFCOUNT] = g_new (gchar, 32);
  sprintf (text[CLIST_COL_REFCOUNT], "%04d", GTK_OBJECT (widget)->ref_count);
  s = gle_widget_get_glename (widget);
  text[CLIST_COL_GLENAME] = g_strconcat ("\"",
					 s ? s : "",
					 "\"",
					 NULL);
  text[CLIST_COL_NAME] = g_strconcat ("\"",
				      widget->name ? widget->name : "",
				      "\"",
				      NULL);
  switch (widget->state)
    {
    case  GTK_STATE_ACTIVE:		text[CLIST_COL_STATE] = g_strdup ("(x) Active"); break;
    case  GTK_STATE_PRELIGHT:		text[CLIST_COL_STATE] = g_strdup ("(x) Prelight"); break;
    case  GTK_STATE_SELECTED:		text[CLIST_COL_STATE] = g_strdup ("(x) Selected"); break;
    case  GTK_STATE_INSENSITIVE:	text[CLIST_COL_STATE] = g_strdup ("(x) Insensitive"); break;
    default:				text[CLIST_COL_STATE] = g_strdup ("(x) Normal"); break;
    }
  text[CLIST_COL_STATE][1] = GTK_WIDGET_VISIBLE (widget) ? 'V' : 'H';
  for (i = 0; i < CLIST_N_COLS; i++)
    {
      gint new_width;

      new_width = gdk_text_width (GTK_WIDGET (clist)->style->font, text[i], strlen (text[i]));
      if (clist->column[i].width < new_width)
	gtk_clist_set_column_width (clist, i, new_width);
    }

  row = gtk_clist_append (clist, text);
  gtk_clist_set_row_data (GTK_CLIST (clist), row, widget);
  gtk_signal_connect_while_alive (GTK_OBJECT (widget),
				  "destroy",
				  GTK_SIGNAL_FUNC (gle_clist_remove_row),
				  clist,
				  gtk_object_get_data (GTK_OBJECT (clist),
						       key_contents_alive_object));
  for (i = 0; i < CLIST_N_COLS; i++)
    g_free (text[i]);

  if (GTK_IS_CONTAINER (widget))
    {
      gchar *old_indent;

      old_indent = clist_indent;
      clist_indent = g_strconcat (old_indent, "	 ", NULL);
      gtk_container_foreach (GTK_CONTAINER (widget),
			     (GtkCallback) gle_clist_append_widget,
			     clist);
      g_free (clist_indent);
      clist_indent = old_indent;
    }
}

static void
gle_main_window_update_tree (GleMainWindow *main_window)
{
  GleMainWindowData *data;
  GleCustomerInfo *info;
  GtkObject *contents_alive_object;
  guint i;

  g_return_if_fail (main_window != NULL);
  g_return_if_fail (GLE_IS_MAIN_WINDOW (main_window));
  data = GLE_MAIN_WINDOW_DATA (main_window);

  info = data->customer_info;

  gtk_clist_freeze (GTK_CLIST (data->clist));
  gtk_clist_clear (GTK_CLIST (data->clist));
  contents_alive_object = gtk_object_get_data (GTK_OBJECT (data->clist), key_contents_alive_object);
  gtk_object_sink (contents_alive_object);
  contents_alive_object = gtk_object_new (gtk_object_get_type (), NULL);
  gtk_object_set_data (GTK_OBJECT (data->clist),
		       key_contents_alive_object,
		       contents_alive_object);
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (data->clist), i, INITIAL_CLIST_COLUMN_WIDTH);
  if (info)
    {
      clist_indent = g_strdup ("");
      gle_clist_append_widget (info->customer, (gpointer) data->clist);
      g_free (clist_indent);
      clist_indent = NULL;
    }

  gtk_clist_thaw (GTK_CLIST (data->clist));
}

static void
gle_main_window_customer_destroy (GtkWidget	*customer,
				  GleMainWindow	*main_window)
{
  GleMainWindowData *data;
  GleCustomerInfo *info = NULL;
  GList *list;

  g_return_if_fail (main_window != NULL);
  g_return_if_fail (GLE_IS_MAIN_WINDOW (main_window));
  data = GLE_MAIN_WINDOW_DATA (main_window);

  list = data->customer_info_list;
  while (list)
    {
      info = list->data;
      if (info->customer == GTK_WIDGET (customer))
	break;
      list = list->next;
    }
  if (list) /* FIXME:   g_assert (list); */
    {
      data->customer_info_list = g_list_remove (data->customer_info_list, info);
      if (data->customer_info == info)
	gle_main_window_set_customer (NULL);
      g_free (info);
    }
}

static gint
gle_main_window_delete_event	(GtkWidget	*widget)
{
  gle_set_flash_widget (NULL);

  gtk_widget_hide (widget);

  return TRUE;
}

static void
gle_main_window_destroy (GtkWidget      *widget,
			 gpointer        func_data)
{
  GleMainWindowData *data;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GLE_IS_MAIN_WINDOW (widget));
  data = GLE_MAIN_WINDOW_DATA (widget);

  gle_set_flash_widget (NULL);

  main_window = NULL;
}

static void
gle_clist_select_row (GtkCList	     *clist,
		      gint	      row,
		      gint	      column,
		      GdkEventButton *event,
		      gpointer	      func_data)
{
  GleMainWindow *main_window;
  GleMainWindowData *data;

  main_window = func_data;
  g_return_if_fail (main_window != NULL);
  g_return_if_fail (GLE_IS_MAIN_WINDOW (main_window));
  data = GLE_MAIN_WINDOW_DATA (main_window);

  if (row >= 0)
    {
      gle_set_flash_widget (NULL);

      if (event)
	{
	  if (event->type == GDK_BUTTON_PRESS)
	    {
	      if (event->button == 2)
		{
		  GtkWidget *widget;
		  
		  widget = gtk_clist_get_row_data (clist, row);
		  widget = gle_editor_popup (widget);
		  if (widget)
		    gtk_signal_connect_object_while_alive (GTK_OBJECT (main_window),
							   "hide",
							   GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   GTK_OBJECT (widget));
		}
	      else if (event->button == 3)
		{
		  GtkWidget *widget;
		  GtkWidget *menu;
		  
		  widget = gtk_clist_get_row_data (clist, row);
		  menu = gle_menu_widegt_ops_popup (widget, event->x_root, event->y_root, event->time);
		  gle_object_set_gle_tag (GTK_OBJECT (menu));
		}
	    }
	  else if (event->type == GDK_2BUTTON_PRESS)
	    {
	      if (column == CLIST_COL_REFCOUNT &&
		  event->button == 1 &&
		  (event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_CONTROL_MASK))
		{
		  GtkWidget *widget;
		  GtkCListRow *clist_row;
		  
		  widget = gtk_clist_get_row_data (clist, row);
		  if (event->state & GDK_SHIFT_MASK)
		    gtk_object_ref (GTK_OBJECT (widget));
		  else
		    gtk_object_unref (GTK_OBJECT (widget));
		  clist_row = (g_list_nth (clist->row_list, row))->data;
		  sprintf (GTK_CELL_TEXT (clist_row->cell[CLIST_COL_REFCOUNT])->text,
			   "%04d",
			   GTK_OBJECT (widget)->ref_count);
		  gtk_widget_draw (GTK_WIDGET (clist), NULL);
		  if (widget)
		    gtk_signal_connect_object_while_alive (GTK_OBJECT (main_window),
							   "hide",
							   GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   GTK_OBJECT (widget));
		}
	    }
	}
    }
}
