// Copyright (C) 1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

//
// Written by Gabriel Dos Reis <gdr@egcs.cygnus.com>
// 
// Note: This program outputs speciliazations of ISO C++ class template
// numeric_limits<> as described in 18.2.1.
// Do not compile with optimization turned on.
//

#include <limits.h>
#include <float.h>
#include <stdio.h>
#include <signal.h>
#include <setjmp.h>
#include <math.h>

const char tab[] = "    ";
const char tab2[] = "        ";
const char* bool_alpha[] = { "false", "true" };
const double log10_of_two = .30102999566398119;
const int bits_per_byte = CHAR_BIT;
const int integer_base_rep = 2;

jmp_buf env;

void signal_handler(int sig) { longjmp(env, sig); }

template<typename Operation>
bool trapping(const Operation& op)
{
    if (setjmp(env) == 0) op();
    else return true;
    return false;
}

template<typename T> struct division_by_zero {
    void operator() () const
    {
        volatile T zero = T();
        volatile T one = T(1);
        volatile T infinity = one / zero;
    }
};

template<typename T> struct overflow {
    void operator() () const
    {
        T i = T(1);
        T j = T();
        while (i>j) {
            j = i;
            i = i * 2 + 1;
        }
    }
};

template<typename T> struct underflow {};

// traps
template<typename T> void traps()
{
    signal(SIGFPE, signal_handler);
    bool trap_flag = trapping(division_by_zero<T>());
    signal(SIGFPE, signal_handler);
    trap_flag = trap_flag && trapping(overflow<T>());
    const char* p = bool_alpha[trap_flag];
    printf("%s%s = %s;\n", tab2, "static const bool traps", p);    
}

#define SPECIALIZE_TRAPPING(T)                                          \
template<> void traps< T >()                                            \
{                                                                       \
    signal(SIGFPE, signal_handler);                                     \
    const char* p = bool_alpha[trapping(division_by_zero<T>())];        \
    printf("%s%s = %s;\n", tab2, "static const bool traps", p);         \
}

SPECIALIZE_TRAPPING(unsigned char);
SPECIALIZE_TRAPPING(unsigned short);
SPECIALIZE_TRAPPING(unsigned int);
SPECIALIZE_TRAPPING(unsigned long);

#undef SPECIALIZE_TRAPPING

template<typename T> struct type_name_trait;

#define DEFINED_TYPE_NAME(T)                                            \
template<> struct type_name_trait< T > {                                \
        static const char type_name[] = #T;                             \
        static const char trait_name[] = "numeric_limits<" #T ">";      \
};                                                                      \
const char type_name_trait< T >::type_name[];                           \
const char type_name_trait< T >::trait_name[];

DEFINED_TYPE_NAME(bool);
DEFINED_TYPE_NAME(char);
DEFINED_TYPE_NAME(signed char);
DEFINED_TYPE_NAME(unsigned char);
DEFINED_TYPE_NAME(wchar_t);
DEFINED_TYPE_NAME(short);
DEFINED_TYPE_NAME(unsigned short);
DEFINED_TYPE_NAME(int);
DEFINED_TYPE_NAME(unsigned int);
DEFINED_TYPE_NAME(long);
DEFINED_TYPE_NAME(unsigned long);
DEFINED_TYPE_NAME(float);
DEFINED_TYPE_NAME(double);
DEFINED_TYPE_NAME(long double);

#undef DEFINED_TYPE_NAME

// declarator
template<typename T> struct declarator : type_name_trait<T> {
    static void start()
    {
        printf("%s%s %s %s\n", tab, "template<> struct", trait_name, "{");
    }

    static void end()
    {
        printf("%s};\n\n", tab);
    }
};

// is_signed
template<typename T> bool is_signed_p() { return T(-1) < 0; }

template<typename T> void is_signed()
{
    const char* p = bool_alpha[is_signed_p<T>()];
    printf("%s%s = %s;\n", tab2, "static const bool is_signed", p);
}

// is_modulo
template<typename T> void is_modulo()
{
    const char* p = bool_alpha[! is_signed_p<T>()];
    printf("%s%s = %s;\n", tab2, "static const bool is_modulo", p);
}

// min/max traits
template<typename T> struct min_max_trait {};

#define DEFINE_MIN_MAX(T, m, M)  DO_DEFINE_MIN_MAX(T, m, M)
#define DO_DEFINE_MIN_MAX(T, m, M)                                      \
template<> struct min_max_trait< T > {                                  \
   static void min()                                                    \
   {                                                                    \
       printf("%s%s\n%s%s\n", tab2, "static " #T " min() throw()",      \
       tab2, "{ return " #m "; }");                                     \
   }                                                                    \
                                                                        \
   static void max()                                                    \
   {                                                                    \
       printf("%s%s\n%s%s\n", tab2, "static " #T " max() throw()",     \
       tab2, "{ return " #M "; }");                                     \
   }                                                                    \
};

DEFINE_MIN_MAX(bool, false, true);
DEFINE_MIN_MAX(char, CHAR_MIN, CHAR_MAX);
DEFINE_MIN_MAX(signed char, SCHAR_MIN, SCHAR_MAX);
DEFINE_MIN_MAX(unsigned char, 0, UCHAR_MAX);
DEFINE_MIN_MAX(short, SHRT_MIN, SHRT_MAX);
DEFINE_MIN_MAX(unsigned short, 0, USHRT_MAX);
DEFINE_MIN_MAX(int, INT_MIN, INT_MAX);
DEFINE_MIN_MAX(unsigned int, 0, UINT_MAX);
DEFINE_MIN_MAX(long, LONG_MIN, LONG_MAX);
DEFINE_MIN_MAX(unsigned long, 0, ULONG_MAX);
DEFINE_MIN_MAX(float, FLT_MIN, FLT_MAX);
DEFINE_MIN_MAX(double, DBL_MIN, DBL_MAX);
DEFINE_MIN_MAX(long double, LDBL_MIN, LDBL_MAX);

#undef DO_DEFINE_MIN_MAX
#undef DEFINE_MIN_MAX

// is_integer/is_exact
template<typename T> struct exact_trait {
    static void is_integer()
    {
        printf("%s%s;\n", tab2, "static const bool is_integer = true");        
    }

    static void is_exact()
    {
        printf("%s%s;\n", tab2, "static const bool is_exact = true");
    }
};

#define SPECIALISE_EXACT_TRAIT(T)                                       \
template<> struct exact_trait< T > {                                    \
    static void is_integer()                                            \
    {                                                                   \
        printf("%s%s;\n", tab2, "static const bool is_integer = false");\
    }                                                                   \
                                                                        \
    static void is_exact()                                              \
    {                                                                   \
        printf("%s%s;\n", tab2, "static const bool is_exact = false");  \
    }                                                                   \
};

SPECIALISE_EXACT_TRAIT(float);
SPECIALISE_EXACT_TRAIT(double);
SPECIALISE_EXACT_TRAIT(long double);

#undef SPECIALISE_EXACT_TRAIT

// precision traits
template<typename T> struct precision_trait {
    static int digits_val()
    {
        int sign_bit = is_signed_p<T>();
        return bits_per_byte * sizeof(T) - sign_bit;
    }
    static void digits()
    {
        printf("%s%s = %d;\n", tab2, "static const int digits",
               digits_val());
    }

    static void digits10()
    {
        printf("%s%s = %d;\n", tab2, "static const int digits10",
               1 + int(log10_of_two * digits_val()));
    }

    // Note: this is always 2 for integer types according to 3.9.1/7
    //       18.2.1.2/18 seems to imply it may be anything else.
    //       Needs clarification.
    static void radix()
    {
        printf("%s%s = %d;\n", tab2, "static const int radix",
               integer_base_rep);
    }

    static void epsilon()
    {
        printf("%s%s %s %s\n%s%s\n", tab2, "static",
               declarator<T>::type_name, "epsilon() throw()",
               tab2, "{ return 0; }");
    }

    static void round_error()
    {
        printf("%s%s %s %s\n%s%s\n", tab2, "static",
               declarator<T>::type_name, "round_error() throw()",
               tab2, "{ return 0; }");
    }
};

#define DEFINE_FLOATING_PRECISION(T, D, D10, FR, E, R)                  \
        DO_DEFINE_FLOATING_PRECISION(T, D, D10, FR, E, R)
#define DO_DEFINE_FLOATING_PRECISION(T, D, D10, FR, E, R)               \
template<> struct precision_trait< T > {                                \
    static void digits()                                                \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int digits", #D);    \
    }                                                                   \
                                                                        \
    static void digits10()                                              \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int digits10", #D10);\
    }                                                                   \
                                                                        \
    static void radix()                                                 \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int radix", #FR);    \
    }                                                                   \
                                                                        \
    static void epsilon()                                               \
    {                                                                   \
        printf("%s%s %s %s\n%s%s\n", tab2, "static", #T ,               \
               "epsilon() throw()", tab2, "{ return " #E "; }");        \
    }                                                                   \
                                                                        \
    static void round_error()                                           \
    {                                                                   \
        printf("%s%s %s %s\n%s%s\n", tab2, "static", #T ,               \
               "round_error() throw()", tab2, "{ return " #R "; }");    \
    }                                                                   \
};


// XXX: find a means to return an appropriate round_error
DEFINE_FLOATING_PRECISION(float, FLT_MANT_DIG, FLT_DIG,
                          FLT_RADIX, FLT_EPSILON, 1.0f);
DEFINE_FLOATING_PRECISION(double, DBL_MANT_DIG, DBL_DIG,
                          FLT_RADIX, DBL_EPSILON, 1.0);
DEFINE_FLOATING_PRECISION(long double, LDBL_MANT_DIG, LDBL_DIG,
                          FLT_RADIX, LDBL_EPSILON, 1.0L);

#undef DO_DEFINE_FLOATING_PRECISION
#undef DEFINE_FLOATING_PRECISION

// exponents traits
template<typename T> struct exponent_trait {
    static void min_exponent()
    {
        printf("%s%s;\n", tab2, "static const int min_exponent = 0");
    }
    static void min_exponent10()
    {
        printf("%s%s;\n", tab2, "static const int min_exponent10 = 0");
    }
    static void max_exponent()
    {
        printf("%s%s;\n", tab2, "static const int max_exponent = 0");
    }
    static void max_exponent10()
    {
        printf("%s%s;\n", tab2, "static const int max_exponent10 = 0");        
    }
};

#define DEFINE_EXPONENT(T, m, m10, M, M10)                              \
        DO_DEFINE_EXPONENT(T, m , m10, M, M10)
#define DO_DEFINE_EXPONENT(T, m , m10, M, M10)                          \
template<> struct exponent_trait< T > {                                 \
    static void min_exponent()                                          \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int min_exponent", #m);\
    }                                                                   \
    static void min_exponent10()                                        \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int min_exponent10", \
               #m10);                                                   \
    }                                                                   \
    static void max_exponent()                                          \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int max_exponent", #M);\
    }                                                                   \
    static void max_exponent10()                                        \
    {                                                                   \
        printf("%s%s = %s;\n", tab2, "static const int max_exponent10", \
               #M10);                                                   \
   }                                                                   \
};

DEFINE_EXPONENT(float, FLT_MIN_EXP, FLT_MIN_10_EXP,
                FLT_MAX_EXP, FLT_MAX_10_EXP);
DEFINE_EXPONENT(double, DBL_MIN_EXP, DBL_MIN_10_EXP,
                DBL_MAX_EXP, DBL_MAX_10_EXP);
DEFINE_EXPONENT(long double, LDBL_MIN_EXP, LDBL_MIN_10_EXP,
                LDBL_MAX_EXP, LDBL_MAX_10_EXP);

#undef DO_DEFINE_EXPONENT
#undef DEFINE_EXPONENT

// inifinity traits





template<typename T> struct infinity_trait {
    static void has_infinity()
    {
        printf("%s%s;\n", tab2, "static const bool has_infinity = false");
    }

    static void has_quiet_NaN()
    {
        printf("%s%s;\n", tab2, "static const bool "
               "has_quiet_NaN = false");
    }

    static void has_signaling_NaN()
    {
        printf("%s%s;\n", tab2, "static const bool "
               "has_signaling_NaN = false");
    }

    static void has_denorm()
    {
        printf("%s%s;\n", tab2, "static const float_denorm_style "
               "has_denorm = denorm_absent");
    }

    static void has_denorm_loss()
    {
        printf("%s%s;\n", tab2, "static const bool "
               "has_denorm_loss = false");
    }

    static void infinity()
    {
        printf("%s%s %s %s\n%s%s %s%s\n", tab2, "static",
               declarator<T>::type_name, "infinity() throw()",
               tab2, "{ return static_cast<", declarator<T>::type_name, 
	       ">(0); }");
    }

    static void quiet_NaN()
    {
        printf("%s%s %s %s\n%s%s %s%s\n", tab2, "static",
               declarator<T>::type_name, "quiet_NaN() throw()",
               tab2, "{ return static_cast<", declarator<T>::type_name, 
	       ">(0); }");
    }

    static void signaling_NaN()
    {
        printf("%s%s %s %s\n%s%s %s%s\n", tab2, "static",
               declarator<T>::type_name, "signaling_NaN() throw()",
               tab2, "{ return static_cast<", declarator<T>::type_name, 
	       ">(0); }");
    }

    static void denorm_min()
    {
        printf("%s%s %s %s\n%s%s %s%s\n", tab2, "static",
               declarator<T>::type_name, "denorm_min() throw()",
               tab2, "{ return static_cast<", declarator<T>::type_name, 
	       ">(0); }");
    }
};


// iec559 compliance
template<typename T> void is_iec559()
{
    printf("%s%s;\n", tab2, "static const bool is_iec559 = false");
}

#define SPECIALIZE_IS_IEC559(T)                                         \
template<> void is_iec559< T >()                                        \
{                                                                       \
    printf("%s%s;\n", tab2, "static const bool is_iec559 = true");      \
}

SPECIALIZE_IS_IEC559(bool);
SPECIALIZE_IS_IEC559(int);
SPECIALIZE_IS_IEC559(unsigned int);
SPECIALIZE_IS_IEC559(long);
SPECIALIZE_IS_IEC559(unsigned long);

#undef SPECIALIZE_IS_IEC559


// tinyness_before
template<typename T> void tinyness_before()
{
    printf("%s%s;\n", tab2, "static const bool tinyness_before = false");
}

// round style
template<typename T> void round_style()
{
    printf("%s%s;\n", tab2, "static const float_round_style "
           "round_style = round_toward_zero");    
}



// type traits
template<typename T> struct type_trait {
    
    type_trait()
    {
        declarator<T>::start();
        printf("%s%s;\n\n", tab2, "static const bool is_specialized = true");
        min_max_trait<T>::min();
        min_max_trait<T>::max();
        printf("\n");
        precision_trait<T>::digits();
        precision_trait<T>::digits10();
        is_signed<T>();
        exact_trait<T>::is_integer();
        exact_trait<T>::is_exact();
        precision_trait<T>::radix();
        precision_trait<T>::epsilon();
        precision_trait<T>::round_error();
        printf("\n");
        exponent_trait<T>::min_exponent();
        exponent_trait<T>::min_exponent10();
        exponent_trait<T>::max_exponent();
        exponent_trait<T>::max_exponent10();
        printf("\n");
        infinity_trait<T>::has_infinity();
        infinity_trait<T>::has_quiet_NaN();
        infinity_trait<T>::has_signaling_NaN();
        infinity_trait<T>::has_denorm();
        infinity_trait<T>::has_denorm_loss();
        printf("\n");
        infinity_trait<T>::infinity();
        infinity_trait<T>::quiet_NaN();
        infinity_trait<T>::signaling_NaN();
        infinity_trait<T>::denorm_min();
        printf("\n");
        is_iec559<T>();
        printf("%s%s;\n", tab2, "static const bool is_bounded = true");
        is_modulo<T>();
        printf("\n");
        traps<T>();
        tinyness_before<T>();
        round_style<T>();
        declarator<T>::end();
    }
};

int main()
{
    type_trait<bool>();

    type_trait<char>();
    type_trait<signed char>();
    type_trait<unsigned char>();

    //    type_trait<wchar_t>();
    
    type_trait<short>();
    type_trait<unsigned short>();

    type_trait<int>();
    type_trait<unsigned int>();

    type_trait<long>();
    type_trait<unsigned long>();

    type_trait<float>();
    type_trait<double>();
    type_trait<long double>();

    // x86/linux gets this weirdness for the min/max functions:
    // static long double min() throw()
    // { return (__extension__ ((union __convert_long_double) 
    // {__convert_long_double_i: {0x0, 0x80000000, 0x1, 0x0}})
    // .__convert_long_double_d); }
}

