/* 
Copyright (C) 1989 Free Software Foundation
    written by Doug Lea (dl@oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/



#ifndef NO_LIBGXX_MALLOC   /* ignore whole file otherwise */

/* compile with -DMALLOC_STATS to collect statistics */
/* collecting statistics slows down malloc by at least 15% */

#ifdef MALLOC_STATS
#define UPDATE_STATS(ARGS) {ARGS;}
#else
#define UPDATE_STATS(ARGS)
#endif




/* 
  A version of malloc/free/realloc tuned for C++ applications.

  Here's what you probably want to know first:

  In various tests, this appears to be about as fast as,
  and usually substantially less memory-wasteful than BSD/GNUemacs malloc.

  Generally, it is slower (by perhaps 20%) than bsd-style malloc
  only when bsd malloc would waste a great deal of space in 
  fragmented blocks, which this malloc recovers; or when, by
  chance or design, nearly all requests are near the bsd malloc
  power-of-2 allocation bin boundaries, and as many chunks are
  used as are allocated. 

  It uses more space than bsd malloc only when, again by chance
  or design, only bsdmalloc bin-sized requests are malloced, or when
  little dynamic space is malloced, since this malloc may grab larger
  chunks from the system at a time than bsd.

  In other words, this malloc seems generally superior to bsd
  except perhaps for programs that are specially tuned to
  deal with bsdmalloc's characteristics. But even here, the
  performance differences are slight.

 
  This malloc, like any other, is a compromised design. 

  The basic overhead for any chunk of memory is represented in
  the malloc_chunk struct, which contains 4 fields,
  although only two of them are used when the chunk is in use
  (malloced but not yet freed). 

  Chunks of memory are maintained using a `boundary tag' method as
  described in e.g., Knuth or Standish.  This means that the size of
  the chunk is stored both in the front of the chunk and at the end.
  This makes consolidating fragmented chunks into bigger chunks very fast.
  The size field is also used to hold bits representing whether a
  chunk is free or in use.

  Available chunks are kept in doubly linked lists. The lists are
  maintained in an array of bins using a power-of-two method, except
  that instead of 32 bins (one for each 1 << i), there are 128: each
  power of two is split in quarters. The use of very fine bin sizes 
  closely approximates the use of one bin per actually used size,
  without necessitating the overhead of locating such bins. It is
  especially desirable in common C++ applications where large numbers
  of identically-sized blocks are malloced/freed in some dynamic
  manner, and then later are all freed. The finer bin sizes make
  finding blocks fast, with little wasted overallocation. The
  consolidation methods ensure that once the collection of blocks is
  no longer useful, fragments are gathered into bigger chunks awaiting new
  roles.

  The bins av[i] serve as heads of the lists. They are malloc_chunk
  structs, but  the field `dirty', used only for
  alignment purposes in chunks, is used to record the number of chunks
  placed on the list since last consolidation.

  On allocation, the bin corresponding to the request size is
  scanned, and if there is a chunk with size >= requested, it
  is used, without splitting off the remainder into a different bin.
  This helps minimize the useless creation of too many very small unusable 
  fragments as often occurs with `best fit' style methods. If
  the bin is not empty, but no candidate chunk is available to
  fill the request, then the bin is sometimes (according to
  a heuristic formula) sent for consolidation, in order to
  try to minimize future useless scans.

  If no chunk exists, bigger bins are scanned. To minimize 
  unsuccessful scanning of a large number of probably empty bins
  (as happens when so many bins are used), a single designated
  ``victim'' is maintained, and is considered first. The designated
  victim is generally the last remaindered or consolidated chunk.
  If the victim is unusable (too small), then a new victim is
  searched for by scanning bigger bins. New victims are scanned approximately
  biggest-first. While this can waste (fragment) big blocks, it is 
  generally preferable, since the same block is often split/reused
  many times without requiring searches for new victims.

  If the victim is usable, it is split. The remainder is placed on
  the back of the appropriate bin list. (All freed chunks are placed
  on fronts of lists. All remaindered or consolidated chunks are
  placed on the rear. Correspondingly, searching within a bin
  starts at the front, but finding victims is from the back. All
  of this approximates  the  effect of having 2 kinds of lists per 
  bin: returned chunks vs unallocated chunks, but without the overhead 
  of maintaining 2 lists.)

  If no victim can be found, then smaller bins are examined for
  consolidation.

  Finally, if consolidation fails to come up with a usable chunk,
  more space is obtained from the system.

  Deallocation (free) consists only of placing the chunk on
  a list.

  Reallocation proceeds in the usual way. If a chunk can be extended,
  it is, else a malloc-copy-free sequence is taken.

  Some other implementation matters:

  8 byte alignment is currently hardwired into the design.

  The basic overhead of a used chunk is 8 bytes: 4 at the front and
  4 at the end.

  When a chunk is free, 8 additional bytes are needed for free list
  pointers. Thus, the minimum allocatable size is 16 bytes.

  The existence of front and back overhead permits some reasonably
  effective fence-bashing checks: The front and back fields must
  be identical. This is checked only within free() and realloc().
  The checks are fast enough to be made non-optional.

  The overwriting of parts of freed memory with the freelist pointers
  can also be very effective (albeit in an annoying way) in helping 
  users track down dangling pointers.

  User overwriting of freed space will often result in crashes
  within malloc or free.
  
  These routines are also tuned to C++ in that free(0) is a noop and
  a failed malloc automatically calls (*new_handler)(). 

  malloc(0) returns a pointer to something of the minimum allocatable size.

  Additional memory is gathered from the system (via sbrk) in a
  way that allows chunks obtained across different sbrk calls to
  be consolidated, but does not require contiguous memory: Thus,
  it should be safe to intersperse mallocs with other sbrk calls.

  This malloc is NOT designed to work in multiprocessing applications.
  No semaphores or other concurrency control are provided to ensure
  that multiple malloc or free calls don't run at the same time,
  which could be disasterous.

  VERY heavy use of inlines is made, for clarity. If this malloc
  is ported via a compiler without inlining capabilities, all
  inlines should be transformed into macros -- making them non-inline
  makes malloc at least twice as slow.


*/


/* preliminaries */

#include "//usr/include/stdio.h"  /* needed for error reporting */
#include <getpagesize.h>          /* needed to find good sbrk size */

#ifdef USG
extern void*     memset(void*, int, int);
extern void*     memcpy(void*,  const void*, int);
inline void      bzero(void* s, int l) { memset(s, 0, l); }
inline void      bcopy(const void* s, void* d, int l) { memcpy(d, s, l); }
#else
extern void      bcopy(void*, void*, unsigned int);
extern void      bzero(void*, unsigned int);
#endif

extern void*     sbrk(unsigned int);
extern int       fputs(const char*, FILE*);
extern int       fprintf(FILE*, const char*, ...);

extern volatile void abort();


/* how to die on detected error */
static volatile void malloc_user_error()
{
  fputs("malloc/free/realloc: clobbered space detected\n", stderr); abort();
}



/*
 Basic overhead for each malloc'ed chunk 
*/

/* malloc chunks and malloc bins have the same structure */

struct malloc_chunk
{
  unsigned int         size;     /* Size in bytes, including overhead. */
                                 /* Or'ed with INUSE if in use. */
                                 /* For bins, size must == 0  */

  struct malloc_chunk* fd;       /* double links -- used only if free. */
  struct malloc_chunk* bk;       /* For bins, these are heads of lists */

  unsigned int         dirty;    /* unused for chunks, but declared so as */
                                 /* to be conservative  wrt alignment */
                                 /* For bins, the number of chunks placed */
                                 /* on list since last consolidation */
};

typedef struct malloc_chunk* mchunkptr;
typedef struct malloc_chunk* mbinptr;


/*
  sizes, alignment
*/


#define SIZE_SZ                   (sizeof(unsigned int))
#define MALLOC_MIN_OVERHEAD       (SIZE_SZ + SIZE_SZ)
#define MALLOC_ALIGN_MASK         (MALLOC_MIN_OVERHEAD - 1)

#define MINSIZE    (sizeof(struct malloc_chunk)) /* MUST == 16! */


/* pad request bytes into a usable size */

static inline unsigned int request2size(unsigned int request)
{
  return  (request == 0) ?  MINSIZE : 
    ((request + MALLOC_MIN_OVERHEAD + MALLOC_ALIGN_MASK) 
      & ~(MALLOC_ALIGN_MASK));
}


static inline int aligned_OK(void* m)  
{
  return ((unsigned int)(m) & (MALLOC_ALIGN_MASK)) == 0;
}


/* size field or'd with INUSE when in use */
#define INUSE  0x1



/* the bins, initialized to have null double linked lists */

#define MAXBIN 120   /* 1 more than needed for 32 bit addresses */

#define FIRSTBIN (&(av[0])) 

static struct malloc_chunk  av[MAXBIN] = 
{
  { 0, &(av[0]),  &(av[0]), 0 },
  { 0, &(av[1]),  &(av[1]), 0 },
  { 0, &(av[2]),  &(av[2]), 0 },
  { 0, &(av[3]),  &(av[3]), 0 },
  { 0, &(av[4]),  &(av[4]), 0 },
  { 0, &(av[5]),  &(av[5]), 0 },
  { 0, &(av[6]),  &(av[6]), 0 },
  { 0, &(av[7]),  &(av[7]), 0 },
  { 0, &(av[8]),  &(av[8]), 0 },
  { 0, &(av[9]),  &(av[9]), 0 },

  { 0, &(av[10]), &(av[10]), 0 },
  { 0, &(av[11]), &(av[11]), 0 },
  { 0, &(av[12]), &(av[12]), 0 },
  { 0, &(av[13]), &(av[13]), 0 },
  { 0, &(av[14]), &(av[14]), 0 },
  { 0, &(av[15]), &(av[15]), 0 },
  { 0, &(av[16]), &(av[16]), 0 },
  { 0, &(av[17]), &(av[17]), 0 },
  { 0, &(av[18]), &(av[18]), 0 },
  { 0, &(av[19]), &(av[19]), 0 },

  { 0, &(av[20]), &(av[20]), 0 },
  { 0, &(av[21]), &(av[21]), 0 },
  { 0, &(av[22]), &(av[22]), 0 },
  { 0, &(av[23]), &(av[23]), 0 },
  { 0, &(av[24]), &(av[24]), 0 },
  { 0, &(av[25]), &(av[25]), 0 },
  { 0, &(av[26]), &(av[26]), 0 },
  { 0, &(av[27]), &(av[27]), 0 },
  { 0, &(av[28]), &(av[28]), 0 },
  { 0, &(av[29]), &(av[29]), 0 },

  { 0, &(av[30]), &(av[30]), 0 },
  { 0, &(av[31]), &(av[31]), 0 },
  { 0, &(av[32]), &(av[32]), 0 },
  { 0, &(av[33]), &(av[33]), 0 },
  { 0, &(av[34]), &(av[34]), 0 },
  { 0, &(av[35]), &(av[35]), 0 },
  { 0, &(av[36]), &(av[36]), 0 },
  { 0, &(av[37]), &(av[37]), 0 },
  { 0, &(av[38]), &(av[38]), 0 },
  { 0, &(av[39]), &(av[39]), 0 },

  { 0, &(av[40]), &(av[40]), 0 },
  { 0, &(av[41]), &(av[41]), 0 },
  { 0, &(av[42]), &(av[42]), 0 },
  { 0, &(av[43]), &(av[43]), 0 },
  { 0, &(av[44]), &(av[44]), 0 },
  { 0, &(av[45]), &(av[45]), 0 },
  { 0, &(av[46]), &(av[46]), 0 },
  { 0, &(av[47]), &(av[47]), 0 },
  { 0, &(av[48]), &(av[48]), 0 },
  { 0, &(av[49]), &(av[49]), 0 },

  { 0, &(av[50]), &(av[50]), 0 },
  { 0, &(av[51]), &(av[51]), 0 },
  { 0, &(av[52]), &(av[52]), 0 },
  { 0, &(av[53]), &(av[53]), 0 },
  { 0, &(av[54]), &(av[54]), 0 },
  { 0, &(av[55]), &(av[55]), 0 },
  { 0, &(av[56]), &(av[56]), 0 },
  { 0, &(av[57]), &(av[57]), 0 },
  { 0, &(av[58]), &(av[58]), 0 },
  { 0, &(av[59]), &(av[59]), 0 },

  { 0, &(av[60]), &(av[60]), 0 },
  { 0, &(av[61]), &(av[61]), 0 },
  { 0, &(av[62]), &(av[62]), 0 },
  { 0, &(av[63]), &(av[63]), 0 },
  { 0, &(av[64]), &(av[64]), 0 },
  { 0, &(av[65]), &(av[65]), 0 },
  { 0, &(av[66]), &(av[66]), 0 },
  { 0, &(av[67]), &(av[67]), 0 },
  { 0, &(av[68]), &(av[68]), 0 },
  { 0, &(av[69]), &(av[69]), 0 },

  { 0, &(av[70]), &(av[70]), 0 },
  { 0, &(av[71]), &(av[71]), 0 },
  { 0, &(av[72]), &(av[72]), 0 },
  { 0, &(av[73]), &(av[73]), 0 },
  { 0, &(av[74]), &(av[74]), 0 },
  { 0, &(av[75]), &(av[75]), 0 },
  { 0, &(av[76]), &(av[76]), 0 },
  { 0, &(av[77]), &(av[77]), 0 },
  { 0, &(av[78]), &(av[78]), 0 },
  { 0, &(av[79]), &(av[79]), 0 },

  { 0, &(av[80]), &(av[80]), 0 },
  { 0, &(av[81]), &(av[81]), 0 },
  { 0, &(av[82]), &(av[82]), 0 },
  { 0, &(av[83]), &(av[83]), 0 },
  { 0, &(av[84]), &(av[84]), 0 },
  { 0, &(av[85]), &(av[85]), 0 },
  { 0, &(av[86]), &(av[86]), 0 },
  { 0, &(av[87]), &(av[87]), 0 },
  { 0, &(av[88]), &(av[88]), 0 },
  { 0, &(av[89]), &(av[89]), 0 },

  { 0, &(av[90]), &(av[90]), 0 },
  { 0, &(av[91]), &(av[91]), 0 },
  { 0, &(av[92]), &(av[92]), 0 },
  { 0, &(av[93]), &(av[93]), 0 },
  { 0, &(av[94]), &(av[94]), 0 },
  { 0, &(av[95]), &(av[95]), 0 },
  { 0, &(av[96]), &(av[96]), 0 },
  { 0, &(av[97]), &(av[97]), 0 },
  { 0, &(av[98]), &(av[98]), 0 },
  { 0, &(av[99]), &(av[99]), 0 },

  { 0, &(av[100]), &(av[100]), 0 },
  { 0, &(av[101]), &(av[101]), 0 },
  { 0, &(av[102]), &(av[102]), 0 },
  { 0, &(av[103]), &(av[103]), 0 },
  { 0, &(av[104]), &(av[104]), 0 },
  { 0, &(av[105]), &(av[105]), 0 },
  { 0, &(av[106]), &(av[106]), 0 },
  { 0, &(av[107]), &(av[107]), 0 },
  { 0, &(av[108]), &(av[108]), 0 },
  { 0, &(av[109]), &(av[109]), 0 },

  { 0, &(av[110]), &(av[110]), 0 },
  { 0, &(av[111]), &(av[111]), 0 },
  { 0, &(av[112]), &(av[112]), 0 },
  { 0, &(av[113]), &(av[113]), 0 },
  { 0, &(av[114]), &(av[114]), 0 },
  { 0, &(av[115]), &(av[115]), 0 },
  { 0, &(av[116]), &(av[116]), 0 },
  { 0, &(av[117]), &(av[117]), 0 },
  { 0, &(av[118]), &(av[118]), 0 },
  { 0, &(av[119]), &(av[119]), 0 }
};

/*
  indexing into bins
*/

static inline mbinptr size2bin(unsigned int sz)
{
  mbinptr b = av;
  while (sz >= (MINSIZE * 2)) { b += 4; sz >>= 1; } /* find power of 2 */
  b += (sz - MINSIZE) >> 2;                         /* find quadrant */
  return b;
}



/* counts maintained if MALLOC_STATS defined */

static unsigned int sbrked_mem;
static unsigned int requested_mem;
static unsigned int malloced_mem;
static unsigned int freed_mem;
static unsigned int max_used_mem;

static unsigned int n_sbrks;
static unsigned int n_mallocs;
static unsigned int n_frees;
static unsigned int n_reallocs;
static unsigned int n_reallocs_with_copy;
static unsigned int n_avail;
static unsigned int max_inuse;

static unsigned int n_malloc_chunks;
static unsigned int n_malloc_bins;

static unsigned int n_split;
static unsigned int n_consol;


/* stat maintainers, for convenience */

#ifdef MALLOC_STATS
static void do_malloc_stats(const mchunkptr p)
{
  ++n_mallocs;
  if ((n_mallocs-n_frees) > max_inuse)
    max_inuse = n_mallocs - n_frees;
  malloced_mem += (p->size & ~(INUSE));
  if (malloced_mem - freed_mem > max_used_mem)
    max_used_mem = malloced_mem - freed_mem;
}

static void do_free_stats(const mchunkptr p)
{
  ++n_frees;
  freed_mem += (p->size & ~(INUSE));
}        

#endif


  
/* operations on  malloc_chunk addresses */


/* return ptr to next physical malloc_chunk */

static inline mchunkptr next_chunk(const mchunkptr p)
{
  return (mchunkptr)((char*)(p) + p->size);
}


/* return ptr to previous physical malloc_chunk */

static inline mchunkptr prev_chunk(const mchunkptr p)
{
  return (mchunkptr)((char*)p - ((((int*)(p))[-1]) & ~(INUSE)));
}

/* place size at front and back of chunk */

static inline void set_size(mchunkptr p, unsigned int sz)
{
  p->size = *((int*)((char*)p + sz - SIZE_SZ)) = sz;
}


/* return pointer to a new malloc_chunk being split off */

static inline mchunkptr split(mchunkptr p, unsigned int offset)
{
  unsigned int room = p->size - offset;
  mchunkptr t = (mchunkptr)((char*)(p) + offset);
  t->size = *((int*)((char*)t + room    - SIZE_SZ)) = room;
  p->size = *((int*)((char*)p + offset  - SIZE_SZ)) = offset;
  UPDATE_STATS (++n_split);
  return t;
}



/* maintaining INUSE via size field */

static inline unsigned int inuse(const mchunkptr p)
{
  return p->size & INUSE;
}

static inline void set_inuse(mchunkptr p)
{
  p->size |= INUSE;
}

static inline void clear_inuse(mchunkptr p)
{
  p->size &= ~INUSE;
}




/* conversion from malloc headers to user pointers, and back */

static inline void* chunk2mem(mchunkptr p) 
{ 
  p->size |= INUSE;
  return (void*)((char*)p + SIZE_SZ); 
}

static inline mchunkptr mem2chunk(void* mem) 
{ 
  mchunkptr p = (mchunkptr)((char*)mem - SIZE_SZ); 

  /* a quick sanity check */
  unsigned int sz = p->size & ~(INUSE);
  if (p->size == sz || sz != *((int*)((char*)p + sz - SIZE_SZ)))
    malloc_user_error();

  p->size = sz;   /* clears INUSE */
  return p;
}



/* maintaining bins & pointers */


/* maximum bin actually used */

static mbinptr malloc_maxbin = FIRSTBIN;


/* The designated victim. */

/* The victim is always checked for size, so it is OK to use bin heads */
/* (which always have size == 0) to represent unusable victims */

static mchunkptr malloc_victim = FIRSTBIN;


/* operations on lists inside bins */


/* take a chunk off a list */

static inline void unlink(mchunkptr p)
{
  mchunkptr b = p->bk;
  mchunkptr f = p->fd;

  f->bk = b;  b->fd = f;

  UPDATE_STATS (--n_avail);
}


/* place a freed chunk on the front of a list */

static inline void frontlink(mchunkptr p)
{
  mbinptr   bn = size2bin(p->size);
  mchunkptr h  = (mchunkptr)(bn);
  mchunkptr f  = h->fd;

  p->bk = h;  p->fd = f;  f->bk = h->fd = p;

  bn->dirty++;   /* mark that bin can be consolidated */

  if (h == f && bn > malloc_maxbin)   /* h==f means was empty */
    malloc_maxbin = bn;  

  UPDATE_STATS(++n_avail);
}


/* place a split or consolidated chunk on the back of a list */

static inline void backlink(mchunkptr p)
{
  mbinptr   bn = size2bin(p->size);
  mchunkptr h  = (mchunkptr)(bn);
  mchunkptr b  = h->bk;

  p->bk = b;  p->fd = h;  h->bk = b->fd = p;

  malloc_victim = p;        /* use as next victim */
  bn->dirty++;              /* mark for consolidation */

  if (h == b && bn > malloc_maxbin) malloc_maxbin = bn; 

  UPDATE_STATS(++n_avail);
}


/* Dealing with sbrk */


/* A good size to call sbrk with, == page size, as computed in morecore */

static unsigned int best_sbrk_size;


/* To link consecutive sbrk regions when possible */

static int* last_sbrk_end;


/*  who to call when sbrk returns failure */

typedef volatile void (*vfp)();
extern vfp __new_handler;


static mchunkptr malloc_from_sys(unsigned nb)
{
  mchunkptr p;
  unsigned int sbrk_size;
  int* ip;
  
  if (best_sbrk_size == 0)  /* first time through */
  {
    best_sbrk_size = getpagesize();
    if (best_sbrk_size < 4096)  /* use something reasonable! */
      best_sbrk_size = 4096;
  }

  /* Minimally, we need to pad with enough space */
  /* to place dummy size/use fields to ends if needed */
  /* and a remander chunk of at least MINSIZE */
  /* (malloc expects that there will be a remainder) */
  /* then round to page size. */

  /* But there is little harm in being a little wasteful here */
  /* and grabbing one more page than minimally needed in anticipation */
  /* of more requests, since the other routines handle large chunks */
  /* of space better than small ones. (They will need to scan */
  /* fewer chunks to consolidate enough to fulfill requests, etc.) */

#ifdef MIN_MALLOC_SBRK_WASTE

  sbrk_size = ((nb + best_sbrk_size - 1 + SIZE_SZ + SIZE_SZ + MINSIZE) 
               / best_sbrk_size) * best_sbrk_size;

#else

  sbrk_size = (1 + (nb + best_sbrk_size-1) / best_sbrk_size) * best_sbrk_size;

#endif

  ip = (int*)(sbrk(sbrk_size));
  if ((char*)ip == (char*)(-1)) /* sbrk returns -1 on failure */
  {
    (*__new_handler) ();
    return 0;
  }

  UPDATE_STATS ((++n_sbrks, sbrked_mem += sbrk_size));


  if (last_sbrk_end != &ip[-1]) 
  {                             
    /* It's either first time through or someone else called sbrk. */
    /* Arrange end-markers at front & back */

    /* Shouldn't be necessary, but better to be safe */
    while (!aligned_OK(ip)) { ++ip; sbrk_size -= SIZE_SZ; }


    /* Mark the front as in use to prevent merging. */
    /* Note we can get away with only 1 word, not MINSIZE overhead here */

    *ip++ = SIZE_SZ | INUSE;

    
    p = (mchunkptr)ip;
    set_size(p,sbrk_size - (SIZE_SZ + SIZE_SZ)); 
    
  }
  else 
  {
    mchunkptr l;  

    /* We can safely make the header start at end of prev sbrked chunk. */
    /* We will still have 1 word left at the end from a previous call */
    /* to place the end marker, below */

    p = (mchunkptr)(last_sbrk_end);
    set_size(p, sbrk_size);


    /* Even better, maybe we can merge with last fragment: */

    l = prev_chunk(p);
    if (!inuse(l))  
    {
      unlink(l);
      set_size(l, p->size + l->size);
      p = l;
    }

  }

  /* mark the end of sbrked space as in use to prevent merging */

  last_sbrk_end = (int*)((char*)p + p->size);
  *last_sbrk_end = SIZE_SZ | INUSE;


  /* make p look like it's in a list, so unlink will work. */
  /* (malloc will properly relink remainder) */

  p->fd = p->bk = p;

  UPDATE_STATS((++n_avail, ++n_malloc_chunks));

  return p;
}



/* Consolidate all chunks in bin b. */
/* If this creates a chunk big enough to satisfy */
/* current malloc request, return it */

/* (It requires much less bookkeeping to consolidate entire bins */
/* at once than to keep records of which chunks might be */
/* consolidatable. So long as the lists are short, which we */
/* try to ensure via small bin ranges, there is little wasted effort.) */

static mchunkptr malloc_consolidate_bin(mbinptr b, unsigned int nb)
{
  mchunkptr h = (mchunkptr)(b);   /* head of list */
  mchunkptr p = h->bk;            /* chunk traverser */
  mchunkptr found = 0;            /* to hold usable consolidated chunk */
  
  b->dirty = 0;                   /* mark that bin is clean */
  
  /* traverse each chunk, looking for consolidations */
  /* proceed backwards to avoid rescanning relinked chunks */

  while (p != h)
  {
    mchunkptr nextp = p->bk;        /* save, in case of relinks */

    mchunkptr l = prev_chunk(p);
    mchunkptr n = next_chunk(p);

    if (!inuse(l) || !inuse(n))
    {
      unlink(p);

      while (!inuse(l)) /* consolidate backward */
      {
        if (l == nextp) nextp = l->bk;
        unlink(l);
        set_size(l, l->size + p->size);
        p = l;
        l = prev_chunk(p);
        UPDATE_STATS (++n_consol);
      }
      
      while (!inuse(n)) /* and forward */
      {
        if (n == nextp) nextp = n->bk;
        unlink(n);
        set_size(p, p->size + n->size);
        n = next_chunk(p);
        UPDATE_STATS (++n_consol);
      }
      
      if (p->size >= nb) found = p;

      backlink(p);
    }

    UPDATE_STATS(++n_malloc_chunks);
    p = nextp;
  }

  return found;  
}




/* Consolidate dirty bins. */
/* Stop if found a chunk big enough to satisfy current malloc request */
/* If can't consolidate to fill request, get more from sys */

static mchunkptr malloc_find_space(unsigned int nb)
{
  mbinptr b;

  /* traverse backwards so as not to rescan chunks we */
  /* just put in bigger bins in previous iterations  */

  for (b = malloc_maxbin; b >= FIRSTBIN; --b)
  {
    UPDATE_STATS(++n_malloc_bins);
   
    if (b->dirty)
    {
      if (b->bk == b)    /* empty - just clear dirty */
        b->dirty = 0; 
      else
      {
        mchunkptr p = malloc_consolidate_bin(b, nb);
        if (p != 0)
          return p;
      }
    }
  }


  /* nothing available -- get some more */

  return malloc_from_sys(nb);
}



/*   Finally, the user-level functions  */

void* malloc(unsigned int bytes)
{
  unsigned int nb  = request2size(bytes);  /* padded request size */
  mbinptr      b   = size2bin(nb);         /* corresponding bin */
  mchunkptr    hd  = (mchunkptr)(b);       /* head of its list */
  mchunkptr    p   = hd->fd;               /* chunk traverser */

  UPDATE_STATS((requested_mem+=bytes, ++n_malloc_bins));

  /* if bin non-empty, try a (near) exact match */

  if (p != hd) 
  {
    do
    {
      UPDATE_STATS(++n_malloc_chunks);

      if (p->size >= nb)         /* found one - take without splitting */
      {

        if (p == malloc_victim)  /* reset victim for next time */
          malloc_victim = (p->bk == hd)? p->fd : p->bk;

        unlink(p);
        UPDATE_STATS(do_malloc_stats(p));
        return chunk2mem(p);
      }

    } while ((p = p->fd) != hd);

    /* If no match, maybe consolidate in order to minimize future searching */

    /* This is just a heuristic, and the criterion is just a magic */
    /* number, but it seems robust across test cases */

    if (b->dirty > 4) 
    { 
      if ((p = malloc_consolidate_bin(b, nb)) != 0)
        malloc_victim = p; /* If we happened to find something, use it */
    }

  }


  /* No exact match, so select a victim */

  p = malloc_victim; 
  UPDATE_STATS(++n_malloc_chunks);

  if (p->size < nb)  
  {
    for (;;)                                   /* search bins for new victim */
    {
      UPDATE_STATS(++n_malloc_bins);
      if (malloc_maxbin <= b)                  /* nowhere to look */
      {
        if ((p = malloc_find_space(nb)) == 0)  /* consolidate or sbrk */
          return 0;                            /* out of space! */
        else
          break;
      }
      else if ((p = malloc_maxbin->bk) != malloc_maxbin) /* found one  */
      {
        break;        /* (p MUST be usable since it's in a bigger bin) */
      }
      else
      {
        --malloc_maxbin;              /* maxbin must be empty, adjust */
      }
    }
  }

  /* use what we found */

  unlink(p);

  if (p->size - nb >= MINSIZE)
    backlink(split(p, nb));        /* (backlink resets victim) */
  else                           
    malloc_victim = p->bk;         /* make sure victim OK for next time */

  UPDATE_STATS(do_malloc_stats(p));
  return chunk2mem(p);
}




void free(void* mem)
{
  if (mem != 0)
  {
    mchunkptr p = mem2chunk(mem);
    UPDATE_STATS(do_free_stats(p));
    frontlink(p);
  }
}


void* calloc(unsigned int elem_size, unsigned int n)
{
  unsigned int sz = n * elem_size;
  void* p = malloc(sz);
  bzero(p, sz);
  return p;
};

/* This is here for compatibility with older systems */
void cfree(void *mem, unsigned int n, unsigned int elem_size)
{
  free(mem);
}
 

unsigned int malloc_usable_size(void* mem)
{
  mchunkptr p = (mchunkptr)((char*)mem - SIZE_SZ); 
  return (p->size & ~INUSE) - MALLOC_MIN_OVERHEAD;
}



void* realloc(void* mem, unsigned int bytes)
{
  if (mem == 0) 
    return malloc(bytes);
  else
  {
    unsigned int nb      = request2size(bytes);
    mchunkptr    p       = mem2chunk(mem);
    unsigned int oldsize = p->size;
    int          room    = oldsize - nb;

    UPDATE_STATS(++n_reallocs);
    
    while (room < 0) /* try to expand */
    {
      mchunkptr nxt = next_chunk(p);
      if (inuse(nxt))
        break;
      else
      {
        UPDATE_STATS ((malloced_mem += nxt->size, ++n_consol));
        if (nxt == malloc_victim) /* maintain usable victim */
          malloc_victim = nxt->bk;
        unlink(nxt);
        set_size(p, p->size + nxt->size);
        room = p->size - nb;
      }
    }

    if (room >= 0)
    {
      if (room >= MINSIZE) 
      {
        backlink(split(p, nb));
        UPDATE_STATS(freed_mem += room);
      }
      return chunk2mem(p);
    }
    else /* do the obvious */
    {
      void* newmem;
      unsigned int s = (oldsize <= nb)? oldsize : nb;
      set_inuse(p);    /* don't let malloc consolidate us yet! */
      newmem = malloc(nb);
      bcopy(mem, newmem, s - SIZE_SZ);
      free(mem);
      UPDATE_STATS(++n_reallocs_with_copy);
      return newmem;
    }
  }
}



void malloc_stats()
{
  int i;
  mchunkptr p;
  double nm = (double)(n_mallocs + n_reallocs);

  fprintf(stderr, "\nmalloc statistics\n\n");

  if (n_mallocs != 0)
  fprintf(stderr, "requests  = %10u total size = %10u\tave = %10u\n",
          n_mallocs, requested_mem, requested_mem/n_mallocs);

  if (n_mallocs != 0)
  fprintf(stderr, "mallocs   = %10u total size = %10u\tave = %10u\n",
          n_mallocs, malloced_mem, malloced_mem/n_mallocs);
  
  if (n_frees != 0)
  fprintf(stderr, "frees     = %10u total size = %10u\tave = %10u\n",
          n_frees, freed_mem, freed_mem/n_frees);
  
  if (n_mallocs-n_frees != 0)
  fprintf(stderr, "in use    = %10u total size = %10u\tave = %10u\n",
          n_mallocs-n_frees, malloced_mem-freed_mem, 
          (malloced_mem-freed_mem) / (n_mallocs-n_frees));

  if (max_inuse != 0)
  fprintf(stderr, "max in use= %10u total size = %10u\tave = %10u\n",
          max_inuse, max_used_mem, max_used_mem / max_inuse);
  
  if (n_avail != 0)
  fprintf(stderr, "available = %10u total size = %10u\tave = %10u\n",
          n_avail, sbrked_mem - (malloced_mem-freed_mem), 
          (sbrked_mem - (malloced_mem-freed_mem)) / n_avail);

  if (n_sbrks != 0)
  fprintf(stderr, "sbrks     = %10u total size = %10u\tave = %10u\n\n",
          n_sbrks, sbrked_mem, sbrked_mem/ n_sbrks);

  if (n_reallocs != 0)
  fprintf(stderr, "reallocs  = %10u with copy  = %10u\n\n",
          n_reallocs, n_reallocs_with_copy);


  if (nm != 0)
  {
    fprintf(stderr, "chunks scanned per malloc = %6.3f\n", 
            n_malloc_chunks / nm);
    fprintf(stderr, "bins scanned per malloc   = %6.3f\n", 
            n_malloc_bins / nm);
    fprintf(stderr, "splits per malloc         = %6.3f\n", 
            n_split / nm);
    fprintf(stderr, "consolidations per malloc = %6.3f\n", 
            n_consol / nm);
  }

  fprintf(stderr, "\nfree chunks:\n");
  for (i = 0; i < MAXBIN; ++i)
  {
    p = av[i].fd;
    if (p != &av[i])
    {
      unsigned int count = 1;
      unsigned int sz = p->size;
      for (p = p->fd; p != &av[i]; p = p->fd)
      {
        if (p->size == sz)
          ++count;
        else
        {
          fprintf(stderr, "\tsize = %10u count = %5u\n", sz, count);
          count = 1;
          sz = p->size;
        }
      }

      fprintf(stderr, "\tsize = %10u count = %5u\n", sz, count);

    }
  }
}
 
#endif
