/* wc - print the number of bytes, words, and lines in files

   AUTHOR: Gregory Pietsch
   
   DESCRIPTION:
   
   The wc program counts the number of bytes, whitespace-separated words,
   and newlines in a given file, or the standard input if none are given or
   when a file named '-' is given.  It prints one line of counts for each
   file, and if the file was given as an argument, it prints the filename
   following the counts. If more than one filename is given, wc prints a
   final line containing the cumulative counts, with the filename 'total'.
   The counts are printed in the order: lines, words, characters (if
   specified), bytes, the maximum number of characters in a line.
   
   By default, wc prints all three counts. Options can specify that only
   certain counts be printed.  Options do not undo others previously given,
   so wc --bytes --words prints both the byte counts and word counts.
   
   OPTIONS:
   
   -c, --bytes                  Print only the byte counts.
   
   -l, --lines                  Print only the newline counts.
   
   -m, --chars                  Print only the character counts.
   
   -w, --words                  Print only the word counts.
   
   -L, --max-line-length        Print only the maximum line width of a file.
                                Tabs are considered to be moving the column
                                to a multiple of 8; nonprintable characters
                                have a width of zero.
                                
   --files0-from=FILE           Get the filenames from a file instead of the
                                command line.  Within the file, the names are
                                delimited with null characters.  This option
                                is useful with the output of the find command
                                with the -print0 option.
   
   --help                       Print a usage message and exit with a non-zero
                                status.
                                
   --version                    Print version information on standard output
                                then exit.
                                
*/

/* include files */

#include "config.h"
#include <ctype.h>
#if defined STDC_HEADERS || defined HAVE_LIMITS_H
#include <limits.h>
#endif
#include <stdio.h>
#if defined STDC_HEADERS || defined HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if defined STDC_HEADERS || defined HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#ifdef HAVE_WCHAR_H
#include <wchar.h>
#endif
#ifdef HAVE_WCTYPE_H
#include <wctype.h>
#endif
#include "getopt.h"

/* typedefs */

typedef struct WC_T
{
  unsigned long long bytes, chars, lines, words, max;
} WC_T;

/* defines */

#define F_BYTES                 (1)
#define F_CHARS                 (2)
#define F_LINES                 (4)
#define F_WORDS                 (8)
#define F_MAX                   (0x10)
#define F_FILES_FROM            (0x20)

/* static data */

unsigned flags = 0U;
int show_help = 0, show_version = 0, files_from_flag = 0;
char *files_from_filename = 0;
const char *shortopts = "clmwL";
const struct option longopts[] = {
  {"bytes", 0, 0, 'c'},
  {"chars", 0, 0, 'm'},
  {"lines", 0, 0, 'l'},
  {"words", 0, 0, 'w'},
  {"max-line-length", 0, 0, 'L'},
  {"files0-from", 1, &files_from_flag, 1},
  {"help", 0, &show_help, 1},
  {"version", 0, &show_version, 1},
  {0, 0, 0, 0}
};

/* functions */

/* help: show the help text */
void
help (void)
{
#include "help.h"
}

/* version: show the version number */
void
version (void)
{
  puts (PACKAGE_STRING);
}

/* wc_zero: Zero out a WC_T */
void
wc_zero (WC_T * w)
{
  w->bytes = w->chars = w->lines = w->words = w->max = 0ULL;
}

/* parse_args: parse the command-line arguments */
void
parse_args (int argc, char *argv[])
{
  int o;

  while ((o = getopt_long (argc, argv, shortopts, longopts, 0)) != EOF)
    {
      switch (o)
	{
	case 'c':		/* --bytes */
	  flags |= F_BYTES;
	  break;
	case 'l':		/* --lines */
	  flags |= F_LINES;
	  break;
	case 'm':		/* --chars */
	  flags |= F_CHARS;
	  break;
	case 'w':		/* --words */
	  flags |= F_WORDS;
	  break;
	case 'L':		/* --max-line-length */
	  flags |= F_MAX;
	  break;
	case 0:		/* --help, --version, --files0-from */
	  if (show_help)
	    {
	      help ();
	      exit (EXIT_FAILURE);
	    }
	  else if (show_version)
	    {
	      version ();
	      exit (EXIT_SUCCESS);
	    }
	  else if (files_from_flag)
	    {
	      flags |= F_FILES_FROM;
	      files_from_filename = optarg;
	    }
	default:
	  break;
	}
    }
  if ((flags & (F_BYTES | F_CHARS | F_LINES | F_WORDS | F_MAX)) == 0)
    flags |= F_BYTES | F_LINES | F_WORDS;
}

/* process one file */
void
wc (WC_T * w, FILE * f)
{
  wint_t c;
  int in_word = 0;
  unsigned long long curmax = 0ULL;
  char s[MB_CUR_MAX + 1];

  wc_zero (w);
  while ((c = fgetwc (f)) != WEOF)
    {
      w->bytes += wctomb (s, c);
      ++(w->chars);
      if (c == L'\n')
	{
	  ++(w->lines);
	  if (curmax > w->max)
	    w->max = curmax;
	  curmax = 0ULL;
	}
      else if (c == L'\t')
	curmax = (curmax | 7) + 1;	/* tabs are every 8 characters */
      else if (iswprint (c))
	++curmax;
      if (iswspace (c) && in_word)
	{
	  in_word = 0;
	  ++(w->words);
	}
      else if (!iswspace (c) && !in_word)
	in_word = 1;
    }
}

/* adjust the totals */
void
adjust_totals (WC_T * t, WC_T * f)
{
  t->bytes += f->bytes;
  t->chars += f->chars;
  t->lines += f->lines;
  t->words += f->words;
  if (f->max > t->max)
    t->max = f->max;
}

/* output the totals */
void
output_totals (WC_T * w, char *s)
{
  if (flags & F_LINES)
    printf ("%7llu ", w->lines);
  if (flags & F_WORDS)
    printf ("%7llu ", w->words);
  if (flags & F_CHARS)
    printf ("%7llu ", w->chars);
  if (flags & F_BYTES)
    printf ("%7llu ", w->bytes);
  if (flags & F_MAX)
    printf ("%7llu ", w->max);
  printf ("%s\n", s);
}

/* process one file given the filename */
void
process_file (WC_T * totals, char *progname, char *s)
{
  WC_T curfile;
  FILE *f;

  if (strcmp (s, "-") == 0)
    f = stdin;
  else
    {
      f = fopen (s, "rb");
      if (f == 0)
	{
	  fprintf (stderr, "%s: Can't open file '%s'\n", progname, s);
	  exit (EXIT_FAILURE);
	}
    }
  wc (&curfile, f);
  output_totals (&curfile, s);
  adjust_totals (totals, &curfile);
  if (f != stdin)
    fclose (f);
}

int
main (int argc, char *v[])
{
  WC_T curfile, total;
  char s[PATH_MAX + 1], *p = s;
  FILE *from;
  size_t n = 0;
  int c;

  parse_args (argc, v);
  if (flags & F_FILES_FROM)
    {
      /* handle --files0-from filenames */
      from = fopen (files_from_filename, "rb");
      if (from == 0)
	{
	  fprintf (stderr, "%s: Can't open file '%s'\n", v[0], v[optind]);
	  exit (EXIT_FAILURE);
	}
      wc_zero (&total);
      while ((c = fgetc (from)) != EOF)
	{
	  if (c != '\0')
	    *p++ = c;
	  else
	    {
	      *p = '\0';
	      process_file (&total, *v, s);
	      ++n;
	      p = s;
	    }
	}
      if (n > 1)
	output_totals (&total, "total");
      fclose (from);
    }
  else if (v[optind] == 0
	   || (v[optind + 1] == 0 && strcmp (v[optind], "-") == 0))
    {
      /* no filenames on command line, input is from stdin */
      wc (&curfile, stdin);
      output_totals (&curfile, "-");
    }
  else if (v[optind + 1] == 0)
    {
      /* one filename on command line */
      process_file (&total, *v, v[optind]);
    }
  else
    {
      /* multiple filenames on command line */
      wc_zero (&total);
      while (v[optind])
	{
	  process_file (&total, *v, v[optind]);
	  ++optind;
	}
      output_totals (&total, "total");
    }
  return 0;
}

/* END OF FILE */
