/*
  The getopt() function parses the command line arguments.  Its
  arguments argc and argv are the argument count and array as passed
  to the main() function on program invocation.  An element of argv
  that starts with `/'is an option element.  If getopt() is called
  repeatedly, it returns successively each of the option characters
  from each of the option elements.

  If there are no more option characters, getopt() returns -1.  Then
  optind is the index in argv of the first argv element that is not an
  option.

  optstring is a string containing the legitimate option characters.
  If such a character is followed by a colon, the option requires an
  argument, so getopt places a pointer to the following text in the
  same argv-element, or the text of the following argv-element, in
  optarg.

  If getopt() does not recognize an option character, it prints an
  error message to stderr, stores the character in optopt, and returns
  `?'.  The calling program MUST TURN ON THIS BEHAVIOR by setting
  opterr to 1.
*/

/*
 * getopt - get option letter from argv
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>			/* for strcmp(), strchr() */


char    *optarg;        /* Global argument pointer. */
int     optind = 1;     /* Global argv index. */
int     opterr = 0;     /* Global error message flag. */
int 	optopt = '?';

static char     *scan = NULL;   /* Private scan pointer. */

int
getopt(argc, argv, optstring)
int argc;
char *argv[];
char *optstring;
{
	register char c;
	register char *place;
	/* Get the actual name of the program, not the full path */
	char * progName = strrchr(argv[0], '\\') + 1;

	optarg = NULL;

	if (scan == NULL || *scan == '\0') {
		if (optind == 0)
			optind++;

		if (optind >= argc || argv[optind][0] != '/' || argv[optind][1] == '\0')
			return -1;

		if (strcmp(argv[optind], "--")==0) {
			optind++;
			return -1;
		}

		scan = argv[optind]+1;
		optind++;
	}

	c = *scan++;
	place = strchr(strlwr(optstring), tolower(c));

	if (place == NULL || c == ':') {
		if (opterr) {

			fprintf(stderr, "%s: unknown option /%c\n", progName, c);
			return('?');
		}
	}

	place++;
	if (*place == ':') {
		if (*scan != '\0') {
			optarg = scan;
			scan = NULL;
		} else if (optind < argc) {
			optarg = argv[optind];
			optind++;
		} else if (opterr) {
			fprintf(stderr, "%s: /%c argument missing\n", progName, c);
			return('?');
		}
	}

	return(c);
}
