/*
 *  MDSAI - Multi-disk split archive installer
 *  Copyright (C) 2016-2017 David McMackins II
 *
 *  Copying and distribution of this file, with or without modification,
 *  are permitted in any medium without royalty provided the copyright
 *  notice and this notice are preserved. This file is offered as-is,
 *  without any warranty.
 */

#include <ctype.h>
#include <errno.h>
#include <process.h>
#include <stdbool.h>
#include <stdio.h>

#define TEMP_DIR "C:\\FDOS\\TEMP\\"
#define TEMP_ZIP TEMP_DIR "MDSAI.ZIP"
#define TARGET "C:\\"

static int
get_num_archives()
{
	int rc, n;
	FILE *info;

	info = fopen("MDSAI.INF", "r");
	if (!info)
		return -1;

	rc = fscanf(info, "%d", &n);
	fclose(info);

	return (1 == rc) ? n : -1;
}

static int
copy(const char *src_path, FILE *dest)
{
	int rc = 0, c;
	long copied = 0;
	FILE *src = fopen(src_path, "rb");
	if (!src)
		return 1;

	printf("Copying files.");
	fflush (stdout);

	while ((c = fgetc(src)) != EOF)
	{
		rc = fputc(c, dest);
		if (EOF == rc)
			break;

		if ((++copied % 100000) == 0)
		{
			printf(".");
			fflush(stdout);
		}
	}

	printf("\r\n");

	fclose(src);
	return EOF == rc;
}

static bool
exists(const char *path)
{
	FILE *f = fopen(path, "rb");
	if (f)
	{
		fclose(f);
		return true;
	}

	return errno != ENOENT;
}

int
main(void)
{
	int rc, i, num_archives;
	FILE *zip;

	printf("Multi-disk split archive installer for DOS\r\nVersion 1.0.1\r\n\r\n");

	num_archives = get_num_archives();
	if (num_archives < 0)
	{
		fprintf(stderr,
			"Error determining number of archive parts\r\n");
		return 1;
	}

	if (0 == num_archives)
	{
		printf("No archive parts to process\r\n");
		return 0;
	}

	if (num_archives > 999)
	{
		fprintf(stderr, "Too many archives to process\r\n");
		return 1;
	}

	zip = fopen (TEMP_ZIP, "wb");
	if (!zip)
	{
		fprintf(stderr, "Error opening %s for writing\r\n", TEMP_ZIP);
		return 1;
	}

	for (i = 1; i <= num_archives; ++i)
	{
		char name[6];
		sprintf(name, "A.%d", i);

		while (!exists(name))
		{
			fprintf(stderr,
				"Doesn't look like disk %d. Try again? ", i);
			rc = getchar();

			if (toupper(rc) == 'Y')
			{
				while (getchar() != '\n')
					;
			}
			else
			{
				fprintf(stderr,
					"Failed to get all files...\r\n");
				goto end;
			}
		}

		rc = copy(name, zip);
		if (rc)
		{
			fprintf(stderr, "Error copying files...\r\n");
			goto end;
		}

		fflush(zip);

		if (i != num_archives)
		{
			printf("Please insert disk %d and press RETURN...",
				i + 1);

			while (getchar() != '\n')
				;
		}
	}

	fclose(zip);
	zip = NULL;

	rc = system("unzip -o -d " TARGET " " TEMP_ZIP);
	if (rc)
	{
		fprintf(stderr, "Error extracting files...\r\n");
		goto end;
	}

	printf("\r\nInstallation complete!\r\n");

end:
	if (zip)
		fclose(zip);

	if (exists(TEMP_ZIP))
		remove(TEMP_ZIP);

	if (exists(TEMP_ZIP))
		fprintf(stderr, "Failed to clean temp files...\r\n");

	return !!rc;
}
