/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2012, 2013. All rights reserved.
 */

#include <stdio.h>
#include <string.h>    /* strlen() */
#include <stdlib.h>    /* free() */
#include <unistd.h>    /* rmdir(), unlink() */
#include "fileexst.h"
#include "getdelim.h"
#include "helpers.h"   /* slash2backslash() */
#include "kprintf.h"
#include "pkgrem.h"
#include "rtrim.h"


/* removes a package from the system. Returns 0 on success, non-zero otherwise. */
int pkgrem(char *pkgname, char *dosdir) {
  struct dirliststruct {
    char *dirname;
    struct dirliststruct *next;
  };
  char fpath[512];
  char shellcmd[512];
  char *lineptr;
  FILE *flist;
  int getdelimlen;
  size_t getdelimcount = 0;
  struct dirliststruct *dirlist = NULL; /* used to remember directories to remove */
  char pkglistfile[512];

  /* Check if the file %DOSDIR%\packages\pkgname.lst exists (if not, the package is not installed) */
  sprintf(fpath, "%s\\packages\\%s.lst", dosdir, pkgname);
  if (fileexists(fpath) == 0) { /* file does not exist */
    kitten_printf(4, 0, "Package %s is not installed, so not removed.", pkgname);
    puts("");
    return(-1);
  }

  /* open the file %DOSDIR%\packages\pkgname.lst */
  flist = fopen(fpath, "r");
  if (flist == NULL) {
    kitten_puts(4, 1, "Error opening lst file!");
    return(-2);
  }

  sprintf(pkglistfile, "packages\\%s.lst", pkgname);

  /* remove all files/folders listed in pkgname.lst but NOT pkgname.lst */
  for (;;) {
    /* read line from file */
    lineptr = NULL;
    getdelimlen = getdelim(&lineptr, &getdelimcount, '\n', flist);
    if (getdelimlen < 0) {
      free(lineptr);
      break;
    }
    rtrim(lineptr);  /* right-trim the filename */
    slash2backslash(lineptr); /* change all / to \ */
    if ((lineptr[0] == 0) || (lineptr[0] == '\r') || (lineptr[0] == '\n')) continue; /* skip empty lines */
    if (lineptr[strlen(lineptr) - 1] == '\\') { /* it's a directory */
        /* remember it for later - I must remove directories last */
        struct dirliststruct *newdir;
        newdir = malloc(sizeof(struct dirliststruct));
        if (newdir == NULL) {
            kitten_printf(4, 3, "Out of memory! Could not store directory %s!", lineptr);
            puts("");
          } else {
            lineptr[strlen(lineptr) - 1] = 0; /* remove the ending \ */
            newdir->next = dirlist;
            newdir->dirname = strdup(lineptr);
            dirlist = newdir;
        }
      } else { /* it's a file */
        if (strcasecmp(pkglistfile, lineptr) != 0) { /* never delete pkgname.lst at this point - it will be deleted later */
          if ((lineptr[0] == '\\') || (lineptr[1] == ':')) { /* this is an absolute path */
              sprintf(shellcmd, "%s", lineptr);
            } else { /* else it's a relative path starting at %dosdir% */
              sprintf(shellcmd, "%s\\%s", dosdir, lineptr);
          }
          kitten_printf(4, 4, "removing %s", shellcmd);
          puts("");
          unlink(shellcmd);
        }
    }
    free(lineptr); /* free the memory occupied by the line */
  }

  /* close the file */
  fclose(flist);

  /* iterate through dirlist and remove directories if empty, from longest to shortest */
  while (dirlist != NULL) {
    struct dirliststruct *dirlistpos, *previousdir;
    /* find the longest path, and put it on top */
    previousdir = dirlist;
    for (dirlistpos = dirlist->next; dirlistpos != NULL; dirlistpos = dirlistpos->next) {
      if (strlen(dirlistpos->dirname) > strlen(dirlist->dirname)) {
          previousdir->next = dirlistpos->next;
          dirlistpos->next = dirlist;
          dirlist = dirlistpos;
          dirlistpos = previousdir;
        } else {
          previousdir = dirlistpos;
      }
    }
    if ((dirlist->dirname[0] == '\\') || (dirlist->dirname[1] == ':')) { /* this is an absolute path */
        sprintf(shellcmd, "%s", dirlist->dirname);
      } else { /* else it's a relative path starting at %dosdir% */
        sprintf(shellcmd, "%s\\%s", dosdir, dirlist->dirname);
    }
    /* printf("RMDIR %s\n", shellcmd); */
    rmdir(shellcmd);
    /* free the allocated memory for this entry */
    dirlistpos = dirlist;
    dirlist = dirlistpos->next;
    free(dirlistpos);
  }

  /* remove %DOSDIR%\packages\pkgname.lst */
  unlink(fpath);
  kitten_printf(4, 5, "Package %s has been removed.", pkgname);
  puts("");
  return(0);
}
