/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2012, 2013
 */

#include <stdio.h>
#include <stdlib.h> /* system() */
#include <string.h> /* strcpy() */
#include <unistd.h> /* read() */
#include "htget.h"
#include "helpers.h"   /* slash2backslash(), strtolower() */
#include "fileexst.h"
#include "kprintf.h"
#include "pkginst.h"
#include "libunzip.h"  /* callunzip(), zip_listfiles()... */
#include "crc32.h"    /* all CRC32 related stuff */



void pkginstall(struct pkgdb *pkgdb, char *pkgname, char *dosdir, char *tempdir, char *localfile, int nosourceflag, char **repolist, struct customdirs *dirlist) {
  char fname[512];
  char shellcmd[512];
  char zipfile[512];
  char appinfofile[64];
  char packageslst[64];
  int appinfopresence;
  char *instrepo, *shortfile;
  struct pkgdb *pkgnode, *lastnode;
  struct pkgrepo *pkgrepo;
  int repoid;
  unsigned long zipfilecrc;
  unsigned char *buff;
  int buffreadres;
  struct ziplist *ziplinkedlist, *curzipnode, *prevzipnode;
  FILE *fd;

  strtolower(pkgname); /* convert pkgname to lower case, because the http repo is probably case sensitive! */
  sprintf(appinfofile, "appinfo\\%s.lsm", pkgname); /* Prepare the appinfo/xxxx.lsm filename string for later use */
  sprintf(packageslst, "packages\\%s.lst", pkgname); /* Prepare the packages/xxxx.lst filename string for later use */

  if (localfile != NULL) {  /* if it's a local file, then we will have to skip all the network stuff */
      strcpy(zipfile, localfile);
    } else {
      zipfile[0] = 0;
  }

  /* check if not already installed, if already here, print a message "you need to remove it first" */
  sprintf(fname, "%s\\packages\\%s.lst", dosdir, pkgname);
  if (fileexists(fname) != 0) { /* file exists -> package already installed */
    kitten_printf(3, 0, "Package %s is already installed! Remove it first if you need to upgrade.", pkgname);
    puts("");
    return;
  }

  if (zipfile[0] == 0) { /* need to download the package from a repository */
    /* look into the db to find the package */
    pkgnode = findpkg(pkgdb, pkgname, &lastnode);
    if (pkgnode == NULL) { /* no such package found in repositories */
      kitten_printf(3, 1, "No package '%s' found in online repositories.", pkgname);
      puts("");
      return;
    }

    /* if found - check the list of repositories */
    if (pkgnode->repolist == NULL) {
      kitten_printf(3, 2, "Package '%s' is not available in repositories.", pkgname);
      puts("");
      return;
    }

    if (pkgnode->repolist->nextrepo != NULL) { /* available from more than 1 repo.. */
        char userchoicestr[8];
        int userchoice;
        struct pkgrepo *xrepo;
        puts("");
        kitten_printf(3, 3, "%s is available from several repositories. Choose which one to use:", pkgname);
        puts("");
        repoid = 1;
        for (xrepo = pkgnode->repolist; xrepo != NULL; xrepo = xrepo->nextrepo) {
          printf(" %d) %s %s (%s)\r\n", repoid, pkgnode->name, xrepo->version, repolist[xrepo->repo]);
          repoid += 1;
        }
        for (;;) {
          kitten_printf(3, 4, "Your choice:");
          printf(" ");
          fgets(userchoicestr, 6, stdin);
          userchoice = atoi(userchoicestr);
          if ((userchoice < 1) || (userchoice >= repoid)) {
              kitten_puts(3, 5, "Invalid choice!");
            } else {
              break;
          }
        }
        repoid = userchoice;
      } else { /* available only from one repository - get it there */
        repoid = 1;
    }
    pkgrepo = pkgnode->repolist;
    for (; repoid > 1; repoid--) pkgrepo = pkgrepo->nextrepo;
    instrepo = repolist[pkgrepo->repo];

    /* if it's a network repo, download the package from repoid into the temp directory */
    if (detect_localpath(instrepo) == 0) {
        sprintf(fname, "%s%s.zip", instrepo, pkgname);
        sprintf(zipfile, "%s\\fdnpkg.tmp", tempdir);
        kitten_printf(3, 6, "Downloading package %s...", fname);
        puts("");
        if (htget(fname, NULL, NULL, 0, zipfile) != 0) {
          kitten_puts(3, 7, "Error downloading package. Aborted.");
          return;
        }
      } else { /* else it's an on-disk repo, so we can use the package right from there */
        sprintf(zipfile, "%s%s.zip", instrepo, pkgname);
    }
    /* check the CRC of the downloaded file */
    buff = malloc(4096);  /* use a 4K buffer to compute file's CRC */
    if (buff == NULL) {
      kitten_puts(3, 15, "Error: Out of memory while computing the CRC of the package!");
      return;
    }
    fd = fopen(zipfile, "rb");
    if (fd == NULL) {
      kitten_puts(3, 14, "Error: Failed to open the downloaded package. Installation aborted.");
      free(buff);
      return;
    }
    zipfilecrc = crc32_init();
    while ((buffreadres = read(fileno(fd), buff, 4096)) > 0) {
      crc32_feed(&zipfilecrc, buff, buffreadres);
    }
    crc32_finish(&zipfilecrc);
    fclose(fd);
    free(buff);
    if (zipfilecrc != pkgrepo->crc32) {
      kitten_puts(3, 13, "Error: Downloaded package had wrong CRC. Installation aborted.");
      return;
    }
  } /* if (zipfile[0] == 0) */

  /* list content of the ZIP file and put it into pkgname.lst - used also to check the validity of the zip file */
  ziplinkedlist = zip_listfiles(zipfile);
  if (ziplinkedlist == NULL) {
    kitten_puts(3, 8, "Error: Invalid zip archive! Package not installed.");
    return;
  }
  /* Verify that there's no collision with existing local files, look for the appinfo presence and get rid of sources if required */
  appinfopresence = 0;
  prevzipnode = NULL;
  for (curzipnode = ziplinkedlist; curzipnode != NULL;) {
    slash2backslash(curzipnode->filename); /* change all slashes to backslashes */
    strtolower(curzipnode->filename);  /* put everything into lowercase */
    if (nosourceflag != 0) { /* if --nosource specified, skip sources */
      if (fdnpkg_strcasestr(curzipnode->filename, "source\\") == curzipnode->filename) { /* drop this file */
        curzipnode->filename[0] = 0; /* in fact, we just mark the file as 'empty' on the filename.. see below */
      }
    }
    if (curzipnode->filename[0] == 0) { /* ignore empty filenames (maybe it was empty originally, or has been emptied because it's a dropped source) */
      if (prevzipnode == NULL) {  /* take the item out of the list */
          ziplinkedlist = curzipnode->nextfile;
          free(curzipnode); /* free the item */
          curzipnode = ziplinkedlist;
        } else {
          prevzipnode->nextfile = curzipnode->nextfile;
          free(curzipnode); /* free the item */
          curzipnode = prevzipnode->nextfile;
      }
      continue; /* go to the next item */
    }
    if (strcmp(curzipnode->filename, appinfofile) == 0) appinfopresence = 1;
    shortfile = computelocalpath(curzipnode->filename, shellcmd, dosdir, dirlist);
    strcat(shellcmd, shortfile);
    if (fileexists(shellcmd)) {
      kitten_puts(3, 9, "Error: Package contains a file that already exists locally:");
      printf(" %s\r\n", shellcmd);
      zip_freelist(&ziplinkedlist);
      return;
    }
    if (curzipnode->isadir != 0) { /* it it's a directory, make sure it will appear first in the list */
      if (prevzipnode != NULL) { /* do it only if it's not already at the top of the list */
        prevzipnode->nextfile = curzipnode->nextfile;
        curzipnode->nextfile = ziplinkedlist;
        ziplinkedlist = curzipnode;
        curzipnode = prevzipnode->nextfile;
        continue;
      }
    }
    prevzipnode = curzipnode;
    curzipnode = curzipnode->nextfile;
  }
  if (appinfopresence != 1) { /* if appinfo file not found, this is not a real FreeDOS package */
    kitten_printf(3, 12, "Error: Package do not contain the %s file! Not a valid FreeDOS package.", appinfofile);
    zip_freelist(&ziplinkedlist);
    return;
  }
  /* open the lst file */
  sprintf(shellcmd, "%s\\%s", dosdir, packageslst);
  fd = fopen(shellcmd, "w");
  if (fd == NULL) {
    kitten_printf(3, 10, "Error: Could not create %s!", shellcmd);
    puts("");
    zip_freelist(&ziplinkedlist);
    return;
  }
  /* write list of files in zip into the lst, and create the directories structure */
  for (curzipnode = ziplinkedlist; curzipnode != NULL; curzipnode = curzipnode->nextfile) {
    if (strcasecmp(curzipnode->filename, packageslst) == 0) continue; /* skip silently the package.lst file, if found */
    shortfile = computelocalpath(curzipnode->filename, fname, dosdir, dirlist); /* substitute paths to custom dirs */
    fprintf(fd, "%s%s\r\n", fname, shortfile); /* log the filename to packages\pkg.lst */
    if (curzipnode->isadir != 0) { /* if a dir, create it */
        printf(" mkdir %s\r\n", fname);
        mkpath(fname);
      } else { /* if a file, extract it */
        /* Now unzip files into %DOSDIR% (via the unzip -d directory switch) */
        sprintf(shellcmd, "unzip -j -qq -C %s %s -d %s", zipfile, curzipnode->filename, fname);
        /* printf("# %s\r\n", shellcmd); */
        printf(" %s -> %s\r\n", curzipnode->filename, fname);
        callunzip(shellcmd);
    }
  }
  /* free the ziplist and close file descriptor */
  zip_freelist(&ziplinkedlist);
  fclose(fd);
  if (nosourceflag == 0) {
      kitten_printf(3, 11, "Package %s installed (with possible sources).", pkgname);
    } else {
      kitten_printf(3, 11, "Package %s installed (without sources).", pkgname);
  }
  puts("");
}
