/*
 * This function provides a few operations to handle zip files.
 * It also provides a wrapper to call UNZIP (when compiled as an object)
 *
 * Copyright (C) Mateusz Viste 2012
 */


#include <stdio.h>  /* printf(), FILE, fclose()... */
#include <stdlib.h> /* NULL */
#include "parsecmd.h"
#include "libunzip.h"
#include "kprintf.h"

extern int UzpMain(); /* UzpMain is the UNZIP's entry point when compiled as an object */


/* this function takes as parameter the full command line that one would use to execute UNZIP (for example 'unzip -Z file.zip'). It returns the error code that would be returned by UNZIP on command line. */
int callunzip(char *params) {
  int argc;
  char *argv[64];
  argc = parsecmd(params, argv, 63);
  argv[argc] = NULL;
  /* call the unzip function */
  return(UzpMain(argc, argv));
}


/* opens a zip file and provides the list of files in the archive.
   returns a pointer to a ziplist (linked list) with all records, or NULL on error.
   The ziplist is allocated automatically, and must be freed via zip_freelist. */
struct ziplist *zip_listfiles(char *zipfile) {
  struct ziplist *reslist = NULL;
  struct ziplist *newentry;
  FILE *fd;
  unsigned int entrysig;
  unsigned int filenamelen, extrafieldlen, filecommentlen, compfilelen;
  int x, bytebuff, eofflag;
  unsigned int ux;
  unsigned char hdrbuff[64];
  fd = fopen(zipfile, "rb");
  if (fd == NULL) return(NULL);
  for (;;) { /* read entry after entry */
    entrysig = 0;
    eofflag = 0;
    /* read the entry signature first */
    for (x = 0; x < 4; x++) {
      if ((bytebuff = fgetc(fd)) == EOF) {
        eofflag = 1;
        break;
      }
      entrysig |= (bytebuff << (x * 8));
    }
    if (eofflag != 0) break;
    /* printf("sig: 0x%08x\r\n", entrysig); */
    if (entrysig == 0x04034b50) { /* local file */
        /* create new entry and link it into the list */
        newentry = malloc(sizeof(struct ziplist));
        if (newentry == NULL) {
          kitten_puts(8, 0, "Out of memory!");
          zip_freelist(&reslist);
          break;
        }
        newentry->filename = NULL;
        newentry->nextfile = reslist;
        newentry->isadir = 0;
        reslist = newentry;
        /* parse header now */
        fread(hdrbuff, 1, 26, fd);
        compfilelen = hdrbuff[14] | (hdrbuff[15] << 8) | (hdrbuff[16] << 16) | (hdrbuff[17] << 24);
        filenamelen = hdrbuff[22] | (hdrbuff[23] << 8);
        extrafieldlen = hdrbuff[24] | (hdrbuff[25] << 8);
        /* printf("Filename len: %d / extrafield len: %d / Compfile len: %d\r\n", filenamelen, extrafieldlen, compfilelen); */
        newentry->filename = calloc(filenamelen + 1, 1); /* alloc space for filename */
        if (newentry->filename == NULL) {
          kitten_puts(8, 0, "Out of memory!");
          zip_freelist(&reslist);
          break;
        }
        for (ux = 0; ux < filenamelen; ux++) newentry->filename[ux] = fgetc(fd); /* store filename */
        if ((newentry->filename[filenamelen - 1] == '/') || (newentry->filename[filenamelen - 1] == '\\')) newentry->isadir = 1; /* if filename ends with / or \ it's a dir */
        /* printf("Filename: %s\r\n", newentry->filename); */
        /* skip rest of fields and data */
        fseek(fd, (extrafieldlen + compfilelen), SEEK_CUR);
      } else if (entrysig == 0x02014b50) { /* central directory */
        /* parse header now */
        fread(hdrbuff, 1, 42, fd);
        filenamelen = hdrbuff[22] | (hdrbuff[23] << 8);
        extrafieldlen = hdrbuff[24] | (hdrbuff[25] << 8);
        filecommentlen = hdrbuff[26] | (hdrbuff[27] << 8);
        compfilelen = hdrbuff[14] | (hdrbuff[15] << 8) | (hdrbuff[16] << 16) | (hdrbuff[17] << 24);
        /* printf("central dir\r\n"); */
        /* skip rest of fields and data */
        fseek(fd, (filenamelen + extrafieldlen + compfilelen + filecommentlen), SEEK_CUR);
      } else { /* unknown sig */
        kitten_printf(8, 1, "unknown zip sig: 0x%08x", entrysig);
        puts("");
        zip_freelist(&reslist);
        break;
    }
  }
  fclose(fd);
  return(reslist);
}


/* Call this to free a ziplist computed by zip_listfiles() */
void zip_freelist(struct ziplist **ziplist) {
  struct ziplist *zipentrytobefreed;
  while (*ziplist != NULL) { /* iterate through the linked list and free all nodes */
    zipentrytobefreed = *ziplist;
    *ziplist = zipentrytobefreed->nextfile;
    /* free memory storing the filename (if allocated) */
    if (zipentrytobefreed->filename != NULL) free(zipentrytobefreed->filename);
    /* free the linked list entry itself */
    free(zipentrytobefreed);
  }
  *ziplist = NULL;
}
