/*

    This file is the source for a series of routines to detect the make and
    model of the CPU the program is running on.
    Copyright (C) 1998 by Phil Brutsche

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA  02111-1307  USA.

*/

#include <dos.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "xms.hpp"

#define CPUID db 0x0F, 0xA2
#define OPND32 db 0x66
#define STR_EQU 0

#define toupper(value) (((value) > 'a' && (value) < 'z') ? value + 32 : value)

char cputype, check_fpu, extensive = 0;

void getDrives (void) {
   char result;
   result = outp (0x70, 0x10);
   result = inp (0x71);
   if (result == 0x00) {
      fprintf (stdout, "No disk drives\n");
      return;
   }
   switch ((result & 0xF0) >> 4) {
      case 0x01: fprintf (stdout, "360kb disk drive on A:");
         break;
      case 0x02: fprintf (stdout, "1.2mb disk drive on A:");
         break;
      case 0x03: fprintf (stdout, "720kb disk drive on A:");
         break;
      case 0x04: fprintf (stdout, "1.44mb disk drive on A:");
         break;
      case 0x00: fprintf (stdout, "No disk drive on A:");
         break;
   }
   printf (", ");
   switch (result & 0x0F) {
      case 0x01: fputs ("360kb disk drive on B:\n", stdout);
         break;
      case 0x02: fputs ("1.2mb disk drive on B:\n", stdout);
         break;
      case 0x03: fputs ("720kb disk drive on B:\n", stdout);
         break;
      case 0x04: fputs ("1.44mb disk drive on B:\n", stdout);
         break;
      case 0x00: fputs ("No disk drive on B:\n", stdout);
         break;
   }
}

void getMath (void)
{
   unsigned fp_status;
   char fpu_type = 0;	
   if (check_fpu == 1) {
      asm fninit;
      asm mov fp_status, 0x5a5a;
      asm fnstsw fp_status;
      asm mov ax, fp_status;
      asm cmp al, 0;
      asm mov fpu_type, 0;
      asm jne end_get_fpuid;
check_control_word:
      asm fnstcw fp_status;
      asm mov ax, fp_status;
      asm and ax, 0x103f;
      asm cmp ax, 0x3f;
      asm mov fpu_type, 0;
      asm jne end_get_fpuid;
      asm mov fpu_type, 1;
check_infinity:
      asm cmp cputype, 3;
      asm jne end_get_fpuid;
      asm fld1;
      asm fldz;
      asm fdiv;
      asm fld st;
      asm fchs;
      asm fcompp;
      asm fstsw fp_status;
      asm mov ax, fp_status;
      asm mov fpu_type, 2;
      asm sahf;
      asm jz end_get_fpuid;
      asm mov fpu_type, 3;
end_get_fpuid:
      switch (fpu_type) {
         case 0: fputs ("Math coprocessor not found\n", stdout);
            break;
         case 2: fputs ("Math coprocessor is Intel287\n", stdout);
            break;
         case 3: fputs ("Math coprocessor is 80387\n", stdout);
            break;
         default: fputs ("Math coprocessor found\n", stdout);
            break;
      }
   }
}

void getMemory (void) {
   unsigned low = outp (0x70, 0x15);
   unsigned high;
   unsigned result = initXMS (), freemem;
   int xmsMajor, xmsMinor;
   char far *ivec;
   char emsVer;
   unsigned pages_free, total_pages;
   low = inp (0x71);
   high = outp (0x70, 0x16);
   high = inp (0x71);
   fprintf (stdout, "%dK base, ", (high << 8) + low);
   low = outp (0x70, 0x17);
   low = inp (0x71);
   high = outp (0x70, 0x18);
   high = inp (0x71);
   fprintf (stdout, "%uK extended memory found\n", (high << 8) + low);
   if (result) {
      freemem = xmsMemAvail ();
      getXMSVersion (&xmsMajor, &xmsMinor);
      fprintf (stdout, "XMS version %d.%d found with %uK free\n\r", xmsMajor, xmsMinor, freemem);
   } else
      fprintf (stdout, "No XMS memory found\n\r");
/* now to check for EMS! */
   ivec = (char far *)_dos_getvect (0x67);
   ivec = (char far *)MK_FP (FP_SEG (ivec), 10);
   if (! _fstrncmp (ivec, "EMMXXXX0", 8)) {
      asm mov ah, 0x46;
      asm int 0x67;
      asm cmp ah, 0x00;
      asm je ok;
      asm mov emsVer, 0;
      asm jmp print_message;
ok:
      asm mov emsVer, al
      asm mov ah, 0x42;
      asm int 0x67;
      asm cmp ah, 0x00;
      asm je get_mem;
      asm mov pages_free, 0;
      asm mov total_pages, 0;
      asm jmp print_message;
get_mem:
      asm mov total_pages, dx;
      asm mov pages_free, bx;
print_message:
      fprintf (stdout, "EMS version %d.%d found with %dK of %dK free\n\r", emsVer % 10, emsVer / 10, pages_free * 16, total_pages * 16);
   } else
      fprintf (stdout, "No EMS memory found\n\r");
}

void getDisk (void)
{
   struct diskfree_t free;
   float avail, total;
   char i;
   unsigned driveNum, _ax, _bx;
   _dos_getdrive (&driveNum);
   if (_dos_getdiskfree (driveNum, &free) != 0) {
      avail = 0;
      total = 0;
   } else {
      avail = ((float) free.avail_clusters * (long) free.bytes_per_sector * (long) free.sectors_per_cluster) / 1024 / 1024;
      total = ((float) free.total_clusters * (long) free.bytes_per_sector * (long) free.sectors_per_cluster) / 1024 / 1024;
   }
   fprintf (stdout, "drive %c: %.2f megs available, %.2f megs total, %.2f%% utilization\n", toupper (driveNum + 0x40), (float)(avail), (float)(total), ((total - avail) / total) * 100.0);
   if (extensive)
      fprintf (stdout, "%5cdrive %c has %lu byte clusters\n", ' ', driveNum + 0x40, (long)(free.bytes_per_sector) * (long)(free.sectors_per_cluster));
   asm mov ax, 0x150b;
   asm mov cx, 0;
   asm int 0x2f;
   asm mov _ax, ax;
   asm mov _bx, bx;
   if (_bx != 0xadad) {
      fputs ("CD-ROM Extensions not loaded\n", stdout);
      return;
   } else
      fputs ("CD-ROM Extensions loaded\n", stdout);
   if (extensive) {
      fprintf (stdout, "%5cCD-ROM drives are ", ' ');
      for (i = 0; i <= 25; i++) {
         asm {
            mov ax, 0x150b
            mov cx, word ptr i
            int 0x2f
            mov _ax, ax
            mov _bx, bx
         }
         if (_ax != 0) {
            fprintf (stdout, "%c: ", i + 0x41);
         }
      }
      fprintf (stdout, "\n");
   }
}

void getProcessor (void)
{
   int cpuidok = 0;
   unsigned long eax;
   char model = 0, vendor_id [12], stepping;
   check_fpu = 1;
//   if (extensive) printf ("Checking for 8088");
   asm {
   /*  check to see if this is a first generation cpu.  if bits 12-16 are
     not toggleable, then it's 8088/8086.  we give our condolences and
     continue.
   */
/*      pushf
      pop ax
      mov cx, ax
      and ax, 0x0fff
      push ax
      popf
      pushf
      pop ax
      and ax, 0x0f000
      cmp ax, 0x0f000
      mov cputype, 0
      jne check80286
      jmp end*/
        pushf                   // push original FLAGS
        pop     ax              // get original FLAGS
        mov     cx, ax          // save original FLAGS
        and     ax, 0fffh       // clear bits 12-15 in FLAGS
        push    ax              // save new FLAGS value on stack
        popf                    // replace current FLAGS value
        pushf                   // get new FLAGS
        pop     ax              // store new FLAGS in AX
        and     ax, 0f000h      // if bits 12-15 are set, then CPU
        cmp     ax, 0f000h      //   is an 8086/8088
        mov     cputype, 0      // turn on 8086/8088 flag
        jne     check80286
        jmp     end             // jump if CPU is 8086/8088
   }
check80286:
//   if (extensive) printf (", 80286");
   /*  we know its at least a second generation cpu.  if bits 12-16 are
     toggleable, then it's 80286.  indistinguishable from a 80186.
   */
   asm {
/*      or cx, 0xF000
      push cx
      popf
      pushf
      pop ax
      and ax, 0xF000
      mov cputype, 2
      jnz check80386
      jmp end*/
        or      cx, 0f000h      // try to set bits 12-15
        push    cx              // save new FLAGS value on stack
        popf                    // replace current FLAGS value
        pushf                   // get new FLAGS
        pop     ax              // store new FLAGS in AX
        and     ax, 0f000h      // if bits 12-15 clear, CPU=80286
        mov     cputype, 2      // turn on 80286 flag
        jnz     check80386
        jmp     end             // if no bits set, CPU is 80286
   }
check80386:
//   if (extensive) printf (", 80386");
   /*  we know that this cpu supports 32-bit instructions, so its at least
     a 80386.  if bit 18 of eflags is toggleable, then we know that its
     a 80386 or clone and we stop.
   */
   asm {
/*      mov bx, sp
      and sp, not 3
      db 0x66
      pushf
      db 0x66
      pop ax
      db 0x66
      mov cx, ax
      db 0x66
      db 0x35
      dd 0x40000
      db 0x66
      push ax
      db 0x66
      popf
      db 0x66
      pushf
      db 0x66
      pop ax
      db 0x66
      xor ax, cx
      mov cputype, 3
      mov sp, bx
      jz end
      and sp, not 3
      db 0x66
      push cx
      db 0x66
      popf
      mov sp, bx*/
        mov     bx, sp          // save current stack pointer to align
        and     sp, not 3       // align stack to avoid AC fault
        OPND32
        pushf                   // push original EFLAGS
        OPND32
        pop     ax              // get original EFLAGS
        OPND32
        mov     cx, ax          // save original EFLAGS
        OPND32
        db 35h
        dd 40000h     // flip AC bit in EFLAGS
        OPND32
        push    ax              // save new EFLAGS value on stack
        OPND32
        popf                    // replace current EFLAGS value
        OPND32
        pushf                   // get new EFLAGS
        OPND32
        pop     ax              // store new EFLAGS in EAX
        OPND32
        xor     ax, cx          // can't toggle AC bit, CPU=80386
        mov     cputype, 3      // turn on 80386 CPU flag
        mov     sp, bx          // restore original stack pointer
        jz      end             // jump if 80386 CPU
        and     sp, not 3       // align stack to avoid AC fault
        OPND32
        push    cx
        OPND32
        popf                    // restore AC bit in EFLAGS first
        mov     sp, bx          // restore original stack pointer
   }
check80486:
//   if (extensive) printf (", 80486");
   /*  we know that its at least a 80486 or clone.  now we check for the
     cpuid instruction.  intel included it in 1992 on all processors, with
     the advent of the pentium.  (i mean, when intel started making the
     pentium, they introduced this instruction.  when they did that, they
     said:  what the hell, lets put it in everything we make from now on.
     and so we have it.
     we try to toggle eflags register 21.  if its toggleable, then the cpuid
     instruction is not supported.
     if its not, then we know that it's a 80486.
   */
   asm {
/*      mov cputype, 4
      OPND32
      mov ax, cx
      db 0x66
      db 0x35
      dd 0x200000
      db 0x66
      push ax
      db 0x66
      popf
      db 0x66
      pushf
      db 0x66
      pop ax
      db 0x66
      xor ax, cx
      mov cputype,4
      je end*/
        mov     cputype, 4     // turn on 80486 CPU flag
        OPND32
        mov     ax, cx          // get original EFLAGS
        OPND32
        db 35h
        dd 200000h              // flip ID bit in EFLAGS
        OPND32
        push    ax              // save new EFLAGS value on stack
        OPND32
        popf                    // replace current EFLAGS value
        OPND32
        pushf                   // get new EFLAGS
        OPND32
        pop     ax              // store new EFLAGS in EAX
        OPND32
        xor     ax, cx          // can't toggle ID bit,
        je      end             //   CPU=80486
   }
pentium:
//   if (extensive) puts (", P5+");
   cpuidok = 1;
   check_fpu = 0;
/* the following code uses the CPUID function to determine the cpu
   on input to cpuid:
      eax contains the function to execute:
         0 : get cpu vendor.  stored in bx, dx, and cx (in that order) in
            MSB order.
         1 : get cpu data.  stored in ax. indicates the family, model, and
            stepping.
         2 : get extended cpu data.  used only in pentium pro and higher
            only. */
   asm {
      OPND32; xor ax, ax
      mov ax, 1
      CPUID
      OPND32; mov word ptr eax, ax
      OPND32; xor ax, ax
      CPUID
      OPND32; mov word ptr vendor_id, bx
      OPND32; mov word ptr vendor_id[+4], dx
      OPND32; mov word ptr vendor_id[+8], cx
   }
   vendor_id [12] = 0;
   cputype = (eax & 0x0F00) >> 8;
   model = (eax & 0x00F0) >> 4;
   stepping = eax & 0x000F;
/*   if (extensive) {
      printf ("eax = 0x%08x, ", eax.data);
      printf ("Vendor ID = '%s'\n", _strncpy (vendor_id, 12));
      printf ("%13cfms\n", ' ');
      check_fpu = 1;
   }*/
end:
   if (cputype == 0) {
      fputs ("8086/8088 CPU found\n", stdout);
      return;
   } else if (cputype == 2) {
      fputs ("80286 CPU found\n", stdout);
      return;
   } else if (cputype == 3) {
      fputs ("80386 CPU found\n", stdout);
      return;
   } else if (cputype >= 4) {
      if (! cpuidok) {
         fputs ("80486 or clone CPU found\n", stdout);
         check_fpu = 1;
         return;
      }
      if (strncmp (vendor_id, "GenuineIntel", 12) == 0) {
         fprintf (stdout, "Intel ");
         if (cputype == 4) {
            fprintf (stdout, "80486");
            switch (model) {
               case 0: fprintf (stdout, "DX");
                  break;
               case 1: fprintf (stdout, "DX50");
                  break;
               case 2: fprintf (stdout, "SX");
                  break;
               case 3: fprintf (stdout, "DX2");
                  break;
               case 4: fprintf (stdout, "SL");
                  break;
               case 5: fprintf (stdout, "SX2");
                  break;
               case 7: fprintf (stdout, "DX2WB");
                  break;
               case 8: fprintf (stdout, "DX4");
                  break;
               case 9: fprintf (stdout, "DX4WB");
                  break;
            }
         } else if (cputype == 5) {
            fprintf (stdout, "Pentium");
            switch (model) {
               case 0:
               case 1: fprintf (stdout, ", 60-66 MHz");
                  break;
               case 2: fprintf (stdout, ", 75+ MHz");
                  break;
               case 3: fprintf (stdout, " P24T");
                  break;
               /*case 4: printf (" OverDrive for 3.3V Pentium");*/
               case 4: fprintf (stdout, " MMX");
                  break;
               case 5: fprintf (stdout, " OverDrive for 80486DX4");
                  break;
               case 6: fprintf (stdout, " OverDrive for 5V Pentium");
                  break;
            }
         } else if (cputype == 6) {
            switch (model) {
               case 0:
               case 2: fprintf (stdout, "Pentium Pro");
                  break;
               case 3: fprintf (stdout, "Pentium II");
                  break;
               case 4: fprintf (stdout, "P54C socket OverDrive");
                  break;
            }
         } else {
            fprintf (stdout, "\rUnidentifiable Intel CPU detected\n");
            check_fpu = 1;
            return;
         }
      } else if (strncmp (vendor_id, "UMC UMC UMC ", 12) == 0) {
         fprintf (stdout, "UMC486");
         switch (model) {
            case 1: fprintf (stdout, " U5D");
               break;
            case 2: fprintf (stdout, " U5S");
               break;
            default: fprintf (stdout, "\rUnidentifiable UMC CPU detected\n");
               return;
         }
      } else if (strncmp (vendor_id, "AuthenticAMD", 12) == 0) {
         fprintf (stdout, "American Micro Devices ");
         switch (cputype) {
            case 4:
               fprintf (stdout, "Am486");
               switch (model) {
                  case 0: fprintf (stdout, "DX");
                    break;
                  case 2: fprintf (stdout, "SX");
                     break;
                  case 3: fprintf (stdout, "DX2");
                     break;
                  case 4: fprintf (stdout, "SL");
                     break;
                  case 5: fprintf (stdout, "SX2");
                     break;
                  case 7: fprintf (stdout, "DX2WB");
                     break;
                  case 8: fprintf (stdout, "DX4");
                     break;
                  case 9: fprintf (stdout, "DX4WB");
                     break;
                  case 0x0F: fprintf (stdout, "\rAmerican Micro Devices Am5x86");
               }
               break;
            case 5:
               fprintf (stdout, "K5");
               switch (model) {
                  case 0: fprintf (stdout, " model 1");
                     break;
                  case 1: fprintf (stdout, " model 2");
                     break;
                  case 6: fprintf (stdout, "\rAmerican Micro Devices K6\n");
                     break;
               }
               break;
            default: fprintf (stdout, "\rUnidentifiable AMD CPU detected\n");
               return;
         }
      } else if (strncmp (vendor_id, "CyrixInstead", 12) == 0) {
         fprintf (stdout, "Cyrix ");
         switch (cputype) {
            case 4: fprintf (stdout, "Cx486");
               break;
            case 6: fprintf (stdout, "6x86");
               break;
            default: fprintf (stdout, "\rUnidentifiable Cyrix CPU detected\n");
               return;
         }
      } else if (strncmp (vendor_id, "NexGenDriven", 12) == 0)
         fputs ("NexGen Nx5x86 CPU found\n", stdout);
      else {
         fputs ("Unidentifiable cpu found\n", stdout);
         fputs ("Data Returned:\n", stdout);
         fprintf (stdout, "%5cvendor identification string = '%s'\n", vendor_id);
         fprintf (stdout, "%5c   model id returned = %d\n", ' ', model);
         fprintf (stdout, "%5c  family id returned = %d\n", ' ', cputype);
         fprintf (stdout, "%5cstepping id returned = %d\n", ' ', stepping);
         check_fpu = 1;
         return;
      }
   }
   fputs (" CPU found\n", stdout);
}

void getDateTime (void)
{
   struct time timeStruct;
   struct date dateStruct;
   unsigned hours, minutes, seconds;
   unsigned month, day, year;
   char locator [3];
   char *months [12] = { "January",
                         "February",
                         "March",
                         "April",
                         "May",
                         "June",
                         "July",
                         "August",
                         "September",
                         "October",
                         "November",
                         "December" };
   gettime (&timeStruct);
   hours = timeStruct.ti_hour;
   if (timeStruct.ti_hour > 12) {
      asm sub hours, 12;
      strcpy (locator, "PM");
   } else
      strcpy (locator, "AM");
   minutes = timeStruct.ti_min;
   seconds = timeStruct.ti_sec;
   fprintf (stdout, "System time is %2d:%02d:%02d %s and", hours, minutes, seconds, locator);
   getdate (&dateStruct);
   month = dateStruct.da_mon - 1;
   day = dateStruct.da_day;
   year = dateStruct.da_year;
   fprintf (stdout, " the system date is %s %d, %d\n\r", months [month], day, year);
}

void getOSVersion (void)
{
   char major, minor, oem;
   char far *e6vector;
   char far *date_string;
   Word dosemu_inst, patchlevel;
   _AH = 0x30;
   _AL = 0x00;
   __emit__ (0xCD, 0x21);
   major = _AL; minor = _AH; oem = _BH;
   if (major == 10)
      fputs ("OS/2 1.x compatibility mode found", stdout);
   else if (major == 20 && minor != 30)
      fputs ("OS/2 2.x compatibility mode found", stdout);
   else if (major == 20 && minor == 30)
      fputs ("OS/2 Warp 3.0 Virtual machine found", stdout);
   else if (major == 40)
     fputs ("OS/2 Warp 4.0 Virtual machine found", stdout);
   else if (major == 4) {
      asm mov al, 0x00;
      asm mov ah, 0x87;
      asm int 0x21;
      if (_AL)
         fputs ("European MS-DOS 4.0 found", stdout);
   } else if (major == 6 && minor == 0 && oem == 0x00)
      fputs ("PC-DOS 6.1 or later found", stdout);
   else {
      switch (oem) {
         case 0x00: fprintf (stdout, "PC-DOS version %d.%d found\n\r", major, minor);
            break;
         case 0x01: fprintf (stdout, "Compaq DOS version %d.%d found\n\r", major, minor);
            break;
         case 0x02: fprintf (stdout, "Microsoft Packaged DOS version %d.%d found\n\r", major, minor);
            break;
         case 0x33: case 0xEF: fprintf (stdout, "Novell DOS version %d.%d found\n\r", major, minor);
            break;
         case 0x5E: fprintf (stdout, "PTS-DOS version %d.%d found\n\r", major, minor);
            break;
         case 0xEE: fprintf (stdout, "DR-DOS version %d.%d found\n\r", major, minor);
            break;
         case 0xFD: fprintf (stdout, "FreeDOS version %d.%d found\n\r", major, minor);
            break;
         case 0xFF: default: fprintf (stdout, "Microsoft DOS version %d.%d found\n\r", major, minor);
            break;
      }
   }
// time to detect ms-windows!
   asm {
      mov ax, 0x1600
      int 0x2F
      mov major, al
      mov minor, ah
   }
   switch (major) {
      case 0x00: fputs ("Running under Standard Mode Windows 3.x or plain MS-DOS", stdout);
         break;
      case 0x01:
      case 0xFF: fputs ("Running under Windows/386 2.x", stdout);
         break;
      default:
         if (major == 4 && minor == 0)
            fprintf (stdout, "Running under Windows95\n");
         else
            fprintf (stdout, "Running under Windows %d.%d\n", major, minor);
   }
   e6vector = (char far *)_dos_getvect (0x67);
   date_string = (char far *)MK_FP (0xf000, 0xfff5);
   if (_fstrncmp (date_string, "02/25/93", 8) == 0 && FP_SEG (e6vector) == 0xf000) {
      asm {
         mov ah, 0x00
         int 0xe6
         mov dosemu_inst, ax
         mov major, bh
         mov minor, bl
         mov patchlevel, cx
      }
      if (dosemu_inst == 0xaa55)
         fprintf (stdout, "Running DOSEmu %d.%d patchlevel %d\n", major,
                  minor, patchlevel);
   } else
      fprintf (stdout, "Not running under DOSEmu\n");
}

void getPorts (void) {
   Word i, j = 0;
   for (i = 0; i < 0x08; i += 2)
      if (*((Word *)MK_FP (0x40, i)) != 0)
         j++;
   /*   printf ("COM%d is 0x%04x\n", j, *((Word *)MK_FP (0x40, i))); */
   printf ("%d communications port(s) found\n", j);
   if (extensive && j) {
      j = 1;
      i = 0;
      fprintf (stdout, "%5c", ' ');
      for (i = 0; i < 0x08; i += 2)
         if (*((Word *)MK_FP (0x40, i)) != 0) {
            fprintf (stdout, " COM%d at 0x%04x", j, *((Word *)MK_FP (0x40, i)));
            j++;
         }
      printf ("\n");
   }
   i = 0x06;
   j = 0;
   do {
     i += 2;
     j++;
   } while ((i <= 0x0e) && (*((Word *)MK_FP (0x40, i)) != 0));
   fprintf (stdout, "%d printer port(s) found\n", j - 1);
   if (extensive && j) {
      j = 0;
      i = 0x06;
      fprintf (stdout, "%5c", ' ');
      do {
         i += 2;
         j++;
         if (*((Word *)MK_FP (0x40, i)) != 0) {
            fprintf (stdout, " LPT%d at 0x%04x", j, *((Word *)MK_FP (0x40, i)));
         }
      } while ((i <= 0x0e) && (*((Word *)MK_FP (0x40, i)) != 0));
      fprintf (stdout, "\n");
   }
}

void getDPMI (void) {
   Word entry_seg, entry_offs, paras, result, bit32progs;
   Byte cpu, major, minor;
   asm {
      mov ax, 0x1687
      int 0x2F
      mov result, ax
      mov cpu, cl
      mov paras, si
      mov entry_seg, es
      mov entry_offs, di
      mov major, dh
      mov minor, dl
      mov bit32progs, bx
   }
   if (result == 0x0000) {
      fprintf (stdout, "DPMI installed\n");
      if (extensive) {
         fprintf (stdout, "%5c80%d86 CPU reported by DPMI server\n", ' ', cpu);
         fprintf (stdout, "%5cServer supports the version %d.%d spec\n", ' ', major, minor);
         fprintf (stdout, "%5c%d paragraphs for extender private data\n", ' ', paras);
         fprintf (stdout, "%5cMode switch entry point at 0x%04x:0x%04x\n", ' ', entry_seg, entry_offs);
         if ((bit32progs & 0x01) != 0)
            fprintf (stdout, "%5c32-bit programs supported\n", ' ');
         else
            fprintf (stdout, "%5c32-bit programs unsupported\n", ' ');
      }
   } else
     fprintf (stdout, "DPMI not installed\n");
}

void main (int argc, char *argv [])
{
   int i;
   for (i = 1; i < argc; i++)
      if (strcmp ("-E", strupr (argv [i])) == STR_EQU)
         extensive = 1;
   getDrives ();
   getDisk ();
   getProcessor ();
   getMath ();
   getMemory ();
   getPorts ();
   getDateTime ();
   getOSVersion ();
   getDPMI ();
}
