/*
 * This file is part of the uHex project.
 * Copyright (C) 2013, 2014, 2015 Mateusz Viste
 *
 * This is the (original) I/O driver for DOS. For non-DOS support, see the
 * io-curse.c file.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 */

#include <dos.h>   /* provides int86() along with the union REGS type */
#include <stdio.h> /* this one contains the NULL definition */
#include "io.h"    /* include self headers for control */

int savedattr = 0; /* used to restore the same video mode before quitting */
int cursor_start = 0, cursor_end = 0; /* remember the cursor's shape */


static void readchar(char *c, int *attr) {
  union REGS regs;
  regs.h.ah = 0x08; /* Read character and attribute at cursor position */
  regs.h.bh = 0;    /* display page */
  int86(0x10, &regs, &regs);
  if (c != NULL) *c = regs.h.al;
  if (attr != NULL) *attr = regs.h.ah;
}


static void cursor_set(int startscanline, int endscanline) {
  union REGS regs;
  regs.h.ah = 0x01;
  regs.h.ch = startscanline;
  regs.h.cl = endscanline;
  int86(0x10, &regs, &regs);
}


/* gets cursor's position and size */
static void cursor_getprops(int *x, int *y, int *start, int *end) {
  union REGS regs;
  regs.h.ah = 3; /* get cursor position and shape */
  regs.h.bh = 0; /* page number (pretty much always 0) */
  int86(0x10, &regs, &regs);
  if (x != NULL) *x = regs.h.dl; /* column */
  if (y != NULL) *y = regs.h.dh; /* row */
  if (start != NULL) *start = regs.h.ch; /* start scan line */
  if (end != NULL) *end = regs.h.cl; /* end scan line */
}


/* inits the IO subsystem */
void io_init(void) {
  readchar(NULL, &savedattr); /* there we save current color attributes for later restoration */
  cursor_getprops(NULL, NULL, &cursor_start, &cursor_end); /* save the current cursor's shape */
}


void io_close(void) {
  cls(savedattr); /* clears the screen and restore the original colors of the shell */
}


void cursor_hide(void) {
  cursor_set(0x0F, 0x0E); /* hide the cursor */
}


void cursor_show(void) {
  cursor_set(cursor_start, cursor_end); /* unhide the cursor */
}


void locate(int row, int column) {
  union REGS regs;
  regs.h.ah = 0x02;
  regs.h.bh = 0;
  regs.h.dh = row;
  regs.h.dl = column;
  int86(0x10, &regs, &regs);
}


void printchar(char c, int attr) {
  union REGS regs;
  regs.h.ah = 0x09; /* Write char and attribute */
  regs.h.al = c;
  regs.h.bh = 0;    /* display page */
  regs.h.bl = attr;
  regs.x.cx = 1;    /* write it 1 time */
  int86(0x10, &regs, &regs);
}


/* waits for a keypress and return it. Returns 0 for extended keystroke, then
   function must be called again to return scan code. */
int getkey(void) {
  union REGS regs;
  regs.h.ah = 0x08;
  int86(0x21, &regs, &regs);
  return(regs.h.al);
}


void cls(int colattr) {
  int termwidth, termheight, colorflag;
  union REGS regs;
  getcurvideomode(&termwidth, &termheight, &colorflag);
  regs.x.ax = 0x0600;  /* Scroll window up - entire window */
  regs.x.bx = (colattr << 8); /* Attribute to write */
  regs.x.cx = 0x0000;  /* Upper left */
  regs.x.dx = ((termheight - 1) << 8) | (termwidth - 1); /* Lower right */
  int86(0x10, &regs, &regs);
  locate(0,0);
}


void getcurvideomode(int *termwidth, int *termheight, int *colorflag) {
  union REGS regs;
  *termheight = peekb(0x40, 0x84) + 1;
  *termwidth = peek(0x40, 0x4A);
  if (*termwidth < 20) *termwidth = 25; /* some pre-EGA adapters don't return the height. Assume 25 then. */
  regs.h.ah = 0x0F;  /* get current video mode */
  int86(0x10, &regs, &regs);
  if ((regs.h.al == 0) || (regs.h.al == 2) || (regs.h.al == 7)) { /* detect mono modes */
      *colorflag = 0;
    } else { /* else we are in color */
      *colorflag = 1;
  }
}


int computecolor(int doscol) {
  return(doscol);
}
