	page	59,132
	title	UHDD -- DOS UltraDMA Disk Driver.
;
; UHDD is a non-caching UltraDMA disk driver for up to 26 BIOS disks
; on up to 10 SATA/IDE controllers.   Any size SATA or IDE disks can
; be used.   UHDD takes 640 bytes of upper/DOS memory plus 768 bytes
; of HMA space with a /H switch, or 1408 bytes of memory without /H.
; It also requests 128K of XMS memory as a buffer, for misaligned or
; other I-O unsuited to UltraDMA.   Without XMS, such input shall be
; "passed" to the BIOS for execution.
;
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
s	equ	<short>		;Make a conditional jump "short".
MAXBIOS	equ	26		;Maximum 26 BIOS disks or diskettes.
STACK	equ	412		;Driver local-stack size.
CMDTO	equ	00Ah		;Disk 500-msec min. command timeout.
STARTTO	equ	07Fh		;Disk 7-second startup timeout.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Error Return Codes.
;
DMAERR	equ	00Fh		;Hard disk DMA error.
CTLRERR equ	020h		;Hard disk controller not-ready.
DISKERR equ	0AAh		;Hard disk drive not-ready.
FAULTED	equ	0CCh		;Hard disk drive FAULTED.
HARDERR equ	0E0h		;Hard disk I-O error.
XMSERR	equ	0FFh		;XMS memory error.
;
; "Legacy IDE" Controller I-O Base Addresses.
;
NPDATA	equ	001F0h		;Normal primary      base address.
NSDATA	equ	00170h		;Normal secondary    base address.
APDATA	equ	001E8h		;Alternate primary   base address.
ASDATA	equ	00168h		;Alternate secondary base address.
;
; IDE Controller Register Offsets.
;
CDATA	equ	0		;Data port offset.
CDSEL	equ	6		;Disk-select and upper LBA offset.
CCMD	equ	7		;Command-register offset.
CSTAT	equ	7		;Primary-status register offset.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
MSEL	equ	0A0h		;"Master" device-select bits.
SSEL	equ	0B0h		;"Slave"  device-select bits.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; LBA "Device Address Packet" Layout.
;
DAP	struc
DapPL	db	?		;Packet length.
	db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
	db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (segment:offset).
DapLBA	dw	?		;48-bit logical block address (LBA).
DapLBA1	dd	?
DAP	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	(I_Stra-@)	;"Strategy" routine pointer.
	dw	(I_Init-@)	;Device-Interrupt routine pointer.
	db	'UHDD-SA$'	;Driver name, fixed to "UHDD-SA$".
;
; General Driver "Data Page" Variables.
;
VLF	db	0		;VDS "lock" flag (001h = buffer lock).
IOF	db	0		;I-O "busy" flag (001h = driver busy).
DMAAd	dw	0		;DMA status/command register address.
IdeDA	dw	0		;IDE data register address.
CStack	dd	0		;Caller's saved stack pointer.
;
; VDS Parameter Block and UltraDMA Command-List.
;
VDSLn	dd	(LMEnd-@)	;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;DMA 32-bit buffer address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; "PRD" Address and IDE Parameter Area.
;
PRDAd	dd	(IOAdr-@)	;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0,0,0		;IDE "upper" LBA bits 24-47.
SecCt	db	0		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
;
; BIOS "Active Unit" Table.   The beginning of this table MUST be at
;   driver address 07Fh or below, for BP-reg. addressing at "Entry".
;
Units	db	MAXBIOS dup (0)	;BIOS unit number for each device.
;
; Int 13h Entry Routine.   For a "CHS" request, the registers are:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number.  We handle 080h+ hard-disks.
;   ES:BX   I-O buffer address.
;
; For an "LBA" request, the registers are:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number.  We handle 080h+ hard-disks.
;   DS:SI   Pointer to Device Address Packet ("DAP") as shown above.
;
Entry:	pushf			;Save CPU flags & BP-reg.
	push	bp
	mov	bp,0		;Reset active-units table index.
@LastU	equ	[$-2].lw	;(Last-unit index, set by Init).
NextU:	dec	bp		;Any more active units to check?
	js s	QuickX		    ;No, not for us -- exit quick!
	cmp	dl,cs:[bp+Units-@]  ;Does request unit match table?
	jne s	NextU		    ;No, see if more units remain.
	cli			;Disable CPU interrupts.
	bts	cs:IOF.lb,0	;Is this driver currently "busy"?
	jc s	InvalF		;Yes?  Set "invalid function" & exit!
	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to our driver stack, to avoid
	pop	ss		;  CONFIG.SYS "STACKS=0,0" problems!!
	mov	sp,0		;Set starting stack address.
@Stack	equ	[$-2].lw	;(Initial stack address, Init set).
	sti			;Re-enable CPU interrupts.
	push	ds		;Save CPU segment registers.
	push	es
	pusha			;Save all 16-bit CPU registers.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	A20Req
	jc s	A20Err		;If "A20" failure, ABANDON request!
	popa			;Restore and restack all Int 13h I-O
	pop	es		;  parameters, after "A20 enable" --
	push	es		;  Looks wasteful, but one learns to
	pusha			;  never-NEVER "trust" external code!
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	db	0EAh		;Go to main Int 13h request handler.
@HDMain	dd	(HDReq-@)	;(Main entry address, set by Init).
PassRq:	call	A20Dis		;Pass request -- Disable "A20" line.
	cli			;Disable CPU interrupts.
	mov	[bx+IOF-@],bx	;Reset "busy" and "VDS lock" flags.
	popa			;Reload all 16-bit CPU registers.
	pop	es		;Reload CPU segment registers.
	pop	ds
	lss	sp,cs:CStack	;Switch back to caller's stack.
QuickX:	pop	bp		;Reload BP-reg. from caller's stack.
	popf			;Reload CPU flags saved on entry.
	db	0EAh		;Go to next routine in Int 13h chain.
@Prev13	dd	0		;(Prior Int 13h vector, set by Init).
InvalF:	mov	ah,001h		;We are BUSY, you fool!  Set "invalid
	jmp s	Quit		;  function" code and get out, QUICK!
Exit:	call	A20Dis		;Exit -- issue "A20 local disable".
	pop	ax		;Reload error code and error flag.
	popf
A20Err:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+15],al	;Set error code in exiting AX-reg.
	cli			;Disable CPU interrupts.
	mov	[bx+IOF-@],bx	;Reset "busy" and "VDS lock" flags.
	popa			;Reload all 16-bit CPU registers.
	pop	es		;Reload CPU segment registers.
	pop	ds
	lss	sp,cs:CStack	;Switch back to caller's stack.
Quit:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-reg. from caller's stack.
	popf			;Discard CPU flags saved on entry.
	iret			;Exit -- Int 13h data reloads flags!
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
A20Dis:	mov	ah,006h		;"A20 local disable" -- get XMS code.
A20Req:	db	09Ah		;Issue "A20" request to the XMS mgr.
@XMgr1:	clc			;(Without XMS, 090h above, and this
	mov	ax,00001h	; logic posts "No XMS error" status).
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	xor	bx,bx		;Rezero BX-reg. for relative logic.
	sub	al,1		;Zero AL-reg. if success, -1 if error.
	ret			;Exit.
LMEnd	equ	($+STACK)	;End of our driver "low-memory" space.
				;  All that follows can go in the HMA.
;
; Main Hard-Disk Request Handler.
;
HDReq:	cmp	dl,002h		;CHS or LBA read/write request?
	jne s	Pass		;No, not for us -- pass this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns s	ValSC		;No, go validate CHS sector count.
	mov	di,sp		;Point DI-reg. to current stack.
	push	ds		;Save this driver's DS-reg.
	mov	ds,[di+18]	;Reload "DAP" segment into DS-reg.
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get "DAP" I-O sector count).
	les	dx,[si].DapLBA1	;(Get "DAP" LBA bits 16-47).
	mov	di,es
	les	cx,[si].DapBuf	;(Get "DAP" I-O buffer address).
	mov	si,[si].DapLBA	;(Get "DAP" LBA bits 0-15).
	pop	ds		;(Reload this driver's DS-reg.).
	jne s	ValSC		;If no 64-bit buffer, check sector ct.
Pass:	push	ss		;Return to request "pass" routine.
	push	(PassRq-@)
	retf
ValSC:	dec	al		;Is sector count zero or over 128?
	js s	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js s	ZeroBX		;Yes, go zero driver's BX-reg.
	xchg	ax,cx		;CHS -- save request code & sectors.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		     ;Swap cylinder/head values.
	mov	al,cs:[bp+(CHSec-@)] ;Get CHS sectors per head.
@CHSec	equ	[$-2].lw
	or	al,al		     ;Are disk CHS values legitimate?
	jz s	Pass		     ;No?  Let BIOS run this request!
	push	ax		     ;Save CHS sectors/head value.
	mul	ah		     ;Convert head to sectors.
	add	si,ax		     ;Add result to starting sector.
	pop	ax		     ;Reload sectors per head.
	mul	cs:[bp+(CHSHd-@)].lb ;Convert cyl. to sectors.
@CHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		   ;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		   ;Get buffer offset in AX-register.
	xchg	ax,cx		   ;Swap offset with request/sectors.
ZeroBX:	xor	bx,bx		   ;Zero BX-reg. for relative logic.
	mov	[bx+VDSOf-@],cx    ;Set VDS I-O buffer address.
	mov	[bx+VDSSg-@],es
	mov	[bx+LBA-@],si	   ;Set 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	mov	bp,cs:[bp+(CtlUn-@)]  ;Get disk ctlr. & unit nos.
@CtlUn	equ	[$-2].lw
	mov	si,offset (Ctl1Tbl-@) ;Point to ctlr. addresses.
@CtlTbl	equ	[$-2].lw
	mov	cx,0007Ch
	and	cx,bp
	add	si,cx
	shr	cx,1
	add	si,cx
	push	cs:[si].dwd	   ;Set controller base address and
	pop	[bx+DMAAd-@].dwd   ;  primary-channel data address.
	rcr	bp,2		   ;Primary channel I-O request?
	jnc s	GetCmd		   ;Yes, get LBA28 or LBA48 commands.
	add	[bx+DMAAd-@].lw,8  ;Use secondary DMA controller.
	mov	cx,cs:[si+4]	   ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],cx
	nop			   ;(Unused alignment "filler").
GetCmd:	shr	dx,12		   ;Shift out LBA bits 16-27.
	or	di,dx		   ;Anything in LBA bits 28-47?
	jz s	LBA28		   ;No, use LBA28 read/write command.
	shl	ah,3		   ;LBA48 -- get command as 020h/030h.
	jmp s	SetCmd		   ;Go set LBA and IDE command bytes.
LBA28:	xchg	dh,[bx+LBA2-@]	   ;LBA28:  Reload & reset bits 24-27.
	or	ah,(DRCMD+1)	   ;Get LBA28 read/write command + 5.
SetCmd:	shl	bp,1		   ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		     ;Or in LBA28 bits 24-27, if any.
	mov	dh,005h		     ;Get final IDE command byte.
	xor	dh,ah		     ;(LBA28 C8h/CAh, LBA48 25h/35h).
	mov	[bx+DSCmd-@],dx	     ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	     ;Set disk I-O sector count.
	mov	ah,0		     ;Set VDS & I-O buffer lengths.
	shl	ax,1		     ;  We set bit 16 for 64K bytes
	mov	[bx+VDSLn+1-@],ax    ;  since a few "odd" chips may
	mov	[bx+IOLen+1-@],ax    ;  NOT work O.K. without it!
	or	[bx+IOAdr-@].dwd,-1  ;Invalidate VDS buffer address.
	mov	ax,08103h	     ;Get VDS "lock" parameters.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)	     ;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		     ;Issue VDS "lock" request.
	call	XMSRst		     ;Restore all critical settings.
	jc	Pass		     ;VDS error?  Pass this request!
	push	eax		     ;Save EAX-reg. for 32-bit work.
	mov	eax,[bx+IOAdr-@]     ;Get our 32-bit VDS address.
	cmp	eax,-1		     ;Is VDS address still all-ones?
	jb s	SetVLF		     ;No, go set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw  ;VDS logic is NOT present --
	shl	eax,4		     ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
	mov	[bx+IOAdr-@],eax     ;Set final 32-bit DMA address.
SetVLF:	adc	[bx+VLF-@],bl	     ;Set VDS "lock" flag from carry.
	pop	eax		     ;Reload EAX-reg. (unneeded now).
	test	[bx+IOAdr-@].lb,003h ;Is user buffer 32-bit aligned?
	jnz s	SetBuf		     ;No, use our XMS memory buffer.
	cmp	[bx+IOAdr-@].hw,009h ;Is DMA beyond our 640K limit?
	ja s	SetBuf		     ;Yes, use our XMS memory buffer.
	mov	cx,[bx+IOLen-@]	     ;Get (I-O length -)1 + I-O addr.
	dec	cx
	add	cx,[bx+IOAdr-@].lw   ;Will I-O cross a 64K boundary?
	jnc s	DirDMA		     ;No, do direct user-buffer DMA.
	nop			     ;(Unused alignment "filler").
SetBuf:	mov	[bx+IOAdr-@].dwd,0   ;Set our 32-bit XMS buffer addr.
@XBAddr	equ	[$-4].dwd	     ;(XMS buffer address, Init set).
	test	[bx+IOCmd-@].lb,012h ;Buffered I-O:  Write request?
	jnz s	BufOut		     ;Yes, use output routine below.
	call	DoDMA		;Buffered input -- read data to XMS.
	jc s	HDDone		;If error, post return code and exit.
	call	XMSMov		;Move data from XMS to user buffer.
	jmp s	HDDone		;Go post any return code and exit.
BufOut:	call	XMSMov		;Buffered output -- move data to XMS.
	jc s	HDDone		;If error, post return code and exit.
DirDMA:	call	DoDMA		;Do direct DMA or buffered output.
HDDone:	pushf			;Save error flag (carry) and code.
	push	ax
	shr	[bx+VLF-@].lb,1	;User I-O buffer "locked" by VDS?
	jnc s	HDEnd		;No, return to hard-disk exit rtn.
	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		;Issue VDS "unlock" request.
HDEnd:	push	ss		;Return to hard-disk exit routine.
	push	(Exit-@)
	retf
;
; Fixed Command-List XMS Block for moving our DMA command-list to XMS.
;
CLXBlk	dd	8		;Move length      (always 8 bytes).
	dw	0		;Source "handle"     (always zero).
CLXSA	dd	(IOAdr-@)	;Source address (seg. set by Init).
CLXDH	dw	0		;Destination handle  (set by Init).
CLXDA	dd	0		;Destination offset  (set by Init).
;
; User-Data XMS Block, used to move data to/from our XMS buffer.
;
XMSBlk	dd	0		;Move length  (user buffer length).
	dw	0		;Source "handle" (0 = seg./offset).
	dd	0		;Source address or offset.
	dw	0		;Destination handle.
	dd	0		;Destination address or offset.
;
; UltraDMA Controller I-O Address Table.   "Legacy" addresses are
;   required in this table, since a PCI BIOS may return zeros for
;   the primary/secondary addresses of a "Legacy IDE" controller.
;   If non-standard "legacy" addresses shall be used, this driver
;   MUST be re-assembled with those addresses in the table below!
;
Ctl1Tbl	dw	0FFFFh		;Controller 1 DMA base address.
Ctl1Pri	dw	NPDATA		;   "Legacy" primary   addresses.
Ctl1Sec	dw	NSDATA		;   "Legacy" secondary addresses.
Ctl2Tbl	dw	0FFFFh		;Controller 2 DMA base address.
Ctl2Pri	dw	APDATA		;   "Legacy" primary   addresses.
Ctl2Sec	dw	ASDATA		;   "Legacy" secondary addresses.
	dw	24 dup (0FFFFh)	;Controller 3 to 10 I-O addresses.
CtlTEnd	equ	$		   ;End of controller I-O tables.
CTLTSIZ	equ	(Ctl2Tbl-Ctl1Tbl)  ;Size of a controller I-O table.
;
; Hard-Disk and Diskette Parameter Tables.
;
CtlUn	db	MAXBIOS dup (0)	;Controller and unit number.
CHSec	db	MAXBIOS dup (0)	;Sectors-per-head table.
CHSHd	db	MAXBIOS dup (0)	;Heads-per-cylinder table.
;
; Subroutine to execute disk UltraDMA read and write requests.
;
DoDMA:	mov	si,(CLXBlk-@)	;Point to command-list XMS block.
CLXPtr	equ	[$-2].lw
	push	cs
	pop	ds
	call	XMSGo		;Move DMA command-list up to XMS.
	jc s	DoDMAE		;Command-list move ERROR?  Exit NOW!
	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	di,[bx+IdeDA-@]	;Get our disk's IDE base address.
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	lea	dx,[di+CDSEL]
	out	dx,al
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,((STARTTO*256)+FLT)	;Get timeout & "fault" mask.
	add	ch,es:[si]		;Set timeout limit in CH-reg.
	call	ChkRdy		      ;Await controller/disk ready.
DoDMAE:	jc s	DoDMAX		      ;If any errors, exit below!
	test	[bx+IOCmd-@].lb,012h  ;Is this a write request?
	jnz s	IniDMA		      ;Yes, init DMA transfer.
	mov	al,008h		;Get "DMA read" command bit.
IniDMA:	push	si		;Save BIOS timer pointer.
	mov	dx,[bx+DMAAd-@]	;Get DMA command-register address.
	mov	si,(PRDAd-@)	;Get Physical-Region Descriptor addr.
	out	dx,al		;Reset DMA commands and set DMA mode.
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	ax,00006h	;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	dx		;Point to PRD address register.
	inc	dx
	outsd			;Set PRD pointer to our DMA address.
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxTPar:	lea	dx,[di+CDATA+1]	;Point to IDE sector count reg. - 1.
IDEPar:	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc s	IDEPar		;Yes, loop back and output next one.
	jnz s	NxTPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	dec	dh		;"Legacy IDE" controller channel?
@DRQ:	jmp s	DMAGo		;No, forget about awaiting 1st DRQ.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ:	cmp	ch,es:[si]	;Too long without 1st data-request?
	je s	ErrDMA		;Yes?  Return carry and DMA error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		   ;Has 1st data-request arrived?
	jz s	ChkDRQ		   ;No, loop back and check again.
DMAGo:	mov	es:[si+HDI_OFS],bl ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	   ;Point to DMA command register.
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	inc	ax
	out	dx,al
ChkDMA:	inc	dx		;Point to DMA status register.
	inc	dx
	in	ax,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,es:[si]	;Has our DMA transfer timed out?
	je s	HltDMA		    ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl  ;Did BIOS get a disk interrupt?
	je s	ChkDMA		    ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
HltDMA:	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	ErrDMA		;No?  Return carry and DMA error!
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz s	ErrDMA		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy:	lea	dx,[di+CSTAT]	;Read IDE primary status.
	in	al,dx
	cmp	ch,es:[si]	;Too long without becoming ready?
	je s	RdyErr		;Yes?  Go see what went wrong.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s	ChkRdy		;Yes, loop back and check again.
	and	al,cl		;Disk "fault" or hard-error?
	jnz s	HdwErr		;Yes?  Go see what went wrong.
DoDMAX:	ret			;End of request -- exit.
ErrDMA:	mov	al,DMAERR	;BAAAD News!  Post DMA error code.
	stc			;Set carry flag (error!) and exit.
	ret
RdyErr:	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp s	WhichE		;Go see which error code to return.
HdwErr:	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:	jz s	Kaput		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
Kaput:	stc			;Set carry flag (error!) and exit.
	ret
;
; Subroutine to move user I-O data to or from our XMS buffer.
;
XMSMov:	push	[bx+VDSSg-@].lw	 ;Save user-buffer address.
	push	[bx+VDSOf-@].lw
	push	[bx+VDSLn-@].dwd ;Get user-buffer length.
	mov	si,(XMSBlk-@)	 ;Point to user-data XMS block.
XMSPtr	equ	[$-2].lw
	push	cs
	pop	ds
	pop	[si].dwd	;Set user-buffer length.
	lea	bx,[si+4]	;Set move indexes for a read.
	lea	di,[si+10]
	jz s	XMSSet		;Is this a read request?
	xchg	bx,di		;No, "swap" indexes for a write.
XMSSet:	mov	[bx].lw,0	;Set our XMS buffer "handle" & offset.
@XMHdl	equ	[$-2].lw
	mov	[bx+2].dwd,0
@XMOffs	equ	[$-4].dwd
	and	[di].lw,0	;Set user buffer "handle" to zero.
	pop	[di+2].dwd	;Set user-buffer address.
XMSGo:	mov	ah,00Bh		;Get XMS "move memory" code.
XMSReq:	db	09Ah		;Issue XMS "move memory" request.
@XMgr2:	clc			;(Without XMS, 090h above, and this
	mov	ax,00001h	; logic posts "No XMS error" status).
	sub	al,1		;Zero AL-reg. if success, -1 if error.
XMSRst:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!
	push	ss		;  UHDD will always set its own stack
	pop	ds		;  and so we can use SS to reset DS!).
	mov	bx,0		;Rezero BX-reg. but save carry flag.
	ret			;Exit.
HMALEN	equ	($-HDReq)	;(Length of all HMA driver logic).
;
; Initialization Variables.
;
InitPkt	dd	0		;DOS "Init" packet address.
PCIBuf	dw	00082h,0	;"Calculated LBA" buffer (8 bytes).
PCISubC	dw	00100h		;PCI subclass/function (Init only).
PCITbl	dw	(Ctl1Tbl-@)	;PCI-test table ptr.   (Init only).
BiosHD	db	0		;BIOS hard-disk count.
HDUnit	db	0		;Current BIOS unit number.
HFlag	db	0		;"Use HMA space" flag.
	db	0,0,0		;(Unused alignment "filler").
;
; Initialization UltraDMA "Mode" Table (digit count in low 4 bits).
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166.
EDDBuff equ	$		;Start of EDD BIOS buffer.
;
; Initialization Messages.
;
TTLMsg	db	CR,LF,'UHDD, 1-12-2014.',CR,LF,'$'
CPUMsg	db	'No 386+ CPU$'
NPMsg	db	'No V2.0C+ PCI!',CR,LF,'$'
DNMsg	db	' disk is '
DName	equ	$
DNEnd	equ	DName+40
BCMsg	db	'BAD$'
PCMsg	db	'IDE0'
PCMsg1	db	' Controller at I-O address '
PCMsg2	db	'PCI h, Chip I.D. '
PCMsg3	db	'        h.',CR,LF,'$'
DPTEMsg	db	'DPTE$'
CHSMsg	db	'CHS'
UnitMsg	db	' data BAD, unit '
UnitNo	db	'A: ',CR,LF,'$'
CtlMsg	db	'IDE'
CtlrNo	db	'0 $'
PriMsg	db	'Primary-$'
SecMsg	db	'Secondary-$'
MstMsg	db	'master$'
SlvMsg	db	'slave$'
IEMsg	db	' Identify ERROR!',CR,LF,'$'
UEMsg	db	' is not UltraDMA!',CR,LF,'$'
NDMsg	db	'Nothing to use$'
VEMsg	db	'VDS init error$'
Suffix	db	'; UHDD not loaded!',CR,LF,'$'
;
; Initialization "Strategy" Routine.   This MUST be placed above all
;   run-time logic, to prevent CPU cache "code modification" ERRORS!
;
I_Stra:	mov	cs:InitPkt.lw,bx ;Save DOS request-packet address.
	mov	cs:InitPkt.hw,es
	retf			 ;Exit and await "Device Interrupt".
;
; Initialization "Device Interrupt" Routine.   This is the main init
;   routine for the driver.
;
I_Init:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	xor	ax,ax		;Get a zero for following logic.
	lds	bx,cs:InitPkt	;Point to DOS "Init" packet.
	cmp	[bx].RPOp,al	;Is this really an "Init" packet?
	jne s	I_Exit			;No?  Reload regs. and exit!
	mov	[bx].RPStat,RPDON+RPERR	;Set "Init" packet defaults
	mov	[bx].RPSeg,cs		;  and "null" driver length.
	and	[bx].RPLen,ax
	push	cs		;NOW point DS-reg. to this driver!
	pop	ds
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes?  CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_Sv32		;Yes, go save 32-bit CPU registers.
I_Junk:	mov	dx,(CPUMsg-@)	;Display "No 386+ CPU" error message.
	call	I_Msg
I_Quit:	mov	dx,(Suffix-@)	;Display "UHDD not loaded!" message.
	call	I_Msg
I_Exit:	jmp	I_Bye		;Go reload 16-bit regs. and exit.
I_VErr:	mov	VEMsg.dwd,eax	;Set prefix in "VDS init error" msg.
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,CStack.lw	;Load our XMS "handle" number.
	or	dx,dx		;Have we reserved any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
I_EMsg:	call	I_Msg		;Display desired error message.
	popad			;Reload all 32-bit CPU registers.
	jmp s	I_Quit		;Go display message suffix and exit!
I_Sv32:	pushad			;Save all 32-bit CPU registers.
	les	si,es:[bx].RPCL	;Get command-line data pointer.
	xor	bx,bx		;Zero BX-reg. for relative logic.
I_NxtC:	cld			;Ensure FORWARD "string" commands!
	lods	es:[si].lb	;Get next command byte and bump ptr.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_Term		;Yes, go validate desired cache size.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_Term		;Yes, go validate desired cache size.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je s	I_Term		;Yes, go validate desired cache size.
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'-'		;Is byte a dash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st byte's lower-case bit.
	cmp	al,'A'		;Is this byte an "A" or "a"?
	jne s	I_ChkH		   ;No, see if byte is "H" or "h".
	mov	al,(ASDATA-0100h)  ;Reverse all "Legacy IDE" addrs.
	mov	Ctl1Sec.lb,al
	mov	al,(APDATA-0100h)
	mov	Ctl1Pri.lb,al
	mov	al,(NSDATA-0100h)
	mov	Ctl2Sec.lb,al
	mov	al,(NPDATA-0100h)
	mov	Ctl2Pri.lb,al
I_ChkH:	cmp	al,'H'		;Is switch byte an "H" or "h"?
	jne s	I_ChkQ		;No, see if byte is "Q" or "q".
	mov	HFlag,al	;Set "use HMA space" flag.
I_ChkQ:	cmp	al,'Q'		;Is switch byte a "Q" or "q"?
	jne s	I_ChkR		;No, see if byte is "R" or "r".
	mov	@DRQ.lb,075h	;Enable "data request" tests above.
I_ChkR:	cmp	al,'R'		;Is switch byte an "R" or "r"?
	jne s	I_NxtC		;No, continue scan for a terminator.
	mov	ax,es:[si+1]	;Get next 2 command-line bytes.
	mov	cx,15296	;Get 15-MB XMS memory size.
	cmp	ax,"51"		;Does user want 15-MB XMS reserved?
	je s	I_CkRA		;Yes, set memory size to reserve.
	mov	ch,(64448/256)	;Get 63-MB XMS memory size.
	cmp	ax,"36"		;Does user want 63-MB XMS reserved?
	jne s	I_NxtC		;No, continue scan for a terminator.
I_CkRA:	mov	CStack.hw,cx	;Set desired XMS memory to reserve.
	jmp	I_NxtC		;Continue scanning for a terminator.
I_Term:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	cmp	al,080h		;Is an XMS manager installed?
	je s	I_GetX		;Yes, get & save its "entry" address.
	mov	al,090h		;Disable calls to the XMS manager.
	mov	A20Req.lb,al
	mov	XMSReq.lb,al
	db	066h,0B8h	;Pass "XMS buffered" I-O to the BIOS.
	db	0E9h
	dw	(Pass-(SetBuf+3))
	db	0
	mov	SetBuf.dwd,eax
	jmp s	I_NoHM		;Go disable loading in the HMA.
I_GetX:	mov	ax,04310h	;Get and save XMS "entry" addresses.
	call	I_In2F
	push	es
	push	bx
	pop	eax
	mov	@XMgr1.dwd,eax
	mov	@XMgr2.dwd,eax
	shl	HFlag,1		;Will we be loading in the HMA?
	jz s	I_NoHM		;No, go set upper/DOS memory size.
	mov	ax,04A01h	;Get total "free HMA" space.
	call	I_In2F
	cmp	bx,HMALEN	;Enough HMA for our driver logic?
	jae s	I_TTL		;Yes, go display driver "title" msg.
I_NoHM:	mov	HFlag,0		;Ensure NO use of HMA space!
	add	VDSLn.lw,HMALEN	;Set driver upper/DOS memory size.
I_TTL:	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	mov	ax,03513h	 ;Get & save current Int 13h vector.
	call	I_In21
	push	es
	push	bx
	pop	@Prev13.dwd
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	mov	al,es:[HDISKS]	;Save our BIOS hard-disk count.
	mov	BiosHD,al
	mov	ax,cs		;Set fixed driver segment values.
	mov	VDSSg.lw,ax
	mov	@HDMain.hw,ax
	mov	CLXSA.hw,ax
	shl	eax,4		;Set driver's VDS 20-bit address.
	mov	IOAdr,eax
	cli			     ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h ;Are "VDS services" active?
	jz s	I_REnI		     ;No, re-enable CPU interrupts.
	mov	ax,08103h	;"Lock" driver in memory forever, so
	mov	dx,0000Ch	;  EMM386 may NOT re-map this driver
	call	I_VDS		;  when doing UltraDMA -- bad CRASH!
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
	jc	I_EMsg		;"Lock" error?  Display msg. & exit!
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,IOAdr	;Get final driver 32-bit address.
	add	PRDAd,eax	;Relocate "No XMS" PRD address.
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	al,001h
	call	I_In1A
	cmp	edx,PCMsg2.dwd	;Is PCI BIOS V2.0C or newer?
	je s	I_ScnC		;Yes, scan for all IDE controllers.
	mov	dx,(NPMsg-@)	;Display "No V2.0C+ PCI" message.
	call	I_Msg
	jmp s	I_XMgr		;Go see if we have an XMS manager.
I_ScnC:	mov	al,PCIBuf.lb	;Get next "interface bit" value.
	and	ax,00003h
	or	ax,PCISubC	;"Or" in subclass & current function.
	call	I_PCIC		;Test for specific PCI class/subclass.
	rol	PCIBuf.lb,4	;Swap both "interface bit" values.
	mov	al,PCIBuf.lb	;Load next "interface bit" value.
	or	al,al		;Both "interface" values tested?
	jns s	I_ScnC		;No, loop back and test 2nd one.
	add	PCISubC.lb,004h	;More PCI function codes to try?
	jnc s	I_ScnC		;Yes, loop back & try next function.
	test	al,001h		;Have we tested "Native PCI" ctlrs.?
	mov	PCIBuf.lb,093h	;(Set "Native PCI" interface bits).
	jz s	I_ScnC		;No, loop back and test them, also.
I_XMgr:	cmp	A20Req.lb,090h	;Did we find an XMS manager?
	je	I_HDCh		;No, scan for any hard-disks to use.
	mov	dx,CStack.hw	;Get "reserved" XMS memory size.
	or	dx,dx		;Does user want any "reserved" XMS?
	jz s	I_XGet		;No, get driver's actual XMS memory.
	mov	ah,009h		;Get 15-MB or 63-MB "reserved" XMS
	call	I_XMS		;  memory, which we "release" below.
	jnz s	I_XErr		;If error, display message and exit!
	mov	CStack.lw,dx	;Save reserved-XMS "handle" number.
I_XGet:	mov	ah,009h		;Request 128K of XMS memory.
	mov	dx,128
	call	I_XMS
	jz s	I_XFre		;If no errors, "free" reserved XMS.
I_XErr:	mov	eax," SMX"	;BAAAD News!  Get "XMS" msg. prefix.
	jmp	I_VErr		;Go display "XMS init error" & exit!
I_XFre:	mov	CLXDH,dx	;Save our XMS "handle" numbers.
	mov	@XMHdl,dx
	xchg	CStack.lw,dx
	or	dx,dx		;Any XMS reserved by /R15 or /R63?
	jz s	I_XLok		;No, go "lock" our XMS memory.
	mov	ah,00Ah		;"Free" our reserved XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
I_XLok:	mov	ah,00Ch		;"Lock" our driver's XMS memory.
	mov	dx,CStack.lw
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	esi,edx		;Initialize command-list XMS offset.
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	mov	@XBAddr,eax	;Save aligned "main buffer" address.
	mov	cx,ax		;Get buffer "offset" in XMS memory.
	sub	cx,dx
	mov	@XMOffs.lw,cx	;Set offset in "SXMMov" subroutine.
	mov	edx,000010000h	;Put command-list after XMS buffer.
	jcxz	I_PRDA		;Is buffer already on a 64K boundary?
	or	edx,-32		;No, put command-list before buffer.
I_PRDA:	add	eax,edx		;Set our 32-bit PRD address.
	mov	PRDAd,eax
	sub	eax,esi		;Set final command-list XMS offset.
	mov	CLXDA,eax
I_HDCh:	cmp	BiosHD,0	;Any BIOS hard-disks to use?
	je	I_AnyD		;No?  Go display error message below.
	mov	HDUnit,080h	;Set 1st BIOS hard-disk unit.
I_Next:	mov	ah,HDUnit	;Set unit no. in error message.
	mov	cx,2
	mov	si,(UnitNo-@)
	call	I_HexA
	mov	[si].lb,'h'
	mov	ah,041h		;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call	I_In13
	jc s	I_IgnJ		;None?  Ignore disk & check for more.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne s	I_IgnJ		;No, ignore disk and check for more.
	test	cl,007h		;Any "EDD extensions" for this disk?
	jz s	I_IgnJ		;No, ignore disk and check for more.
	push	cx		;Save disk "EDD extensions" flags.
	mov	si,(EDDBuff-@)	;Point to "EDD" input buffer.
	mov	[si].lw,30	;Set 30-byte buffer size.
	or	[si+26].dwd,-1	;Init NO "DPTE" data!  A bad BIOS may
				;  NOT post this dword for USB sticks
				;  etc.!  Many Thanks to Daniel Nice!
	mov	ah,048h		;Get this disk's "EDD" parameters.
	call	I_In13
	pop	cx		;Reload disk "EDD extensions" flags.
	jc s	I_IgnJ		;Error?  Ignore disk & check for more.
	test	[si+2].lb,004h	;Is this HARD disk flagged "removable"?
	jnz s	I_IgnJ		;If so, we have NO logic to SUPPORT IT!
	cmp	[si].lw,30	;Did we get at least 30 bytes?
	jb s	I_IgnJ		;No, ignore disk and check for more.
	test	cl,004h		;Does this disk provide "DPTE" data?
	jz s	I_IgnJ		;No, ignore disk and check for more.
	cmp	[si+26].dwd,-1	;"Null" drive parameter-table pointer?
	je s	I_IgnJ		;Yes, ignore disk and check for more.
I_DPTE:	les	si,[si+26]	;Get this disk's "DPTE" pointer.
	mov	bx,15		;Calculate "DPTE" checksum.
	xor	cx,cx
I_CkSm:	add	cl,es:[bx+si]
	dec	bx
	jns s	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
	mov	dx,(DPTEMsg-@)	;Display "DPTE data BAD" message.
	call	I_Msg
	mov	dx,(UnitMsg-@)
	call	I_Msg
I_IgnJ:	jmp	I_IgnD		;Ignore this disk and check for more.
I_EDOK:	call	I_CHSB		;Get and save this disk's CHS values.
	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,es:[si+4]
	shr	bl,4
	mov	ax,es:[si]	;Get disk's IDE base address.
	mov	CtlrNo.lb,'0'	;Reset display to "Ctlr. 0".
	mov	si,(Ctl1Tbl-@)	;Point to IDE address table.
I_ITbl:	cmp	[si].lw,-1	;Is this IDE table active?
	je s	I_IgnJ		;No, ignore disk and check for more.
	cmp	ax,[si+2]	;Is disk on this primary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust index for secondary channel.
	inc	bx
	cmp	ax,[si+4]	;Is disk on this secondary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust values for next controller.
	inc	bx
	add	si,CTLTSIZ
	inc	CtlrNo.lb
	cmp	si,(CtlTEnd-@)	;More IDE controllers to check?
	jb s	I_ITbl		;Yes, loop back and check next one.
	jmp s	I_IgnJ		;Ignore this disk and check for more.
I_ChMS:	push	bx		;Save disk's caching unit number.
	mov	IdeDA,ax	;Save disk's base IDE address.
	add	ax,CDSEL	;Point to IDE device-select register.
	xchg	ax,dx
	mov	al,bl		;Get drive-select command byte.
	shl	al,4
	or	al,LBABITS
	out	dx,al		;Select master or slave disk.
	push	ax		;Save drive-select and caching unit.
	push	bx
	mov	dx,(CtlMsg-@)	;Display IDE controller number.
	call	I_Msg
	pop	ax		;Reload caching unit number.
	mov	dx,(PriMsg-@)	;Point to "Primary" channel name.
	test	al,002h		;Is this a primary-channel disk?
	jz s	I_PRNm		;Yes, display "Primary" channel.
	mov	dx,(SecMsg-@)	;Point to "Secondary" channel name.
I_PRNm:	call	I_Msg		;Display disk's IDE channel name.
	pop	ax		;Reload drive-select byte.
	mov	dx,(MstMsg-@)	;Point to "master" disk name.
	test	al,010h		;Is this disk the master?
	jz s	I_MSNm		;Yes, display "master" name.
	mov	dx,(SlvMsg-@)	;Point to "slave" disk name.
I_MSNm:	call	I_Msg		;Display disk's master/slave name.
	call	I_ValD		;Validate disk as an UltraDMA unit.
	pop	ax		;Reload caching unit number.
	jc s	I_UDEr		;If any errors, display message.
	mov	bx,@LastU	;Change this disk's device-type to
	mov	[bx+CtlUn-@],al	;  "IDE" and include channel/unit.
	jmp s	I_More		;Go check for any more BIOS disks.
I_UDEr:	call	I_Msg		;NOT UltraDMA -- Display error msg.
	jmp s	I_IgnD		;Ignore this disk and check for more.
I_More:	inc	@LastU		;Bump our unit-table indexes.
I_IgnD:	inc	HDUnit		;Bump BIOS unit number.
	cmp	@LastU,MAXBIOS	;More entries in our units table?
	jae s	I_AnyD		;No, see if we have any disks to use.
	dec	BiosHD		;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyD:	mov	dx,(NDMsg-@)	;Point to "Nothing to use" message.
	cmp	@LastU,0	;Did we find any hard-disks to use?
	jz	I_Err		;No?  Display error message and exit!
	shr	HFlag,1		;Will we be loading in the HMA?
	jz s	I_Done		;No, go post "Init" packet results.
	mov	ax,04A02h	;Request needed memory in the HMA.
	mov	bx,HMALEN
	call	I_In2F
	push	es		;Get 32-bit HMA segment/offset addr.
	push	di
	pop	eax
	mov	@HDMain,eax	;Set caching logic HMA entry address.
	inc	eax		;Is our HMA address = -1 (no HMA)?
	jnz s	I_HMA1		;No, do all needed HMA adjustments.
	mov	eax," AMH"	;Get "HMA" error-message prefix.
I_HMAX:	jmp	I_VErr		  ;Go display error message & exit!
I_HMA1:	lea	ax,[di-(HDReq-@)] ;Get caching HMA logic offset.
	add	CLXPtr,ax	  ;Adjust XMS block addresses.
	add	XMSPtr,ax
	add	@CtlTbl,ax	;Adjust disk controller-table addr.
	add	@CtlUn,ax	;Adjust disk parameter-table addrs.
	add	@CHSec,ax
	add	@CHSHd,ax
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
I_A20E:	mov	eax," 02A"	;Get "A20" error-message prefix.
	jnz s	I_HMAX		;If any "A20" error, bail out QUICK!
	mov     cx,(HMALEN/2)	;Move required logic up to the HMA.
	mov	si,(HDReq-@)
	les	di,@HDMain
	rep	movsw
	mov	ah,006h		;Issue "A20 local-disable" request.
	call	I_XMS
I_Done:	xor	ax,ax		;Done!  Load & reset driver length.
	xchg	ax,VDSLn.lw
	les	bx,InitPkt	;Set results in DOS "Init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	mov	@Stack,ax	;Set initial driver stack pointer.
	xor	bx,bx		;Clear driver stack (helps debug).
	xchg	ax,bx
	mov	cx,STACK
I_ClrS:	dec	bx
	mov	[bx],al
	loop	I_ClrS
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
	int	021h
	popad			;Reload all 32-bit CPU registers.
I_Bye:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Init subroutine to convert 4-digit hex values to ASCII for display.
;   At entry, the value is in the AX-reg., and the message pointer is
;   in the SI-reg.   The SI-reg. is updated.   The CX-reg. is zeroed.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
I_HexX:	ret			;Exit.
;
; Init subroutine to test for and set up a PCI disk controller.
;
I_PCIC:	mov	IdeDA,ax	;Save subclass & function codes.
	and	PCIBuf.hw,0	   ;Reset PCI device index.
I_PCI1:	cmp	PCITbl,(CtlTEnd-@) ;More space in address tables?
 	jae s	I_HexX		   ;No, go exit above.
	mov	ax,IdeDA	;Test PCI class 1, subclass/function.
	mov	ecx,000010003h	;(Returns bus/device/function in BX).
	xchg	ax,cx
	mov	si,PCIBuf.hw
	call	I_In1A
	jc s	I_HexX		;Controller not found -- exit above.
	inc	PCIBuf.hw	;Bump device index for another test.
	xor	di,di		;Get controller Vendor & Device I.D.
	call	I_PCID
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Save DMA controller base address.
	call	I_PCID
	xchg	ax,cx
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,(PCMsg2-@)	;Set controller address in msg.
	call	I_Hex
	mov	si,(PCMsg3-@)	;Set vendor I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax		;Set Device I.D. in message.
	call	I_Hex
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	not	cl		;Get "Bus Master" & "I-O Space" bits.
	and	cl,005h		;Is controller using both BM and IOS?
	jz s	I_PCI2		;Yes, save this controller's data.
	mov	dx,(BCMsg-@)	;Display "BAD controller!", and hope
	call	I_Msg		;  our user can find a better BIOS!
	mov	dx,(PCMsg1-@)
	call	I_Msg
I_PCIJ:	jmp s	I_PCI1		;Go test for more same-class ctlrs.
I_PCI2:	mov	si,PCITbl	;Get current I-O address table ptr.
	mov	ax,DMAAd	;Set controller DMA base address.
	mov	[si],ax
	test	PCIBuf.lb,001h	;Is this a "Native PCI" controller?
	jz s	I_PCI3		;No, go display controller data.
	mov	di,16		;Set primary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+2],cx
	mov	di,24		;Set secondary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+4],cx
I_PCI3:	mov	dx,(PCMsg-@)	;Display all controller data.
	call	I_Msg
	inc	[PCMsg+3].lb	;Bump controller number in message.
	add	PCITbl,CTLTSIZ	;Bump controller address-table ptr.
	jmp s	I_PCIJ		;Go test for more same-class ctlrs.
;
; Init subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD:	xor	bx,bx		;Zero BX-reg. for timeout checks.
	mov	bp,[bx+IdeDA-@]	;Get drive's IDE address in BP-reg.
	lea	dx,[bp+CCMD]	;Point to IDE command reg.
	mov	al,0ECh		;Issue "Identify ATA Device" command.
	out	dx,al
	mov	ah,CMDTO	;Use 500-msec command timeout.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	add	ah,es:[si]	;Set timeout limit in AH-reg.
I_VDTO:	cmp	ah,es:[si]	;Has our I-O timed out?
	je s	I_DErr		;Yes?  Exit & display "Identify" msg.
	lea	dx,[bp+CSTAT]	;Point to IDE status reg.
	in	al,dx		;Read IDE primary status.
	test	al,BSY+DRQ	;Controller still busy, or no DRQ?
	jle s	I_VDTO		;If either, loop back & test again.
	test	al,ERR		;Did command cause any errors?
	jz s	I_PIO		;No, read I.D. data using PIO.
I_DErr:	mov	dx,(IEMsg-@)	;Point to "Identify error" msg.
I_SErr:	stc			;Set carry flag (error) & exit.
	ret
I_PIO:	mov	dx,bp		;Point to IDE data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	shl	ax,1		;Save "ATA/ATAPI" flag in carry bit.
	mov	cx,27		;Skip I.D. bytes 2-53 and
I_Skp1:	in	ax,dx		;  read I.D. bytes 54-55.
	loop	I_Skp1
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,(DName-@)
	mov	cl,26		;Read & swap disk name into message.
I_RdNm:	xchg	ah,al		;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw			;  also read but are ignored.   Bytes
	in	ax,dx		;  106-107 are left in the AX-reg.).
	loop	I_RdNm
	xchg	ax,bx		;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35		;Skip I.D. bytes 108-175 &
I_Skp2:	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bh,ah		;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3:	in	ax,dx
	loop	I_Skp3
	mov	dx,(UEMsg-@)	;Point to "is not UltraDMA" message.
	rcr	bl,1		;Shift "ATA/ATAPI" flag into BL-reg.
	and	bl,082h		;ATAPI disk, or UltraDMA bits invalid?
	jle s	I_SErr		;Yes?  Exit & display error message.
	mov	di,(Modes-@)	;Point to UltraDMA mode table.
	or	bh,bh		;Will disk do UltraDMA mode 0?
	jz s	I_SErr		;No?  Exit & display message!
I_NxtM:	cmp	bh,bl		;Will disk do next UltraDMA mode?
	jb s	I_GotM		;No, use current mode.
	inc	di		;Point to next mode table value.
	inc	di
	shl	bl,1		;More UltraDMA modes to check?
	jnz s	I_NxtM		;Yes, loop back.
I_GotM:	mov	si,(DNEnd-@)	;Point to end of disk name.
I_NxtN:	cmp	si,(DName-@)	;Are we at the disk-name start?
	je s	I_Name		;Yes, disk name is all spaces!
	dec	si		;Decrement disk name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_NxtN		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma & space.
	inc	si
	inc	si
I_Name:	mov	[si].dwd,"-ATA"	;Set "ATA-" after drive name.
	add	si,4
	mov	ax,[di]		;Set UltraDMA "mode" in message.
	mov	cl,00Fh
	and	cl,al
	call	I_HexA
	mov	[si].dwd,0240A0D2Eh ;Set message terminators.
	mov	dx,(DNMsg-@)	    ;Display mfr. name/"mode" & exit.
	jmp s	I_Msg
;
; Init subroutine to set BIOS CHS data for a hard-disk or diskette.
;
I_CHSB:	mov	al,080h		;Get "BIOS disk" device-type.
I_CHSD:	push	ax		;Save unit's device-type flag.
	mov	ah,008h		;Get BIOS CHS data for this unit.
	call	I_In13
	jc s	I_CHSE		;If BIOS error, zero CHS sectors.
	and	cl,03Fh		;Get sectors/head value (low 6 bits).
	jz s	I_CHSE		;If zero, ERROR!  Zero CHS sectors.
	inc	dh		;Get heads/cylinder (BIOS value + 1).
	jnz s	I_CHST		;If non-zero, save data in our table.
I_CHSE:	xor	cl,cl		;Error!  Zero unit's CHS sectors.
I_CHST:	mov	di,@LastU	;Point to "active units" table.
	mov	al,HDUnit	;Set BIOS unit number in our table.
	mov	[di+Units-@],al
	mov	[di+CHSec-@],cl ;Set unit's CHS data in our table.
	mov	[di+CHSHd-@],dh
	pop	ax		;Reload and set device-type flag.
	mov	[di+CtlUn-@],al
	mov	dx,(CHSMsg-@)	;Point to "CHS data BAD" message.
	or	cl,cl		;Valid CHS values for this unit?
	jz s	I_Msg		;No?  Display error msg. and exit!
	ret			;All is well -- exit.
;
; Init subroutines to issue "external" calls.
;
I_XMS:	call	@XMgr2.dwd	;XMS -- issue desired request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_VDS:	mov	di,(VDSLn-@)	;VDS -- Point to parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In13:	mov	dl,HDUnit	;BIOS data -- set BIOS unit in DL-reg.
	int	013h		;Issue BIOS data interrupt.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_PCID:	push	bx		;Save PCI bus/device/function codes.
	push	si		;Save IDE address-table pointer.
	mov	al,00Ah		;Set "PCI doubleword" request code.
	call	I_In1A		;Get desired 32-bit word from PCI.
	pop	si		;Reload IDE address-table pointer.
	pop	bx		;Reload PCI bus/device/function.
	ret			;Exit.
I_In1A:	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	int	01Ah
	jmp s	I_IntX		;Restore driver settings, then exit.
I_Msg:	push	bx		;Message -- save our BX-register.
	mov	ah,009h		;Issue DOS "display string" request.
	int	021h
	pop	bx		;Reload our BX-register.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In21:	int	021h		;General DOS request -- issue Int 21h.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In2F:	int	02Fh		;"Multiplex" -- issue XMS/HMA request.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
I_Ret:	ret			;Exit.
CODE	ends
	end
