	page	59,132
	title	UDMA -- DOS UltraDMA Disk Caching Driver.
;
; This is a general-purpose caching driver for DOS drives addressed by
; "Int 13h" BIOS I-O requests.   UDMA has an "internal" driver for all
; UltraDMA hard disks, and it can have up to 3 more "internal" drivers
; for other disk types.   At present, UDMA caches up to 17 BIOS disks,
; 2 of which may be standard A: and B: diskettes.   I-O not handled by
; its "internal" drivers (diskettes, etc.) will be passed to the BIOS,
; and on return to UDMA, it caches data from the request's I-O buffer.
; UDMA caches data-input requests from the UDVD driver for up to three
; CD/DVD drives.   It can also cache I-O for up to 236 more "external"
; drives (USB, Firewire, etc.).   Write-Through caching is done on all
; output.    UDMA uses XMS and caches from 5-MB to 1-Gigabyte of data!
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
s	equ	<short>		;Default conditional jumps to "short".
STACK	equ	568		;Driver local-stack size.
RDYTO	equ	008h		;389-msec minimum I-O timeout.
HDWRFL	equ	00410h		;BIOS hardware-installed flag.
DKTSTAT	equ	00441h		;BIOS diskette status byte.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
MCHDWFL	equ	0048Fh		;Diskette hardware media-change flags.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver-Size and Cache-Size Definitions.
;
MAXBIOS	equ	17		;Maximum BIOS disks handled.  Can be
				;  up to 255, using "align" commands
				;  as below.   Recommended:  9 to 29
				;  in steps of 4 for best alignment.
EXTYPE	equ	010h		;"External" device type.   See the
				;  NOTE about the "DvrTbl", below.
TBLCT	equ	640		;Starting cache-table entry count.
				;  The "CachSiz" table (below) has
				;  all other initialization values.
;
; Driver Error Return Codes.
;
MCHANGE	equ	006h		;Media change (diskettes only).
DMATIMO equ	008h		;DMA timeout.
DMAERR	equ	00Fh		;DMA error.
CTLRERR equ	020h		;Controller busy.
DISKERR equ	0AAh		;Disk not-ready.
FAULTED	equ	0CCh		;Disk FAULTED.
HARDERR equ	0E0h		;Hard I-O error.
XMSERR	equ	0FFh		;XMS memory error.
;
; "Legacy IDE" Controller I-O Base Addresses.
;
NPDATA	equ	001F0h		;Normal primary      base address.
NSDATA	equ	00170h		;Normal secondary    base address.
APDATA	equ	001E8h		;Alternate primary   base address.
ASDATA	equ	00168h		;Alternate secondary base address.
;
; IDE Controller Register Offsets.
;
CDATA	equ	0		;Data port offset.
CDSEL	equ	6		;Disk-select and upper LBA offset.
CCMD	equ	7		;Command-register offset.
CSTAT	equ	7		;Primary-status register offset.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; LBA "Device Address Packet" Layout.
;
DAP	struc
DapPL	db	?		;Packet length.
	db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
	db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (segment:offset).
DapLBA	dw	?		;48-bit logical block address (LBA).
DapLBA1	dw	?
DapLBA2	dw	?
DAP	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
	db	2 dup (?)	;(Unused by us).
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	9 dup (?)	;(Unused by us).
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	(Strat-@)	;"Strategy" routine pointer.
VarPtr	equ	[$-2].lb	;(UDVD variables ptr. after Init).
	dw	(DevInt-@)	;Device-Interrupt routine pointer.
DvrNam	db	'UDMA5X$',0	;"Type 5" UDMA driver name, declaring
				;  17 BIOS units!  (Was 9 in "Type 4").
;
; General Driver "Data Page" Variables, Part 1.
;
VLF	db	0		;VDS lock flag (001h buffer is locked).
IOF	db	001h		;I-O busy flag (080h busy, 001h flush).
DMAAd	dw	0		;DMA status/command register address.
IdeDA	dw	0		;IDE data register address.
PRDAd	dd	0		;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0,0,0		;IDE "upper" LBA bits 24-47.
SecCt	db	0		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
STLmt	dw	0		;Cache binary-search table limit.
;
; VDS Parameter Block.
;
VDSLn	dd	(NormEnd-@)	;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
VDSAd	dd	0		;VDS 32-bit physical buffer address.
;
; UltraDMA Command-List.
;
IOAdr	dd	0		;DMA 32-bit buffer address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; General Driver "Data Page" Variables, Part 2.   Note that some of
;   these overlay the driver "Final Initialization" routine, below.
;
LUTop	dw	0		;Least-recently-used "top" indexes.
LUEnd	dw	0		;Least-recently-used "end" indexes.
RqBuf	dd	0		;Request I-O buffer address.
RqXMS	dd	128		;Request XMS buffer offset.
RqSec	db	0		;Request I-O sector count.
RqCmd	equ	[$].lb		;Request I-O command bits.
RqCSC	equ	[$+1].lb	;Request current-block sector count.
RqTyp	equ	[$+2].lb	;Request device-type flag.
RqLBA	equ	[$+3].lw	;Request initial LBA address.
RqUNo	equ	[$+9].lb	;Request "cache unit" number.
CBLBA	equ	[$+11].lw	;Current-buffer initial LBA.
CBUNo	equ	[$+17].lb	;Current-buffer "cache unit" number.
CBSec	equ	[$+18].lb	;Current-buffer sector count.
CBLst	equ	[$+19].lw	;Current-buffer "last" LRU index.
CBNxt	equ	[$+21].lw	;Current-buffer "next" LRU index.
WBLBA	equ	[$+23].lw	;Working-buffer initial LBA.
WBUNo	equ	[$+29].lb	;Working-buffer "cache unit" number.
WBSec	equ	[$+30].lb	;Working-buffer sector count.
WBLst	equ	[$+31].lw	;Working-buffer "last" LRU index.
WBNxt	equ	[$+33].lw	;Working-buffer "next" LRU index.
;
; Driver "Final Initialization" Routine.   The driver stack must be
;   cleared here (not above!), since this driver varies in size and
;   its final run-time stack may be set WITHIN the main init logic!
;
I_End:	cld			;Clear driver stack (helps debug).
	xor	ax,ax
	mov	cx,STACK
	mov	di,@Stack
	sub	di,cx
	push	cs
	pop	es
	shr	cx,1
	rep	stosw
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,offset (Entry-@)
	int	021h
I_Fail:	popad			;Reload all 32-bit CPU registers.
I_Exit:	pop	dx		;Reload 16-bit CPU regs. we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; "External Variables" Table.   Each of these variables is fixed to
;   the offsets listed below.   The address of the table divided by
;   4 is put in byte 6 of the driver's "header" so external drivers
;   can find UDMA and this table through the Int 13h segment.   The
;   stand-alone UltraDMA driver zeroes "ExEntrP", bytes 4-5.
;
	align	4		;(4-byte table alignment REQUIRED!).
XMBuf	dw	0		; 0:  High-order XMS buffer address.
XMOfs	dw	0		; 2:  Buffer "offset" in XMS memory.
ExEntrP	dw	(ExEntr-@)	; 4:  External Entry routine offset.
MoveP	dd	(MvData-@)	; 6:  "XMS move" subroutine address.
;
; BIOS "Active Units" Tables.
;
Units	db   MAXBIOS dup (0FFh)	;BIOS unit-number table, 0FFh unused.
TypeF	db   MAXBIOS dup (0)	;Device type and "sub-unit":
				;  000h - 007h:  SATA/UltraDMA disk.
				;  008h - 00Fh:  SATA/UltraDMA disk.
				;  010h - 017h:  Driver 2 devices.
				;   ...
				;   ...    (8 devices per pointer).
				;   ...
				;  070h - 077h:  Driver 14 devices.
				;  078h:  CD/DVD or other externals.
				;  080h:  Hard disk handled by BIOS.
				;  0C0h:  Diskettes handled by BIOS.
CHSec	db   MAXBIOS dup (0)	;Sectors-per-head table.
CHSHd	db   MAXBIOS dup (0)	;Heads-per-cylinder table.
;
; Controller I-O Address Table, set during Initialization.
;
Ctl1Tbl	dw	0FFFFh		;Controller 1 DMA base address.
Ctl1Pri	dw	NPDATA		;   "Legacy" primary   addresses.
Ctl1Sec	dw	NSDATA		;   "Legacy" secondary addresses.
Ctl2Tbl	dw	0FFFFh		;Controller 2 DMA base address.
Ctl2Pri	dw	APDATA		;   "Legacy" primary   addresses.
Ctl2Sec	dw	ASDATA		;   "Legacy" secondary addresses.
	dw	0FFFFh,0,0	;Controller 3 I-O addresses.
	dw	0FFFFh,0,0	;Controller 4 I-O addresses.
CtlTEnd	equ	$		   ;End of controller I-O tables.
CTLTSIZ	equ	(Ctl2Tbl-Ctl1Tbl)  ;Size of a controller I-O table.
;
; "Miscellaneous" Caching Variables.
;
LBABuf	dw	00082h,0,0	;"Calculated LBA" buffer.
;
; "Internal Driver" Handler Table.   Each points to a subroutine that
;   handles the "device types" shown below.    All initially point to
;   a "null driver" handler, which simply discards the return address
;   and calls the BIOS.    Device types 004h - 007h are unused, since
;   UltraDMA allows only 4 hard disks!
;
;   An "internal" driver works on the UDMA stack, can access all UDMA
;   variables, and must exit with a "retf" command.   It can post bit
;   7 of "RqCmd", indicating that an I-O request fit within one cache
;   block, and that block's XMS buffer was used to handle I-O.    The
;   main caching routine will then assume data has been "cached" into
;   XMS memory DURING the I-O request!!   For an example of how to do
;   this, see the "UdmaIO" subroutine below.
;
; NOTE:  "UdmaIO" and up to 13 other "internal" drivers ARE possible,
;   as shown for the "TypeF" table above.    But for now, "UdmaIO" is
;   the only "internal" driver used, and the "EXTYPE" external-device
;   type which indexes "ExtDvr" below is 010h (not 078h).    To add a
;   new "internal" driver, add a pointer for it below, then increment
;   "EXTYPE" above by 008h.
;
DvrTbl	dd	(UdmaIO-@)	;Type 000h-007h:  SATA/UltraDMA disk.
Ud2Ptr	dd	(UdmaIO-@)	;Type 008h-00Fh:  SATA/UltraDMA disk.
ExtDvr	dd	000000100h	;Type 010h:       "External" device.
;
; Global Descriptor Table Pointer, used by our "lgdt" command.
;
GDTP	dw	GDTLen		;GDT length (always 24 bytes).
GDTPAdr	dd	(OurGDT-@)	;GDT 32-bit address (Set By Init).
RqIndex	dw	(Ctl1Tbl-@)	;Request cache-table index, here
				;  for 32-bit alignment of "GDTP".
;
; Global Descriptor Table, used during our own "real-mode" moves.
;
OurGDT	dd	0,0		;"Null" segment to start.
	dw	0FFFFh		;Conforming code segment.
GDT_CSL	dw	0
GDT_CSM	db	0
	dw	0009Fh
GDT_CSH	db	0
GDT_DS	dd	00000FFFFh,000CF9300h  ;4-GB data segment in pages.
GDTLen	equ	($-OurGDT)
;
; Int 15h Descriptor Table, used for BIOS "protected-mode" moves.
;
MoveDT	dd	0,0		       ;"Null" descriptor.
	dd	0,0		       ;GDT descriptor.
SrcDsc	dd	00000FFFFh,000009300h  ;Source segment descriptor.
DstDsc	dd	00000FFFFh,000009300h  ;Dest.  segment descriptor.
	dd	0,0		       ;(Used by BIOS).
	dd	0,0		       ;(Used by BIOS).
;
; Main Int 13h Entry Routine.  For a CHS request, the registers are:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   ES:BX   I-O buffer address.
;
; For an LBA request, the registers are:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   DS:SI   Pointer to Device Address Packet ("DAP") as shown above.
;
Entry:	pushf			;Save CPU flags and BP-reg.
	push	bp
	call	Switch		;Switch to this driver's stack.
	mov	bp,0		;Reset active-units table index.
@LastU	equ	[$-2].lw	;(Last-unit index, set by Init).
NextU:	dec	bp		;Any more active units to check?
	js s	QuickX		;No, not our request -- exit quick!
	cmp	dl,[bp+Units-@]	;Does request unit match our table?
	jne s	NextU		;No, see if more units remain.
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	mov	dl,[bp+TypeF-@]	;(Set unit "type" flag in any case).
	mov	cs:RqTyp,dl
	je s	RSetup		;Yes, do setup for this request.
	cmp	dl,0C0h		;Is this request for a diskette?
	jne s	Pass		;No, "flush" cache and pass request.
	cmp	ah,005h		;Diskette track-format request?
	jne s	QuickX		;No, "pass" other diskette requests.
Pass:	or	cs:IOF.lb,001h	;"Flush" cache before next request.
QuickX:	cli			;Pass to BIOS -- disable interrupts.
	and	cs:IOF.lb,001h	;Reset "busy" but save "flush" flag.
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
	pop	bp		;Reload BP-register and CPU flags.
	popf
	db	0EAh		;Go to next routine in Int 13h chain.
@Prv13A	dd	0		;(Prior Int 13h vector, set by Init).
RSetup:	shl	ah,1		;Is this an LBA read or write request?
	jns s	ValSC		;No, go validate CHS sector count.
	mov	al,[si].DapSC	;Get "DAP" I-O sector count.
	mov	dx,[si].DapLBA1	;Get "DAP" LBA bits 16-47.
	mov	di,[si].DapLBA2
	les	cx,[si].DapBuf	;Get "DAP" I-O buffer address.
	cmp	[si].DapBuf,-1	;Is this a 64-bit "DAP" I-O buffer?
	mov	si,[si].DapLBA	;(Get "DAP" LBA bits 0-15).
	je s	Pass		;Yes?  Let BIOS handle this request!
ValSC:	dec	al		;Is sector count zero or over 128?
	js s	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js s	SetDS		;Yes, go set driver's DS-reg.
	xchg	ax,cx		;CHS -- save request code and sectors.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		;Swap cylinder & head values.
	mov	al,[bp+CHSec-@] ;Get disk CHS sectors/head value.
	or	al,al		;Were disk CHS values legitimate?
	jz s	Pass		;No?  Let BIOS handle this request!
	push	ax		;Save CHS sectors/head value.
	mul	ah		;Convert head to sectors.
	add	si,ax		;Add result to starting sector.
	pop	ax		;Reload CHS sectors/head value.
	mul	[bp+CHSHd-@].lb ;Convert cylinder to sectors.
	mul	dx
	add	si,ax		;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
SetDS:	push	cs		;Set this driver's DS-reg.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative logic.
	and	ah,006h		;Save sector count & command bits.
	mov	[bx+RqSec-@],ax
	mov	[bx+VDSOf-@].lw,cx   ;Set VDS I-O buffer address.
	mov	[bx+VDSOf-@].hw,bx
	mov	[bx+VDSSg-@],es
	mov	ah,0		     ;Set VDS buffer length.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	pusha			     ;Save LBA and "cache unit" no.
	or	[bx+VDSAd-@].dwd,-1  ;Invalidate VDS address.
	mov	ax,08103h	     ;VDS "lock" user buffer.
	mov	dx,0000Ch
	call	VDLock
	popa			     ;Reload LBA and "cache unit" no.
	jc	Pass		     ;VDS error:  Pass this request!
	mov	eax,[bx+VDSAd-@]     ;Get 32-bit VDS buffer address.
	cmp	eax,-1		     ;Is VDS address valid?
	jb s	SetVLF		     ;Yes, set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw  ;No VDS:  Get 20-bit segment.
	shl	eax,4
	add	eax,[bx+VDSOf-@]     ;Add in buffer offset value.
SetVLF:	adc	[bx+VLF-@],bl	     ;Set VDS "lock" flag from carry.
	cmp	[bx+RqTyp-@].lb,0C0h ;Is this a diskette request?
	jne s	SetBuf		     ;No, go set user buffer address.
	mov	es,bx		     ;Point ES-reg. to low memory.
	mov	cl,01Fh		     ;Get BIOS diskette status code.
	and	cl,es:[DKTSTAT]
	cmp	cl,MCHANGE	   ;Diskette media-change detected?
	jne s	SetBuf		   ;No, go set 32-bit buffer address.
	or	[bx+IOF-@].lb,1	   ;Flush cache before next request.
SetBuf:	mov	[bx+RqBuf-@],eax   ;Set user I-O buffer address.
	mov	[bx+RqLBA-@],di	   ;Set initial request LBA.
	mov	[bx+RqLBA+2-@],dx
	mov	[bx+RqLBA+4-@],si
	mov	[bx+RqUNo-@],bp	   ;Set "cache unit" number.
	mov	ah,005h		   ;Issue "A20 local-enable" request.
	call	A20Req
	jnz s	A20Er		;If "A20" error, use routine below.
GoMain: db	0EAh		;Go to main cache or driver routine.
@GMAddr	dd	(Cache-@)	;(Main entry address, set by Init).
Exit:	pushf			;Save error flag (carry) & error code.
	push	ax
	mov	ah,006h		;Issue "A20 local-disable" request.
	call	A20Req
	nop			;(Unused alignment "filler").
Abandn:	shr	[bx+VLF-@].lb,1	;Was I-O buffer "locked" by VDS?
	jnc s	LoadEC		;No, reload error code & flag.
	call	VDUnlk		;Do VDS "unlock" of user I-O buffer.
LoadEC:	pop	ax		;Reload error code and error flag.
	popf
	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+33],al	;Set error code in exiting AH-reg.
	cli			;Disable CPU interrupts.
	rcl	[bx+IOF-@].lb,1	;Reset "busy" but save "flush" flag.
	shr	[bx+IOF-@].lb,1
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
GetOut:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-register and CPU flags.
	popf
	iret			;Exit.
A20Er:	or	[bx+IOF-@].lb,1	;"A20" error!  Request cache "flush".
	stc			;Stack error flag (carry bit).
	pushf
	push	ax		;Stack "XMS error" code (AL = 0FFh).
	jmp s	Abandn		;Abandon this request and exit QUICK!
InvalF:	mov	ah,001h		;We are BUSY, you fool!  Set "invalid
	jmp s	GetOut		;  function" code and get out, QUICK!
;
; "External Reset" Routine (always 8 bytes behind "ExEntr" below).
;
ExRSet:	or	cs:IOF.lb,001h	;"Flush" cache before next request.
	retf			;Exit.
	db	0		;(Unused alignment "filler").
;
; "External Drive" Entry Routine.    This logic sets up a read/write
;   request for handling at "SetBuf"above.   At entry, the registers
;   contain:
;
;   EAX     32-bit user buffer address.    MUST be valid, and must
;	      already have been "VDS locked" if needed.
;   DS:BX   "Call back" I-O routine address, used to do I-O for an
;	      LBA address that is NOT found to be "cached".    See
;	      the UDVD driver for an example of such a routine.
;	      NOTE:  The upper half of the EBX-reg. will be LOST!
;   CL      I-O sector count (1 to 128) of 512-byte sectors.
;   CH      000h for a read request, 002h for a write request.
;   SI      LBA address bits 0-15.
;   DX      LBA address bits 16-31.
;   DI      LBA address bits 32-47.
;   BP      Bits 0-7 are the "cache unit" number, MAXBIOS through
;	      MAXBIOS+2 for CD/DVD drives 0 to 2 handled by UDVD,
;	      and MAXBIOS+3 or higher for other external drivers.
;	      Bits 8-15 are ignored.
;
ExEntr:	push	bx		;Save unit no. in lower BX-reg. and
	push	bp		;  "call back" addr. in upper BX-reg.
	pop	ebx
	pushf			;Save CPU flags and BP-reg.
	push	bp
	call	Switch		;Switch to this driver's stack.
	push	ds		;Stack "call back" address & unit no.
	push	ebx
	call	VDExit			;Do critical driver settings.
	pop	bp			;Set desired unit in BP-reg.
	pop	ExtDvr			;Set "call back" routine ptr.
	mov	[bx+RqSec-@],cx		;Set sectors and I-O command.
	mov	[bx+RqTyp-@].lb,EXTYPE	;Post "external" device type.
	jmp	SetBuf			;Go post other data & do I-O.
	db	0			;(Unused alignment "filler").
;
; Subroutine to do physical XMS moves.   At entry, the ECX-reg. has
;   the byte length, and the ESI/EDI-regs. have the move addresses.
;   At exit, the carry flag is reset for no error or is SET for any
;   XMS move error.   If so, the AL-reg. has a 0FFh XMS error code.
;   The DS- and ES-regs. are saved and restored, and the BX-reg. is
;   zeroed at exit.   The other registers are NOT saved, for speed.
;   This subroutine must be called with 32-bit "call far" commands.
;   Note that the "real mode" move logic has no XMS errors to post!
;   Only "protected mode" moves (BIOS or EMM386) may return errors.
;
MvData:	push	ds		;Save caller's segment registers.
	push	es
	push	cs		;Set this driver's data segment.
	pop	ds
	smsw	ax		;Get CPU "machine status word".
	shr	ax,1		;Are we running in "protected" mode?
	jc s	MvBIOS		;Yes, use BIOS move routine below.
	shr	ecx,2		;Convert byte count to dword count.
	mov	bp,cx		;Set move byte count in BP-reg.
	mov	dx,512		;Set 512-dword section count.
MvRNxt:	mov	cx,dx		;Get move section count.
	cmp	cx,bp		;At least 512 dwords left?
	jbe s	MvRGo		;Yes, use full section count.
	mov	cx,bp		;Use remaining dword count.
MvRGo:	cli			;Disable CPU interrupts.
	db	02Eh,00Fh	;"lgdt cs:GDTP", coded the hard-way
	db	001h,016h	;   to avoid any annoying V5.1 MASM
	dw	(GDTP-@)	;   warning messages about it!
	mov	eax,cr0		;Set "protected-mode" control bit.
	or	al,001h
	mov	cr0,eax
	mov	bx,(GDT_DS-OurGDT)  ;Set DS- and ES-reg. "selectors".
	mov	ds,bx
	mov	es,bx
	cld			;Ensure FORWARD "string" commands.
	db	0F3h,067h	;Move all 32-bit words using "rep"
	movsd			;  and "address-override" prefixes.
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	dec	ax		;Clear "protected-mode" control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	sub	bp,dx		;Any more data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	xor	ax,ax		;Success!  Reset carry and error code.
MvExit:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	pop	es		;Reload caller's segment registers.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	retf			;Exit ("retf" for calls from UDVD).
MvBIOS:	mov	edx,ecx		;BIOS move:  Set length in EDX-reg.
	xor	ecx,ecx		;Initialize 16K section count the
	mov	ch,040h		;  "5-byte" way (helps alignment!).
MvBNxt:	push	ecx		;Save all 32-bit move parameters.
	push	edx
	push	esi
	push	edi
	cmp	ecx,edx		;At least 1 full section left?
	jbe s	MvBGo		;Yes, use full section count.
	mov	ecx,edx		;Use remaining data byte count.
MvBGo:	cld			;Ensure FORWARD "string" commands!
	shr	cx,1		;Convert byte count to word count.
	push	ds		;Ensure ES-reg. points to our driver.
	pop	es
	push	edi		;Set up destination descriptor.
	mov	di,(DstDsc+2-@)
	pop	ax
	stosw
	pop	ax
	stosb
	mov	[di+2],ah
	push	esi		      ;Set up source descriptor ("sub"
	sub	di,(DstDsc-SrcDsc+3)  ;  will zero carry for Int 15h).
	pop	ax
	stosw
	pop	ax
	stosb
	mov	[di+2],ah
	mov	si,(MoveDT-@)	;Point to move descriptor table.
	mov	ah,087h		;Have BIOS move next data section.
	int	015h
	pop	edi		;Reload all 32-bit move parameters.
	pop	esi
	pop	edx
	pop	ecx
	mov	al,XMSERR	;Get our XMS error code.
	jc s	MvExit		;If any BIOS error, exit immediately.
	sti			;Ensure CPU interrupts are enabled.
	push	cs		;Ensure DS-reg. points to our driver.
	pop	ds
	add	esi,ecx		;Update source and dest. addresses.
	add	edi,ecx
	sub	edx,ecx		;Any more data sections to move?
	ja s	MvBNxt		;Yes, go move next section of data.
	jmp s	MvDone		;Success!  Reset carry flag and exit.
;
; Subroutine to check for "busy" and switch to this driver's stack.
;
Switch:	cli			;Disable CPU interrupts.
	pop	bp		;Load our return address into BP-reg.
	bts	cs:IOF,7	;Is this driver currently "busy"?
	jc	InvalF		;Yes?  Set "invalid function" & exit!
	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to this driver's stack.
	pop	ss
	mov	sp,offset (NormEnd-@)
@Stack	equ	[$-2].lw	;(Initial stack pointer, set by Init).
	sti			;Re-enable CPU interrupts.
	cld			;Ensure FORWARD "string" commands.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	jmp	bp		;Exit.
CStack	dd	0		;Caller's saved stack pointer,
				;  placed here for "alignment".
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
A20Req:	db	09Ah		;Call XMS manager for "A20" request.
@XEntry	dd	0		;(XMS "entry" address, set by Init).
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	VDExit		;Go restore critical settings & exit.
;
; Subroutine to do a VDS "lock" or "unlock".
;
VDUnlk:	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
VDLock:	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute desired VDS request.
VDExit:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	ret			;Exit.
NormEnd	equ	$+STACK		;End of all "normal memory" logic.
;
; Ye Olde UltraDMA "Internal Driver" Subroutine.   This routine sets
;   up I-O requests, then calls "DoDMA" below to perform actual I-O.
;
UdmaIO:	cld			   ;Ensure FORWARD "string" commands.
	mov	bp,[bx+RqTyp-@]	   ;Get this disk's "type" flags.
	mov	ax,bp		   ;Get disk's IDE controller number.
	shr	al,2
	mov	ah,CTLTSIZ	   ;Point to disk's I-O addresses.
	mul	ah
	add	ax,(Ctl1Tbl-@)
	xchg	ax,si
	lodsd			   ;Set controller base address and
	mov	[bx+DMAAd-@],eax   ;  primary-channel data address.
	test	bp,00002h	   ;Primary channel I-O request?
	jz s	GetSec		   ;Yes, get sector count & commands.
	add	[bx+DMAAd-@].lw,8  ;Use secondary DMA controller.
	lodsw			   ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],ax
GetSec:	mov	ax,[bx+RqSec-@]	   ;Get sector count and command bits.
	les	di,[bx+RqLBA-@]	   ;Set LBA bits 16 thru 47.
	mov	dx,es
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	shr	dx,12		   ;Shift out LBA bits 16-27.
	or	di,dx		   ;Anything in LBA bits 28-47?
	jnz s	LBA48		   ;Yes, use LBA48 read/write command.
	xchg	dh,[bx+LBA2-@]	   ;LBA28:  Reload & reset bits 24-27.
	or	ah,(DRCMD+1)	   ;Get LBA28 read/write command + 5.
	jmp s	SetCmd		   ;Go set LBA and IDE command bytes.
LBA48:	shl	ah,3		   ;LBA48 -- get command as 020h/030h.
SetCmd:	shr	bp,1		   ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		   ;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		   ;Get final IDE command byte.
	xor	dh,ah		   ;(LBA28 C8h/CAh, LBA48 25h/35h).
	mov	[bx+DSCmd-@],dx	   ;Set LBA and IDE command bytes.
	mov	si,[bx+RqLBA+4-@]  ;Set LBA bits 0 thru 15.
	mov	[bx+LBA-@],si
	mov	[bx+SecCt-@],al	   ;Set I-O sector count.
	mov	ah,0		   ;Set DMA buffer length.
	shl	ax,1
	mov	[bx+IOLen+1-@],ax
	mov	eax,[bx+RqBuf-@]   ;Get user I-O buffer address.
	mov	cl,[bx+RqCSC-@]	   ;Get current I-O sector count.
	cmp	cl,[bx+RqSec-@]	   ;All I-O within one cache entry?
@UDSA:	jb s	SetIOA		      ;No, use normal I-O methods.
	or	[bx+RqCmd-@].lb,080h  ;Do NOT cache data, after I-O!
	mov	eax,[bx+RqXMS-@]      ;Do I-O thru the cache buffer!
SetIOA:	mov	[bx+IOAdr-@],eax  ;Set 32-bit DMA buffer address.
	or	[bx+RqCmd-@],bl	  ;Using a cache buffer for I-O?
	js s	UBufIO		  ;Yes, use buffered I-O logic below.
	test	al,003h		  ;Is user I-O buffer 32-bit aligned?
	jnz s	UBfSet		  ;No, use buffered I-O logic below.
	mov	cx,[bx+IOLen-@]	  ;Get lower ending DMA address.
	dec	cx		  ;(IOLen - 1 + IOAdr).
	add	ax,cx		  ;Will I-O cross a 64K DMA boundary?
	jc s	UBfSet		      ;Yes, do buffered I-O below.
	cmp	[bx+IOAdr-@].hw,009h  ;Will DMA occur beyond 640K?
	jbe s	DoDMA		      ;No, do DMA with user buffer.
	nop			      ;(Unused alignment "filler").
UBfSet:	mov	[bx+IOAdr-@].dwd,0    ;Buffered:  Use our XMS buffer.
@XBufAd	equ	[$-4].dwd	      ;(XMS buffer addr., Init set).
UBufIO:	test	[bx+RqCmd-@].lb,002h  ;Is this a read request?
	jz s	UBufIn		      ;Yes, use input routine below.
	push	cs		  ;Move user data to our XMS buffer.
	call	UBufMv
	jc s	DoDMAE		  ;If any XMS errors, exit below!
	jmp s	DoDMA		  ;Go do UltraDMA output, then exit.
UBufIn:	push	cs		  ;Buffered input:  Read data to XMS.
	call	DoDMA
	jc s	DoDMAE		     ;If any XMS errors, exit below!
UBufMv:	movzx	ecx,[bx+SecCt-@].lb  ;Get number of sectors to move.
	mov	esi,[bx+IOAdr-@]     ;Set move addresses for a read.
	mov	edi,esi
	xchg	edi,[bx+RqBuf-@]
	jz s	UBfMv1		  ;Is this a read from XMS memory?
	xchg	esi,edi		  ;No, "swap" source & destination.
UBfMv1:	shl	ecx,9		  ;Convert sectors to byte count.
	jmp	[bx+MoveP-@].dwd  ;Go do desired XMS move, then exit.
;
; Subroutine to execute UltraDMA read and write commands.
;
DoDMA:	xor	ecx,ecx		 ;MUST move DMA command-list to XMS
	mov	cl,8		 ;  for output, or UMBPCI will FAIL!
	mov	esi,(IOAdr-@)
@CLAddr	equ	[$-4].dwd	 ;(Command-list address, Init set).
	mov	edi,[bx+PRDAd-@]
	call	[bx+MoveP-@].dwd
	jc s	DoDMAE		;Command-list move ERROR?  Exit NOW!
	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	di,[bx+IdeDA-@]	;Get our disk's IDE base address.
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	lea	dx,[di+CDSEL]
	out	dx,al
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,((RDYTO*256)+FLT)  ;Get timeout & "fault" mask.
	add	ch,es:[si]	      ;Set timeout limit in CH-reg.
	push	cs		;Await controller- and disk-ready.
	call	ChkRdy
DoDMAE:	jc s	DoDMAX		;If any errors, exit below!
	push	si		;Save BIOS timer pointer.
	mov	si,(PRDAd-@)		  ;Point to IDE parameters.
	test	[si+IOCmd-PRDAd].lb,012h  ;Is this a write request?
	jnz s	SetDMA			  ;Yes, reset command reg.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA:	mov	dx,[bx+DMAAd-@]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	outsd
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxtPar:	lea	dx,[di+CDATA+1]	;Point to IDE sector count reg. - 1.
IDEPar:	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc s	IDEPar		;Yes, loop back and output next one.
	jnz s	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	dec	dh		;"Legacy IDE" controller channel?
@DRQ:	jmp s	DMAGo		;No, forget about awaiting 1st DRQ.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ:	cmp	ch,es:[si]	;Too long without 1st data-request?
	je s	DMAEnd		;Yes?  Return carry and DMA timeout!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz s	ChkDRQ		    ;No, loop back and check again.
DMAGo:	mov	es:[si+HDI_OFS],bl  ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	    ;Point to DMA command register.
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	inc	ax
	out	dx,al
ChkDMA:	inc	dx		;Point to DMA status register.
	inc	dx
	in	ax,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,es:[si]	;Has our DMA transfer timed out?
	je s	HltDMA		    ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl  ;Did BIOS get a disk interrupt?
	je s	ChkDMA		    ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
HltDMA:	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	ErrDMA		;No?  Go see what went wrong.
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz s	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy:	lea	dx,[di+CSTAT]	;Read IDE primary status.
	in	al,dx
	cmp	ch,es:[si]	;Too long without becoming ready?
	je s	RdyErr		;Yes?  Go see what went wrong.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s	ChkRdy		;Yes, loop back and check again.
	and	al,cl		;Disk "fault" or hard-error?
	jnz s	HdwErr		;Yes?  Go see what went wrong.
DoDMAX:	retf			;End of this request -- exit.
ErrDMA:	test	al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd:	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp s	WhichE		;Go see which error code to return.
RdyErr:	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp s	WhichE		;Go see which error code to return.
HdwErr:	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:	jz s	Kaput		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
Kaput:	stc			;Set carry flag to denote "error"!
	retf			;Exit.
	db	0		;(Unused alignment "filler").
DvrEnd	equ	$+STACK		;Size of stand-alone driver.
;
; Main Caching Routine.
;
Cache:	btr	[bx+IOF-@],bl	;Does cache need to be "flushed"?
	jnc s	NextIO		;No, proceed with this I-O request.
	mov	ax,0FFFFh	;Get binary-search table data.
@TC1	equ	[$-2].lw	     ;(Table count - 1, Init set).
	mov	si,offset (SrchT-@)
@TE1	equ	[$-2].lw	     ;(Table upper-limit, Init set).
FlushT:	dec	si		     ;Flush our binary-search table.
	dec	si
	mov	cs:[si],ax
	dec	ax
	jns s	FlushT
	mov	[bx+STLmt-@],si	     ;Reset "free" cache-index ptr.
	or	[bx+LUTop-@].dwd,-1  ;Reset LRU start & end indexes.
NextIO:	push	ds		     ;Set ES-reg. same as DS-reg.
	pop	es
	mov	al,0		;Get cache block "granularity".
@GRAN1	equ	[$-1].lb	;(Block "granularity", set by Init).
	cmp	al,[bx+RqSec-@]	;Will we need multiple blocks?
	jbe s	SetSC		;Yes, use full sector count.
	mov	al,[bx+RqSec-@]	;Use remaining sector count.
SetSC:	mov	[bx+RqCSC-@],al	;Set maximum I-O sector count.
	mov	[bx+RqXMS-@],bx ;Reset lower XMS buffer offset.
	mov	dx,0		;Set up for our binary search.
@MP1	equ	[$-2].lw	;(Beginning offset, set by Init).
	mov	bp,offset (SrchT-2-@)
@MP2	equ	[$-2].lw	;(Midpoint address, set by Init).
Search:	shr	dx,1		;Divide binary-search offset by 2.
	and	dl,0FEh		;Ensure offset remains "even".
	cmp	bp,[bx+STLmt-@]	;Is our search pointer too high?
	jae s	SrchLo		;Yes, cut search pointer by offset.
	mov	ax,cs:[bp]	;Get next binary-search table index.
	call	CBGet		;Get next cache entry into buffer.
	jc	EndIO		;BAAAD News if XMS error -- go exit!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	CComp
	jae s	ChkEnd		;Request >= table:  Check for found.
SrchLo:	sub	bp,dx		;Subtract offset from search ptr.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	jmp s	NoFind		;Handle this request as "not found".
ChkEnd:	je s	Found		;Request = table:  Treat as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	CalcEA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	CComp1
	jb s	Found		   ;Request < Entry:  Handle as found.
	ja s	SrchHi		   ;Request > Entry:  Bump search ptr.
	cmp	[bx+CBSec-@].lb,0  ;Is this cache entry "full"?
@GRAN2	equ	[$-1].lb	   ;(Block "granularity", Init set).
	jb s	Found		;No, handle this request as "found".
SrchHi:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	inc	bp		;Bump pointer to next table entry.
	inc	bp
NoFind:	push	cs		;Unfound:  Set search table segment.
	pop	es
	mov	di,[bx+STLmt-@]	;Get next "free" cache-entry index.
	mov	ax,es:[di]
	std			;Enable a "backward" move, and set
	lea	si,[di-2]	;  move-up source address & count.
	mov	cx,di		;  We do the "std" early, for more
	sub	cx,bp		;  time margin v.s. AMD CPU "bugs"!
	shr	cx,1
	rep	movs es:[di].lw,cs:[si].lw  ;Move up higher indexes.
	stosw				    ;Store new search index.
	cld			   ;Re-enable FORWARD "string" cmds.
	push	ds		   ;Reload this driver's ES-reg.
	pop	es
	add	[bx+STLmt-@].lw,2  ;Bump binary-search table limit.
	mov	si,(RqLBA-@)	   ;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			   ;Set "cache unit" in new entry.
	mov	[di],bl		   ;Reset new entry's sector count.
Found:	mov	RqIndex,ax	   ;Post current cache-entry index.
	mov	cx,[bx+RqLBA+4-@]  ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	   ;Set starting XMS sector offset.
	mov	[bx+LBA2+1-@],cx   ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	   ;Is this a new cache-table entry?
	je s	ReLink		   ;Yes, relink entry as top-of-list.
	push	ax		   ;Unlink this entry from LRU list.
	call	UnLink
	pop	ax
ReLink:	mov	[bx+RqXMS+2-@],bx  ;Reset upper XMS buffer offset.
	movzx	edx,ax		   ;Get 32-bit cache block number.
	shl	edx,2		   ;Shift number to starting sector.
@GRANSC	equ	[$-1].lb	   ;("Granularity" shift, Init set).
	add	[bx+RqXMS-@],edx   ;Add to "preset" sector offset.
	shl	[bx+RqXMS-@].dwd,9	  ;Convert sectors to bytes.
	add	[bx+RqXMS-@].dwd,020000h  ;Add in XMS "base" address.
@XBase	equ	[$-4].dwd		  ;(XMS "base", set by Init).
	mov	cx,0FFFFh	;Make this entry "top of list".
	or	[bx+CBLst-@],cx	;Set this entry's "last" index.
	mov	dx,ax		;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	[bx+CBNxt-@],dx ;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	ReLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	UnLnk3
	pop	ax
	jmp s	ReLnk2		;Go deal with I-O sector count.
ReLnk1:	mov	[bx+LUEnd-@],ax	  ;Make entry last on LRU list, too!
ReLnk2:	mov	cx,[bx+LBA2+1-@]  ;Reload initial I-O sector offset.
	mov	ch,0		  ;Get entry's available sectors.
@GRAN3	equ	[$-1].lb	  ;(Block "granularity", Init set).
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	;More I-O sectors than available?
	jae s	Larger		;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	;Reduce current I-O sector count.
	nop			;(Unused alignment "filler").
Larger:	add	cl,[bx+RqCSC-@]	;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	;More sectors than entry has now?
	ja s	Update		;Yes, update entry sectors.
	or	cl,cl		;Reset Z-flag for "put" into XMS.
	call	CBPut		;Update this cache-table entry.
	jc s	EndIOJ		      ;If any XMS error, exit fast!
	test	[bx+RqCmd-@].lb,002h  ;Is this a write request?
	jnz s	CachIO		      ;Yes, output and cache data.
	jmp	BufMov		;Go move cache data to user buffer.
Update:	mov	[bx+CBSec-@],cl	;Update this entry's total sectors.
	call	CBPut		;Update this cache-table entry.
	jc s	EndIOJ		;If any XMS error, exit fast!
	movzx	cx,[bx+RqCSC-@]	;Calculate ending LBA for this I-O.
	mov	si,(RqLBA-@)
	call	CalcEA
	inc	bp		;Point to next search index.
	inc	bp
Ovrlap:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	CachIO		;No, O.K. to handle "found" entry.
	mov	ax,cs:[bp]	;Set next binary-search table index.
	call	CBGet		;Get next cache entry into buffer.
EndIOJ:	jc s	EndIO		;BAAAD News if XMS error -- go exit!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	CComp
	jbe s	CachIO		;Request <= entry:  O.K. to proceed.
	call	Delete		;Delete this overlapping table entry.
	jmp s	Ovrlap		;Go check for more entry overlap.
	db	0		   ;(Unused alignment "filler").
CachIO:	bts	[bx+RqCmd-@].lb,6  ;I-O done during a prior block?
	jc s	BfMore		   ;Yes, buffer more data into cache.
	mov	ax,000F8h	;Get device-type flags for this unit.
	and	al,[bx+RqTyp-@]	;I-O for this unit done by the BIOS?
	js s	BiosIO		;Yes, use our BIOS I-O logic, below.
	shr	ax,1		;Get "internal driver" table offset.
	xchg	ax,si
	call	DvrTbl[si].dwd	;Call this "internal driver" for I-O.
	jmp s	CachRS		;Go restore critical driver settings!
BiosIO:	lds	si,CStack	;BIOS I-O:  Reload caller CPU flags.
	push	[si+2].lw
	popf
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	pushf			;"Call the BIOS" to do this request.
	db	09Ah
@Prv13B	dd	0		;(Prior Int 13h vector, set by Init).
	pushad			;Save all CPU registers again.
	push	ds
	push	es
	mov	al,ah		;Move any BIOS error code to AL-reg.
CachRS:	sti			;Restore critical driver settings.
	cld
	push	08000h		;Reload this driver's DS- & ES-regs.
@CRSSeg	equ	[$-2].lw	;(Driver segment address, Init set.
	pop	ds		;   MUST reload this way, since we
	push	ds		;   could be running from the HMA!).
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	jnc s	InCach		;If good I-O, see if data was cached.
IOErr:	push	ax		;BAAAD News!  Save posted error code.
	mov	ax,RqIndex	;Delete current entry from the cache.
	call	ScanD
	pop	ax		;Reload returned I-O error code.
	stc			;Set carry again to denote error.
EndIO:	db	0EAh		;Return to driver "exit" logic.
	dw	(Exit-@)
@RtnSeg	dw	0		;(Driver segment address, Init set).
	db	0		;(Unused alignment "filler").
InCach:	or	[bx+RqCmd-@],bl	;Did user data get cached during I-O?
	js s	ChkFul		;Yes, check if cache tables are full.
BfMore:	or	ax,1		     ;Ensure we LOAD data into cache!
BufMov:	movzx	ecx,[bx+RqCSC-@].lb  ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]     ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]   ;Set user buffer address as dest.
	jz s	BufMv1		   ;Will we be reading from cache?
	xchg	esi,edi		   ;No, "swap" source & destination.
BufMv1:	shl	ecx,9		   ;Convert sectors to byte count.
	call	[bx+MoveP-@].dwd   ;Move data between user and XMS.
	jc s	IOErr		   ;If move error, use routine above.
	nop				   ;(Alignment "filler").
ChkFul:	cmp	[bx+STLmt-@].lw,(SrchT-@)  ;Binary-search table full?
@TE2	equ	[$-2].lw		   ;(Table "end", Init set).
	jb s	MoreIO		   ;No, check for more sectors to go.
	mov	ax,[bx+LUEnd-@]	   ;Delete least-recently-used entry.
	call	ScanD
MoreIO:	xor	ax,ax		   ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	   ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	   ;More data sectors left to handle?
	jz s	EndIO		   ;No, return to our "exit" routine.
	add	[bx+RqLBA+4-@],cx  ;Update current LBA for next I-O.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		   ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl  ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	NextIO		   ;Go handle next cache data block.
;
; Subroutine to calculate ending request or cache-table LBA values.
;
CalcEA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
	ret			;Exit.
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
CComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
CComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	CCompX		;If units are different, exit now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
CCompX:	ret			;Exit -- Main routine checks results.
;
; Subroutine to set up and execute cache-entry XMS moves.
;
WBGet:	xor	di,di		;Work buffer:  Set Z-flag for "get".
WBPut:	mov	edi,(WBLBA-@)	;Set 32-bit "working" buffer address.
@WBAdr	equ	[$-4].dwd	;(Working-buffer address, Init set).
	jmp s	CEMov		;Go move data to/from current buffer.
CBGet:	xor	di,di		;Entry buffer:  Set Z-flag for "get".
CBPut:	mov	edi,(CBLBA-@)	;Set 32-bit "current" buffer address.
@CBAdr	equ	[$-4].dwd	;(Current-buffer address, Init set).
CEMov:	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	xor	ecx,ecx		;Set cache-table entry size.
	mov	cl,12
	mul	cx		;Multiply cache index by entry size.
	push	dx		;Get cache-entry offset in ESI-reg.
	push	ax
	pop	esi
	add	esi,010000h	;Add cache-table "base" address.
@CTBase	equ	[$-4].dwd	;(Cache-table "base", Init set).
	popf			;Reload CPU flags.
	jz s	CEMov1		;Are we "getting" a cache entry?
	xchg	esi,edi		;No, "swap" source and destination.
CEMov1:	push	bp		;Save BP-reg. now (helps alignment).
	call	MoveP		;"Get" or "put" desired cache entry.
	pop	bp		;Reload all 16-bit regs. we used.
	pop	dx
	pop	cx
	pop	ax		;Reload cache-entry index.
	jc s	CEMovE		;Any BIOS or EMM386 move errors?
	ret			;No, all is well -- exit.
CEMovE:	or	[bx+IOF-@].lb,1	;BAAAD News!  Post our "flush" flag.
	stc			;Set carry again after "or" command.
	mov	al,XMSERR	;Post "XMS error" return code.
	ret			;Exit.
;
; Subroutine to scan for a binary-search table index and delete it.
;
ScanD:	mov	cx,0		;Set up a 16-bit "scan" command.
@TC2	equ	[$-2].lw	;(Search-table count, set by Init).
	mov	di,offset (SrchT-@)
@TA1	equ	[$-2].lw	;(Search-table start, set by Init).
	push	cs
	pop	es
	repne	scasw		;"Scan" for index in search table.
	lea	bp,[di-2]	;Index is at [di-2], due to "scasw".
	call	CBGet		;Get cache entry from XMS to buffer.
	jc s	UnLnkE		;If any XMS error, go exit below.
;
; Subroutine to delete a cache index from the binary-search table.
;
Delete:	mov	di,bp		;Point DI-reg. to index in table.
	lea	si,[di+2]	;Set move-down source and count.
	mov	cx,[bx+STLmt-@]
	sub	cx,si
	shr	cx,1
	push	cs		;Set binary-search table segment and
	pop	es		;  move down remaining table indexes.
	rep	movs es:[di].lw,cs:[si].lw
	stosw			;Put deleted index on free-list.
	push	ds		;Reload this driver's ES-reg.
	pop	es
	sub	[bx+STLmt-@].lw,2  ;Decrement table limit.
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
UnLink:	mov	cx,[bx+CBLst-@]	;Get current entry's "last" index.
	mov	dx,[bx+CBNxt-@]	;Get current entry's "next" index.
	cmp	cx,-1		;Is this entry top-of-list?
	je s	UnLnk1		;Yes, "promote" next entry.
	mov	ax,cx		;Get "prior" entry into work buffer.
	call	WBGet
	jc s	UnlnkE		;If XMS error, go exit below!
	mov	[bx+WBNxt-@],dx	;Update working entry's "next" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	WBPut
	jnc s	UnLnk2		;If no XMS error, check end-of-list.
UnLnkE:	pop	cx		;XMS error!  Discard return address.
	jmp	EndIO		;Go post XMS error code and get out!
UnLnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
UnLnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	UnLnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
UnLnk3:	mov	ax,dx		;Get "next" entry into work buffer.
	call	WBGet
	jc s	UnLnkE		;If XMS error, go exit above!
	mov	[bx+WBLst-@],cx	;Update working entry's "last" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	WBPut
	jc s	UnLnkE		;If any XMS error, go exit above!
	ret			;Exit.
	dw	0,0,0		;(Unused alignment "filler").
HMALEN	equ	($-UdmaIO)	;Maximum length of logic moved to HMA.
;
; Cache Table Definitions.
;
SrchT	equ	$		;Start of binary-search table.
ResEnd	equ	SrchT+STACK	;End of "resident" logic and stack.
;
; Initialization Cache-Sizes Table.
;
; NOTE:  /Snnnn "large caches" of more than 250-MB should be used only
; on a Pentium IV CPU or better, due to all the required binary-search
; table handling.   At least 512-MB of upper memory must be available,
; and more is HIGHLY recommended, so other applications will also have
; a bit of XMS memory!
;
CachSiz	dw	TBLCT		;/S0 5-MB:  Fixed 640 cache blocks.
	db	16		;    8K granularity:  16 sectors per
	db	4		;       block, 4 sector-shift count.
	db	(1024/256)	;    1024 binary-search midpoint.
	db	8		;    8K cache-table XMS memory.
	db	"   5"		;    5-MB "title" message bytes.
	dw	(TBLCT*2)	;/S1 10-MB:  Fixed 1280 cache blocks.
	db	16,4		;    8K granularity values.
	db	(2048/256)	;    2048 binary-search midpoint.
	db	16		;    16K cache-table XMS memory.
	db	"  10"		;    10-MB "title" message bytes.
	dw	(TBLCT*2)	;/S2 20-MB:  Fixed 1280 cache blocks.
	db	32,5		;    16K granularity values.
	db	(2048/256)	;    2048 binary-search midpoint.
	db	16		;    16K cache-table XMS memory.
	db	"  20"		;    20-MB "title" message bytes.
	dw	(TBLCT*2)	;/S3 40-MB:  Fixed 1280 cache blocks.
	db	64,6		;    32K granularity values.
	db	(2048/256)	;    2048 binary-search midpoint.
	db	16		;    16K cache-table XMS memory.
	db	"  40"		;    40-MB "title" message bytes.
VTBlkCt	dw	(80*16)		;/Snnnn:  nnnn * 16 cache blocks.
	db	128,7		;    64K granularity values.
	db	(2048/256)	;    2048 to 16384 search midpoint.
	db	24		;    24K to 192K cache-table XMS.
	db	"  80"		;    nnnn-MB "title" message bytes.
;
; Initialization Variables.
;
XMHdl	dw	0		;XMS memory "handle" number.
XMAlloc	db	009h		;XMS V2.0 "allocate memory" command.
RFlag	db	0		;"Restricted memory" flag (no HMA).
SFlag	db	4		;User cache-size flag (default = 4).
BiosHD	db	0		;Number of BIOS hard disks.
HDUnit	db	0		;Current BIOS unit number.
	db	0		;(Unused alignment "filler").
;
; Initialization UltraDMA "Mode" Table (digit count in low 4 bits).
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166.
;
; Initialization Messages.
;
EDDBuff equ	$
TTLMsg	db	CR,LF,'UDMA, 9-23-2007'
TTL1	db	'.   '
TTL2	db	'   5-MB Cache.',CR,LF,'$'
IBMsg	db	'PCI test failed, BIOS I-O only!',CR,LF,'$'
NXMsg	db	'No XMS manager$'
DNMsg	db	'is '
DName	equ	$
DNEnd	equ	DName+40
BCMsg	db	'BAD$'
PCMsg	db	'IDE1'
PCMsg1	db	' Controller at I-O address '
PCMsg2	db	'XXXXh, Chip I.D. '
PCMsg3	db	'XXXXXXXXh.',CR,LF,'$'
NonUMsg	db	'Disks run by the BIOS:  '
NonUCt	db	'0. ',CR,LF,'$'
EDDMsg	db	'EDD BIOS$'
CHSMsg	db	'CHS'
UnitMsg	db	' data ERROR, D'
UnitNam	db	'rive '
UnitNo	db	'A: ',CR,LF,'$'
CtlMsg	db	'IDE1 $'
PriMsg	db	'Primary-$'
SecMsg	db	'Secondary-$'
MstMsg	db	'master disk $'
SlvMsg	db	'slave disk $'
IEMsg	db	'Identify ERROR!',CR,LF,'$'
UEMsg	db	'is not UltraDMA!',CR,LF,'$'
VEMsg	db	'VDS init error$'
NDMsg	db	'No disk to use$'
PRMsg	db	'No 80386+ CPU'
Suffix	db	'; UDMA not loaded!',CR,LF,'$'
;
; DOS "Strategy" Routine.
;
Strat:	mov	cs:RqBuf.lw,bx	;Save DOS "Init" packet pointer.
	mov	cs:RqBuf.hw,es
	retf			;Exit & await "Device Interrupt".
;
; DOS "Device Interrupt" Routine.   This logic initializes the driver.
;
DevInt:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,RqBuf	  ;Point to DOS "Init" packet.
	cmp	es:[bx].RPOp,0	  ;Is this really an "Init" packet?
	jne s	I_Quit		  ;No?  Reload regs. and exit quick!
	mov	ax,es:[bx].RPLen  ;Save "free" regular-memory limit.
	mov	@XEntry.lw,ax	  ;("Free" HMA is determined below).
	mov	ax,es:[bx].RPSeg
	mov	@XEntry.hw,ax
	mov	ax,cs		  ;Set "fixed" driver segment values.
	mov	MoveP.hw,ax
	mov	@GMAddr.hw,ax
	mov	@CRSSeg,ax
	mov	@RtnSeg,ax
	mov	DvrTbl.hw,ax	;Set "UdmaIO" driver segment pointers.
	mov	Ud2Ptr.hw,ax

;*** NOTE ***	Set other "internal" driver segment pointers here!

	mov	es:[bx].RPStat,RPDON+RPERR  ;Set init packet defaults.
	mov	es:[bx].RPSeg,cs
	and	es:[bx].RPLen,0
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes, CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_386		;Yes, CPU is at least an 80386.
I_Junk:	mov	dx,(PRMsg-@)	;Display "No 80386+ CPU" message.
	call	I_Msg
I_Quit:	jmp	I_Exit		;Go reload registers and exit quick!
I_386:	pushad			;80386+ CPU:  Save 32-bit registers.
	les	si,es:[bx].RPCL	;Get command-line data pointer.
I_NxtC:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	inc	si		;Bump pointer past first byte.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_Term		;Yes, save current Int 13h vector.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_Term		;Yes, save current Int 13h vector.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je s	I_Term		;Yes, save current Int 13h vector.
	cmp	al,'-'		;Is byte a dash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'/'		;Is byte a slash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	al,ah		;Get byte after the slash or dash.
	and	al,0DFh		;Mask out lower-case bit (020h).
	cmp	al,'A'		;Is this byte an "A" or "a"?
	jne s	I_ChkQ			   ;No, test for "Q" or "q".
	mov	Ctl1Pri.lb,(APDATA-0100h)  ;Reverse all "Legacy
	mov	Ctl1Sec.lb,(ASDATA-0100h)  ;  IDE" I-O addresses.
	mov	Ctl2Pri.lb,(NPDATA-0100h)
	mov	Ctl2Sec.lb,(NSDATA-0100h)
	inc	si			   ;Point to next cmd. byte.
I_ChkQ:	cmp	al,'Q'		;Is this byte a "Q" or "q"?
	jne s	I_ChkR		;No, see if byte is "R" or "r".
	mov	@DRQ.lb,075h	;Enable "data request" tests above.
	inc	si		;Point to next command-line byte.
I_ChkR:	cmp	al,'R'		;Is this byte an "R" or "r"?
	jne s	I_ChkS		;No, see if byte is "S" or "s".
	mov	RFlag,al	;Set "restricted memory" flag.
	inc	si		;Point to next command-line byte.
I_ChkS:	cmp	al,'S'		;Is this byte an "S" or "s"?
	jne s	I_ChkU		;No, see if byte is "U" or "u".
	call	GetSiz		;Get desired user cache size.
I_NxtJ:	jmp s	I_NxtC		;Continue scanning for a terminator.
I_ChkU:	cmp	al,'U'		;Is this byte a "U" or "u"?
	jne s	I_NxtC		;No, go see if byte is a terminator.
	and	ExEntrP,0	;Reset "External Entry" routine ptr.
	mov	ax,(DvrEnd-@)	;Set stand-alone driver size.
	mov	@Stack,ax
	mov	VDSLn.lw,ax
	db	066h,0B8h	     ;Enable the stand-alone driver
	db	(A20Er-GoMain),09Ah  ;  with a "call UdmaIO" at the
	dw	(UdmaIO-@)	     ;  label "GoMain" above.
	mov	[GoMain-1].dwd,eax
	mov	al,0EBh		     ;MUST have user buffer for I-O!
	mov	@UDSA.lb,al
	mov	RFlag,al	     ;Set "restricted" to exit below.
	mov	TTL1.dwd,0240A0D2Eh  ;Disable cache size in "title".
	inc	si		     ;Point to next command-line byte.
	jmp s	I_NxtJ		;Continue scanning for a terminator.
I_Term:	mov	ax,03513h	;Get and save current Int 13h vector.
	call	I_In21
	push	es
	push	bx
	pop	eax
	mov	@Prv13A,eax
	mov	@Prv13B,eax
	cmp	GoMain.lb,09Ah	;Will this be the stand-alone driver?
	je	I_TTL		;Yes, go display driver "title" msg.
	cmp	VTBlkCt,250*16	;Is a 250-MB cache or less desired?
	jbe s	I_Rstr		;Yes, see if driver is "restricted".
	inc	RFlag		;Greater may CRASH in some DOS HMAs!
I_Rstr:	cmp	RFlag,0		;"Restricted" to normal memory?
	jne s	I_UprM		;Yes, set "upper-memory" driver size.
	mov	ax,04A01h	;Get total "free" HMA memory.
	call	I_In2F
	sub	bx,HMALEN	;Enough memory for our HMA routines?
	jb s	I_NoHM		;No, we are stuck with normal memory!
	mov	dx,bx		;Set remaining "free" HMA in EDX-reg.
	cmp	dx,(TBLCT*2)	;Enough HMA for a small search table?
	jb s	I_NoHM		;No, we are stuck with normal memory!
	cmp	SFlag,0		;Does our boy want only a 5-MB cache?
	je s	I_Size		;Yes, put binary-search table in HMA.
	cmp	dx,(TBLCT*4)	;Enough for our medium search table?
	jae s	I_Size		;Yes, put binary-search table in HMA.
I_NoHM:	inc	RFlag		;Set "restricted memory" flag.
I_UprM:	mov	ax,(ResEnd-@)	;Set our "normal memory" driver size.
	mov	VDSLn.lw,ax
	mov	@Stack.lw,ax
	xor	edx,edx		;Get 32-bit "free" memory limit.
	xchg	dx,@XEntry.hw
	shl	edx,4
	add	edx,@XEntry
	xor	eax,eax		;Subtract 32-bit driver "base" size
	mov	ax,cs		;  for available search-table space.
	shl	eax,4
	add	eax,(SrchT+STACK-@)
	sub	edx,eax
	jb s	I_Min1		;Limit < driver?  Use minimum 5-MB!
	cmp	edx,65536	;64K or more of free memory?
	jb s	I_MinM		;No, see if we have minimum needed.
	mov	dx,0FFFFh	;Use 64K - 1 as free memory size.
I_MinM:	cmp	dx,(TBLCT*2)	;Enough for our minimum 5-MB cache?
	jae s	I_Size		;Yes, go see what cache is desired.
I_Min1:	mov	dx,(TBLCT*2)	;Our .SYS is 4K!  Use minimum 5-MB!
I_Size:	mov	si,(CachSiz-@)	;Point to our cache-sizes table.
	movzx	cx,SFlag	;Did user want only the 5-MB cache?
	jcxz	I_Siz2		;Yes, go "set it up" below.
I_Siz1:	mov	ax,[si+10]	;Get next binary-search table size.
	shl	ax,1
	cmp	dx,ax		;Enough table space for next cache?
	jb s	I_Siz2		;No, default to current cache size.
	add	si,10		;Point to data for next cache size.
	dec	cx		;Did user request this cache size?
	jnz s	I_Siz1		;No, see if next cache can be used.
I_Siz2:	mov	ax,[si]		;Set cache-entry counts.
	add	@TC1,ax
	mov	@TC2,ax
	shl	ax,1		;Set binary-search limit addresses.
	add	@TE1,ax
	add	@TE2,ax
	cmp	RFlag,0		;"Restricted" to normal memory?
	je s	I_Siz3		;No (using HMA), set "granularity".
	add	VDSLn.lw,ax	;Set final resident driver size.
	add	@Stack,ax	;Set final driver stack pointer.
I_Siz3:	mov	ax,[si+2]	;Set cache "granularity" values.
	mov	@GRAN1,al
	mov	@GRAN2,al
	mov	@GRAN3,al
	mov	@GRANSC,ah
	and	al,080h		;80-MB cache and up (64K granularity)
	or	XMAlloc,al	;  must issue V3.0 XMS "allocate" cmd.
	xor	ax,ax		;Set binary-search midpoint values.
	mov	ah,[si+4]
	mov	@MP1,ax
	add	@MP2,ax
	mov	ax,@TC2		;Multiply number of cache blocks
	movzx	dx,@GRAN1	;  times (sectors-per-block / 2).
	shr	dx,1
	mul	dx
	push	dx		;Get 32-bit cache data XMS size.
	push	ax
	pop	eax
	movzx	ecx,[si+5].lb	;Get cache-table XMS memory size.
	add	eax,ecx		;Save total required XMS memory.
	add	RqXMS,eax
	mov	eax,[si+6]	;Set cache size in "title" message.
	mov	TTL2.dwd,eax
I_TTL:	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	or	al,es:[HDISKS]	;Did BIOS find any hard-disks?
	jz	I_None		;No?  Display "No disk" and exit!
	mov	BiosHD,al	;Save our BIOS hard-disk count.
	mov	ax,cs		;Set 20-bit VDS driver address.
	mov	VDSSg.lw,ax
	shl	eax,4
	mov	VDSAd,eax
	cli			      ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h  ;Are "VDS services" active?
	jz s	I_REnI		      ;No, re-enable CPU interrupts.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	I_VDS
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
	jc	I_Err		;If "lock" error, display msg. & exit!
	inc	VDSOf.lb	;Set initialization VDS "lock" flag.
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,VDSAd	;Get final driver 32-bit address.
	add	GDTPAdr,eax	;Relocate "real mode" GDT base addr.
	add	@WBAdr,eax	;Relocate "working" buffer address.
	add	@CBAdr,eax	;Relocate cache-entry buffer address.
	add	@CLAddr,eax	;Relocate command-list source address.
	mov	GDT_CSL,ax	;Set lower CS-descriptor address.
	shr	eax,16		;Set upper CS-descriptor address.
	mov	GDT_CSM,al
	mov	GDT_CSH,ah
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	al,001h
	call	I_In1A
	cmp	edx," ICP"	;Is PCI BIOS V2.0C or newer?
	je s	I_ScnC		;Yes, scan for all IDE controllers.
	mov	dx,(IBMsg-@)	;Display "No V2.0C+ PCI" message.
	call	I_Msg
	jmp s	I_XMgr		;Go see if we have an XMS manager.
I_ScnC:	mov	al,LBABuf.lb	;Get next "interface bit" value.
	and	ax,00003h
	or	ax,ExtDvr.lw	;"Or" in subclass & current function.
	call	I_PCIC		;Test for specific PCI class/subclass.
	rol	LBABuf.lb,4	;Swap both "interface bit" values.
	mov	al,LBABuf.lb	;Load next "interface bit" value.
	or	al,al		;Both "interface" values tested?
	jns s	I_ScnC		;No, loop back and test 2nd one.
	add	ExtDvr.lb,004h	;More PCI function codes to try?
	jnc s	I_ScnC		;Yes, loop back & try next function.
	test	al,001h		;Have we tested "Native PCI" ctlrs.? 
	mov	LBABuf.lb,093h	;(Set "Native PCI" interface bits).
	jz s	I_ScnC		;No, loop back and test them, also.

;
; These additional controller tests could be added, if necessary:
;
;	mov	ax,00400h	;Scan for "RAID" controller.
;	call	I_PCIC
;	mov	ax,00520h	;Scan for ADMA "Single-stepping" ctlr.
;	call	I_PCIC
;	mov	ax,00530h	;Scan for ADMA "Continuous DMA" ctlr.
;	call	I_PCIC
;	mov	ax,00600h	;Scan for standard SATA controller.
;	call	I_PCIC
;	mov	ax,00601h	;Scan for "AHCI" SATA controller.
;	call	I_PCIC

I_XMgr:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	mov	dx,(NXMsg-@)	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne s	I_ErrJ		;No?  Display error message and exit!
	mov	ax,04310h	;Get XMS manager "entry" address.
	call	I_In2F
	push	es		;Save XMS manager "entry" addresses.
	push	bx
	pop	@XEntry
	mov	ah,XMAlloc	;Request all necessary XMS memory.
	mov	edx,RqXMS
	call	I_XMS
	jz s	I_XLok		;If no error, "lock" our XMS memory.
I_XErr:	mov	dx,(VEMsg-@)	;Set up "XMS init error" message.
	mov	VEMsg.lw,"MX"
I_ErrJ:	jmp	I_Err		;Go display XMS error message & exit!
I_XLok:	mov	XMHdl,dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	mov	@XBufAd,eax	;Save aligned "main buffer" address.
	push	@XBufAd.hw	;Post upper address for other drivers.
	pop	XMBuf
	add	@XBase,eax	;Initialize cache-data base address.
	mov	cx,ax		;Save buffer "offset" in XMS memory.
	sub	cx,dx
	mov	XMOfs,cx
	mov	edx,000010000h	;Put command-list after XMS buffer.
	jcxz	I_PRDA		;Is buffer already on a 64K boundary?
	mov	edx,-32		;No, put command-list before buffer.
	dec	@XBase.hw	;Decrement cache-data base by 64K.
I_PRDA:	add	eax,edx		;Set our 32-bit PRD address.
	mov	PRDAd,eax
	mov	ax,[RqXMS+1].lw	;Get needed XMS in 256-KB increments.
	and	al,0FCh		;Reset lower 2 bits = Megabytes only.
	shl	eax,18		;Convert from Megabytes to bytes.
	add	eax,@XBase	;Add cache-data XMS base address.
	mov	@CTBase,eax	;Set cache-table XMS base address.
	cmp	GoMain.lb,09Ah	;Will this be the stand-alone driver?
	je s	I_HDCh		;Yes, forget about running diskettes!
	xor	ax,ax		;Point ES-reg. to low memory.
	mov	es,ax
	mov	al,es:[HDWRFL]	;Get BIOS "hardware installed" flag.
	test	al,001h		;Any diskettes on this system?
	jz s	I_HDCh		;No, scan for available hard-disks.
	mov	al,es:[MCHDWFL]	;Use diskette media-change bits
	and	al,011h		;  for our number of diskettes.
I_FScn:	test	al,001h		;Can next unit signal media-changes?
	jz s	I_FMor		;No?  CANNOT use this old "clunker"!
	push	ax		;Save our diskette counter.
	mov	al,0C0h		;Get "diskette" device-type flags.
	call	I_CHSD		;Get and save diskette's CHS values.
	pop	ax		;Reload our diskette counter.
	inc	@LastU		;Bump unit-table index.
I_FMor:	inc	HDUnit		;Bump BIOS unit number.
	inc	UnitNo.lb	;Bump error-message unit number.
	shr	al,4		;Another diskette to check?
	jnz s	I_FScn		;Yes, loop back and do next one.
I_HDCh:	mov	HDUnit,080h	    ;Set 1st BIOS hard-disk unit.
	mov	UnitNam.dwd," ksi"  ;Set messages for hard-disks.
	mov	[UnitNo+1].lw,".h"
I_Next:	mov	ah,HDUnit	    ;Set unit no. in error message.
	mov	cx,2
	mov	si,(UnitNo-1-@)
	call	I_HexA
	mov	al,080h		;Get "BIOS disk" device-type.
	call	I_CHSD		;Get and save next disk's CHS values.
	mov	ah,041h		;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call	I_In13
	jc s	I_NoUJ		;If none, bump non-UltraDMA disk count.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne s	I_NoUJ		;No, bump non-UltraDMA disk count.
	test	cl,004h		;Does this disk have "EDD" extensions?
	jz s	I_NoUJ		;No, bump non-UltraDMA disk count.
	mov	si,(EDDBuff-@)	;Point to "EDD" input buffer.
	mov	[si].lw,30	;Set 30-byte buffer size.
	mov	ah,048h		;Get this disk's "EDD" parameters.
	call	I_In13
	jc s	I_ErED		;Error?  Display msg. & ignore!
	cmp	[si].lw,30	;Did we get at least 30 bytes?
	jb s	I_NoUJ		;No, bump non-UltraDMA disk count.
	cmp	[si+26].dwd,-1	;Valid "DPTE" pointer?
	je s	I_NoUJ		;No, bump non-UltraDMA disk count.
	les	si,[si+26]	;Get this disk's "DPTE" pointer.
	mov	bx,15		;Calculate "DPTE" checksum.
	xor	cx,cx
I_CkSm:	add	cl,es:[bx+si]
	dec	bx
	jns s	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErED:	mov	dx,(EDDMsg-@)	;Display "EDD BIOS ERROR" message.
	call	I_Msg
	mov	dx,(UnitMsg-@)
	call	I_Msg
I_NoUJ:	jmp	I_NoUD		;Go bump number of non-UltraDMA disks.
I_EDOK:	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,es:[si+4]
	shr	bl,4
	mov	ax,es:[si]	   ;Get disk's IDE base address.
	mov	[CtlMsg+3].lb,'1'  ;Reset display to "Ctlr. 1".
	mov	si,(Ctl1Tbl-@)	   ;Point to IDE address table.
I_ITbl:	cmp	[si].lw,-1	;Is this IDE table active?
	je s	I_NoUJ		;No, go bump non-UltraDMA disk count.
	cmp	ax,[si+2]	;Is disk on this primary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust index for secondary channel.
	inc	bx
	cmp	ax,[si+4]	;Is disk on this secondary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust values for next controller.
	inc	bx
	add	si,CTLTSIZ
	inc	[CtlMsg+3].lb
	cmp	si,(CtlTEnd-@)	;More IDE controllers to check?
	jb s	I_ITbl		;Yes, loop back and check next one.
	jmp s	I_NoUJ		;Go bump non-UltraDMA disk count.
I_ChMS:	push	bx		;Save disk's caching unit number.
	mov	IdeDA,ax	;Save disk's base IDE address.
	add	ax,CDSEL	;Point to IDE device-select register.
	xchg	ax,dx
	mov	al,bl		;Get drive-select command byte.
	shl	al,4
	or	al,LBABITS
	out	dx,al		;Select master or slave disk.
	push	ax		;Save drive-select and caching unit.
	push	bx
	mov	dx,(CtlMsg-@)	;Display IDE controller number.
	call	I_Msg
	pop	ax		;Reload caching unit number.
	mov	dx,(PriMsg-@)	;Point to "Primary" channel name.
	test	al,002h		;Is this a primary-channel disk?
	jz s	I_PRNm		;Yes, display "Primary" channel.
	mov	dx,(SecMsg-@)	;Point to "Secondary" channel name.
I_PRNm:	call	I_Msg		;Display disk's IDE channel name.
	pop	ax		;Reload drive-select byte.
	mov	dx,(MstMsg-@)	;Point to "master" disk name.
	test	al,010h		;Is this disk the master?
	jz s	I_MSNm		;Yes, display "master" name.
	mov	dx,(SlvMsg-@)	;Point to "slave" disk name.
I_MSNm:	call	I_Msg		;Display disk's master/slave name.
	call	I_ValD		;Validate disk as an UltraDMA unit.
	pop	ax		;Reload caching unit number.
	jc s	I_UDEr		;If any errors, display message.
	mov	bx,@LastU	;Change this disk's device-type to
	mov	[bx+TypeF-@],al	;  "IDE" and include channel/unit.
	jmp s	I_More		;Go check for any more BIOS disks.
I_UDEr:	call	I_Msg		;NOT UltraDMA!  Display error msg.
I_NoUD:	cmp	GoMain.lb,09Ah	;Non-UDMA disk:  Stand-alone driver?
	je s	I_IgnD		;Yes, ignore disk and scan for more.
	inc	NonUCt		;Bump number of non-UltraDMA disks.
	cmp	NonUCt,'9'	;Over 9 non-UltraDMA hard disks?
	jbe s	I_More		       ;No, check for more disks.
	mov	NonUCT.dwd,00D2E2B39h  ;Set 9+ non-UltraDMA count.
I_More:	inc	@LastU		       ;Bump our unit-table index.
I_IgnD:	inc	HDUnit		;Bump BIOS unit number.
	cmp	@LastU,MAXBIOS	;More entries left in units table?
	jae s	I_AnyN		;No, check for non-UltraDMA disks.
	dec	BiosHD		;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyN:	cmp	NonUCt.lb,'0'	;Do we have any non-UltraDMA disks?
	je s	I_AnyU		;No, check for any UltraDMA disks.
	mov	dx,(NonUMsg-@)	;Display "Non-UltraDMA disks" msg.
	call	I_Msg
I_AnyU:	cmp	@LastU,0	;Did we find any drives to use?
	jne s	I_HMA		;Yes, see about loading in the HMA.
	cmp	GoMain.lb,09Ah	;Stand-alone driver requested?
	je	I_None		;Yes?  Display "No disks" and exit!
I_HMA:	shl	RFlag,1		;"Restricted" to normal memory?
	jnz	I_Done		;Yes, set DOS "init" packet results.
	mov	ax,04A02h	;Request needed memory in the HMA.
	mov	bx,@TC2
	shl	bx,1
	add	bx,HMALEN
	call	I_In2F
	mov	ax,es		;Get returned HMA segment address.
	cmp	ax,-1		;Is segment = 0FFFFh?
	jne s	I_HMA1		;No, VERY unusual but not an error.
	cmp	ax,di		;Is offset also = 0FFFFh?
	jne s	I_HMA1		;No, set binary-search table addr.
	mov	eax," AMH"	;BAAAD News!  Set up HMA error msg.
	jmp s	I_HMA2		;Go display error message and exit!
I_HMA1:	push	ax		;Set our HMA routine entry address.
	lea	dx,[di+Cache-UdmaIO]
	push	dx
	pop	@GMAddr

;*** NOTE ***	Update other "internal" driver pointers here!

	push	ax		;Update "UdmaIO" driver pointers.
	push	di
	pop	eax
	mov	DvrTbl,eax
	mov	Ud2Ptr,eax
	add	di,HMALEN	;Set binary-search table address.
	mov	@TA1,di
	mov	ax,@TC2		;Set binary-search table limits.
	shl	ax,1
	add	ax,di
	mov	@TE1,ax
	mov	@TE2,ax
	mov	ax,@MP1		;Set binary-search midpoint address.
	dec	ax
	dec	ax
	add	ax,di
	mov	@MP2,ax
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
	jz s	I_HMA3		;If no error, move logic into HMA.
	mov	eax," 02A"	;BAAAD News!  Set up A20 error msg.
I_HMA2:	mov	VEMsg.dwd,eax
	mov	dx,(VEMsg-@)
	jmp s	I_Err		;Go display error message & exit!
I_HMA3:	cld			;Move our cache logic up to the HMA.
	mov	cx,(HMALEN/2)
	mov	si,(UdmaIO-@)
	les	di,@GMAddr
	sub	di,(Cache-UdmaIO)
	rep	movsw
	mov	ah,006h		;Issue "A20 local-disable" request.
	call	I_XMS
I_Done:	mov	al,(XMBuf-@)/4	;Post UDVD "External Variables" ptr.
	mov	VarPtr,al
	xor	ax,ax		;Load & reset driver length.
	xchg	ax,VDSLn.lw
	les	bx,RqBuf	;Set result values in "init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	jmp	I_End		;Go clear driver's stack, then exit.
I_None:	mov	dx,(NDMsg-@)	;Point to "No disk to use" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	shr	VDSOf.lb,1	;Was driver "locked" by VDS?
	jnc s	I_XDis		;No, check on XMS memory.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,XMHdl	;Load our XMS memory "handle".
	or	dx,dx		;Did we reserve any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
	call	I_Msg		;Display desired error message.
	mov	dx,(Suffix-@)	;Display error-message suffix.
	call	I_Msg
	jmp	I_Fail		;Go reload all registers and exit.
;
; Subroutine to get the user's desired /Snnnn cache size.
;
GetSiz:	inc	si		;Bump scan pointer past "S" or "s".
	mov	di,(VTBlkCt-@)	;Point to variable cache-block count.
	mov	al,(2048/256)	;Set default 80-MB cache parameters.
	mov	ah,24
	mov	[di+4],ax
	mov	[di+6].dwd,"    "  ;Reset variable "title" bytes.
GtSiz0:	mov	[di].lw,08000h	   ;Invalidate cache-block count.
GtSiz1:	mov	al,es:[si]	;Get next command-line byte.
	cmp	al,'9'		;Is byte greater than a '9'?
	ja s	GtSiz2		;Yes, go see what is in bucket.
	cmp	al,'0'		;Is byte less than a '0'?
	jb s	GtSiz2		;Yes, go see what is in bucket.
	inc	si		;Bump scan pointer past digit.
	cmp	[di+6].lb,' '	;Have we already found 4 digits?
	jne s	GtSiz0		;Yes, set INVALID & keep scanning.
	push	[di+7].dwd	;Shift "title" bytes 1 place left.
	pop	[di+6].dwd
	mov	[di+9],al	;Insert next "title" message byte.
	mov	cx,[di]		;Multiply current block size by 10.
	shl	[di].lw,2
	add	[di],cx
	shl	[di].lw,1
	and	ax,0000Fh	;"Add in" next cache size digit.
	add	[di],ax
	jmp s	GtSiz1		;Go scan for more cache-size digits.
GtSiz2:	mov	ax,[di]		;Get user's desired cache Megabytes.
	mov	SFlag,0		;Set 5-MB cache size flag.
	cmp	ax,10		;Does user want only a 5-MB cache?
	jb s	GTSizX		;Yes, exit below.
	inc	SFlag		;Set 10-MB cache size flag.
	cmp	ax,20		;Does user want the 10-MB cache?
	jb s	GtSizX		;Yes, exit below.
	inc	SFlag		;Set 20-MB cache size flag.
	cmp	ax,40		;Does user want the 20-MB cache?
	jb s	GtSizX		;Yes, exit below.
	inc	SFlag		;Set 40-MB cache size flag.
	cmp	ax,80		;Does user want the 40-MB cache?
	jb s	GtSizX		;Yes, exit below.
	inc	SFlag		;Set "large" cache size flag.
	cmp	ax,1023		;Is user cache size invalid or > 1-GB?
	ja s	GtSizE		;Yes, ignore it & reset 80-MB default.
	cmp	ax,128		;Is cache size 80-MB to 127-MB?
	jb s	GTSiz3		;Yes, set user cache-block count.
	shl	[di+4].lw,1	;Double variable cache parameters.
	cmp	ax,256		;Is cache size 128-MB to 255-MB?
	jb s	GTSiz3		;Yes, set user cache-block count.
	shl	[di+4].lw,1	;Double variable cache parameters.
	cmp	ax,512		;Is cache size 256-MB to 511-MB?
	jb s	GTSiz3		;Yes, set user cache-block count.
	shl	[di+4].lw,1	;Double variable cache parameters.
GtSiz3:	shl	[di].lw,4	;Multiply blocks by 16 per Megabyte.
GtSizX:	ret			;Exit.
GtSizE:	mov	[di],(TBLCT*2)	   ;Error!  Reset default 80-MB cache.
	mov	[di+6].dwd,"08  "
	ret			   ;Exit.
;
; Subroutine to test for and set up a specific PCI disk controller.
;
I_PCIC:	mov	IdeDA,ax	     ;Save subclass & function codes.
	and	LBABuf.hw,0	     ;Reset PCI device index.
I_PCI1:	cmp	RqIndex,(CtlTEnd-@)  ;More space in address tables?
 	jae s	GtSizX		     ;No, go exit above.
	mov	ax,IdeDA	;Test PCI class 1, subclass/function.
	mov	ecx,000010003h	;(Returns bus/device/function in BX).
	xchg	ax,cx
	mov	si,LBABuf.hw
	call	I_In1A
	jc s	GtSizX		;Controller not found -- exit above.
	inc	LBABuf.hw	;Bump device index for another test.
	xor	di,di		;Get controller Vendor & Device I.D.
	call	I_PCID
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Save DMA controller base address.
	call	I_PCID
	xchg	ax,cx
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,(PCMsg2-@)	;Set controller address in message.
	call	I_Hex
	mov	si,(PCMsg3-@)	;Set vendor I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax		;Set Device I.D. in message.
	call	I_Hex
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	not	cl		;Get "Bus Master" & "I-O Space" bits.
	and	cl,005h		;Is controller using both BM and IOS?
	jz s	I_PCI2		;Yes, go save all controller data.
	mov	dx,(BCMsg-@)	;Display "BAD controller!", and hope
	call	I_Msg		;  our user can find a "better" BIOS!
	mov	dx,(PCMsg1-@)
	call	I_Msg
I_PCIJ:	jmp s	I_PCI1		;Go test for more same-class ctlrs.
I_PCI2:	mov	si,RqIndex	;Get current I-O address table ptr.
	mov	ax,DMAAd	;Set controller DMA base address.
	mov	[si],ax
	test	LBABuf.lb,001h	;Is this a "Native PCI" controller?
	jz s	I_PCI3		;No, display all controller data.
	mov	di,16		;Set primary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+2],cx
	mov	di,24		;Set secondary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+4],cx
I_PCI3:	mov	dx,(PCMsg-@)	;Display all this controller's data.
	call	I_Msg
	inc	[PCMsg+3].lb	;Bump controller number in message.
	add	RqIndex,CTLTSIZ	;Bump controller address-table ptr.
	jmp s	I_PCIJ		;Go test for more same-class ctlrs.
;
; Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD:	xor	bx,bx		;Point ES-reg. to low memory.
	mov	es,bx
	mov	dx,[bx+IdeDA-@]	;Issue "Identify Device" command.
	add	dx,CCMD
	mov	al,0ECh
	out	dx,al
	mov	bx,BIOSTMR	;Point to low-memory BIOS timer.
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,es:[bx]
I_IDly:	cmp	cl,es:[bx]	;Has our command timed out?
	je s	I_DErr		;Yes, set "Identify" message & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s 	I_IDly		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz s	I_PIO		;No, read I.D. data using PIO mode.
I_DErr:	mov	dx,(IEMsg-@)	;Point to "Identify error" message.
I_SErr:	stc			;Set carry flag (error!) and exit.
	ret
I_PIO:	add	dx,(CDATA-CCMD)	;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	shl	ax,1		;Save "ATA/ATAPI" flag in carry bit.
	mov	cx,27		;Skip I.D. bytes 2-53 and
I_Skp1:	in	ax,dx		;  read I.D. bytes 54-55.
	loop	I_Skp1
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,(DName-@)
	mov	cl,26		;Read & swap disk name into message.
I_RdNm:	xchg	ah,al		;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw			;  also read but are ignored.   Bytes
	in	ax,dx		;  106-107 are left in the AX-reg.).
	loop	I_RdNm
	xchg	ax,bx		;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35		;Skip I.D. bytes 108-175 &
I_Skp2:	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bh,ah		;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3:	in	ax,dx
	loop	I_Skp3
	mov	dx,(UEMsg-@)	;Point to "is not UltraDMA" message.
	rcr	bl,1		;Shift "ATA/ATAPI" flag into BL-reg.
	and	bl,082h		;ATAPI disk, or UltraDMA bits invalid?
	jle s	I_SErr		;Yes?  Exit & display error message.
	mov	di,(Modes-@)	;Point to UltraDMA mode table.
	or	bh,bh		;Will disk do UltraDMA mode 0?
	jz s	I_SErr		;No?  Exit & display message!
I_NxtM:	cmp	bh,bl		;Will disk do next UltraDMA mode?
	jb s	I_GotM		;No, use current mode.
	inc	di		;Point to next mode table value.
	inc	di
	shl	bl,1		;More UltraDMA modes to check?
	jnz s	I_NxtM		;Yes, loop back.
I_GotM:	push	[di].lw		;Save disk mode value.
	mov	si,(DNEnd-@)	;Point to end of disk name.
I_NxtN:	cmp	si,(DName-@)	;Are we at the disk-name start?
	je s	I_Name		;Yes, disk name is all spaces!
	dec	si		;Decrement disk name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_NxtN		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma & space.
	inc	si
	inc	si
I_Name:	mov	[si].dwd,"-ATA"	;Set "ATA-" after disk name.
	add	si,4
	pop	ax		;Get disk "mode" value.
	mov	cl,00Fh		;Set "mode" value after disk name.
	and	cl,al
	call	I_HexA
	mov	[si].dwd,0240A0D2Eh  ;Set message terminators.
	mov	dx,(DNMsg-@)	     ;Display disk name and "mode".
	call	I_Msg
	clc			;Reset carry (no error) and exit.
I_ValX:	ret
;
; Subroutine to set the BIOS CHS data for a hard-disk or diskette.
;
I_CHSD:	push	ax		;Save unit's device-type flag.
	mov	ah,008h		;Get BIOS CHS data for this unit.
	call	I_In13
	jc s	I_CHSE		;If BIOS error, zero both values.
	and	cl,03Fh		;Get sectors/head value (low 6 bits).
	jz s	I_CHSE		;If zero, ERROR!  Zero both values.
	inc	dh		;Get heads/cylinder (BIOS value + 1).
	jnz s	I_CHST		;If non-zero, save data in our table.
I_CHSE:	xor	cl,cl		;Error!  Zero CHS data for this unit.
	xor	dh,dh
I_CHST:	mov	di,@LastU	;Point to "active units" table.
	mov	al,HDUnit	;Set BIOS unit number in our table.
	mov	[di+Units-@],al
	mov	[di+CHSec-@],cl ;Set unit's CHS data in our table.
	mov	[di+CHSHd-@],dh
	pop	ax		;Reload and set device-type flag.
	mov	[di+TypeF-@],al
	or	cl,cl		;Valid CHS values for this unit?
	jnz s	I_ValX		;Yes, go exit above.
	mov	dx,(CHSMsg-@)	;Display "CHS data error" and exit.
	jmp s	I_Msg
;
; Subroutines to issue initialization "external" calls.
;
I_XMS:	call	@XEntry		;Issue desired XMS request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_VDS:	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute desired VDS "lock"/"Unlock".
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In13:	mov	dl,HDUnit	;Set BIOS unit in DL-reg.
	int	013h		;Issue BIOS data interrupt.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_PCID:	push	bx		;Save PCI bus/device/function codes.
	push	si		;Save IDE address-table pointer.
	mov	al,00Ah		;Set "PCI doubleword" request code.
	call	I_In1A		;Get desired 32-bit word from PCI.
	pop	si		;Reload IDE address-table pointer.
	pop	bx		;Reload PCI bus/device/function.
	ret			;Exit.
I_In1A:	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	int	01Ah
	jmp s	I_IntX		;Restore driver settings, then exit.
I_Msg:	mov	ah,009h		;Get DOS "display string" opcode.
I_In21:	int	021h		;Issue desired DOS request.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In2F:	int	02Fh		;Issue XMS interrupt.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	ret			;Exit.
;
; Subroutine to convert a 16-bit number to ASCII for messages.
;
I_Hex:	mov	cx,4		;Set 4-digit display count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Set next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
	ret			;Exit.
CODE	ends
	end
