/* Partial implementation of the DOS MODE command by Eric Auer 2003 ..., */
/* for inclusion in MODECON (by Aitor Merino), which does the codepages! */
/* This code is primarily for Aitor, but you may use it as is under the  */
/* terms of the GNU GPL (v2) license if you want to, see www.gnu.org ... */
/* If you have questions, mail me: eric%coli.uni-sb.de (replace % by @). */

/* This file: control for COMx and LPTx things */

#include "mode.h"

static union REGS r;

/* ------------------------------------------------ */

/* handle printer port related commands */
/* show status if no argument or argument "/STA..." */
/* style 1: cols,lines,retry (80/132, 6/8, b/e/r/p/n) */
/* style 2: colr=... lines=... retry=... */
int printer(int pnum, char * what)
{
  unsigned int cols, lines, xretry;
  char retry;

#define lp(ch) r.h.ah=0; r.h.al=ch; r.x.dx=pnum; int86(0x17, &r, &r); \
  /* TODO: check returned r.h.ah status...? */

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    static char * parbits[] = { "timeout", "", "", "I/O-error",
      "selected", "out-of-paper", "acknowledge", "ready" /* not busy */
    }; /* parbits */
    int n;
    printf("*** PRINTER PORT %d STATUS ***\r\n", pnum+1);
    r.h.ah = 2;
    r.x.dx = pnum;
    if ( (r.h.ah & 0x30) == 0x30 ) {
      printf("Both out-of-paper and selected: no printer attached?\r\n");
    }
    printf("Port status: [ ");
    for (n=7; n>=0; n--) {
      if ( (r.h.ah & (1<<n)) != 0)
        printf("%s ", parbits[n]);
      /* a typical okay status is 0x90, ready, selected, no errors, no ack */
    }
    printf("]\r\n");
    return 0;
  }

  if (what[0] == '=') {
    what++;
    if (strncmp(what,"COM",3)) {
      printf("You can only redirect to serial ports (COMn).\r\n");
      return 1;
    }
    what+=3;
    if ((what[0] < '0') || (what[0] > '5')) {
      printf("You can only redirect to COM1-COM4. Alternatively, you can...\r\n");
      printf("- redirect to 'COM0' to stop redirection or\r\n");
      printf("- redirect to 'COM5' to redirect data to the bit bucket.\r\n");
      return 1;
    }
    return RedirectLptCom(pnum, what[0]-'1'); /* redirect LPTn to COMm */
  }

  if (strchr(what,'=') != NULL) {
    cols   = grabarg(what, "COLS=");
    lines  = grabarg(what, "LINES=");
    xretry  = grabarg(what, "RETRY=");
  } else {
    what = skipspace(what);
    if ( (what[0]!=',') && (!isdigit(what[0])) ) {
      printf("Syntax error in 'MODE LPTn ...', please check 'MODE /?'.\r\n");
      return 1;
    }
    cols   = posarg(what, 0);
    lines  = posarg(what, 1);
    xretry  = posarg(what, 2);
  }

  retry = (char)xretry;
  xretry = xlatretry(xretry);
  if (xretry<0) return 1; /* error */
  if (xretry>0) SetRetry(pnum, xretry); /* update resident part */
    /* to disable retry stuff you select "return error if busy" for now */
    /* xretry values: "if busy..." 0 busy 1 error 2 ignore 3 retry 4 noretry */

  if ( (cols!=0) && (cols!=80) && (cols!=132) ) {
    printf("Columns must be 80 or 132.\r\n");
    return 1; /* failed */
  }

  if ( (lines!=0) && (lines!=6) && (lines!=8) ) {
    printf("Lines per inch must be 6 or 8.\r\n");
    return 1; /* failed */
  }

  if ( (cols==0) && (lines==0) && (retry==0) ) {
    printf("Syntax error in 'MODE LPTn ...', please check 'MODE /?'.\r\n");
    return 1;
  }

#ifdef DMODE
  printf("MODE LPT%d cols=%d lines=%d retry=%c\r\n",
    pnum+1, cols, lines, retry ? retry : '-');
#endif

  if (lines!=0) {
    lp(27); /* ESC */
    if (lines==6) {
      lp('2'); /* ESC 2: 6lpi */
    } else {
      lp('0'); /* ESC 0: 8lpi */
    }
  } /* lpi selection */

#if 0
  /* TODO: what are the escape sequences for 10cpi / 15cpi ??? */
  if (cols!=0) {
    lp(27); /* ESC */
    if (cols==80) {
      lp('X'); /* ESC X: 10cpi, 80 columns */
    } else {
      lp('X'); /* ESC X: 15lpi, 132 columns */
    }
  } /* cpi selection */
#endif

  /* TODO: send RETRY setting to TSR / .SYS part */

#undef lp

  return 0;
} /* printer */

/* ------------------------------------------------ */

/* handle serial port related commands */
/* show status if no argument or argument "/STA..." */
/* style 1: baud,parity,data,stop,retry (..., o/e/n/s/m, 5..8, 1..2, ...) */
/* style 2: baud=... parity=... data=... stop=... retry=... */
int serial(int snum, char * what)
{
  unsigned int baud, xparity, data, stop, xretry;
  char parity, retry;

#ifdef DMODE
  static long int baudlist[] = { 110, 150, 300, 600,
    1200, 2400, 4800, 9600,   19200, 38400L, 57600L, 115200L
  }; /* baudlist */
#endif

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    int i;
    static char * serbits[] = {
      "delta-CTS", "delta-DSR", "RI-trail", "delta-CD",
      "CTS", "DSR", "RI", "CD",
      /* modem: cleartosend, datasetready, ringindicator, carrierdetect */
      "data-received", "overrun", "parity-error", "frame-error",
      "break-received", "xmit-hold-empty", "xmit-shift-empty", "timeout"
    }; /* serbits */
    printf("*** SERIAL PORT %d STATUS ***\r\n", snum+1);
    printf("SETTINGS unknown: Would have to read UART directly.\r\n");
    r.x.ax = 0x0300; /* read status */
    r.x.dx = snum;
    int86(0x14, &r, &r);
    printf("Port status: [ ");
    for (i=15; i>=0; i--) {
      if ( (r.x.ax & (1<<i)) != 0)
        printf("%s ", serbits[i]);
    }
    printf("]\r\n");
    return 0;
  }

  if (strchr(what,'=') != NULL) {
    baud   = grabarg(what, "BAUD=");
    xparity = grabarg(what, "PARITY=");
    data   = grabarg(what, "DATA=");
    stop   = grabarg(what, "STOP=");
    xretry  = grabarg(what, "RETRY=");
  } else {
    what = skipspace(what);
    if ( (what[0]!=',') && (!isdigit(what[0])) ) {
      printf("Syntax error in 'MODE COMn ...', please check 'MODE /?'.\r\n");
      return 1;
    }
    baud   = posarg(what, 0);
    xparity = posarg(what, 1);
    data   = posarg(what, 2);
    stop   = posarg(what, 3);
    xretry  = posarg(what, 4);
  }

  if ( (baud==0) && (xparity==0) && (data==0) && (stop==0) && (xretry==0) ) {
    printf("Syntax error in 'MODE COMn ...', please check 'MODE /?'.\r\n");
    return 1;
  }

  if ( (data>8) || ((data<5) && (data!=0)) ) {
    printf("Data bits must be 5, 6, 7 or 8.\r\n");
    return 1; /* failed */
  }

  if ( (stop>2) || (stop<0) ) {
    printf("Stop bits must be 1 or 2.\r\n");
    printf("(2 stopbits treated as 1.5 if 5 or 6 data bits).\r\n");
    return 1; /* failed */
  }

  retry = (char)xretry;
  xretry = xlatretry(retry);
  if (xretry<0) return 1; /* error */
  if (xretry>1) /* if busy: 0 busy 1 ignore 2 ready 3 keep trying 4 no retry */
    printf("Not yet supported RETRY value - ignored.\r\n");
    /* Reason: We do not hook int 14h at all (yet) ;-) ! */

  parity = (char)xparity;
  switch (parity) {
    case 'N':
      xparity = 0; /* none */
      break;
    case 'O':
      xparity = 1; /* odd */
      break;
    case 'E':
      xparity = 2; /* even (3 for old style API) */
      break;
    case 'S':
      xparity = 3; /* space? (new style API only) */
      break;
    case 'M':
      xparity = 4; /* mark? (new style API only) */
      break;
    case 0:
      xparity = 0; /* default parity setting: no parity */
      parity = '-';
      break;
    default:
      printf("Parity must be N, O, E, S or M (none, odd, even, space, mark).\r\n");
      return 1; /* failed */
  } /* switch */

  if (stop==0)
    stop = (baud==110) ? 2 : 1; /* default number of stop bits */

  if ((baud % 10) == 0) baud /= 10; /* strip at most 2 trailing zeroes */
  if ((baud % 10) == 0) baud /= 10; /* strip at most 2 trailing zeroes */

  switch (baud) {
    case 11:  /*   110 (caveat: 11 could also abbreviate 115200 (*)) */
      baud = 0;
      break;
    case 15:  /*   150 */
      baud = 1;
      break;
    case 3:   /*   300 (caveat: 3 could also abbreviate 38400 (*)) */
      baud = 2;
      break;
    case 6:   /*   600 (*) */
      baud = 3;
      break;
    case 12:  /*  1200 */
      baud = 4;
      break;
    case 0: /* default baud value is 2400 in most DOS versions */
    case 24:  /*  2400 */
      baud = 5;
      break;
    case 48:  /*  4800 */
      baud = 6;
      break;
    case 96:  /*  9600 */
      baud = 7;
      break;
    /* 14400 ??? */
    case 19:  /* 19200 */
    case 192:
      baud = 8; /* from here on we have to use the new API */
      break;
    /* 28800 ??? */
    case 38:  /* 38400 (supported?) */
    case 384:
      baud = 9;
      break;
    case 57:  /* 57600 (supported?) */
    case 576:
      baud = 10;
      break;
    case 115: /* 115200 (supported?) */
    case 1152:
    case 0xc200: /* 115200 & 0xffff */
      baud = 11;
      break;
    default:
      printf("Unsupported baud rate, sorry.\r\n");
      return 1;
  } /* switch */

  /* (*) MS MODE only allows full values or 2 or 3 digit abbreviations, */
  /*  while we also allow "omit trailing zeroes" style 1 digit abbrev.! */

  /* TODO: maybe better to program the UART directly, allows more rates? */
  /* (for example the Assembly language version of MODE does this) */

  if (data==0)
    data = 8; /* default number of data bits */

  r.x.dx = snum; /* port number */
  if ( (baud > 7) || (xparity > 2) ) { /* need new API? */
    r.x.ax = 0x0401; /* extended setup, no break */
    r.h.bh = xparity;
    r.h.bl = stop-1; /* 0 means 1, 1 means 2 (1.5 if 5 data bits) stop bits */
    r.h.ch = data-5; /* 0..3 means 5..8 data bits */
    r.h.cl = baud;   /* baud rate selector, values 0..8 same for all BIOSes */
  } else {
    if (xparity==2)
      xparity=3; /* translate to old style value */
    r.h.ah = 0; /* initialize port */
    r.h.al = (baud<<5) | (xparity<<3) | ((stop-1)<<1) | (data-5);
  }
  int86(0x14, &r, &r);
  /* returns status in AX */

#ifdef DMODE
  printf("MODE COM%d baud=%ld parity=%c data=%d stop=%d (ignored: retry=%c)\r\n",
    snum+1, baudlist[baud], parity, data, stop, retry ? retry : '-');
#endif

  /* TODO: send RETRY setting to TSR / .SYS part */

  return 0;
} /* serial */

