/* Partial implementation of the DOS MODE command by Eric Auer 2003 ..., */
/* for inclusion in MODECON (by Aitor Merino), which does the codepages! */
/* This code is primarily for Aitor, but you may use it as is under the  */
/* terms of the GNU GPL (v2) license if you want to, see www.gnu.org ... */
/* If you have questions, mail me: eric%coli.uni-sb.de (replace % by @). */

#include <stdlib.h> /* getenv, putenv, exit */
#include <string.h> /* strlen */
#include <stdio.h>  /* sprintf, printf */
#include <dos.h>    /* MK_FP, _psp, int86, union REGS */
#include <ctype.h>  /* toupper */

#define DMODE 1 /* define for debug mode */

static char * helptext[] = {
  "MODE LPTn[:] cols[,[lines][,retry]] (80/132 cols, 6/8 lpi)",
  "MODE LPTn[:] [COLS=...] [LINES=...] [RETRY=...]",
  "  (retry: b/e/r return busy/error/ready if busy, p/n infinite/no retry)",
  "MODE COMn[:] baud,parity,data,stop,retry (same optionality as LPT)",
  "MODE COMn[:] [BAUD=...] [PARITY=...] [DATA=...] [STOP=...] [RETRY=...]",
  "  (baud can be abbreviated to unique prefix, parity can be o/e/n/s/m,",
  "  the latter 2 meaning space/mark, data can be 5..8, stop 1..2)",
  "MODE [device] [/STA[TUS]] (show status of one or all devices)",
  "MODE LPTn[:]=COMn[:] (redirect printer data to serial port)",
  "MODE device [CP|CODEPAGE] REFRESH",
  "MODE device [CP|CODEPAGE] SELECT=...",
  "MODE device [CP|CODEPAGE] PREPARE=((...) filename)",
  "  (... can be one or more , separated numbers or empty strings)",
  "MODE [BW40|BW80|CO40|CO80|MONO][,rows] (rows can be 25, 43, 50)",
  "MODE CON[:] [COLS=...] [LINES=...] (cols can be 40 or 80)",
  "MODE CON[:] [RATE=...] [DELAY=...] (default rate 20, default delay 1)",
  "  (rate: 1..32 for 2..30 char/sec, delay: 1..4 for 1/4sec..4/4sec)",
  "** Retry-parameters and redirection need a TSR or separate .sys driver",
  "** (I prefer the latter) which MODE would control. CGA horizontal shift",
  "** not planned, other special features not planned either...)",
  NULL
}; /* helptext */

/* TODO: improve help screen (e.g. look at the FreeDOS MODE(.asm) one). */

/* TODO: use KITTEN i18n library and tiny prf.c printf implementation! */

static volatile unsigned char far * biosdatap = (unsigned char far *)MK_FP(0x40, 0);
#define VIDMAXROW biosdatap[0x84]

static union REGS r;

/* ------------------------------------------------ */

/* skip over whitespace, return NULL if hitting EOF */
char * skipspace(char * str)
{
  if (str == NULL)
    return str; /* the NULL string is empty, too */

  while (!isgraph(str[0])) { /* skip over leading whitespace */
    if ((str[0] == 13) || (str[0] == 0)) { /* nothing but whitespace? */
      return NULL;
    }
    str++;
  }
  return str;
} /* skipspace */

/* ------------------------------------------------ */

/* find a numerical or one-char value after the label string */
unsigned int grabarg(char * str, const char * label)
{
  str = strstr(str, label);
  if (str == NULL)
    return 0; /* failed */
  str = skipspace(str+strlen(label));
  if (isdigit(str[0])) {
    return (int)(0xffff & atol(str)); /* numerical argument */
  } else {
    return (int)str[0]; /* single char argument */
  }
} /* grabarg */

/* ------------------------------------------------ */

/* find a numerical or one-char value after the nth comma */
unsigned int posarg(char * str, int commas)
{
  char * termp;
  char term;
  unsigned int value;

  while (commas>0) {
    str = strchr(str, ',');
    if (str == NULL)
      return 0; /* failed */
    str++; /* advance past the comma */
    commas--;
  }
  str = skipspace(str);

  if (str==NULL) { /* string ended at the comma that would start the value */
    return 0;
  }
  
  termp = str;
  while ( (termp[0]!=0) && (termp[0]!=' ') && (termp[0]!=',') ) {
    termp++; /* find END of argument (next space or comma) */
  }
  term = termp[0]; /* save char */
  termp[0] = 0; /* temporarily terminate string here */

  if (isdigit(str[0])) {
    value = (int)(0xffff & atol(str)); /* numerical argument */
  } else {
    value = (int)str[0]; /* single char argument */
  }
  
  /* printf("[%s] -> %u\r\n", str, value); */
  termp[0] = term; /* restore string */
  return value;
} /* posarg */

/* ------------------------------------------------ */

/* translate char retry B/E/R/P/N into value 0..4 or -1 */
int xlatretry(char retry)
{
  static char * retryname[] = {
    "report B_USY", "report E_RROR", "report R_EADY",
    "INFINITE retry (P)", "N_O retry"
  }; /* retryname */
  int i = -1;

  switch (retry) {
    case 0: /* default: return BUSY when printer is BUSY */
      return 0;
    case 'B':
      i = 0;
      break;
    case 'E':
      i = 1;
      break;
    case 'R':
      i = 2;
      break;
    case 'P':
      i = 3;
      break;
    case 'N':
      i = 4;
      break;
  } /* switch */

#ifdef DMODE
  if (i>=0) printf("User selected reaction on busy port: %s\r\n",
    retryname[i]);
#endif

  return i;
} /* xlatretry */

/* ------------------------------------------------ */

/* show the help screen */
void help(void)
{
  int i;
  for (i=0; helptext[i]!=NULL; i++) {
    printf("%s\r\n", helptext[i]);
  }
  return;
} /* help */

/* ------------------------------------------------ */

/* handle printer port related commands */
/* show status if no argument or argument "/STA..." */
/* style 1: cols,lines,retry (80/132, 6/8, b/e/r/p/n) */
/* style 2: colr=... lines=... retry=... */
int printer(int pnum, char * what)
{
  unsigned int cols, lines, xretry;
  char retry;

#define lp(ch) r.h.ah=0; r.h.al=ch; r.x.dx=pnum; int86(0x17, &r, &r); \
  /* TODO: check returned r.h.ah status...? */

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    static char * parbits[] = { "timeout", "", "", "I/O-error",
      "selected", "out-of-paper", "acknowledge", "ready" /* not busy */
    }; /* parbits */
    int n;
    printf("*** PRINTER PORT %d STATUS ***\r\n", pnum+1);
    r.h.ah = 2;
    r.x.dx = pnum;
    if ( (r.h.ah & 0x30) == 0x30 ) {
      printf("Both out-of-paper and selected: no printer attached?\r\n");
    }
    printf("Port status: [ ");
    for (n=7; n>=0; n--) {
      if ( (r.h.ah & (1<<n)) != 0)
        printf("%s ", parbits[n]);
      /* a typical okay status is 0x90, ready, selected, no errors, no ack */
    }
    printf("]\r\n");
    return 0;
  }

  if (what[0] == '=') {
    what++;
    printf("Redirection of LPT%d to %s not yet possible!\r\n",
      pnum+1, what);
    return 1;
    /* TODO: communicate with .sys or TSR part to allow redirection! */
  }

  if (strchr(what,'=') != NULL) {
    cols   = grabarg(what, "COLS=");
    lines  = grabarg(what, "LINES=");
    xretry  = grabarg(what, "RETRY=");
  } else {
    what = skipspace(what);
    if ( (what[0]!=',') && (!isdigit(what[0])) ) {
      printf("Syntax error in 'MODE LPTn ...', please check 'MODE /?'.\r\n");
      return 1;
    }
    cols   = posarg(what, 0);
    lines  = posarg(what, 1);
    xretry  = posarg(what, 2);
  }

  retry = (char)xretry;
  xretry = xlatretry(xretry);
  if (xretry<0) return 1; /* error */

  if ( (cols!=0) && (cols!=80) && (cols!=132) ) {
    printf("Columns must be 80 or 132.\r\n");
    return 1; /* failed */
  }

  if ( (lines!=0) && (lines!=6) && (lines!=8) ) {
    printf("Lines per inch must be 6 or 8.\r\n");
    return 1; /* failed */
  }

  if ( (cols==0) && (lines==0) && (retry==0) ) {
    printf("Syntax error in 'MODE LPTn ...', please check 'MODE /?'.\r\n");
    return 1;
  }

#ifdef DMODE
  printf("MODE LPT%d cols=%d lines=%d retry=%c\r\n",
    pnum+1, cols, lines, retry ? retry : '-');
#endif

  if (lines!=0) {
    lp(27); /* ESC */
    if (lines==6) {
      lp('2'); /* ESC 2: 6lpi */
    } else {
      lp('0'); /* ESC 0: 8lpi */
    }
  } /* lpi selection */

#if 0
  /* TODO: what are the escape sequences for 10cpi / 15cpi ??? */
  if (cols!=0) {
    lp(27); /* ESC */
    if (cols==80) {
      lp('X'); /* ESC X: 10cpi, 80 columns */
    } else {
      lp('X'); /* ESC X: 15lpi, 132 columns */
    }
  } /* cpi selection */
#endif

  /* TODO: send RETRY setting to TSR / .SYS part */

#undef lp

  return 0;
} /* printer */

/* ------------------------------------------------ */

/* handle serial port related commands */
/* show status if no argument or argument "/STA..." */
/* style 1: baud,parity,data,stop,retry (..., o/e/n/s/m, 5..8, 1..2, ...) */
/* style 2: baud=... parity=... data=... stop=... retry=... */
int serial(int snum, char * what)
{
  unsigned int baud, xparity, data, stop, xretry;
  char parity, retry;

#ifdef DMODE
  static long int baudlist[] = { 110, 150, 300, 600,
    1200, 2400, 4800, 9600,   19200, 38400L, 57600L, 115200L
  }; /* baudlist */
#endif

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    int i;
    static char * serbits[] = {
      "delta-CTS", "delta-DSR", "RI-trail", "delta-CD",
      "CTS", "DSR", "RI", "CD",
      /* modem: cleartosend, datasetready, ringindicator, carrierdetect */
      "data-received", "overrun", "parity-error", "frame-error",
      "break-received", "xmit-hold-empty", "xmit-shift-empty", "timeout"
    }; /* serbits */
    printf("*** SERIAL PORT %d STATUS ***\r\n", snum+1);
    printf("SETTINGS unknown: Would have to read UART directly.\r\n");
    r.x.ax = 0x0300; /* read status */
    r.x.dx = snum;
    int86(0x14, &r, &r);
    printf("Port status: [ ");
    for (i=15; i>=0; i--) {
      if ( (r.x.ax & (1<<i)) != 0)
        printf("%s ", serbits[i]);
    }
    printf("]\r\n");
    return 0;
  }

  if (strchr(what,'=') != NULL) {
    baud   = grabarg(what, "BAUD=");
    xparity = grabarg(what, "PARITY=");
    data   = grabarg(what, "DATA=");
    stop   = grabarg(what, "STOP=");
    xretry  = grabarg(what, "RETRY=");
  } else {
    what = skipspace(what);
    if ( (what[0]!=',') && (!isdigit(what[0])) ) {
      printf("Syntax error in 'MODE COMn ...', please check 'MODE /?'.\r\n");
      return 1;
    }
    baud   = posarg(what, 0);
    xparity = posarg(what, 1);
    data   = posarg(what, 2);
    stop   = posarg(what, 3);
    xretry  = posarg(what, 4);
  }

  if ( (baud==0) && (xparity==0) && (data==0) && (stop==0) && (xretry==0) ) {
    printf("Syntax error in 'MODE COMn ...', please check 'MODE /?'.\r\n");
    return 1;
  }

  if ( (data>8) || ((data<5) && (data!=0)) ) {
    printf("Data bits must be 5, 6, 7 or 8.\r\n");
    return 1; /* failed */
  }

  if ( (stop>2) || (stop<0) ) {
    printf("Stop bits must be 1 or 2.\r\n");
    printf("(2 stopbits treated as 1.5 if 5 or 6 data bits).\r\n");
    return 1; /* failed */
  }

  retry = (char)xretry;
  xretry = xlatretry(retry);
  if (xretry<0) return 1; /* error */

  parity = (char)xparity;
  switch (parity) {
    case 'N':
      xparity = 0; /* none */
      break;
    case 'O':
      xparity = 1; /* odd */
      break;
    case 'E':
      xparity = 2; /* even (3 for old style API) */
      break;
    case 'S':
      xparity = 3; /* space? (new style API only) */
      break;
    case 'M':
      xparity = 4; /* mark? (new style API only) */
      break;
    case 0:
      xparity = 0; /* default parity setting: no parity */
      parity = '-';
      break;
    default:
      printf("Parity must be N, O, E, S or M (none, odd, even, space, mark).\r\n");
      return 1; /* failed */
  } /* switch */

  if (stop==0)
    stop = (baud==110) ? 2 : 1; /* default number of stop bits */

  if ((baud % 10) == 0) baud /= 10; /* strip at most 2 trailing zeroes */
  if ((baud % 10) == 0) baud /= 10; /* strip at most 2 trailing zeroes */

  switch (baud) {
    case 11:  /*   110 (caveat: 11 could also abbreviate 115200 (*)) */
      baud = 0;
      break;
    case 15:  /*   150 */
      baud = 1;
      break;
    case 3:   /*   300 (caveat: 3 could also abbreviate 38400 (*)) */
      baud = 2;
      break;
    case 6:   /*   600 (*) */
      baud = 3;
      break;
    case 12:  /*  1200 */
      baud = 4;
      break;
    case 0: /* default baud value is 2400 in most DOS versions */
    case 24:  /*  2400 */
      baud = 5;
      break;
    case 48:  /*  4800 */
      baud = 6;
      break;
    case 96:  /*  9600 */
      baud = 7;
      break;
    /* 14400 ??? */
    case 19:  /* 19200 */
    case 192:
      baud = 8; /* from here on we have to use the new API */
      break;
    /* 28800 ??? */
    case 38:  /* 38400 (supported?) */
    case 384:
      baud = 9;
      break;
    case 57:  /* 57600 (supported?) */
    case 576:
      baud = 10;
      break;
    case 115: /* 115200 (supported?) */
    case 1152:
    case 0xc200: /* 115200 & 0xffff */
      baud = 11;
      break;
    default:
      printf("Unsupported baud rate, sorry.\r\n");
      return 1;
  } /* switch */

  /* (*) MS MODE only allows full values or 2 or 3 digit abbreviations, */
  /*  while we also allow "omit trailing zeroes" style 1 digit abbrev.! */

  /* TODO: maybe better to program the UART directly, allows more rates? */
  /* (for example the Assembly language version of MODE does this) */

  if (data==0)
    data = 8; /* default number of data bits */

  r.x.dx = snum; /* port number */
  if ( (baud > 7) || (xparity > 2) ) { /* need new API? */
    r.x.ax = 0x0401; /* extended setup, no break */
    r.h.bh = xparity;
    r.h.bl = stop-1; /* 0 means 1, 1 means 2 (1.5 if 5 data bits) stop bits */
    r.h.ch = data-5; /* 0..3 means 5..8 data bits */
    r.h.cl = baud;   /* baud rate selector, values 0..8 same for all BIOSes */
  } else {
    if (xparity==2)
      xparity=3; /* translate to old style value */
    r.h.ah = 0; /* initialize port */
    r.h.al = (baud<<5) | (xparity<<3) | ((stop-1)<<1) | (data-5);
  }
  int86(0x14, &r, &r);
  /* returns status in AX */

#ifdef DMODE
  printf("MODE COM%d baud=%ld parity=%c data=%d stop=%d retry=%c\r\n",
    snum+1, baudlist[baud], parity, data, stop, retry ? retry : '-');
#endif

  /* TODO: send RETRY setting to TSR / .SYS part */

  return 0;
} /* serial */

/* ------------------------------------------------ */

/* set number of screen lines */
int set_lines(int lines)
{
  int havevga = 0;
  int oldlines = VIDMAXROW+1; /* only valid for EGa or better */

  r.h.ah = 0x12; /* ah=0x12, bl=0x10: get EGA info */
  r.x.bx = 0xff10; /* init bh to -1 */
  r.x.cx = 0xffff; /* init cx to -1 */
  int86(0x10, &r, &r);
  if ( (r.x.cx == 0xffff) || (r.h.bh == 0xff) ) {
    if (lines == 25) {
#ifdef DMODE
      printf("Not EGA or better, assuming 25 lines.\r\n");
#endif
      return 0; /* 25 lines is tautologic for less-than-EGA */
    }
    printf("Only EGA or better allows more than 25 lines!\r\n");
    return 1;
  }

  r.x.ax = 0x1a00; /* get VGA display combination - VGA install check */
  int86(0x10, &r, &r);
  /* returned active / alternate display in BL / BH (use al=1 to SET...) */
  /* 0 none, 1 mono, 2 CGA, 4..5 EGA 6 PGA 7..8 VGA a..c MCGA */
  if (r.h.al == 0x1a)
    havevga = 1; /* VGA found */

  if ( (lines == 50) && (havevga == 0) ) { /* 50 lines requires VGA */
    printf("Only VGA or better allows more than 43 lines!\r\n");
    return 1;
  }

  if (oldlines == lines) {
#ifdef DMODE
    printf("Number of lines already as requested.\r\n");
#endif
    return 0; /* no action needed */
  }

  /* only reached for EGA, VGA or better */
  switch (lines) {
    case 25:
#if 0
      r.h.ah = 0x0f; /* get current video mode */
      int86(0x10, &r, &r);
      r.h.ah = 0;
      /* AL: simply re-activate current mode */
      if (r.h.al > 3)
        r.h.al = 3; /* better use only known modes like 80x25 color */
      int86(0x10, &r, &r);
#endif
      r.x.ax = (havevga==1) ? 0x1114 : 0x1111;
      /* activate 8x16 / 8x14 VGA / EGA (or MONO) default font */
      /* use AL 1x for full mode reset, 0x for font load only */
      r.h.bl = 0; /* font bank 0 */
      int86(0x10, &r, &r);
      break;
    case 43:
    case 50:
      if (havevga==1) { /* we need extra 43 <-> 50 line switching in VGA */
        r.h.ah = 0x12; /* set resolution (with BL 0x30) */
        r.h.bl = 0x30;
        r.h.al = (lines==43) ? 1 : 2; /* 1 means 350 pixels rows, 2 400 */
        int86(0x10, &r, &r);
        if (r.h.al==0x12) { /* did the call succeed? */
          r.h.ah = 0x0f; /* get current video mode */
          int86(0x10, &r, &r);
          r.h.ah = 0; /* set mode again, to let resolution change take effect! */
          r.h.al |= 0x80; /* mode |= flag "do not clear screen" */
          int86(0x10, &r, &r);
#ifdef DMODE
          printf("Selecting vertical resolution %d on VGA.\r\n",
            (lines==43) ? 350 : 400);
#endif
        } else {
          printf("Could not select ???x%d pixel resolution, using default.\r\n",
            (lines==43) ? 350 : 400);
        }
      }
      r.x.ax = 0x1112; /* activate 8x8 default font */
      /* use AL 1x for full mode reset, 0x for font load only */
      r.h.bl = 0; /* font bank 0 */
      int86(0x10, &r, &r);
  } /* switch */

  oldlines = lines; /* desired value */
  lines = VIDMAXROW+1; /* value from BIOS */
  if (lines < 25)
    lines = 25;
  if (lines!=oldlines) {
    printf("Could only set %d rows, not %d.\r\n", lines, oldlines);
  }

  return 0; /* success */
} /* set_lines */

/* ------------------------------------------------ */

/* handle console related commands */
/* show status if no argument or argument "/STA..." */
/* chains to CODEPAGE code if first argument "CP" or "CODEPAGE" */
/* otherwise, keyboard repeat OR screen size can be set: */
/* style 1: cols=... lines=... (40/80 and 25/43/50) */
/* style 2: rate=... delay=... (1..32 and 1..4) */
int console(char * what)
{
  unsigned int cols, lines, rate, delay, i;

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    unsigned int j;
    printf("*** CONSOLE STATUS ***\r\n");
    r.x.ax = 0x0306; /* get keyboard repeat settings */
    r.x.bx = 0xffff; /* to notice if function not possible (common!) */
    int86(0x16, &r, &r);
    if (r.x.bx == 0xffff) {
      printf("Keyboard repeat rate and delay could not be read.\r\n");
    } else {
      i = r.h.bh;
      j = r.h.bl;
      printf("Keyboard repeat after %u msec, rate %u (0=2/s...32=30/s).\r\n",
        (i+1)*250, 32-j);
    }
    lines = VIDMAXROW+1;
    if (lines < 25) /* CGA/Mono does not set this value */
      lines = 25;
    r.h.ah = 0x0f; /* get current video mode */
    int86(0x10, &r, &r);
    cols = r.h.ah;
    i = r.h.al & 0x7f; /* current video mode, without "no clear" flag. */
    printf("Screen size: %d columns, %d rows, mode %d.\r\n",
      cols, lines, i);
    return 0;
  }

  if ( !strncmp(what, "CP", 2) ) {
#ifdef DMODE
    printf("MODE CON CODEPAGE '%s'\r\n", skipspace(what+2));
#endif
    return 0;
  }

  if ( !strncmp(what, "CODEPAGE", 8) ) {
#ifdef DMODE
    printf("MODE CON CODEPAGE '%s'\r\n", skipspace(what+8));
#endif
    return 0;
  }

  cols  = grabarg(what, "COLS=");
  lines = grabarg(what, "LINES=");
  rate  = grabarg(what, "RATE=");
  delay = grabarg(what, "DELAY=");

  if ( (cols!=0) && (cols!=40) && (cols!=80) ) {
    printf("Columns must be 40 or 80.\r\n");
    return 1; /* failed */
  }

  if ( (lines!=0) && (lines!=25) && (lines!=43) && (lines!=50) ) {
    printf("Lines must be 25, 43 or 50.\r\n");
    return 1; /* failed */
  }

  if ( (rate<0) || (rate>32) ) {
    printf("Repeat rate code must be between 1 and 32.\r\n");
    return 1; /* failed */
  }

  if ( (delay<0) || (delay>4) ) {
    printf("Repeat delay must be between 1 and 4 (unit is 0.25 seconds).\r\n");
    return 1; /* failed */
  }

  if ( (delay==0) && (rate==0) && (cols==0) && (lines==0) ) {
    printf("Syntax error in 'MODE CON ...', please check 'MODE /?'.\r\n");
    return 1;
  }

#ifdef DMODE
  printf("MODE CON cols=%d lines=%d rate=%d delay=%d\r\n",
    cols, lines, rate, delay);
#endif

  if ( (rate!=0) || (delay!=0) ) {
    if (rate>0) {
      rate = 32-rate; /* turn 1..32 into 31..0 */
    } else {
      rate = 12; /* default: 20 chars per second */
    }
    if (delay==0)
      delay = 2; /* default: 2/4 seconds */
    r.x.ax = 0x0305; /* set keyboard repeat */
    r.h.bh = delay-1; /* unit 250ms, value 0 means 250ms */
    r.h.bl = rate; /* 0 is 30/sec, default 12 is 10/sec, 31 is 2/sec */
    int86(0x16, &r, &r);
  }

  if ( (cols!=0) || (lines!=0) ) {
    r.h.ah = 0x0f; /* get video mode */
    int86(0x10, &r, &r);
    if (cols != r.h.ah) { /* current column count not as desired? */
      i = r.h.al & 0x7f;  /* current video mode */
      r.h.al &= 0x80;
      if ( (i < 2) && (cols==80) ) { /* 40x... mode but 80x... requested? */
        i = (i==1) ? 3 : 2; /* mode 2/3 work on MONO, too. */
      }
      if ( (i > 1) && (cols==40) ) { /* 80x... mode but 40x... requested? */
        i = ( (i==2) || (i==7) ) ? 0 : 1; /* 0 if mode was 2 or 7, 1 else */
      }
      r.h.ah = 0;  /* set video mode */
      r.h.al |= i; /* the new mode */
      int86(0x10, &r, &r);
    } /* mode change needed for column selection */

    return set_lines(lines); /* SET NUMBER OF TEXT ROWS */
  }

  return 0;
} /* console */

/* ------------------------------------------------ */

/* find out whether a CRTC with editable cursor register exists at port. */
/* CGA/MDA/HGC 6845 has registers 0..0x11 (only 0x0e..0x0f readable)     */
/* 0x0e/0x0f: cursor location (high, low!) (12bit in 6845)               */
/* EGA: 0x0c..0x0f readable, VGA: 0..7 write-protectable, regs 0..0x18   */
int testCRTC(unsigned int port)
{
  unsigned char one, other;
  disable(); /* block IRQs during test */
  outportb(port,0x0f);     /* 0x0f is cursor position register, lower byte */
  one = inportb(port+1);   /* current value */
  outportb(port,0x0f);
  other = one ^ 1;
  outportb(port+1,other);  /* a modified value */
  outportb(port,0x0f);
  other = inportb(port+1); /* did new value get stored? */
  outportb(port,0x0f);
  outportb(port+1,one);    /* restore value */
  enable();  /* enable IRQs again */
  return (one != other);   /* if we could modify the value, success! */
} /* testCRTC */

/* ------------------------------------------------ */

/* handle video mode related commands */
/* optional argument: ",rows" (25/43/50) */
int screen(int mode, char * what)
{
  static volatile unsigned char far * sysinfop = (char far *)MK_FP(0x40, 0x10);
  unsigned int lines = 0;
  /* unsigned int flags = SYSINFO & 0x30; ... */
  unsigned int havevga = 0;

  if (mode==4) /* not 0..3? */
    mode=7; /* adjust MONO mode number */

  if (mode == 7) {
    if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
      sysinfop[0] |= 0x30; /* force MONO mode (if mono CARD installed!) */
      /* 0x00: card-with-BIOS, 0x10 CGA40, 0x20 CGA80, 0x30 MONO */
      /* the sysinfop trick is needed to switch if TWO cards exist... */
    } /* dual monitor */
    /* other special MONO preparations needed if single monitor? */
  } else {
    if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
      r.x.ax = 0x1a00; /* get VGA display combination - VGA install check */
      int86(0x10, &r, &r);
      /* returned active / alternate display in BL / BH (use al=1 to SET...) */
      /* 0 none, 1 mono, 2 CGA, 4..5 EGA 6 PGA 7..8 VGA a..c MCGA */
      if (r.h.al == 0x1a)
        havevga = 1; /* VGA found */
      sysinfop[0] = (sysinfop[0] & ~0x30) |
        ( (havevga==1) ? 0 : 0x20 ); /* switch to EGA/VGA BIOS or CGA80. */
        /* EGA seems to work fine in CGA80 mode - we only check for VGA. */
    } /* dual monitor */
    /* other special COLOR preparations needed if single monitor? */
  }

  /* how do we detect cases where ONLY mono or ONLY color works? */
  /* single monitor VGA can be in either mode but still can switch! */

  r.h.ah = 0; /* set mode */
  r.h.al = mode;
  int86(0x10, &r, &r);

#ifdef DMODE
  printf("MODE: screen mode %d, lines %d\r\n", mode, lines);
  if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
    printf("Dual monitor system detected (MONO plus CGA or newer).\r\n");
  } /* dual monitor */
#endif

  if ((what!=NULL) && (what[0]==',')) {
    what++;
    lines = atoi(what);
    if ( (lines!=25) && (lines!=43)  && (lines!=50) ) {
        printf("Rows must be 25, 43 or 80.\r\n");
        return 1;
    }

    return set_lines(lines); /* SET NUMBER OF TEXT ROWS */
  } /* argument given */

  return 0;
} /* screen */

/* ------------------------------------------------ */

/* find main keyword and jump to matching subroutine(s) */
int main(void) /* (int argc, char ** argv) */
{
  static char * topics[] = { "/?", "LPT", "COM", "CON",
    "BW40", "CO40", "BW80", "CO80", "MONO", NULL
  }; /* topic */
  int i;
  char far * pspargs = (char far *)MK_FP(_psp, 0x81);
  char argstr[128];
  char * args;
  int far * serports = (int far *)MK_FP(0x40, 0);
  int far * parports = (int far *)MK_FP(0x40, 8);
  /* useful: n = atol(string) */

  for (i=0; (pspargs[i]!=0) && (pspargs[i]!=13); i++) { /* until zero/CR */
    argstr[i] = toupper(pspargs[i]); /* convert to upper case */
    if ((argstr[i] == 9) || (argstr[i] == ':'))
      argstr[i]=' '; /* convert tab and colon to space */
  }
  argstr[i] = 0; /* use zero to terminate string */

  args = skipspace(argstr);
  if ( (args == NULL) || (!strncmp(args, "/STA", 4)) ) {
      int n=0;
      for (i=0; i<4; i++) {
        if (serports[i] != 0)
	  n |= serial(i, NULL); /* SERIAL PORT INFORMATION */

      }
      for (i=0; i<3; i++) {
        if (parports[i] != 0)
	  n |= printer(i, NULL); /* PRINTER PORT INFORMATION */

      }
      n |= console(NULL); /* CONSOLE INFORMATION */
      exit(n); /* success only if all information requests succeeded */
  }

  for (i=0; topics[i]!=NULL; i++) { /* find topic selecting first word */
    if ( !strncmp(args, topics[i], strlen(topics[i])) ) {
      int n;
#ifdef DMODE
      printf("Topic: %s\r\n",topics[i]);
#endif
      for (n=0; n < (int)strlen(topics[i]); n++) {
	args++; /* skip over keyword */
      }

      switch (i) {
        case 0: /* /? */
	  help(); /* SHOW HELP SCREEN */
	  i=0; /* return success */
          break;
        case 1: /* LPTx */
	  if ((args[0]<'1') || (args[0]>'3')) {
	    printf("Invalid printer port number %c\r\n", args[0]);
	    exit(1);
          }
	  i = args[0] - '1';
          args++;
	  if (parports[i] == 0) {
	    printf("LPT%d not existing in this PC.\r\n", i+1);
	    exit(1);
	  }
	  i = printer(i, skipspace(args)); /* PRINTER PORT COMMAND */
          break;
        case 2: /* COMx */
	  if ((args[0]<'1') || (args[0]>'4')) {
	    printf("Invalid serial port number %c\r\n", args[0]);
            exit(1);
          }
	  i = args[0] - '1';
	  args++;
	  if (serports[i] == 0) {
	    printf("COM%d not existing in this PC.\r\n", i+1);
	    exit(1);
	  }
	  i = serial(i, skipspace(args)); /* SERIAL PORT COMMAND */
          break;
        case 3: /* CON */
	  i = console(skipspace(args)); /* CONSOLE COMMAND */
          break;
        case 4: /* BW40 */
        case 5: /* C040 */
        case 6: /* BW80 */
        case 7: /* CO80 */
        case 8: /* MONO */
	  i = screen(i-4, skipspace(args)); /* VIDEO MODE COMMAND */
          /* 0..3 for modes 0..3, 4 for mode mono */
          break;
#ifdef DMODE
        default:
	  printf("Internal error %d\r\n", i);
#endif
      } /* case */
      exit(i); /* errorlevel from topic handler */
    } /* found topic */
  }

  printf("Unknown device or syntax error! 'MODE /?' shows help.\r\n");
  exit(1);
  return 1;

} /* main */

