{===========================================================================
||  FreeDOS Keyb  2.0                                                     ||
||  (Prototype 4 - 22.August.2003)                                        ||
|| ---------------------------------------------------------------------- ||
||  Open Source keyboard driver for DOS                                   ||
||  License:       GNU-GPL 2.0  (see copying.txt)                         ||
||  Platform:      DOS v.1.0 or later                                     ||
||                 PC XT/AT with keyboard interface at int9h/int15h       ||
||  Compiler:      Borland Turbo Pascal 7.0                               ||
|| ---------------------------------------------------------------------- ||
|| (C) Aitor SANTAMARIA MERINO                                            ||
|| (aitor.sm@wanadoo.es)                                                  ||
|| ---------------------------------------------------------------------- ||
|| Contributions:                                                         ||
||   - Dietmar  HOEHMANN   (XStrgins, some non resident code)             ||
||   - Matthias PAUL       (reboot and flush cache code, general help)    ||
||   - Axel     FRINKE     (general help)                                 ||
===========================================================================}



{$A-,B-,D-,E-,F-,I-,L-,N-,O-,R-,S-,V-,Y-}
{$M $2000,0,0}

PROGRAM Keyb;

USES DOS;


CONST
        {Version constants}
        VerS          = '2.0 (Prototype4)';
        Version       = $0100 + 94;

        E0Prefixed    = 2;   {flags in KStatusByte1}

        { Installable KEYB functions}
        KF_ExtInt9    = 1;
        KF_StoreKey   = 2;    {REMOVABLE!!}
        KF_APMFunc    = 3;
        KF_DoCommand  = 4;
        KF_Translate  = 5;
        KF_ExtCommands= 6;
        KF_Combine    = 7;
        KF_ExecCombi  = 8;

        { APMProc function }
        APM_FlushCache = 0;
        APM_WarmReboot = 1;
        APM_ColdReboot = 2;
        APM_PowerOff   = 3;
        APM_Suspend    = 4;

{$I KEYBMSG.NLS}   { message string constants }


TYPE
        pbyte=^byte;
        SimpleProc     = procedure;     { parameter-less callable function }
        PtrRec         = record         { pointer record }
                            Ofs,Seg : Word;
                         End;

{=== BIOS Data segment variables ===}
VAR
       KStatusFlags1     : Byte absolute 0:$417;  { Listing active keys }
                                            { 0 : Right SHIFT pressed}
                                            { 1 : Left SHIFT  pressed}
                                            { 2 : CTRL  pressed}
                                            { 3 : ALT  pressed}
                                            { 4 : SCROLL LOCK active }
                                            { 5 : NUM LOCK active }
                                            { 6 : CAPS LOCK  active}
                                            { 7 : INSERT active}
       KStatusFlags2     : Byte absolute 0:$418;  { Currently pressed modifier keys. }
       AltInput          : Byte absolute 0:$419;  { Char entered by Alt and numberkeys. }
       KStatusByte1      : Byte absolute 0:$496;  { Status of right ALT and Control. }
       KStatusByte2      : Byte absolute 0:$497;  { Status of LEDs. }
       BufferStart       : Word absolute 0:$480;  { Start address of key buffer. }
       BufferEnd         : Word absolute 0:$482;  { End address of key buffer. }
       BufferHead        : Word absolute 0:$41C;  { Next free char in buffer. }
       BufferTail        : Word absolute 0:$41A;  { Next char to read. }


    {------- KEYB2 P4-style layout information ---------------------------}
    {------- To be REPLACED in prototype 5  ------------------------------}
    Type
         KCBuffer = array[0..32767] of byte;    {max: 32Kb}
         PKCBuffer = ^KCBuffer;
    
    {------- END ---------------------------------------------------------}

CONST
        {======= KEYB GLOBAL DATA SEGMENT (later accessed through CS) ====}

       { Old interrupt vectors }
       OldInt9h       : Pointer = NIL;         { Chain for int9h. }
       OldInt2Fh      : Pointer = NIL;         { Chain for multiplexer interrupt. }
       OldInt16h      : Pointer = NIL;         { Chain for Int 16h. }
       OldInt15h      : Pointer = NIL;         { Chain for Int 15h. }

       {Layout pointers}
       CurrentLayout  : PKCBuffer = NIL;  { Pointer to the P4 layout block }

       {Other global variables}
       PIsActive      : Pointer = NIL; { Pointer to a byte, 0:inactive, <>0: active }
       CombStat       : Word = 0;      { Status of char combination. 0=No combination signs in work. }
                                        { Else pointer to combination table. }
                                        { will probably be integrated into COMBI procedure }
       DecimalChar    : char ='.';
       VIsActive      : byte = 1;      { we are active }
       NewFileName    : NameStr = '';

       { Installable functions vectors }
       { NOTE: if ExtInt9Proc is MOVED to other position in the table, you
         should also modify CallKeybFunc }
       ExtInt9Proc : SimpleProc   = NIL;    { Int9h management extensions }
       BIOSStoreKey: SimpleProc   = NIL;    { procedure EXCLUSIVE to StoreKey }
       APMProcedure: SimpleProc   = NIL;    { APM (power) procedure       }
       DoCommands  : SimpleProc   = NIL;    { perform commands 100..199   }
       TranslateK  : SimpleProc   = NIL;    { translate scode in Fn, ^cursor }
       ExtCommands : SimpleProc   = NIL;    { extended commands }
       CombineProc : SimpleProc   = NIL;    { combi pending, Combine }
       ExecCombi   : SimpleProc   = NIL;    { execute a COMBI sequence }

       { End of the KEYB Global data segment }
       EoDS        : Byte = 0;


{------------------------------------------------------------------------
---- KEYB 2 CORE ROUTINES (non-discardable)                         -----
------------------------------------------------------------------------}


{*** SPACE for the KEYB Global Data Segment, accessible through CS ***}
Procedure GlobalDS;
Begin
   ASM
   DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
   End;
End;


{************************************************************************
** CallKEYBFunc:   calls a KEYB installable function                   **
** ------------------------------------------------------------------- **
** IN:   CX: Function code (see constants above)                       **
**       CS: KEYB Global data segment                                  **
** OUT:  CF set if function not found                                  **
*************************************************************************}
procedure CallKEYBFunc; assembler;
label CallFunc;
asm
        { CX = FirstFunction + (CX-1)*4 }
        DEC    CX
        SHL    CX, 2
        ADD    CX, Offset ExtInt9Proc

        { Preserve DI, BX }
        PUSH   DI
        PUSH   BX

        { CS:DI -> Pointer to function; load and test if zero }
        MOV    DI, CX
        MOV    BX,CS:[DI]
        OR     BX,CS:[DI+2]

        { Recover registers and test if zero }
        TEST   BX,$FF
        POP    BX
        JNZ    CallFunc
        POP    DI

        { function not found: exit }
        STC
        RET

        { function found: call it and return }
CallFunc:
        CALL   DWORD PTR CS:[DI]
        POP    DI
        CLC
end;



{TODO: check if all called functions are DI-preserving safe}
{************************************************************************
** StoreKey: the unified common entry point to store a key in buffer   **
** ------------------------------------------------------------------- **
** IN:   AH: scancode                                                  **
**       AL: character code                                            **
** OUT:  Does not preserve registers (because of DoCombi)              **
*************************************************************************}
Procedure StoreKey; assembler;
label
    Combi, Store, Ende;
      asm
      
           push dx
           push es
           push di

           xor  dx,dx
           mov  es,dx

           mov  dl,[es:offset KStatusFlags2]
           and  dl,8
           cmp  dl,8
           jne  Combi

           Mov  AL,ES:[Offset KStatusFlags2]     { ES still remains from the check for 0000! }
           And  AL,$F7
           Mov  ES:[Offset KStatusFlags2],AL
           jmp  ende

combi:     cmp  CombStat, 0
           je   Store

           mov  bl,ah
           MOV       CX, KF_Combine
           CALL      CallKeybFunc
           jmp  ende

store:     call BIOSStoreKey


ende:      pop  di
           pop  es
           pop  dx
           
End;

{TODO: check if all called functions are DI-preserving safe}
{************************************************************************
** KEYBexecute: executes a KEYB command (except for 0 and 160)         **
** ------------------------------------------------------------------- **
** IN:   DL: command code (excluded 0,160)                             **
** OUT:  registers possibly destroyed                                  **
*************************************************************************}
procedure KEYBExecute; assembler;
label other, ende, command;
asm
   PUSH DI
      CMP       DL,80                     { 1..79:     XString }
      JAE       Other
      MOV       AL,DL
      MOV       AH,$FF
      CALL      StoreKey
      JMP       Ende

other:
      CMP       DL,199                  { 200..255:     COMBIs  }
      JNA       Command
      MOV       AL,DL
      MOV       CX, KF_ExecCombi
      CALL      CallKeybFunc
      JMP       ende

Command:                                {  80..199:     Other commands }
      MOV       CX, KF_DoCommand
      CALL      CallKeybFunc

ende:
  POP DI
end;


{TODO: func$AD80: LES DI,CurrentLayout}
{************************************************************************
** Int2Fh: handler for the DOS-MuX Keyb services                       **
** ------------------------------------------------------------------- **
** INT:  services interrupt 2Fh (AX=AD80h)                             **
*************************************************************************}
procedure Int2Fh; assembler;
label ChainOld, kmux80, kmux81, kmux9x, ExitMux;
asm
      CMP  AH, $AD
      JNE  ChainOld

      CMP  AL,$8F
      JA   KMux9X
      
      CMP  AL, $80
      JE   KMux80
      CMP  AL, $81
      JE   KMux81
ChainOld:
      JMP  CS:[OldInt2Fh]

KMux80:
      MOV AX,$FFFF
      PUSH CS
      POP  ES
      MOV  DI,ES:[Offset CurrentLayout]
      MOV  DX,Version
      IRET
KMux81:
      POPF
      CLC
      RETF

KMux9x:
      CMP  AL,$93
      JNE  ChainOld

      PUSH      AX
      PUSH      BX
      PUSH      CX
      PUSH      DX
      PUSH      DI
      PUSH      SI
      PUSH      ES
      PUSH      DS
      CALL      KEYBexecute
      POP       DS
      POP       ES
      POP       SI
      POP       DI
      POP       DX
      POP       CX
      POP       BX
      POP       AX

      IRET
      
end;

procedure DefaultTable; assembler;
asm
    DB  83             {DEL:  83  !0   !163}
    DB  2              {N is preprocessed with the columns}
    DB  3
    DB  0
    DB  161

    DB  59             {F1:   59   !100}
    DB  1
    DB  1
    DB  100
    
    DB  60             {F2:   60   !101}
    DB  1
    DB  1
    DB  101
    DB  0              {STOP}
end;


{ TableLookUp procedure }
{   IN:  AL: normalized KEYB scancode
         DL: break?
         CX: column
         DS:DI -> NewTable
         ES=0
    OUT: DL: break?
         ES=0
         CF set if was not found
            AL: normalized KEYB scancode
            CX: column
         CF clear if was found:
            AH:  unnormalized KEYB scancode, or 0 if IGNORE
                 if NOT ignore:
                    AL:  character value
                    BL: X?                                          }
procedure TableLookUp; assembler;
label CompareSC, ScancodeFound, NotFound, Ende, ContFound, FoundEnd,
      NotReplace, XBit;
asm

        { loop to find it }
CompareSC:
        xor     bx,bx
        mov     bl, [DS:DI+1]           { attrib byte  }
        and     bl, 7                   { get number of colums }
        
        cmp     al,[DS:DI]
        je      ScancodeFound
        add     bl, 3                   { not found: find next }
        add     di, bx
        mov     bl, [DS:DI]
        test    bl, $FF
        jz      NotFound
        jmp     CompareSC


ScancodeFound:
        inc     di                      { DI-> Attributes }

        { Step 1: ignore scancode }

        mov     bh, [DS:DI]             { attributes }

        test    bh, 16                  { ignore? }
        jz      ContFound

        xor     ah,ah                   { YES, AH=0, then }
        jmp     FoundEnd                { we are done! }

ContFound:

        { step 2: replace scancode }
        inc     di                      { DI-> Attributes+1 }
        test    bh, 128                 { replace scancode? }
        jz      NotReplace

        mov     al, [DS:DI]
        inc     di

NotReplace:                             { DI-> XBits }

        { step 3: check length }

        push    bx                      { save BH: attributes }
        shr     bx,8
        and     bl,7
        cmp     bl,cl
        pop     bx
        jb      NotFound

        { step 4: check NumLock, CapsLock }

        push    cx
        dec     cl                      { column - 1 (starts on 0) }

        cmp     cl,1                    { it must be columns 1 or 2 }
        ja      XBit

        shr     bx,8
        and     bl,64+32
        and     bl,[ES:KStatusFlags1]
        jz      XBit

        xor     cl,1
XBit:
        { step 5: Xbit }

        mov     bl, [DS:DI]             { X-bit }
        inc     di                      { DI-> First column }

        ror     bl, cl
        and     bl, 1                   { BL is READY }

        { step 6: column data }

        add     di,cx
        shl     ax, 8                   { scancode to AH }
        mov     al,[DS:DI]              { AL is READY }

        pop     cx

        { step 7: if !0 => not found }
        
        test    al,$ff
        jnz     FoundEnd

        test    bl,$ff
        jz      FoundEnd

        shr     ax,8
        jmp     NotFound

FoundEnd:
        and     ah, $7F                 { AH is READY }
                                        { DL is UNTOUCHED }
        clc
        jmp     ende                    { hence, we are done! }
        

NotFound:                               { DL, AL and CL are untouched! }
        stc

Ende:
end;



{TODO in P5: DS: segment of layout}
{************************************************************************
** Int15h: handler for int15h                                          **
** ------------------------------------------------------------------- **
** INT:  services interrupt 15h (AH=4Fh)                               **
*************************************************************************}
procedure Int15h; assembler;
label
  { General management labels }
  IsFunc4F, KeyboardHandler,
  { pre-table lookup }
  PreTable1, PreTable2, PreTable2a, PreTable3, PreTable3a, PreTable3b, PreTable3c,
  { table lookup block }
  TableBlock,skp1_2a, skp1_2b, skp1_2c,NotFound,NotFound2,NotFound3,Found,
  { post-table lookup }
  PostTable1, ChainScancodeAH, PostTable2, PostTable3, X,
  { store blocks }
  Store, Store2, Store2a, Store3, Store4,
  { exit points }
  FinishScancode, ChainScancode, Ende, Ende2, EndeT;

  
asm

{================= INITIAL CONTROL BLOCK ================================}

      PUSHF                 { preserve flags! }

      CMP       AH, $4F     { is it function 4Fh? }
      JE        IsFunc4F    { check functions }
      POPF
      JMP       DWORD PTR CS:[OldInt15h]

IsFunc4F:
      STC
      CALL      DWORD PTR CS:[OldInt15h] { the old one first! }
      JC        KeyboardHandler          { if Carry clear, }
      RETF      2                        { return and preserve flags! }

{================= KEYB's KEYBOARD HANDLER ==============================}

KeyboardHandler:

{====================== PRE-TABLE BLOCK ============================}
{  IN:      AL: scancode                                            }
{  NOTES:   - code is linear
            - Uses AX,CX,ES, and later BX,DX                        }

      {--- 1.- E0?, F0? (discard them) ---}

      CMP       AL,$E0
      JE        PreTable1
      CMP       AL,$F0
      JE        PreTable1
      JMP       PreTable2
PreTable1:
      STC
      JMP       EndeT

      {--- 2.- Save some registers ---}

PreTable2:

      PUSH      AX                            { save some registers }
      PUSH      CX
      PUSH      DS
      PUSH      ES
      
      XOR       CX,CX               { set our segments:             }
      MOV       ES,CX               {   CS: Keyb and global data    }
      PUSH      CS                  {   ES: 0 (BIOS data)           }
      POP       DS                  {   DS: Layout data             }

      PUSH      BX
      PUSH      DX

      {--- Get break flag and correct scancode ---}

      XOR       DL,DL
      TEST      AL,$80
      JZ        PreTable2a
      INC       DL

PreTable2a:
      MOV       BL,[ES:Offset KStatusByte1]
      TEST      BL,E0Prefixed
      JZ        PreTable3
      OR        AL,$80

      {--- 3.- Compute the column ---}


      {FIXED SCHEME: Normal, Shift, Ctrl, Alt, AltGr}
PreTable3:
      XOR       CX,CX
      MOV       BL,ES:[Offset KStatusFlags1]

      TEST      BL,$08         { any alt? }
      JZ        PreTable3b

      TEST      BL,$07         { with ctrl or shift? }
      JNZ       PreTable3a

      MOV       CL,ES:[Offset KStatusByte1]
      SHR       CL,3           { no: Alt=3, AltGr=4 }
      AND       CL,1
PreTable3a:                    { pretable3a: Ctrl/Shift+Alt => 4 }
      ADD       CL,4
      JMP       TableBlock

PreTable3b:                    { no ALT }
      TEST      BL,$01
      JZ        PreTable3c
      OR        BL,$02         { convert bit1 into AnyShift }
PreTable3c:
      SHR       BL,1
      AND       BL,$03         { BL bit0: Shift, bit1: Ctrl }
      INC       CL             { CL=1 }
      ADD       CL,BL


TableBlock:

{====================== TABLE LOOKUP BLOCK =========================}
{   IN:  AL: normalized KEYB scancode
         DL: break?
         CX: column
    OUT: CF: set if has to Chain scancode, clear to continue        }

      {--- 1.- First Table: user defined table ---}

      PUSH      DI
      MOV       DI,[CS:Offset CurrentLayout]
      ADD       DI,9

      CALL      TableLookUp

      JNC       Found

      {--- 2.- Second Table ---}

      SHL       AX,8               { scancode -> AH }

      { compute the column }

      MOV       AL,[ES:KStatusFlags1]
      MOV       BL,AL
      TEST      AL,3                       {shift}
      JZ        Skp1_2a
      MOV       AL,1
Skp1_2a:
      AND       AL,1

      SHR       BL,5                       {numlock?}
      AND       BL,1
      XOR       AL,BL
      JZ        Skp1_2b

      MOV       CX,2
      JMP       Skp1_2c

Skp1_2b:
      MOV       AL,[ES:KStatusFlags1]
      AND       AL,12
      CMP       AL,12
      JNE       NotFound
      MOV       CX,1

Skp1_2c:
      SHR       AX,8
      MOV       DI,Offset DefaultTable

      CALL      TableLookUp

      JNC       Found

      SHL       AX,8            { scancode -> AH }

      (*
push ax         { &&& aqu llega en AL }
mov al,ah
mov ah,33
call BIOSStoreKey
pop ax
*)
      {--- 3.- Results ---}

NotFound:       { &&& aqu llega en AH si press, AL si release }
(*
push ax
mov al,ah
mov ah,44
call BIOSStoreKey
pop ax

push ax
mov ah,55
call BIOSStoreKey
pop ax
*)
      POP       DI

      TEST      DL,$FF                      { if breakcode, chain }
      JNZ       NotFound3

      CMP       AH,42                       { avoid: Shifts, Ctrls, Alts, Caps }
      JE        NotFound3
      CMP       AH,54
      JE        NotFound3
      CMP       AH,29
      JE        NotFound3
      CMP       AH,157
      JE        NotFound3
      CMP       AH,56
      JE        NotFound3
      CMP       AH,184
      JE        NotFound3
      CMP       AH,58
      JE        NotFound3

      PUSH      AX                  { else drop any pending COMBI }
      PUSH      DI
      MOV       DI,CombStat
      TEST      DI,$FFFF
      JZ        NotFound2
      MOV       AL,[CS:DI-1]
      XOR       AH,AH
      CALL      BIOSStoreKey
      XOR       AX,AX
      MOV       CombStat,AX
NotFound2:
      POP       DI
      POP       AX

NotFound3:
      AND       AL,$7F                       { normalize scancode }
      JMP       ChainScancode



Found:
      POP       DI
      TEST      AL,$FF
      JZ        FinishScancode              { IGNORE }

      CLC



{====================== POST-TABLE BLOCK ===========================}
{ IN:  AL: character
       AH: scancode
       BL: X?
       DL: break?
       ES=0                                                         }

      {--- 1.- Keyb disabled? ---}
PostTable1:

      PUSH      DI
      MOV       DI,[CS:Offset PIsActive]        {is it disabled?}
      MOV       CL,[CS:DI]
      POP       DI
      TEST      CL,$FF
      JNZ       PostTable2

      TEST      DL,DL                           {is it makecode?}
      JNZ       ChainScancodeAH

      TEST      BL,$FF                          {is it X?}
      JZ        ChainScancodeAH

      CMP       AL,101                          {command 100..119?}
      JB        ChainScancodeAH
      CMP       AL,119
      JNA       PostTable2                      {YES: continue processing!}

ChainScancodeAH:
      SHR       AX,8
      JMP       ChainScancode

      {--- 2.- BreakCode? ---}

PostTable2:

      TEST      DL,DL                          { break code?}
      JZ        PostTable3                     { if no, then continue}

      TEST      BL,$FF                         { is it X?}
      JZ        ChainScancodeAH

      CMP       AL,80                          { is it 80-85 or 90-97?}
      JB        ChainScancodeAH
      CMP       AL,86
      JB        X
      CMP       AL,90
      JB        ChainScancodeAH
      CMP       AL,98
      JB        X                              { if YES: goto X }
      JMP       ChainScancodeAH

      {--- 3.- Determine if Store or Execute ---}
PostTable3:

      TEST      BL,$FF      { XOperation: command, XString or COMBI }
      JZ        Store

      {--- 4.- Execute Subblock (!0 was already processed) ---}

      CMP       AL,160                         { 160: NOP (absorve) }
      JE        FinishScancode
X:
      TEST      DL,DL                             { is it makecode? }
      JNZ       ChainScancodeAH
{ TEMPORARY: we reject them }

      MOV       DL,AL                       { the rest: KEYBexecute }
      CALL      KEYBexecute
      JMP       FinishScancode


{====================== STORE BLOCK ================================}
{   IN:  AL=charcode
         AH=scancode                                                }

Store:
      {--- 1.- Scancode modification routines ---}

      AND       AH,$7F                       { unnormalize scancode }

      MOV       CX, KF_Translate
      CALL      CallKeybFunc

      {--- 2.- ASCII=E0,F0 ---}
Store2:
      CMP       AL,$E0
      JE        Store2a
      CMP       AL,$F0
      JNE       store3
Store2a:
      XOR       AH,AH           { scancode=0 }


      {--- 3.- E0-prefixed keys modifications ---}
Store3:
      MOV       CL,[ES:KStatusByte1]
      TEST      CL,E0Prefixed
      JZ        Store4
      MOV       AH,$E0


      {--- 4.- Finally, store the key ---}
Store4:
      CALL      StoreKey

{====================== END OF ROUTINE: EXIT POINTS ================}

      {--- 1.- Scancode was processed ---}
FinishScancode:
      CLC
      JMP       Ende

      {--- 2.- Scancode was rejected ---}
ChainScancode:
      MOV       AH,$4F
      STC

Ende:
      POP       DX
      POP       BX

Ende2:
      POP       ES
      POP       DS
      POP       CX
      POP       AX

{========== END OF KEYB's KEYBOARD HANDLER ==============================}
EndeT:

      RETF      2

end;

{************************************************************************
** EfStoreKey0: Stores AX into BIOS KEYB buffer (AT/PS machines, enh K **
** ------------------------------------------------------------------- **
** IN:   AX: word to be stored                                         **
** OUT:  -                                                             **
*************************************************************************}
Procedure EfStoreKey0; far; assembler;
label ende;
asm
   mov cx,ax
   mov ah, $05
   int $16
end;



{------------------------------------------------------------------------
---- KEYB 2 DISCARDABLE FUNCTIONS BEGIN HERE                        -----
------------------------------------------------------------------------}


{------------------------------------------------------------------------
---- MODULE 1: COMBI routines                                       -----
------------------------------------------------------------------------}


procedure DoCombi; far; assembler;
label Lop2, Skp5, NoCombine, Done;
asm
    {== Entry: AL: character to combine with the pending COMBI char
               BL: scancode of the key pressed (in case it fails) }
      Push SI

      Cmp Bl,$FF          { no combine with XStrings }
      Je  NoCombine

      Mov Di,CombStat
      Mov CombStat,0

      Cmp AL,8            { is it backspace? }
      JE  Done            { yes, simply forget the combi, do nothing }

      Mov Si,Di
      Mov CL,CS:[DI]              { Number of chars to check. }
      Xor Ch,Ch
      Inc Di

Lop2: Cmp Al,CS:[DI]              { Found 2nd char of combination? }
      JNZ Skp5                    { No. }

      Inc Di
      Mov Al,CS:[Di]              { Replacement char. }
      XOR AH,AH                   { ScanCode should be 0 with COMBI!}
      Call BIOSStoreKey
      Jmp Done

Skp5: Inc Di                      { Next entry. }
      Inc Di
      Dec Cx                      { Another candidate? }
      JNZ Lop2                    { Then go on. }

NoCombine:
      mov   ah,bl
      push  ax
      mov   al,CS:[SI-1]
      xor   ah,ah
      call  BIOSStoreKey
      pop   ax
      call  BIOSStoreKey
      jmp   done

Done:

      POP   SI
end;


procedure StartCombi; far; assembler;
          { ENTRY: AL = COMBI code }
label lop3, skp7, ende;
asm
      MOV       DI,[CS:Offset CurrentLayout]
      ADD       DI,[CS:DI+4]

Lop3: Cmp  Byte Ptr CS:[Di],0      { End of list? }
      Jz   Ende                    { Yes.: ERROR, shouldn't happen!! }
      Cmp  Al,200                  { Is this the char? }
      JE   Skp7                    { Yes. }
      Inc  Di
      Mov  Cl,CS:[DI]
      Xor  Ch,Ch
      Shl  Cx,1
      Add  Di,Cx
      Inc  Di                      { Next char. }
      Dec  Al
      Jmp  Lop3

Skp7: Inc Di
      Mov CombStat,Di
Ende:
end;


{------------------------------------------------------------------------
---- MODULE 2: BEEP (nothing yet)                                   -----
------------------------------------------------------------------------}


{------------------------------------------------------------------------
---- MODULE 3: Basic commands                                       -----
------------------------------------------------------------------------}


{************************************************************************
** BasicCommand: executes KEYB commands 80-199 (excluded 160)          **
** ------------------------------------------------------------------- **
** IN:   DL: KEYB command number (see KEYB documentation)              **
** OUT:  Many registers are probably trashed                           **
**       On APMCommands, may never return                              **
** NOTE: for extended commands, it calls the Extended procedure        **
*************************************************************************}
{**** List of commands already implemented ****
--- BASIC
100: disable keyb
101: reenable keyb
161: decimal char
--- EXTENDED
89: simulate INS
140: int5h
141: int19h
142: int1bh
150..154: APMcommand (n-150)
163: set pause bit
168: reset character-by-code buffer
169: release alternative character from character-by-code buffer
170-179: alternative character introduction }
procedure BasicCommand ; far; assembler;
label  no100, no101, no140, no141, no142, no161, no163, no89, noAPM, noAltNum,
       no168, delAltNum, endPC;
ASM
         push       es
         xor        ax, ax
         mov        es, ax

         {---- individual quick commands ---}
         cmp        dl, 100
         jne        no100

         {100: disable KEYB}
         xor        cx,cx
         push       di
         mov        di,[cs: Offset PIsActive]
         mov        [cs:di],cl
         pop        di
         jmp        endPC

         {101: reenable KEYB}
no100:   cmp        dl, 101
         jne        no101

         mov        cl,1
         push       di
         mov        di,[cs: Offset PIsActive]
         mov        [cs:di],cl
         pop        di
         jmp        endPC

no101:   cmp        dl, 161
         jne        no161

         xor        ah,ah
         mov        al,[cs: offset DecimalChar]
         call       StoreKey
         jmp        endPC

no161:
         MOV       CX, KF_ExtCommands
         CALL      CallKeybFunc

endPC:   pop        es
end;




{==================  /9- MODE LIMIT (/9* STARTS HERE) =============}


{------------------------------------------------------------------------
---- MODULE 4: Extended commands                                    -----
------------------------------------------------------------------------}


{TODO: XBufferTail=XBufferHead in break}
{************************************************************************
** ExtCommand: executes KEYB extended commands                         **
** ------------------------------------------------------------------- **
** IN:   DL: KEYB command number (see KEYB documentation)              **
** OUT:  Many registers are probably trashed                           **
**       On APMCommands, may never return                              **
*************************************************************************}
{**** List of commands already implemented ****
89: simulate INS
140: int5h
141: int19h
142: int1bh
150..154: APMcommand (n-150)
163: set pause bit
168: reset character-by-code buffer
169: release alternative character from character-by-code buffer
170-179: alternative character introduction }
procedure ExtCommand ; far; assembler;
label  no100, no101, no140, no141, no142, no161, no163, no89, noAPM, noAltNum,
       no168, delAltNum, endPC;
ASM
         push       es
         xor        ax, ax
         mov        es, ax

         {---- individual quick commands ---}
         cmp        dl, 140
         jne        no140

         {140: int 5h}
         int        $5
         jmp        endPC

no140:   cmp        dl, 141
         jne        no141

         {141: int 19h}
         int        $19
         jmp        endPC

no141:   cmp        dl, 142
         jne        no142

         {142: int 1Bh}
         mov        al, [es: offset BufferHead]
         mov        [es: offset BufferTail], al
         int        $1b

no142:   cmp        dl, 163
         jne        no163

         {161: set pause}
         mov        al, [es: offset KStatusFlags2]
         or         al, 8
         mov        [es: offset KStatusFlags2], al
         jmp        endPC

no163:   cmp        dl, 89
         jne        no89

         {164: simulate INS}
         mov        al, [es:offset KStatusFlags1]
         xor        al, $80
         mov        [es:offset KStatusFlags1], al
         xor        al,al
         Mov        ah,82
         call       StoreKey
         jmp        endPC

         {----  APM Commands (150-154) ----}
no89:    cmp        dl, 150
         jb         noAPM
         cmp        dl,154
         ja         noAPM
         sub        dl,150
         mov        cx, KF_APMFunc
         call       CallKeybFunc
         jmp        endPC

         {---- Alternative character introduction ---}
noAPM:   cmp        dl, 170
         jb         noAltNum
         cmp        dl,179
         ja         noAltNum
         sub        dl,170
         mov        cl,dl
         mov        dx,10
         xor        ah,ah
         mov        al,[es:offset AltInput]
         mul        dx
         add        al,cl
         mov        [es:offset AltInput],al
         jmp        endPC

noAltNum:
         cmp        dl,168
         jne        no168
delAltNum:
         xor        al,al
         mov        [es:offset AltInput],al
         jmp        endPC

no168:   cmp        dl,169
         jne        endPC
         xor        ah,ah
         mov        al,[es:offset AltInput]
         call       StoreKey
         jmp        delAltNum
         

endPC:   pop        es

end;


{------------------------------------------------------------------------
---- MODULE 5: Default shift/locks (to be moved/patched)            -----
------------------------------------------------------------------------}


{------------------------------------------------------------------------
---- MODULE 6: Extended column computation (todo)                   -----
------------------------------------------------------------------------}


{------------------------------------------------------------------------
---- MODULE 7: XT specific routines                                 -----
------------------------------------------------------------------------}

{************************************************************************
** EfStoreKey1: Stores AX into BIOS KEYB buffer                        **
** ------------------------------------------------------------------- **
** IN:   AX: word to be stored                                         **
** OUT:  ES is zeroed                                                  **
**       DI, SI: trashed                                               **
*************************************************************************}
Procedure EfStoreKey1; far; assembler;
label ende, rende;
asm
         CLI
{
         Xor Si,Si
         Mov Es,Si
}
         Mov DI,ES:BufferHead     { Pointer to end of buffer. }
         Mov ES:[DI+$400],AX      { Key in [40:BufferHead]. }

         Add Di,2                 { Pointer to next entry. }
         Cmp DI,ES:BufferEnd      { Reached end of buffer ? }
         Jne ende                 { No -> Proceed in Text. }
         Mov DI,ES:BufferStart    { Pointer back to start. }

ende:
         Mov ES:BufferHead,DI
rende:
         STI
end;



CONST
        { keyboard commands }
        DisableKeyboard = $AD;
        EnableKeyboard  = $AE;

{************************************************************************
** EnableKeyboardProc: enable keyboard hardware                        **
** ------------------------------------------------------------------- **
** IN:   -                                                             **
** OUT:  AX and CX are trashed                                         **
*************************************************************************}
Procedure EnableKeyboardProc; assembler;
label WaitReady2;
asm
        XOR    CX, CX       { counter to 65535! }
WaitReady2:
        IN     AL, $64      { read status register }
        TEST   AL, 2        { bit 1 set: input buffer full? }
        LOOPNZ WaitReady2   { loop until timeout or bit 1 clear }

        MOV AL, EnableKeyboard
        OUT $64, AL         { send the enable command }

        RET
end;


procedure ProcessPause; forward;


{************************************************************************
** Int9H: Basic int9 management                                        **
** ------------------------------------------------------------------- **
** INT:  services interrupt 9h (keyboard IRQ)                          **
*************************************************************************}
procedure Int9H; assembler;
Label  IsActive, Start, CallExtInt9, Reenable, LeaveInt, Jmp1,
       WaitReady1, ProcessIt, ChainToOld;
ASM

    {******* SECTION 1: COMMON INT9h ************}

        {-- Workaround for APL software --}
        JMP    Start
IsActive:DW    1             { modified by APL software}

        {-- Save the registers --}
Start:  PUSH   AX            { scancode }
        PUSH   BX            { bool: ExtInt9 }
        PUSH   CX            { counter }

        XOR    BX,BX
        XOR    CX,CX

        {-- Compute if we should handle the extended --}
        MOV    BL, CS:[Offset IsActive]
        TEST   BL, $FF
        JNZ    Jmp1

        {-- Disable the interrupts and the keyboard --}
Jmp1:   CLI                  { disable hardware ints }
        DEC    CX            { counter to 65535! }
WaitReady1:
        IN     AL, $64       { read status register }
        TEST   AL, 2         { bit 1 set: input buffer full? }
        LOOPNZ WaitReady1    { loop until timeout or bit 1 clear }
        MOV    AL, DisableKeyboard 
        OUT    $64, AL       { send the disable command }

        {-- Read and authenticate scancode --}
        IN      AL, $60      { get scancode to AL }
        MOV     Ah, $4F      { Authenticate scancode}
        STC
        INT     15h          { Return: CF clear if key needs no further processing }

        {-- What to do next? --}
        JC      ProcessIt    { No further processing of pressed keys if CARRY cleared! }
        XOR     BX, BX       { no extended processing }
        JMP     Reenable     { reenable all }

ProcessIt:
        TEST    BX,$FF       { is driver active? }
        JNZ     CallExtInt9  { if not: chain to older driver }

    {******* SECTION 2: CHAIN TO PREVIOUS INT9 HANDLER ************}

ChainToOld:
        CALL   EnableKeyboardProc
        PUSHF
        STI
        CALL   DWORD PTR CS:[Offset OldInt9h]
        JMP    LeaveInt

    {******* SECTION 3: CALLING EXTINT9 ************}

CallExtInt9:
        MOV    CX, KF_ExtInt9
        CALL   CallKEYBFunc
        JNC    Reenable
        XOR    BX, BX        { if function not found: no longer extended handling }
        JMP    ChainToOld    { whenever NO function, try to chain to previous }

    {******* SECTION 4: REENABLE EVERYTHING (process PAUSE if needed) ************}

Reenable:
        {-- Reenable Interrupt controller, Interrupts and Keyboard --}
        CALL   EnableKeyboardProc  { reenable keyboard }
        MOV    AL, $20       { report End of Interrupt to interrupt controller }
        OUT    $20, AL
        STI                  { restore interrupts }

        {-- Process PAUSE, if needed (this is EXTENDED handling) --}
        TEST   BX, $FF
        JZ     LeaveInt
        CALL   ProcessPause

    {******* SECTION 5: LEAVE INTERRUPT ROUTINE ************}
LeaveInt:

        POP CX
        POP BX
        POP AX
        IRET
end;



{------------------------------------------------------------------------
---- MODULE 8: APM routines                                         -----
------------------------------------------------------------------------}


{************************************************************************
** APMProc: process an APM command                                     **
** ------------------------------------------------------------------- **
** IN:   DL: (0-based) APM command number (see constants above)        **
** OUT:  Many registers are probably trashed                           **
**       On some commands, may never return                            **
*************************************************************************}
procedure APMProc; far; assembler;
label FlushNLCACHE, NLCACHE_callf, NLCACHE_farentry, NLCACHE_reentry,
      FlushSMARTDRV, FlushCacheForce, FlushCacheExit,
      EndAPMProc, jmp1, jmp2, jmp3, doReset, Skp1, Skp2, Skp3;

const CallReboot : pointer = ptr ($FFFF,$0000);

asm

        push dx { place parameter on DX }

(*  BEGIN FlushCache code (by Matthias Paul)
== Last edit: 2001-06-18 MPAUL
;
; - Taken from Axel C. Frinkes & Matthias Pauls FreeKEYB sources
;   with some modifications to possibly work as a drop-in replacement
;   in 4DOS.
; - While the implied actions are different for SMARTDRV 4.0+
;   and NWCACHE 1.00+, the result is the same for both of
;   them - the cache will be flushed unconditionally.
; - Works around a problem with DBLSPACE loaded, where DBLSPACE
;   may terminate the current process with an error message, that
;   it would not be compatible with SMARTDRV before 4.10+.
; - Works around a problem, where the cache would not be flushed
;   with NWCACHE 1.00+, if the CX register accidently happened
;   to be 0EDCh.
; - Is enabled to continue to work with future issues of NWCACHE,
;   which might no longer flush the cache just on calling
;   SMARTDRV's install check...
; - Supports NetWare Lite's NLCACHE and Personal NetWare's
;   NWCACHE sideband API to asynchronously flush the cache.
;   This ensures system integrity even when the NetWare Lite or
;   Personal NetWare SERVER is loaded on the machine.
; - Furthermore, under some conditions on pre-386 machines
;   NWCACHE cannot intercept a reboot broadcast by itself, and
;   hence it must be called explicitely before a possible reboot.  *)


FlushNLCACHE:
        mov     ax,0D8C0h       { NLCACHE/NWCACHE install check  }
        mov     cl,ah           { (sanity check: preset CL >= 10h) }
        push    cs              { (preset to ourselves for safety) }
        pop     es              
        mov     di,offset NLCACHE_farentry
        int     2Fh             { (NLCACHE/NWCACHE modify AL,CL,DX,DI,ES)}

        cmp     ax,0D8FFh       { cache installed? (AL = FFh)             }
        jne    FlushSMARTDRV

        cmp     cl,al           { CL=FFh? (workaround for NWCACHE before  }
         je     NLCACHE_callf   { BETA 17 1993-09-28, CL=FFh,00h,01h)     }
        cmp     cl,10h          { (sanity check: CL < 10h on return,      }
         jae    FlushSMARTDRV   { only CL=01h,02h,03h are defined so far) }

NLCACHE_callf:
        xor     bx,bx           { BX=0: asynch. flush request from server }
        push    cs              { push return address on stack            }
        mov     ax,offset NLCACHE_reentry
        push    ax              
        push    es              { push ES:DI -> entry point into far API  }
        push    di              
        clc                     { assume pure cache                       }
NLCACHE_farentry:               { (dummy entry point)                     }
        retf                    { simulate a CALLF into sideband function }

NLCACHE_reentry:                { return from sideband (AX/flags modified)}
         jc     FlushNLCACHE    { if error retry because still dirty cache}
        test    ax,ax           { CF clear and AX=0000h?                  }
         jnz    FlushNLCACHE    { if not, retry until everything is OK    }

(*NLCACHE/NWCACHE is pure now, so it would be safe to exit here.
; However, it doesn't harm to play it extra safe and
; just fall through into the normal SMARTDRV flush sequence...
; Who knows, multiple caches might be loaded at the same time...  *)

FlushSMARTDRV:
        mov     ax,4A10h        { SMARTDRV 4.00+ API                      }
        xor     bx,bx           { install check (BX=0)                    }
        mov     cx,0EBABh       { mimic SMARTDRV 4.10+ (CX=EBABh)         }
                                { to workaround DBLSPACE problem.         }
                                { CX<>0EDCh to avoid NWCACHE's /FLUSH:OFF }
                                { special case! Flush regardless          }
                                { of NWCACHE's configuration setting.     }
        int     2Fh             { (modifies AX,BX,CX,DX,DI,SI,BP,DS;ES?)  }
                                { NWCACHE 1.xx has flushed its buffers    }
                                { now, but we should not rely on this.    }
        cmp     ax,6756h        { NWCACHE 1.00+ magic return?             }
         je     FlushCacheForce {  (extra-safe for future NWCACHE 2.00+)  }
        cmp     ax,0BABEh       { SMARTDRV 4.0+ magic return?             }
         jne    FlushCacheExit  {  nothing we can do                      }
         jcxz   FlushCacheExit  { any dirty cache elements?               }
FlushCacheForce:
        mov     cx,ax           { CX<>0EDCh to avoid NWCACHE special case,}
                                { hence we preset with magic return for   }
                                { possible future broadcast expansion.    }
        mov     ax,4A10h        { SMARTDRV 4.00+ API                      }
        mov     bx,0001h        { force synchronous cache flush           }
        push    cx              
        int     2Fh             { (modifies BP???)                        }
        pop     cx              { (safety only, not necessary)            }
        cmp     cx,6756h        { retry for any cache but NWCACHE         }
        jne    FlushSMARTDRV    { probably obsolete, but safer            }
                                { at the risk of a possible deadlock in   }
                                { case some hyphothetical SMARTDRV        }
                                { clone would not support the CX return   }
FlushCacheExit:

(** END FlushCache code (by Matthias Paul)  **)

        pop ax
        test ax,ax
        clc
        jz EndAPMProc

        push   es
        xor    cx,cx
        mov    es,cx

        dec al
        jnz skp1

        mov    cx,$1234        { YES: warm reboot }
        jmp    doReset

        stc
        jmp EndAPMProc

skp1:
        dec al
        jnz skp2

        doReset:mov    [es:$472],cx    { YES: cold reboot }
        call   callReboot

        stc
        jmp EndAPMProc

skp2:
        pop  es
        push ax

     MOV AX, $5301      {Real Mode interface connect}
     XOR BX, BX
     INT $15

     MOV AX, $530F      {Engage power management}
     MOV BX, 1
     MOV CX, BX
     INT $15

     MOV AX, $5308      {Enable APM for all devices}
     MOV BX, 1
     MOV CX, BX
     INT $15

     MOV AX, $530E      {force version 1.1}
     XOR BX, BX
     MOV CX, $0101
     INT $15

        pop ax
        dec al
        jnz skp3

     MOV AX, $5307              {First attempt: switch all off}
     MOV BX, 1
     MOV CX, 3
     INT $15

     MOV AX, $5307              {Second attempt: system bios}
     XOR BX, BX
     MOV CX, 3
     INT $15
     
        stc
        jmp EndAPMProc

skp3:

     MOV AH,$2
     MOV DL,'H'
     INT $21

     MOV AX, $5307
     MOV BX, 1
     MOV CX, 2
     INT $15
     clc

EndAPMProc:

end;


{------------------------------------------------------------------------
---- MODULE 9: Key scancode translation routines                    -----
------------------------------------------------------------------------}


{************************************************************************
** TranslateScanCode: Does some translation on Fn and ^cursor keys     **
** ------------------------------------------------------------------- **
** IN:   BL: scancode                                                  **
**       ES=0                                                          **
** OUT:  BL: scancode possibly translated                              **
*************************************************************************}
Procedure translateScanCode; far; assembler;
label NoAltNum, NoFx, FxNoAlt, FxNoCtrl, NoF1112, F1xNoAlt, F1xNoCtrl, F1xNoShift,
      TranTableFrom, TranTableTo,  endeCX, ende;
asm
       push   ax
       mov    al,[es: offset KStatusFlags1]

       TEST      AL,8           {*** Alt + upper numbers ***}
       JZ        NoAltNum
       CMP       BL,2
       JB        NoAltNum
       CMP       BL,13
       JA        NoAltNum
       ADD       BL,118
       jmp       ende

NoAltNum:
       cmp    bl,59       {*** F1-F12 ***}
       jb     NoFx
       cmp    bl,68
       ja     NoFx

       test   al,8        { with alt }
       jz     FxNoAlt
       add    bl,45
       jmp    ende

FxNoAlt:
       test   al,4        { with ctrl }
       jz     FxNoCtrl
       add    bl,35
       jmp    ende

FxNoCtrl:
       test   al,3        { with shift }
       jz     ende
       add    bl,25
       jmp    ende
       
       
NoFx:  cmp    bl,87       { F11 and F12 }
       jb     NoF1112
       cmp    bl,88
       jb     NoF1112

       test   al,8        { with alt }
       jz     F1xNoAlt
       add    bl,52
       jmp    ende

F1xNoAlt:
       test   al,4        { with ctrl }
       jz     F1xNoCtrl
       add    bl,50
       jmp    ende

F1xNoCtrl:
       test   al,3        { with shift }
       jz     F1xNoShift
       add    bl,48
       jmp    ende
       
F1xNoShift:               { no other key }
       add    bl, 46
       jmp    ende

                { Ctrl + cursors and Print }
TranTableFrom:  DB  55, 71, 72, 73, 75, 76, 77, 79, 80, 81, 82, 83, 0
TranTableTo  :  DB 114,119,141,132,115,143,116,117,145,118,146,147

NoF1112:
       and    al, $C
       cmp    al, 4
       jne    ende

       mov    bl,al
       push   cx
       mov    cx,13   {total size + 1 }
       cld
       push   cs
       pop    es
       mov    di, Offset TranTableFrom

repnz  scasb

       test  cx,cx
       jz    endeCX

       mov   di, 12   {total size}
       sub   di,cx
       mov   bl,cs:[di + offset TranTableTo]

endeCX:
       pop    cx

ende:  pop ax
end;








{==================  /9* MODE LIMIT (/9+ STARTS HERE) =============}


{------------------------------------------------------------------------
---- MODULE 10: XString management                                  -----
------------------------------------------------------------------------}


{************************************************************************
** ExtInt9Data: variables for XStrings management                      **
*************************************************************************}
procedure XStrData;
Begin
   ASM
     DD   0               { XBuffer: starts on 0 }
     DW   0,0             { XBufferHead and XBufferTail }
     DB   0               { PutXStr (bool): XString in process? }
     DD   0               { XString (ptr):  being processed }
     DB   0               { XStrPos (byte): position in string }
   End;
End;


{************************************************************************
** EfStoreKey2: store a key in the secondary buffer                    **
** ------------------------------------------------------------------- **
** IN:   AX: scancode/char pair to be stored                           **
** OUT:  BX, ES:DI are trashed                                         **
*************************************************************************}
Procedure EfStoreKey2; far; assembler;  { effectively store key in XBuffer. }
asm
        MOV  BX,CS:[Offset XStrData+4]   { buffer head to BX }

        LES  DI,CS:[Offset XStrData]
        PUSH BX                          { Increase to DI the Head offset }
        SHL  BX,1                        { (it is counted in words) }
        ADD  DI,BX
        POP  BX
        MOV  [ES:DI],AX                  { store the value }

        INC  BX                          { increase pointer, maximally is 64 }
        AND  BX,63
        MOV  [CS:Offset XStrdata+4], BX  { store again the buffer head }
end;


{TODO: this SHOULD use EfStoreKey1 or EfStoreKey0 }

{************************************************************************
** Int16h: int 16h prologue                                            **
** ------------------------------------------------------------------- **
** INT:  before int16h is served, transfers keys from secondary to     **
**       primary/BIOS buffer                                           **
*************************************************************************}

Procedure Int16h; assembler;
label Skp1,Lop1,FindXStr,Lop2,Skp2,NextEntry,PutXStrChar,Skp3,
      ContinueLoop,NoAction;

ASM
      {** 1.- Push the registers **}
      PUSHF
      PushA       {to be replaced by the appropriate code!!!}
      Push DS
      Push CS
      Pop  DS
      Push ES

      {** 2.- Compute free space in primary buffer **}

      Xor Bx,Bx
      Mov Es,Bx
      Mov BX,ES:BufferEnd
      Sub BX,ES:BufferStart                 { Buffer size in Byte. }
      Mov AX,ES:BufferHead
      Sub ax,ES:BufferTail                  { Number of Bytes in buffer. }
      JNC Skp1                              { If BX negative, -BX is the free space! }
      Xor Bx,Bx
Skp1: Sub Bx,ax                             { AX = Free Bytes. }
      Shr Bx,1                              { AX = Free Entries. }
      Dec BX                                { One entry stays empty for administration. }

      {** 3.- Loop while room in buffer (BX) **}
Lop1: OR  bx,bx
      push bx
      JZ  NoAction

      {** INSIDE LOOP on free primary buffer space **}

           { if string being put, get character from there}
           mov al,CS:[offset XStrData+8]
           test al,$FF
           jne PutXStrChar

           { if secondary buffer is empty, exit the loop directly}
           Mov Bx,CS:[offset XStrData+4]   {XBufferHead}
           Cmp Bx,CS:[offset XStrData+6]   {XBufferTail}
           JE  NoAction

           { == No string being processed, secondary buffer non empty ==}

           { Get a key from secondary buffer to AX }
           Push ES
           Mov AX,CS:[offset XStrData+6] {XBufferTail}
           Shl Ax,1
           LES DI,CS:[offset XStrData]  {XBuffer}
           Add Di,Ax                   { Address of next key. }
           Mov Ax,ES:[DI]              { Read key. }
           Pop ES

           { if scancode is $FF, then it is XString, else store it }
           Cmp  Ah,$FF
           Je   FindXStr
           
           Call EfStoreKey1

           Jmp  NextEntry

FindXStr:



               { find the XString in the buffer }
               MOV       DI,[CS:Offset CurrentLayout]  {compare to LastXStr}
               cmp       AL,[CS:DI+8]      { AL is the string number }
               ja        NextEntry         { if above: IGNORE }


               ADD       DI,[CS:DI+6]
(*
               MOV       CS
               POP       ES             { ES:DI -> XStrings }
*)

               { skip n-1 XStrings }
               xor cx, cx        
               mov cl, al
               dec cl
               jz skp2

Lop2:          Mov Al,CS:[DI]              { Address of next XStr. }
               Xor Ah,Ah
               SHL ax,1                    { each Xchar is a WORD }
               Inc AX
               Add DI,Ax
               LOOP Lop2
Skp2:

               { store address of XString }
               Mov [CS:offset XStrData+9],DI   {Word Ptr XString}
               Mov [CS:offset XStrData+11],CS  {Word Ptr XString+2}
               CMP Byte Ptr CS:[DI],0     { length of string = 0 ? }
               JZ  NextEntry                           { Then end. }

               MOV AL, 1
               MOV CS:[offset XStrData+13],AL {XStrPos}           { pointer in XString = 1 }
               MOV CS:[offset XStrData+8],AL     {PutXStr}     { processing XString = TRUE }

           { Move on the secondary buffer pointer }
NextEntry: Mov AX,CS:[offset XStrData+6]  {XBufferTail}
           Inc Ax
           And Ax,63
           Mov CS:[offset XStrData+6],Ax          { Pointer to next entry. } {XBufferTail}

           { Are we processing a XString? }
           mov al,CS:[offset XStrData+8]
           test al,$ff
           jz  ContinueLoop

           { Put a character from the XString buffer }
PutXStrChar:
           Mov Al,CS:[offset XStrData+13]  {XStrPos}
           dec al
           Xor Ah,AH
           shl al,1     { each Xchar is a word }
           inc al
           Mov Di,Ax
           Add Di,Word Ptr offset XStrData+9 {XString}    {!}
           mov ax,[cs:di]
           call EfStoreKey1     { store from AL }

           Mov Al,CS:[offset XStrData+13]  {XStrPos}
           inc al
           mov CS:[offset XStrData+13],al

           { PutXStr:= XStrPos <= Length(XString^) }
{           xor  al,al }
           mov  di, [Offset XStrData+9]  {[Offset XString]}   {!}
           mov  al, [cs:di]
           cmp  al, CS:[offset XStrData+13] {XStrPos}
           jae  ContinueLoop
           xor  al,al
           mov  CS:[offset XStrData+8],al

(*
           jb   Skp3
           inc  al
Skp3:      mov  CS:[offset XStrData+8],al   {PutXStr}
*)

ContinueLoop:
      {** End of loop **}
      pop  bx
      dec  bx
      jmp  lop1

NoAction:
      pop  BX

      {** 4.- END: recover registers}
      Pop  ES
      Pop  DS
      PopA
      PopF

      JMP CS:OldInt16h
End;



{------------------------------------------------------------------------
---- MODULE 11: Full int9h management routines                      -----
------------------------------------------------------------------------}



{************************************************************************
** ExtInt9Data: variables for extended int9h management                **
*************************************************************************}
Procedure ExtInt9Data;
Begin
   ASM
     DB   0                              { LoopRunning: locks access to loop running }
     DB   54, 42, 29, 56, 70, 69, 58, 82 { ShiftKeys }
   End;
End;


{************************************************************************
** ProcessPause: process the system pause                              **
** ------------------------------------------------------------------- **
** IN:   -                                                             **
** OUT:  -                                                             **
*************************************************************************}
procedure ProcessPause; assembler;
label Lop1, EndProcessPause;
asm
        MOV AL, CS:[Offset ExtInt9Data]  {LoopRunning}
        TEST AL, $FF               { is Pause loop running?}
        JNZ EndProcessPause        { yes: leave interrupt! }

        PUSH ES
        MOV AL, 1                  { lock this region }
        MOV CS:[Offset ExtInt9Data], AL   {LoopRunning}
        XOR AX, AX                 { see Pause bit in KStatusFlags2}
        MOV ES, AX
Lop1:   TEST ES:KStatusFlags2, 8   { in Pause? then continue loop }
        JNZ Lop1
        XOR AX, AX
        MOV CS:[Offset ExtInt9Data], AL { exit locked region }  {LoopRunning}
        POP ES

EndProcessPause:
end;


{ DoShiftKeys: updates BIOS variables according to the SHIFT key that has
  been pressed }
procedure DoShiftKeys (shiftkey: byte; isbreak: boolean);  {0..7}
var
   shiftedbit, v1, v2 : byte;
begin
    shiftedbit := 1 SHL shiftkey;
    case shiftkey of
                    0..1: {shift}
                          if isBreak then KStatusFlags1 := KStatusFlags1 XOR shiftedbit
                                     else KStatusFlags1 := KStatusFlags1 OR shiftedbit;
                    2..3: begin {ctrl,alt}
                            if (KStatusByte1 AND E0Prefixed)>0 then begin
                                if isBreak then KStatusByte1 := KStatusByte1 XOR shiftedbit
                                           else KStatusByte1 := KStatusByte1 OR shiftedbit
                            end else begin
                                shiftedbit := shiftedbit SHR 2;
                                if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                           else KStatusFlags2 := KStatusFlags2 OR shiftedbit
                            end;
                            v1 := (KStatusByte1 OR (KStatusFlags2 SHL 2)) AND $0C;
                            v2 := KStatusFlags1 AND $F3;
                            KStatusFlags1 := v1 OR v2;

                            if (shiftkey=3) AND isBreak AND (AltInput<>0)
                               then begin
                                      asm
                                         xor    ah, ah
                                         mov    al, AltInput
                                         call   StoreKey
                                      end;
                                 AltInput := 0
                               end;

                          end;
                    4..6: begin  {caps, num, scroll}
                              if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                         else KStatusFlags2 := KStatusFlags2 OR shiftedbit;
                              if not isBreak then KStatusFlags1 := KStatusFlags1 XOR shiftedbit
                          end;
                    7   : if (KStatusFlags2 AND $0C)=0 then begin  {no ctrl, alt}
                               v1 := ord ( (KStatusFlags2 AND $03)>0 );
                               v2 := (KStatusFlags2 AND $20) SHR 5;
                               if (v1 xor v2)=0 then begin  {no shifted!}
                                  if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                             else KStatusFlags2 := KStatusFlags2 OR shiftedbit;
                                  if not isBreak then begin
                                      KStatusFlags1 := KStatusFlags1 XOR shiftedbit;
                                      asm
                                         mov    al, al
                                         mov    ah, 82
                                         call   BIOSStoreKey
                                      end;
                                  end;
                               end;
                          end
    end;
end;

CONST
        { keyboard signals }
        AckSignal    = $FA;
        NakSignal    = $FE;


{************************************************************************
** ExtInt9h: extended int 9h management                                **
** ------------------------------------------------------------------- **
** IN:   AL: scancode                                                  **
** OUT:  Several registers trashed                                     **
*************************************************************************}
procedure ExtInt9h; far; assembler; 
label  Jmp1, Jmp2, Jmp3, Jmp4, NoBreak, Lop2, ShiftFound, Jmp5, Jmp6,
       NoPause, SimplyStore, Ende;
asm
        PUSH DX
        PUSH DI
        PUSH DS
        PUSH ES

        XOR    CX, CX 
        MOV    ES, CX        { we annihilate ES, for the extended handling }

        PUSH CS       {CS->DS}
        POP  DS


        {**** Test for Ack, Nak, E0-Key }

        CMP AL, $E0          { E0? then jump to Ende, and there E0prefixed will be updated }
        JE  Ende
        CMP AL, AckSignal    { Acknowledge signal? }
        JNE Jmp1             
        MOV CL, $10
        JMP Jmp2
Jmp1:   CMP AL, NakSignal    { No-acknowledge signal? }
        JNE Jmp3
        MOV CL, $20
Jmp2:   OR  ES:KStatusByte2, CL   { Ack or Nak signals:  }
        JMP Ende                  { update extended keyboard state }


        {**** Determine if it's break or make code }
        { Fill CX: BreakCode? }

Jmp3:
        XOR  CX, CX          { this will store BreakCode }
        TEST AL, $80         { break or make? }
        JZ   Jmp4            { if zero -> make }
        MOV  CL,1            { breakcode}
        AND  AL, $7F         { disable the break bit}

        { ********** Test for Break ********** }

Jmp4:
        TEST CL,$FF         { if NOT breakcode }
        JNZ  NoBreak
        MOV  BL, ES:[Offset KStatusByte1]
        TEST BL, E0Prefixed { and it is Extended }
        JZ   NoBreak
        CMP  AL, 70         { scancode 70 (break) }
        JNE  NoBreak

        MOV BX, CS:[offset XStrData+6]  {XBufferHead}
        MOV CS:[offset XStrData+8], BX  {XBufferTail}
        MOV BX, ES:[Offset BufferHead]
        MOV ES:BufferTail, BX
        INT $1B
        JMP Ende

        { ********** Scan the shifting keys ********** }


NoBreak:
        XOR BX, BX
        MOV DI, offset ExtInt9Data + 1
Lop2:   MOV DL, CS:[DI]
        CMP DL, AL
        JE  ShiftFound
        INC DI
        INC BL
        CMP BL, 8
        JB  Lop2
        JMP Jmp5
ShiftFound:
        PUSH BX         { FOUND: call DoShiftKeys with appropriate params }
        PUSH CX
        CALL DoShiftKeys
        JMP  Ende

        { ********** BreakCodes are no longer processed ********** }


Jmp5:   TEST CL, $FF    { no more breakcodes processed }
        JNZ  Ende 


        { ********** Test for Pause (=Ctrl+NumLock) ********** }

        CMP BL, 5                   { IF NumLock }
        JNE NoPause
        XOR AX, AX
        MOV ES, AX
        TEST ES:KStatusFlags1, 4    { AND Control... }
        JZ  NoPause                 { then PAUSE!! }
        OR  ES:KStatusFlags2,8      { Set pause bit. }
        JMP Ende                    { Goto final waiting loop. }


         { ********** If we reached this point, simply store with ASCII=0 ********** }

NoPause:

(*** NO: This should be made through commands
         CMP   AL, 83         { first: Ctrl+Alt+Del }
         JNE   SimplyStore
         MOV   BL, [ES:KStatusFlags1]
         AND   BL, 12
         CMP   BL, 12
         JNE   SimplyStore
         MOV   DL, APM_WarmReboot  { warm reboot! }
         CALL  APMProc
         JMP   Ende            { just in case }
         
***)
SimplyStore:
         XOR   AH,AH
         Call  StoreKey         { otherwise, store (0,AL) }

Ende:

        {===== update the E0 flag in KStatusByte1 (0040h:0096h) ==}
        XOR CX,CX
        MOV ES,CX
        MOV CL, ES:[Offset KStatusByte1]      { CL := KStatusByte1 }
        AND CL, $FD          { Clear E0Prefixed flag }
        CMP AL, $E0          { AL=Scancode = E0h ? }
        JNE Jmp6
        OR  CL, E0Prefixed   { Put E0Prefixed flag }
Jmp6:   MOV ES:[Offset KStatusByte1], CL

        POP ES
        POP DS
        POP DI
        POP DX

end;



{------------------------------------------------------------------------
---- KEYB 2 LAYOUT DATA INFORMATION, FIRST BLOCK                    -----
------------------------------------------------------------------------}

Procedure Data;                               {  about 2'75K free memory. }
Begin ASM
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0

DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
End; End;

{------------------------------------------------------------------------
---- KEYB 2 INITIALIZATION STUB BEGINS HERE                         -----
-------------------------------------------------------------------------
---- From here onwards, code is no longer clean, as it is mostly       --
---- coming from xkeyb, and will mostly be replaced/removed in         --
---- prototype 4                                                       --
------------------------------------------------------------------------}

var
     s           : string;


function maxb (a,b: byte): byte;
begin
    if a>b then maxb:=a else maxb:=b
end;


Procedure Keep(EndP : Pointer);     { Terminate program but stay resident in memory. }

type   pWord       = ^Word;
Var    Laenge      : Word;
       vES         : word;
Begin

   {free the environment block}
   vES := PWord (ptr(PrefixSeg,$2C))^;

   asm
      MOV ES, vES
      MOV AX, $4900
      INT $21
   end;

   {compute size and exit}
   Laenge:=ptrRec(Endp).Seg-PrefixSeg +     { Calculate program length. }
           (ptrrec(Endp).Ofs+15) Div 16;

   ASM
      MOV AH, 49
      XOR AL, AL
      MOV DX, Laenge
      INT $21
   END;
End;


Procedure Error(B : Byte);
Begin
   Case B of
      1 : Writeln(STR2_1);
      2 : Writeln(STR2_2);
      3 : Writeln(STR2_3);
      4 : Writeln(s,': ',STR2_4);
      5 : Writeln(STR2_5);
      6 : Writeln(STR2_6);
      7 : Writeln(STR2_7);
      9 : WriteLn(STR2_9);
      10: WriteLn(STR2_10);
      11: WriteLn(STR1_7,S);
   End;
   Halt(b);
End;


{$S+,I+,R+}

const  ShiftKeys   : Array[0..7] of Byte = (54, 42, 29, 56, 70, 69, 58, 82);   { ScanCodes of Shift keys. }
Var    KCBlockSize : word;


Function ProgramName:String;                { Finds out name and path of the current program. }
Var    EnvSeg,                              { Replaces ParamStr(0) as that won't work if environment is empty. }
       EnvOfs      : Word;
       S           : String;
Begin
   EnvSeg:=MemW[PrefixSeg:$2C];             { Environment Adress. }
   EnvOfs:=0;
   While MemW[EnvSeg:EnvOfs]>0 Do           { Seek end of environment. }
      Inc(EnvOfs);
   Inc(EnvOfs,4);
   S:='';
   While Mem[EnvSeg:EnvOfs]>0 Do            { Program name. Terminated by zero. }
   Begin
      S:=S+Char(Mem[EnvSeg:EnvOfs]);
      Inc(EnvOfs);
   End;
   ProgramName:=S;                          { Let's give it back ... }
End;



{keeps trailing \}
Function ProgramPath: string;
var
   s: string;
begin
     s := ProgramName;
     while s[length(s)]<>'\' do
        delete (s,length(s),1);
     ProgramPath := s
End; {ProgramPath}


Procedure ExpandFileName(Var Name : String);{ If needed, extend name of config file. }
Begin
  { first, we are loading a .KC file }
   If Pos('.',Name)=0 Then    { filename has no extension -> add .KC }
      Name:=Name+'.KC';

   If (Pos('\',Name)=0) and
      (Pos(':',Name)=0) Then  { filename contains no path -> add program path }
   Begin
      Name := FSearch (Name, ProgramPath+';'+GetEnv('PATH'));  {RQ 1.6-1.7}
      Name := FExpand (Name);
   End;
End;


procedure ReadConfigFile (name: string);
var d: dirstr;
    e: extstr;
    p: PKCBuffer;
    f: file;
    count: word;  {counter}
    totalsize: word;  {total size read}
begin
     { 1.- Find file in path and paramstr(0) }
     ExpandFileName (name);

     { 2.- Store the file name for information }
     fsplit (name, d, NewFileName, e);
     

     { 3.- Get the file information }
     p := CurrentLayout;
     assign (f,name);
     s := name;                 { in case of error }
     count := 0;

     {$I-}
     Reset(f,1);
     {$I+}
     If IOResult<>0 Then error (4);
     repeat
         BlockRead (f,p^[count],128,totalsize);
         count := count+totalsize;
     until (totalsize<128) or eof(f);
     close (f);
     WriteLn (STR1_3,name);
     KCBlockSize := count;
end;

{***********************************************************
 Main program & Check for already installed driver.
 ***********************************************************}

(**
Type   BCDString   = String[2];

Function BCD(B : Byte) : BCDString;         { Translate BCD number into string. }
Begin
   If B>15 Then BCD:=Char(B shr 4 + 48)+Char(B and 15 + 48)
           Else BCD:=Char(B+48);
End;


Function VS(W : Word) : String;             { Returns version number as string. }
Var    S           : String;
Begin
   S:=BCD(LO(W));                           { Turn second value into string. }
   If Length(S)<2 Then S:='0'+S;            { Eventually add leading zero. }
   VS:=BCD(Hi(W))+'.'+S;                    { Add first value and point. }
End;
**)

Function TestInstallation : Byte;
{ Check whether a copy of FD-Keyb is already installed. }
{ Result:   0 -> No keyboard driver installed. }
{           1 -> Identical version of FD-Keyb installed. }
{                >> CurrentLayout will be set to data region of resident copy. }
{           2 -> Different version of FD-Keyb installed. }
{           3 -> Different keyboard driver installed. }
label  NoKeyb, OtherKeyb, OtherVersion, EndTest;
var    r: byte;
begin

  asm
      mov  ax, $AD80
      xor  cx,cx
      mov  bx,cx
      int  $2F

      cmp  al, $FF      {=== Part 1: any-KEYB? ==}
      jne  NoKeyb

      mov  ax,1

      test bx,$FFFF     {=== Part 2: which KEYB? ==}
      jnz  OtherKeyb
      test cx,$FFFF
      jnz  OtherKeyb

      cmp  dx,Version   {=== Part 3: which version? ==}
      jne  OtherVersion

      mov  [ds:offset CurrentLayout],di
      mov  [ds:offset CurrentLayout+2],es
      jmp  endTest      { ax=1 already! }

OtherKeyb:
      inc  ax

OtherVersion:
      inc  ax
      jmp  endTest
      
NoKeyb:
      xor  ax,ax

endTest:
      mov r, al
  end;
  TestInstallation := r
end;


Procedure Remove;                           { Remove Keyb from memory. }
Var    IntVec      : Array[Byte] of Pointer absolute 0:0;
       MultiHand   : Pointer;
       Int16Hand   : Pointer;
       Int9Hand    : Pointer;
       Int15Hand   : Pointer;
       s           : word;
Begin
{ Check whether removing is possible. }

   MultiHand:=CurrentLayout;
   ptrrec(MultiHand).Ofs := Ofs(Int2Fh);
   Int16Hand:=CurrentLayout;
   ptrrec(Int16Hand).Ofs := Ofs(Int16h);
   Int9Hand :=CurrentLayout;
   ptrrec(Int9Hand).Ofs  := Ofs(Int9h);
   Int15Hand :=CurrentLayout;
   ptrrec(Int15Hand).Ofs := Ofs(Int15h);

   if not (
         ((IntVec[$09]=Int9Hand) or (not assigned(OldInt9h))) and
         ((IntVec[$16]=Int16Hand) or (CurrentLayout^[8]=0)) and  {LastXStr}
         (IntVec[$2F]=MultiHand) and
         (IntVec[$15]=Int15Hand) )

   Then Error(5);      { Removing impossible. Interrupt vectors were changed by another program. }

{ Uninstall it. }
      if assigned (OldInt9h) then SetIntVec ($9, OldInt9h);
      if CurrentLayout^[8]>0 then SetIntVec ($16, OldInt16h);
      SetIntVec ($15, OldInt15h);
      SetIntVec ($2F, OldInt2Fh);


      s := PtrRec(Int15Hand).seg-16;           { Program segment. }

      asm
         mov  ah,$49
         mov  es,s
         int $21
      end;

   Writeln(STR1_4);
   Halt (0);
End;

Procedure ShowInfo;
Begin
      Writeln(STR1_5,NewFileName);
      Writeln(STR1_6,CurrentLayout^[8]);

      Halt (0);
End;


procedure ShowFastHelp;
begin
    Writeln (STR3_1);
    WriteLn (STR3_2);
    WriteLn ('(c) Aitor Santamara  - 2003');
    WriteLn;
    WriteLn (STR3_3);
    WriteLn ('KEYB');
    WriteLn ('KEYB  /U');
    WriteLn ('KEYB  /?');
    WriteLn;
    WriteLn (STR3_4);
    WriteLn (STR3_5);
    WriteLn (STR3_6);
    WriteLn (STR3_7);
    Writeln (STR3_8);
    WriteLn (STR3_9);
    WriteLn (STR3_10);
    WriteLn (STR3_11);
    WriteLn (STR3_12);
    WriteLn (STR3_13);
    WriteLn;
    WriteLn (STR3_14);
    halt (0)
end;




{obtain the minimum neccessary module from the KC}
function  GetLastModule : byte;
type
   pWord = ^word;
var
   pw : pWord;
   i,b: byte;
   w  : word;
begin
     pW := pword(CurrentLayout);
     b  := 0;

     for i:=0 to 11 do begin
        w := 1 shl i;
        if (w and pW^)>0 then b := i+1;
     end;

     GetLastModule := b
end;

label NoMoreMod;


Var
     CutHere     : pointer;        { pointer where to cut }

     lastXstr    : byte;           { last defined XString }
     isAT        : boolean;        { do we have an AT keyboard? }
     Installed   : Byte;           { existing KEYB status }
     ConFileName : String;         { name of the configuration file }
     Int9Type    : byte;           { 0=none, 1=chain, 2=extended }
     LastModule  : byte;           { last module to be installed }

     SChar       : char;           { switchar variables }
     switches    : string;

     b           : byte;
     pb          : pByte;
     pp          : ^pointer;
     q           : PKCBuffer;

Begin

   {********** MAIN ENTRY POINT **********}

   {--- 1.- Test self-integrity ---}

   If Ofs(EoDS)>Ofs(CallKEYBFunc) Then Error(6);        {this should never happen}

   {--- 2.- Welcome message ---}

   WriteLn ('KEYB ',VerS, STR1_1);

   {--- 3.- Initialise some data ---}
   LastModule := 0;
   
   CurrentLayout := @KCBuffer(@Data^);     { Pointer to Layout information }
                                           { temporary buffer, later moved }


   Installed     := TestInstallation;      { check if installed }
   {obtain data segment (old style, to be improved) }
   If Installed=1 Then                      { If already installed get certain values from resident GlobalDS. }
      Move(Ptr(ptrrec(CurrentLayout).Seg , Ofs(OldInt9h))^,OldInt9h,41);

   ConFileName   := '';                    { no configuration file yet }

   asm                                     { admissible switches: }
      mov  ax,$3700
      int  $21
      mov  SChar,dl
   end;                                    { DOS system switch AND }
   switches := '/-'+SChar;                 { / AND - }

   {--- 3.- Parse commandline ---}

   if ParamCount=0 then begin                       { obtain info that was }
        If Installed=1 Then ShowInfo Else Error(3); { read in the GlobalDS }
   end else                                         { (no return) }
   for b:=1 to ParamCount do begin
       s := ParamStr (b);
       if pos(s[1],switches)>0 then begin             { we are in a switch }
          if length(s)=2 then                          { one letter switch }
             case upcase(s[2]) of
                                 { IMMEDIATE SWITCHES (no return, program exits) }
                                 'U': If Installed=1 Then Remove Else Error(3);
                                 '?': ShowFastHelp;

                                 { OTHER SWITCHES }
                                 'I': Installed := 0; {force: ignore the other}
                                 'E': IsAT := TRUE; { user has an AT, says }

                                 else Error (11);
             end
          else case upcase(s[2]) of        { switch with more than 1 letter }
                                 '9': if length(s)>3 then error (11)
                                      else case s[3] of
                                                       '+': Int9Type := 2;
                                                       '*': Int9Type := 1;
                                                       '-': Int9Type := 0;
                                                       else error (11)
                                           end;
                                 'D': if length(s)>3 then error (11)
                                      else begin
                                           DecimalChar := s[3];
                                           LastModule  := maxb (LastModule,3);
                                      end
                                 else error (11)
               end
       end else begin                   { case of a layout }
           if ConFileName<>'' then error (11);   {in fact, a too many files }
           ConFileName := s
       end
   end;

   { if we are here, it means we proceed to install }

   case Installed of
                    1: error(9);        { no overloading (YET) }
                    2: error(1);        { wrong version of FD-KEYB }
                    3: error(2);        { other driver was installed }
   end;
   
   ReadConfigFile (ConFileName);


   {********** INSTALL, PREPARE ROUTINES **********}
   { if we are here, we have to install }

   {--- 1.- Variable initialisation ---}
   isAT          := Mem[$F000:$FFFE] < $FD;
   Int9Type      := 0;
   CutHere       := NIL;

   {--- 2.- Compute last module ---}
   if not isAT and (int9type=0) then int9type := 1;

   if (int9Type=2) and (GetLastModule<11) then Error (10);  {/9+ only with FULL layouts}

   case int9type of
                   1: LastModule := maxb (LastModule, 7);
                   2: LastModule := 11
   end;

   LastModule := maxb (GetLastModule,LastModule);

   LastModule := maxb (LastModule,3); {TEMPORARY: we ALWAYS process basic table}
   { when we implement how to avoid processing the default table, we'll remove }
   { this line }

   if CurrentLayout^[8]>0 then maxb (LastModule,10);  { XStrings }

   WriteLn (STR1_2, LastModule);

   {--- 3.- Compute the core functions ---}

   if isAT then BIOSStoreKey := EfStoreKey0      { BIOSStoreKey }
           else BIOSStoreKey := EfStoreKey1;

   GetIntVec($2F,OldInt2Fh);                     { old 15h and 2Fh vectors }
   GetIntVec($15,OldInt15h);

   PIsActive  := @VIsActive;

   {--- 4.- Compute last module to be discarded ---}

   if LastModule>0 then begin

      {== MODULE 1 ==}
      CombineProc  := DoCombi;
      ExecCombi    := StartCombi;
      CutHere      := @BasicCommand;
      if LastModule<2 then goto NoMoreMod;

      {== MODULE 3 ==}
      DoCommands   := BasicCommand;
      CutHere      := @ExtCommand;
      if LastModule<4 then goto NoMoreMod;

      {== MODULE 4 ==}
      ExtCommands  := ExtCommand;
      CutHere      := @EfStoreKey1;
      if LastModule<5 then goto NoMoreMod;

      {== MODULE 7 ==}
      if int9type=0 then int9Type := 1;
      GetIntVec($9 ,OldInt9h);
      PIsActive := Ptr (CSeg, ofs(Int9H)+3);
      CutHere      := @APMProc;
      if LastModule<8 then goto NoMoreMod;

      {== MODULE 8 ==}
      APMProcedure := APMProc;
      CutHere      := @TranslateScancode;
      if LastModule<9 then goto NoMoreMod;

      {== MODULE 9 ==}
      TranslateK   := TranslateScanCode;
      CutHere      := @XStrData;
      if lastmodule<10 then goto NoMoreMod;

      {== MODULE 10 ==}
      pp           := @XStrData;                      { set XBuffer }
      pp^          := Ptr(PrefixSeg,128);

      GetIntVec    ($16,OldInt16h);
      BIOSStoreKey := EfStoreKey2;

      CutHere      := @ProcessPause;
      if lastmodule<11 then goto NoMoreMod;
      
      {== MODULE 11 ==}
      Move ( ShiftKeys, Ptr(CSeg, ofs(ExtInt9Data)+1)^, 8);  { shiftkeys }
      pb := ptr (CSeg, ofs(ExtInt9Data));                    { looprunning }
      pb^ := 0;
      ExtInt9Proc := ExtInt9h;
      CutHere     := NIL;
   
   end;
NoMoreMod:

   {--- 5.- Make driver active ---}

   pb      := PIsActive;
   pb^     := 1;       {make the driver active}


   {--- 6.- Move all the data ---}

   lastXstr := CurrentLayout^[8];
   if assigned (CutHere) then begin             { Layout block }
       Move ( pbyte(@data)^, pbyte(CutHere)^, KCBlockSize );
       CurrentLayout := CutHere;
   end;

   Move(Ptr(DSeg,0)^,Ptr(CSeg,0)^,Ofs(EoDS));   { GlobalDS }

   {--- 7.- Set our vectors ---}

   if lastXStr>0 then SetIntVec($16,@Int16h);
   if int9Type>0 then SetIntVec($9, @Int9H);
   SetIntVec  ($15,@Int15h);
   SetIntVec  ($2F,@Int2Fh);

   {--- 8.- Make it resident ---}
   q := PKCBuffer(CurrentLayout);
   Keep (@q^[KCBlockSize]);

   {--- 9.- Data procedures, not to be removed by the linker ---}
   GlobalDS;
   ExtInt9Data;
   XStrData;
   DefaultTable;
END.

