/*
 *      format.c
 *
 *      FORMAT: a generic DOS disk-formatting utility
 *      Copyright (c) 1987-96, Robert Nordier
 *      All rights reserved
 *
 *      Written by Robert Nordier   Mar 1996
 *
 *      Redistribution and use in source and binary forms are freely
 *      permitted provided that the above copyright notice and
 *      attribution and date of work and this paragraph are duplicated
 *      in all such forms.
 *
 *      This software is provided "AS IS" and without any express or
 *      implied warranties, including, without limitation, the implied
 *      warranties of merchantability and fitness for a particular
 *      purpose.
 */

/*
 *      Revision history:
 *
 *      0.8   Robert Nordier   31 Mar 1996
 *            Beta release.
 */

#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>

char sccsid[] = "@(#)format.c 0.8 (rnordier) 96/3/31";

static char syntax[] =
   "FORMAT: a generic DOS disk-formatting utility\n"
   "Copyright (c) 1987-96, Robert Nordier\n\n"
   "Formats floppy disks and hard drive partitions for use with DOS\n\n"
   "FORMAT drive: [options]\n\n"
   "  /8          Format eight sectors per track\n"
   "  /1          Format a single side of a floppy disk\n"
   "  /4          Format a 360K floppy disk in a 1.2M drive\n"
   "  /T:tracks   Specify the number of tracks per disk\n"
   "  /N:sectors  Specify the number of sectors per track\n"
   "  /F:size     Specify the floppy disk format by capacity, eg:\n"
   "              160K, 180K, 320K, 360K, 720K, 1.2M, 1.44M, 2.88M\n"
   "  /V:label    Specify the volume label\n\n"
   "This free software is supplied with absolutely no warranty.\n"
   "Refer to the documentation for conditions of use and distribution.";

#define FAR _far
#define HUGE _huge
#define ASM _asm

#define BYTE unsigned char
#define WORD unsigned short
#define DWORD unsigned long

struct BPB {
   WORD secsiz;                 /* sector size */
   BYTE spc;                    /* sectors per cluster */
   WORD ressec;                 /* reserved sectors */
   BYTE fats;                   /* FATs */
   WORD dirents;                /* root directory entries */
   WORD secs;                   /* total sectors */
   BYTE media;                  /* media descriptor */
   WORD spf;                    /* sectors per FAT */
   WORD spt;                    /* sectors per track */
   WORD heads;                  /* drive heads */
   DWORD hidsec;                /* hidden sectors */
   DWORD lsecs;                 /* huge sectors */
};

struct BOOTSECTOR {
   BYTE jmp[3];                 /* jmp xxxx or jmp short xxxx */
   char oem[8];                 /* OEM name and version */
   struct BPB bpb;              /* BIOS parameter block */
   BYTE drive;                  /* drive number */
   BYTE reserved;               /* reserved */
   BYTE extsig;                 /* extended boot signature */
   DWORD volid;                 /* volume ID */
   char label[11];              /* volume label */
   char fstype[8];              /* file system type */
};

struct DEVICEPARAMS {
   BYTE func;                   /* special functions */
   BYTE dev;                    /* device type */
   WORD attr;                   /* device attributes */
   WORD cyls;                   /* cylinders */
   BYTE media;                  /* media type */
   struct BPB bpb;              /* BPB */
};

struct FVBLOCK {
   BYTE func;                   /* special functions */
   WORD head;                   /* head */
   WORD cyl;                    /* cylinder */
   WORD trks;                   /* tracks */
};

struct RWBLOCK {
   BYTE func;                   /* special functions */
   WORD head;                   /* head */
   WORD cyl;                    /* cylinder */
   WORD sec;                    /* starting sector */
   WORD nsecs;                  /* number of sectors */
   void far *buf;               /* buffer */
};

struct DIRENTRY {
   BYTE name[8];                /* name */
   BYTE ext[3];                 /* extension */
   BYTE attr;                   /* attributes */
   BYTE reserved[10];           /* reserved */
   WORD time;                   /* time */
   WORD date;                   /* date */
   WORD clus;                   /* starting cluster */
   DWORD size;                  /* file size */
};

struct FILEINFO {
   BYTE reserved[21];           /* reserved */
   BYTE attr;                   /* attributes */
   WORD time;                   /* time */
   WORD date;                   /* date */
   DWORD size;                  /* file size */
   char name[13];               /* filename and extension */
};

struct DOSDATE {
   WORD y;                      /* year */
   BYTE d;                      /* day */
   BYTE m;                      /* month */
};

struct DOSTIME {
   BYTE m;                      /* minutes */
   BYTE h;                      /* hours */
   BYTE f;                      /* hundredths */
   BYTE s;                      /* seconds */
};

#define FA_HIDDEN  0x02         /* file attribute: hidden */
#define FA_SYSTEM  0x04         /* file attribute: system */
#define FA_ARCH    0x20         /* file attribute: archive */
#define FA_LABEL   0x08         /* file attribute: label */

#define DOSMINVER  0x031e       /* minimum DOS version */
#define SECSIZ     0x200        /* sector size */
#define MAXCYLS    0x400        /* maximum cylinders */
#define MAXSECS    0x40         /* maximum sectors per track */
#define MAGICOFS   0x1fe        /* magic number offset */
#define DOSMAGIC   0xaa55       /* DOS magic number */
#define BAD12      0xff7        /* bad cluster: 12-bit FAT */
#define BAD16      0xfff7       /* bad cluster: 16-bit FAT */
#define TKLOFS     0x26         /* track layout offset */
#define TKLSIZ     (2 + 4 * MAXSECS)   /* track layout size */
#define EPOCH      1980         /* base year for DOS dates */
#define SWITCHAR   '/'          /* switch character */
#define LABELEN    11           /* volume label length */
#define FSTYPELEN  8            /* fstype length */

#define IOSETDEV   0x40         /* IOCTL: set device parameters */
#define IOWRITE    0x41         /* IOCTL: write track */
#define IOFORMAT   0x42         /* IOCTL: format/verify track */
#define IOGETDEV   0x60         /* IOCTL: get device parameters */
#define IOREAD     0x61         /* IOCTL: read track */

#define DEV4009    0            /* 5.25-inch 360K drive */
#define DEV8015    1            /* 5.25-inch 1.2M drive */
#define DEV8009    2            /* 3.5-inch 720K drive */
#define DEVHARD    5            /* hard drive */
#define DEV8018    7            /* 3.5-inch 1.44M drive */
#define DEV8036    9            /* 3.5-inch 2.88M drive */

#define XE_WRTPRO  0x13         /* extended error: write-protected */
#define XE_NOTRDY  0x15         /* extended error: not ready */
#define XE_CRCERR  0x17         /* extended error: CRC error */
#define XE_SECNOT  0x1b         /* extended error: sector not found */
#define XE_WRITE   0x1d         /* extended error: write fault */
#define XE_READ    0x1e         /* extended error: read fault */
#define XE_GFAIL   0x1f         /* extended error: general failure */

#define X_INT      3            /* exit code: interrupted */
#define X_ERR      4            /* exit code: error */
#define X_USR      5            /* exit code: user cancelled */

#define ERRARG     0xffff       /* unsigned error result */

#define OPT_8      0x01
#define OPT_1      0x02
#define OPT_4      0x04
#define OPT_N      0x08
#define OPT_T      0x10
#define OPT_F      0x20
#define OPT_V      0x40

#define OPT_81     (OPT_8 | OPT_1)
#define OPT_14     (OPT_1 | OPT_4)
#define OPT_814    (OPT_8 | OPT_14)
#define OPT_NT     (OPT_N | OPT_T)
#define OPTARG     (OPT_NT | OPT_F | OPT_V)
#define OPTFMT     (OPT_814 | OPT_NT | OPT_F)

#define K_ARGS     8
#define M_ARGS     3
#define F_ARGS     (K_ARGS + M_ARGS)

static char *f_args[F_ARGS] = {
   "160", "180", "320", "360",        /* K_ARGS */
   "720", "1200", "1440", "2880",
   "1.2", "1.44", "2.88"              /* M_ARGS */
};

#define BPBCNT     8

static struct BPB stdbpb[BPBCNT] = {
   {512, 1, 1, 2,  64,  320, 0xfe, 1,  8, 1, 0, 0},   /* 160K */
   {512, 1, 1, 2,  64,  360, 0xfc, 2,  9, 1, 0, 0},   /* 180K */
   {512, 2, 1, 2, 112,  640, 0xff, 1,  8, 2, 0, 0},   /* 320K */
   {512, 2, 1, 2, 112,  720, 0xfd, 2,  9, 2, 0, 0},   /* 360K */
   {512, 2, 1, 2, 112, 1440, 0xf9, 3,  9, 2, 0, 0},   /* 720K */
   {512, 1, 1, 2, 224, 2400, 0xf9, 7, 15, 2, 0, 0},   /* 1.2M */
   {512, 1, 1, 2, 224, 2880, 0xf0, 9, 18, 2, 0, 0},   /* 1.44M */
   {512, 2, 1, 2, 240, 5760, 0xf0, 9, 36, 2, 0, 0}    /* 2.88M */
};

static unsigned char bootcode[] = {
   0xfa,                 /* cli              */
   0x31, 0xc0,           /* xor     ax,ax    */
   0x8e, 0xd0,           /* mov     ss,ax    */
   0xbc, 0x00, 0x7c,     /* mov     sp,7c00h */
   0xfb,                 /* sti              */
   0x8e, 0xd8,           /* mov     ds,ax    */
   0xe8, 0x00, 0x00,     /* call    $ + 3    */
   0x5e,                 /* pop     si       */
   0x83, 0xc6, 0x19,     /* add     si,+19h  */
   0xbb, 0x07, 0x00,     /* mov     bx,0007h */
   0xfc,                 /* cld              */
   0xac,                 /* lodsb            */
   0x84, 0xc0,           /* test    al,al    */
   0x74, 0x06,           /* jz      $ + 8    */
   0xb4, 0x0e,           /* mov     ah,0eh   */
   0xcd, 0x10,           /* int     10h      */
   0xeb, 0xf5,           /* jmp     $ - 9    */
   0x30, 0xe4,           /* xor     ah,ah    */
   0xcd, 0x16,           /* int     16h      */
   0xcd, 0x19,           /* int     19h      */
   0x0d, 0x0a,
   'N', 'o', 'n', '-', 'S', 'y', 's', 't',
   'e', 'm', ' ', 'd', 'i', 's', 'k',
   0x0d,  0x0a,
   'P', 'r', 'e', 's', 's', ' ', 'a', 'n',
   'y', ' ', 'k', 'e', 'y', ' ', 't', 'o',
   ' ', 'r', 'e', 'b', 'o', 'o', 't',
   0x0d,  0x0a,
   0
};

#define E_DOSVER    1
#define E_SYNTAX    2
#define E_BADARG    3
#define E_DEFDRV    4
#define E_INVDRV    5
#define E_NETDRV    6
#define E_SUBDRV    7
#define E_NOMEM     8
#define E_GIOCTL    9
#define E_HDDOPT   10
#define E_FMTOPT   11
#define E_DRVOPT   12
#define E_VOLOPT   13
#define E_LABEL    14
#define E_NOTRDY   15
#define E_WRTPRO   16
#define E_SYSTRK   17
#define E_IOBOOT   18
#define E_IOFATS   19
#define E_IOROOT   20

static char *errmsg[] = {
   "Internal error",
   "This program requires DOS 3.3 or higher",
   syntax,
   "Invalid parameter",
   "Drive must be specified",
   "Invalid drive specification",
   "Cannot format a network drive",
   "Cannot format an ASSIGNed or SUBSTed drive",
   "Insufficient memory",
   "Error in IOCTL call",
   "Parameters not compatible with fixed drive",
   "Parameters not compatible",
   "Parameters not supported by drive",
   "Volume label not supported for this format",
   "Invalid volume label",
   "Drive not ready",
   "Attempted write-protect violation",
   "Invalid media or bad track zero",
   "Unable to write BOOT",
   "Error writing FAT",
   "Error writing directory"
};

struct ARG {
   int drv;                     /* drive */
   int opt;                     /* option flags */
   WORD n;                      /* /N argument */
   WORD t;                      /* /T argument */
   int f;                       /* /F argument */
   char *v;                     /* /V argument */
};

struct VAR {
   char HUGE *fat;              /* FAT under construction */
   char *buf;                   /* buffer for boot and root */
   DWORD totsec;                /* total sectors */
   DWORD volid;                 /* volume serial number */
   WORD dirsec;                 /* root directory sectors */
   WORD syssec;                 /* system sectors */
   WORD badcnt;                 /* bad cluster count */
   int fat12;                   /* 12-bit FAT */
};

struct LH {
   WORD l;                      /* DWORD low word */
   WORD h;                      /* DWORD high word */
};

static jmp_buf jmpbuf;

static void ctrlc(int sig);
static int getargs(int argc, char *argv[], struct ARG *arg);
static int chkdrv(int drv);
static int setfmt(struct ARG *arg, struct DEVICEPARAMS *dp);
static int format(struct ARG *arg, struct DEVICEPARAMS *dp);
static int format_tracks(struct ARG *arg, struct DEVICEPARAMS *dp,
                         struct VAR *v);
static int write_meta(struct ARG *arg, struct DEVICEPARAMS *dp,
                      struct VAR *v);
static int confirm(struct ARG *arg, struct DEVICEPARAMS *dp);
static char *get_label(void);
static DWORD make_volid(struct DOSDATE *dt, struct DOSTIME *tm);
static void make_label(struct DIRENTRY *de, const char *label,
                       struct DOSDATE *dt, struct DOSTIME *tm);
static int another(void);
static int dos1xx(struct DEVICEPARAMS *dp);
static int labelok(const char *label);
static int argfmt(char *s, char **endptr);
static void errout(int err);
static unsigned argopt(char *s, char **endptr, unsigned lo, unsigned hi);
static void hbufset(char HUGE *p, int c, unsigned long n);
static int absio(int drive, struct DEVICEPARAMS *dp, int op, WORD nsecs,
                 DWORD lsec, void FAR *buf);
static int dos_getver(void);
static int dos_remote(int drive, int *flags);
static int dos_truename(char *d, const char *s);
static int dos_gioctl(int func, int drive, void FAR *data);
static int dos_findfirst(const char *path, int attr,
                         struct FILEINFO *finfo);
static void dos_getdate(struct DOSDATE *dt);
static void dos_gettime(struct DOSTIME *tm);
static void dos_flush(void);

int main(int argc, char *argv[])
{
   struct ARG arg;
   struct DEVICEPARAMS *dp, def_dp;
   int retc, i;

   retc = X_ERR;
   arg.drv = arg.opt = 0;
   signal(SIGINT, SIG_IGN);
   if (dos_getver() < DOSMINVER)
      errout(E_DOSVER);
   else if ((i = getargs(argc, argv, &arg)) != 0)
      errout(i);
   else if ((i = chkdrv(arg.drv)) != 0)
      errout(i);
   else if ((dp = malloc(TKLOFS + TKLSIZ)) == NULL)
      errout(E_NOMEM);
   else {
      memset(dp, 0, TKLOFS + TKLSIZ);
      if (dos_gioctl(IOGETDEV, arg.drv, dp))
         errout(E_GIOCTL);
      else {
         def_dp = *dp;
         if ((i = setfmt(&arg, dp)) != 0)
            errout(i);
         else if (arg.opt & OPT_V && dos1xx(dp))
            errout(E_VOLOPT);
         else if (arg.opt & OPT_V && !labelok(arg.v))
            errout(E_LABEL);
         else {
            {
               WORD *tkl;
               WORD s;
               tkl = (WORD *)((char *)dp + TKLOFS);
               *tkl++ = dp->bpb.spt;
               for (s = 1; s <= dp->bpb.spt; s++) {
                  *tkl++ = s;
                  *tkl++ = SECSIZ;
               }
            }
            dp->func = 5;
            if (dos_gioctl(IOSETDEV, arg.drv, dp))
               errout(E_GIOCTL);
            else {
               if ((i = setjmp(jmpbuf)) == 0) {
                  signal(SIGINT, ctrlc);
                  retc = format(&arg, dp);
               }
               else
                  retc = X_INT;
               *dp = def_dp;
               *(WORD *)((char *)dp + TKLOFS) = 0;
               dp->func = 4;
               if (dos_gioctl(IOSETDEV, arg.drv, dp))
                  errout(E_GIOCTL);
            }
         }
      }
   }
   return retc;
}

static void ctrlc(int sig)
{
   signal(sig, SIG_IGN);
   longjmp(jmpbuf, 1);
}

static int getargs(int argc, char *argv[], struct ARG *arg)
{
   static char optstr[] = "814NTFV";
   char *p, *q;
   int i, x;

   for (i = 1; i < argc; i++) {
      p = argv[i];
      if (*p != SWITCHAR) {
         if (arg->drv || !isalpha(*p) || p[1] != ':')
            return E_BADARG;
         arg->drv = 1 + toupper(*p) - 'A';
         p += 2;
      }
      while (*p) {
         if (*p++ != SWITCHAR)
            return E_BADARG;
         if (*p == '?')
            return E_SYNTAX;
         if ((q = strchr(optstr, toupper(*p))) == NULL)
            return E_BADARG;
         p++;
         x = 1 << (q - optstr);
         if (x & OPTARG) {
            if (*p++ != ':')
               return E_BADARG;
            switch (x) {
               case OPT_N:
                  if ((arg->n = argopt(p, &q, 1, MAXSECS)) == ERRARG)
                     return E_BADARG;
                  break;
               case OPT_T:
                  if ((arg->t = argopt(p, &q, 1, MAXCYLS)) == ERRARG)
                     return E_BADARG;
                  break;
               case OPT_F:
                  if ((arg->f = argfmt(p, &q)) == -1)
                     return E_BADARG;
                  break;
               case OPT_V:
                  if (!*(arg->v = strupr(p)))
                     return E_BADARG;
                  q = strchr(p, 0);
            }
            p = q;
         }
         arg->opt |= x;
      }
   }
   return 0;
}

static int chkdrv(int drv)
{
   char str[6], buf[7];
   int i;

   if (!drv)
      return E_DEFDRV;
   if (dos_remote(drv, &i))
      return E_INVDRV;
   if (i & 0x1000)
      return E_NETDRV;
   if (i & 0x200)
      return E_SUBDRV;
   *strcpy(str, "A:CON") += (char)(drv - 1);
   return dos_truename(buf, str) || *buf != *str ? E_SUBDRV : 0;
}

static int setfmt(struct ARG *arg, struct DEVICEPARAMS *dp)
{
   int i;

   if ((i = arg->opt & OPTFMT) != 0) {
      if (dp->dev == DEVHARD)
         return E_HDDOPT;
      if (i == OPT_NT) {
         if (arg->t == dp->cyls && arg->n == dp->bpb.spt)
            return 0;
         for (i = 0; i < BPBCNT; i++)
            if (arg->t == (i <= 3  ? 40U : 80U) &&
                dp->bpb.heads == stdbpb[i].heads &&
                arg->n == stdbpb[i].spt)
               break;
         if (i == BPBCNT)
            return E_FMTOPT;
      }
      else {
         if (i <= OPT_814) {
            if (dp->dev == DEV8015 && (i & OPT_14) == OPT_1)
               return E_FMTOPT;
            i = 3 - (i & OPT_81);
         }
         else if (i == OPT_F)
            i = arg->f;
         else
            return E_FMTOPT;
         if ((i <= 3 ? 40U : 80U) == dp->cyls &&
             stdbpb[i].heads == dp->bpb.heads &&
             stdbpb[i].spt == dp->bpb.spt)
            return 0;
      }
      switch (dp->dev) {
         case DEV4009:
            if (i == 5)
               return E_DRVOPT;
            /* falls through */
         case DEV8015:
            if (i == 4 || i >= 6)
               return E_DRVOPT;
            break;
         case DEV8009:
            if (i == 6)
               return E_DRVOPT;
            /* falls through */
         case DEV8018:
            if (i == 7)
               return E_DRVOPT;
            /* falls through */
         case DEV8036:
            if (i <= 3 || i == 5)
               return E_DRVOPT;
            break;
         default:
            return E_DRVOPT;
      }
      if (dp->dev == DEV8015 && i <= 3) {
         dp->media = 1;
         dp->cyls = 40;
      }
      dp->bpb = stdbpb[i];
   }
   return 0;
}

static int format(struct ARG *arg, struct DEVICEPARAMS *dp)
{
   struct VAR v;
   DWORD fatsiz;
   WORD clusts;
   int i;

   v.totsec = dp->bpb.secs ? dp->bpb.secs : dp->bpb.lsecs;
   v.dirsec = sizeof(struct DIRENTRY) * dp->bpb.dirents / SECSIZ;
   v.syssec = dp->bpb.ressec + dp->bpb.spf * dp->bpb.fats + v.dirsec;
   clusts = (WORD)((v.totsec - v.syssec) / dp->bpb.spc);
   v.fat12  = clusts < 0xff6;
   if ((v.fat = _fmalloc(fatsiz = (DWORD)dp->bpb.spf * SECSIZ)) == NULL ||
       (v.buf = malloc(SECSIZ)) == NULL)
      return E_NOMEM;
   do {
      if (!confirm(arg, dp))
         return X_USR;
      hbufset(v.fat, 0, fatsiz);
      *v.fat = (char)dp->bpb.media;
      _fmemset(v.fat + 1, 0xff, v.fat12 ? 2 : 3);
      v.badcnt = 0;
      if ((i = format_tracks(arg, dp, &v)) == 0)
         i = write_meta(arg, dp, &v);
      if (i) {
         errout(i);
         if (i == E_BADARG)
            return X_ERR;
         printf("Format terminated\n\n");
      }
      else {
         printf("%10lu bytes total disk space\n",
                (DWORD)clusts * dp->bpb.spc * SECSIZ);
         if (v.badcnt)
            printf("%10lu bytes in bad sectors\n",
                   (DWORD)v.badcnt * dp->bpb.spc * SECSIZ);
         printf("%10lu bytes available on disk\n\n",
                (DWORD)(clusts - v.badcnt) * dp->bpb.spc * SECSIZ);
         printf("%10u bytes in each allocation unit\n",
                dp->bpb.spc * SECSIZ);
         printf("%10u allocation units available on disk\n\n",
                clusts - v.badcnt);
         if (!dos1xx(dp))
            printf("Volume Serial Number is %04X-%04X\n\n",
                   (WORD)(v.volid >> 16), (WORD)v.volid);
      }
   } while (dp->dev != DEVHARD && another());
   return i ? X_ERR : 0;
}

int format_tracks(struct ARG *arg, struct DEVICEPARAMS *dp, struct VAR *v)
{
   struct FVBLOCK fv;
   WORD HUGE *f;
   DWORD sec;
   WORD trkcnt, trk, c;
   int i;

   if (!dos1xx(dp)) {
      fv.func = 1;
      if (!dos_gioctl(IOFORMAT, arg->drv, &fv)) {
         if (fv.func == 2)
            return E_BADARG;
         if (fv.func == 3)
            return E_NOTRDY;
      }
   }
   trkcnt = (WORD)(v->totsec / dp->bpb.spt);
   fv.func = 0;
   sec = dp->bpb.hidsec / dp->bpb.spt;
   fv.cyl = (WORD)(sec / dp->bpb.heads);
   fv.head = (WORD)(sec % dp->bpb.heads);
   for (trk = 0; trk < trkcnt; trk++) {
      printf("Cylinder: %-4u Head: %u\r", fv.cyl, fv.head);
      if ((i = dos_gioctl(IOFORMAT, arg->drv, &fv)) != 0) {
         if (i == XE_WRTPRO)
            return E_WRTPRO;
         if (i == XE_NOTRDY)
            return E_NOTRDY;
         if (i != XE_CRCERR && i != XE_SECNOT && i != XE_WRITE &&
             i != XE_READ && i != XE_GFAIL)
            return E_GIOCTL;
         sec = trk * dp->bpb.spt;
         if (sec < v->syssec)
            return E_SYSTRK;
         sec -= v->syssec;
         for (c = (WORD)(2 + sec / dp->bpb.spc);
              c <= 2 + (sec + dp->bpb.spt - 1) / dp->bpb.spc; c++) {
            f = (WORD HUGE *)(v->fat + c + (v->fat12 ? c >> 1 : c));
            if (!(v->fat12 ? c & 1 ? *f >> 4 : *f & 0xfff : *f)) {
               *f = v->fat12 ? *f | (c & 1 ? BAD12 << 4 : BAD12) : BAD16;
               v->badcnt++;
            }
         }
      }
      if (++fv.head == dp->bpb.heads) {
         fv.cyl++;
         fv.head = 0;
      }
   }
   printf("%24cFormat complete\n\n", '\r');
   return 0;
}

int write_meta(struct ARG *arg, struct DEVICEPARAMS *dp, struct VAR *v)
{
   struct DOSDATE dt;
   struct DOSTIME tm;
   struct BOOTSECTOR *bs;
   struct DIRENTRY *de;
   char *label;
   unsigned i;

   if (!dos1xx(dp)) {
      if (arg->opt & OPT_V)
         label = arg->v;
      else
         for (;;) {
            printf("Volume label (11 characters, ENTER for none)? ");
            if (labelok(label = get_label()))
               break;
            errout(E_LABEL);
         }
      dos_getdate(&dt);
      dos_gettime(&tm);
   }
   memset(v->buf, 0, SECSIZ);
   bs = (struct BOOTSECTOR *)v->buf;
   bs->jmp[0] = 0xeb;
   bs->jmp[2] = 0x90;
   if (dos1xx(dp)) {
      bs->jmp[1] = 0x27;
      if (dp->bpb.heads == 1)
         bs->oem[0] = 0x08;
      else {
         bs->oem[0] = 0x03;
         bs->oem[1] = 0x01;
      }
      bs->oem[2] = 0x14;
   }
   else {
      bs->jmp[1] = 0x3c;
      memcpy(bs->oem, "MSDOS5.0", 8);
      bs->extsig = 0x29;
      bs->volid = v->volid = make_volid(&dt, &tm);
      memset(bs->label, 0x20, LABELEN + FSTYPELEN);
      if ((i = strlen(label)) != 0)
         memcpy(bs->label, label, i);
      else
         memcpy(bs->label, "NO NAME", 7);
      memcpy(bs->fstype, v->fat12 ? "FAT12" : "FAT16", 5);
      *(WORD *)(v->buf + MAGICOFS) = DOSMAGIC;
   }
   bs->bpb = dp->bpb;
   memcpy(v->buf + 2 + bs->jmp[1], bootcode, sizeof(bootcode));
   if (absio(arg->drv, dp, IOWRITE, 1, 0, v->buf))
      return E_IOBOOT;
   for (i = 0; i < dp->bpb.fats; i++)
      if (absio(arg->drv, dp, IOWRITE, dp->bpb.spf,
                dp->bpb.ressec + dp->bpb.spf * i, v->fat))
         return E_IOFATS;
   memset(v->buf, 0, SECSIZ);
   de = (struct DIRENTRY *)v->buf;
   if (dos1xx(dp))
      de->attr = (de + 1)->attr = FA_HIDDEN | FA_SYSTEM;
   else if (*label)
      make_label(de, label, &dt, &tm);
   for (i = 0; i < v->dirsec; i++) {
      if (absio(arg->drv, dp, IOWRITE, 1, v->syssec - v->dirsec + i,
                v->buf))
         return E_IOROOT;
      if (!i)
         memset(v->buf, 0, SECSIZ);
   }
   return 0;
}

static int confirm(struct ARG *arg, struct DEVICEPARAMS *dp)
{
   static char *prompt[2] = {
      "Insert new diskette for drive %c:\n"
      "and press ENTER when ready",
      "\n"
      "WARNING: ALL DATA ON NON-REMOVABLE DISK\n"
      "DRIVE %c: WILL BE LOST!\n"
      "Proceed with format (Y/N)?"
   };
   struct FILEINFO fi;
   char buf[SECSIZ];
   size_t i;
   int c;

   dos_flush();
   printf(prompt[dp->dev == DEVHARD], 'A' + arg->drv - 1);
   c = getchar();
   fflush(stdin);
   putchar('\n');
   if (dp->dev != DEVHARD)
      return 1;
   if (toupper(c) != 'Y')
      return 0;
   if (absio(arg->drv, dp, IOREAD, 1, 0, buf) ||
       *(WORD *)(buf + MAGICOFS) != DOSMAGIC)
      return 1;
   *strcpy(buf, "A:\\*.*") += (char)(arg->drv - 1);
   if (dos_findfirst(buf, FA_LABEL, &fi))
      return 1;
   if ((i = strlen(fi.name)) > 8)
      memcpy(buf + 8, fi.name + 9, i - 8);
   if (!labelok(memcpy(buf, fi.name, i > 8 ? 8 : i + 1)))
      return 1;
   printf("Enter current volume label for drive %c: ", 'A' + arg->drv - 1);
   if ((c = strcmp(get_label(), buf)) != 0)
      printf("Incorrect volume label\n");
   return !c;
}

static char *get_label(void)
{
   static char buf[LABELEN + 2];
   char *p;

   dos_flush();
   fgets(buf, sizeof(buf), stdin);
   fflush(stdin);
   putchar('\n');
   if ((p = strchr(buf, '\n')) == NULL)
      p = buf + LABELEN;
   do
      *p = 0;
   while (p > buf && *--p == ' ');
   return strupr(buf);
}

static DWORD make_volid(struct DOSDATE *dt, struct DOSTIME *tm)
{
   return ((DWORD)(((struct LH *)dt)->h + ((struct LH *)tm)->h) << 16) |
          ((struct LH *)dt)->l + ((struct LH *)tm)->l;
}

static void make_label(struct DIRENTRY *de, const char *label,
                       struct DOSDATE *dt, struct DOSTIME *tm)
{
   memcpy(de->name, label, strlen(label));
   de->attr = FA_ARCH | FA_LABEL;
   de->time = (tm->h << 11) | (tm->m << 5) | (tm->s >> 1);
   de->date = ((dt->y - EPOCH) << 9) | (dt->m << 5) | dt->d;
}

static int another(void)
{
   int c;

   dos_flush();
   printf("Format another (Y/N)?");
   c = getchar();
   fflush(stdin);
   putchar('\n');
   return toupper(c) == 'Y';
}

static int dos1xx(struct DEVICEPARAMS *dp)
{
   return dp->dev <= DEV8015 && dp->cyls == 40 && dp->bpb.spt == 8;
}

static int labelok(const char *label)
{
   int i, c;

   for (i = 0; i < LABELEN && (c = label[i]) != 0; i++)
      if (!(c & 0x80) && (c < 0x20 || islower(c) ||
                          strchr("\"*+,./:;<=>?[\\]|", c)))
         return 0;
   return !c;
}

static int argfmt(char *s, char **endptr)
{
   int i;

   for (i = 0; i < F_ARGS; i++)
      if (!strncmp(s, f_args[i], strlen(f_args[i]))) {
         s += strlen(f_args[i]);
         if (toupper(*s) == (i < K_ARGS ? 'K' : 'M')) {
            s++;
            if (toupper(*s) == 'B')
               s++;
         }
         break;
      }
   *endptr = s;
   return i < F_ARGS ? i < K_ARGS ? i : i - M_ARGS : -1;
}

static void errout(int err)
{
   fprintf(stderr, "%24c%s\n", '\r', errmsg[err]);
}

static unsigned argopt(char *s, char **endptr, unsigned lo, unsigned hi)
{
   unsigned long lu;

   lu = strtoul(s, endptr, 10);
   return *endptr > s && lu >= lo && lu <= hi ? (unsigned)lu : ERRARG;
}

static void hbufset(char HUGE *p, int c, unsigned long n)
{
   size_t i;

   for (; (i = n > 0xfff0 ? 0xfff0 : (size_t)n) != 0; p += i, n -= i)
      _fmemset(p, c, i);
}

static int absio(int drive, struct DEVICEPARAMS *dp, int op, WORD nsecs,
                 DWORD lsec, void FAR *buf)
{
   struct RWBLOCK rw;
   char HUGE *p;
   DWORD s;
   WORD n;
   int i;

   for (p = buf; (n = nsecs < 16 ? nsecs : 16) != 0;
        p += n * SECSIZ, lsec += n, nsecs -= n) {
      rw.func = 0;
      s = dp->bpb.hidsec + lsec;
      rw.sec = (WORD)(s % dp->bpb.spt);
      s /= dp->bpb.spt;
      rw.head = (WORD)(s % dp->bpb.heads);
      rw.cyl = (WORD)(s / dp->bpb.heads);
      rw.nsecs = n;
      rw.buf = p;
      if ((i = dos_gioctl(op, drive, &rw)) != 0)
         return i;
   }
   return 0;
}

static int dos_getver(void)
{
   int ver;

   ASM  mov ax,3000h
   ASM  int 21h
   ASM  xchg ah,al
   ASM  mov ver,ax
   return ver;
}

static int dos_remote(int drive, int *flags)
{
   ASM  mov bx,drive
   ASM  mov ax,4409h
   ASM  int 21h
   ASM  jc done
   ASM  mov drive,dx
   *flags = drive;
   ASM  xor ax,ax
done:
   ASM  mov drive,ax
   return drive;
}

static int dos_truename(char *d, const char *s)
{
   int err;

   ASM  push ds
   ASM  pop es
   ASM  mov si,s
   ASM  mov di,d
   ASM  mov ah,60h
   ASM  int 21h
   ASM  jc done
   ASM  xor ax,ax
done:
   ASM  mov err,ax
   return err;
}

static int dos_gioctl(int func, int drive, void FAR *data)
{
   ASM  push ds
   ASM  mov bx,drive
   ASM  mov cl,byte ptr func
   ASM  mov ch,08h
   ASM  lds dx,data
   ASM  mov ax,440dh
   ASM  int 21h
   ASM  mov ax,0
   ASM  jnc done
   ASM  xor bx,bx
   ASM  push bp
   ASM  push di
   ASM  push si
   ASM  mov ah,59h
   ASM  int 21h
   ASM  pop si
   ASM  pop di
   ASM  pop bp
done:
   ASM  mov func,ax
   ASM  pop ds
   return func;
}

static int dos_findfirst(const char *path, int attr,
                         struct FILEINFO *finfo)
{
   int err;

   ASM  push ds
   ASM  mov ah,2fh
   ASM  int 21h
   ASM  push es
   ASM  push bx
   ASM  mov dx,finfo
   ASM  mov ah,1ah
   ASM  int 21h
   ASM  mov dx,path
   ASM  mov cx,attr
   ASM  mov ah,4eh
   ASM  int 21h
   ASM  jc done
   ASM  xor ax,ax
done:
   ASM  mov err,ax
   ASM  pop dx
   ASM  pop ds
   ASM  mov ah,1ah
   ASM  int 21h
   ASM  pop ds
   return err;
}

static void dos_getdate(struct DOSDATE *dt)
{
   ASM  mov ah,2ah
   ASM  int 21h
   ASM  mov bx,dt
   ASM  mov [bx],cx
   ASM  mov [bx+2],dx
}

static void dos_gettime(struct DOSTIME *tm)
{
   ASM  mov ah,2ch
   ASM  int 21h
   ASM  mov bx,tm
   ASM  mov [bx],cx
   ASM  mov [bx+2],dx
}

static void dos_flush(void)
{
   ASM  mov ax,0c00h
   ASM  int 21h
}

/* end of format.c */
