/* Standard C version of fdisk :-( That was the hope anyway.. */
/* This is a read only version. The code is released under the GNU license.
   Go to the GNU site for a full version of it. As long as you use the code,
   whatever you do with it has to be free and the code for that available to
   the whole world.

   Robert Sandilands (robert@vps.co.za)
   */

// #define __INT13_48_DEBUGGING__
// #define __REAL_PARTITION_INFO__
/* __REAL_PARTITION_INFO__ really is junk... It was based on an idea I had
   that you can easily handle disks without LBA and more than 1024 tracks.
   */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>
#include <conio.h>

/***************************************************************************/
/* 		Global Variables					   */
/***************************************************************************/
extern unsigned _stklen=32*124L;
/* Borland C 3.1 specific thing to help stop stack overflows. */

/***************************************************************************/
/* 		Constants - should be in a .h				   */
/***************************************************************************/
#define FDISK_MAX_PHYS_DISKS 0x10
/* Number of physical disks to check for... */

/* Partition magic numbers */
#define PART_EMPTY		0x00
#define PART_DOS12		0x01
#define PART_XENIXROOT		0x02
#define PART_XENIXUSR		0x03
#define PART_DOS16SMALL		0x04
#define PART_EXTENDED		0x05
#define PART_DOS16		0x06
#define PART_OS2HPFS		0x07
#define PART_AIX		0x08
#define PART_AIXBOOT		0x09
#define PART_OS2BOOTMANAGE      0x0A
#define PART_FAT32		0x0B
#define PART_FAT32EXTENDED	0x0C
#define PART_VENIX		0x40
#define PART_NOVELL		0x51
#define PART_MICROPORT		0x52
#define PART_GNUHURD            0x63
#define PART_NOVELL2		0x64
#define PART_NOVELL3		0x65
#define PART_PCIX		0x75
#define PART_OLDMINUX		0x80
#define PART_LINUXMINUX		0x81
#define PART_LINUXSWAP		0x82
#define PART_LINUXNATIVE	0x83
#define PART_AMOEBA		0x93
#define PART_AMOEBA_BBT		0x94
#define PART_BSD386		0xA5
#define PART_BSDIFS		0xB7
#define PART_BSDISWAP		0xB8
#define PART_SYRINX		0xC7
#define PART_CPM		0xDB
#define PART_DOSACCESS		0xE1
#define PART_DOSRO		0xE3
#define PART_SECOND		0xF2
#define PART_BBT		0xFF

/* Number of known partition types */
#define PART_COUNT              35

/* Standard error codes */
#define ERROR_OK	 	0
#define ERROR_WRONGPARAM 	1
#define ERROR_DISKNOPARAM 	2
#define ERROR_NOMEM		3
#define ERROR_BADRANGE		4
#define ERROR_NOSPACE		5
#define ERROR_NOREAD		6

/***************************************************************************/
/* 		Macro's - should be in a .h				   */
/***************************************************************************/
/* These macro's do not expand correctly under Watcom C 10, 11. Used to
   translate CX value to track and sector values */
#define CX2TRACK(x)  (((x)>>8) | (((x)&0x00C0)<<2))
#define CX2SECTOR(x) ((x) & 0x003F)

/***************************************************************************/
/* 		Structure defs - should be in a .h			   */
/***************************************************************************/
/* Variable size defs. Copied from another file for one of the structure defs */
#ifndef __WIN32__
#define UCHAR unsigned char
#endif
#ifdef __MSDOS__
#define _INT int
#define LINT long int
#define LUINT long unsigned int
#define _UINT unsigned int
#else
#define _INT short int
#define _UINT unsigned short int
#define LINT long int
#define LUINT long unsigned int
#endif

/* Structure to hold partition names, whether it is an extended partition
   and their magic numbers. The constant array defining this is later */
struct part_names
{
 int id;
 int next;
 char *name;
};

/* Partition info as contained in partition table */
struct partspec
{
 UCHAR boot;
 UCHAR beg_h;
 _UINT beg_ts;
 UCHAR type;
 UCHAR end_h;
 _UINT end_ts;
 LUINT relsec;
 LUINT size;
}; /* Structure of partition table entry */

/* Partition table + code and signature */
struct partrec
{
 UCHAR dummy[0x1BE];
 struct partspec part[4];
 _UINT sig;
}; /* Structure of Master Boot Record */

#ifdef __REAL_PARTITION_INFO__
struct realpart
{
 UCHAR boot;
 LUINT beg_h;
 LUINT beg_track;
 LUINT beg_sector;
 UCHAR type;
 LUINT end_h;
 LUINT end_track;
 LUINT end_sector;
 LUINT relsec;
 LUINT size;
};
#endif

/* This holds info on every partition, position and the sector itself */
struct partition
{
 LUINT sector;
 LUINT track;
 LUINT head;
 struct partrec *part;
#ifdef __REAL_PARTITION_INFO__
 struct realpart mpart[4];
#endif
};

/* This holds info on every disk in the system */
struct disk_info
{
 _INT saved;
 LUINT max_track;
 LUINT max_sector;
 LUINT max_head;
 _UINT Drive;
 _INT num_partitions;
 struct partition **part;
};

/* Global container structure. */
struct fdisk
{
 _INT num_disks;
 struct disk_info **di;
};

/* Structure used by Int 13 function 48h */
struct int13ext_diskinfo
{
 _UINT size;
 _UINT info;
 LUINT cylinders;
 LUINT heads;
 LUINT sectors;
 LUINT total_sectors[2];
 _UINT bytes_per_sector;
 LUINT edd;
};

/***************************************************************************/
/* 		Constant arrays						   */
/***************************************************************************/
static struct part_names partition_names[PART_COUNT] =
 { { PART_EMPTY, 	0,	"No Partition" },
   { PART_DOS12,  	0,	"DOS-12" },
   { PART_XENIXROOT, 	0,	"Xenix  - Root" },
   { PART_XENIXUSR,  	0,	"Xenix  - User" },
   { PART_DOS16SMALL,   0,	"DOS-16 - Small" },
   { PART_EXTENDED,	1,	"Extended" },
   { PART_DOS16,	0,	"DOS-16" },
   { PART_OS2HPFS,	0,	"OS/2 - HPFS" },
   { PART_AIX,		0,	"AIX"	    },
   { PART_AIXBOOT,	0,	"AIX - Boot" },
   { PART_OS2BOOTMANAGE,0,	"OS/2 - Boot manager" },
   { PART_FAT32,	0,	"DOS-32" },
   { PART_FAT32EXTENDED,1,	"DOS-32 Extended" },
   { PART_VENIX,	0,	"Venix" },
   { PART_NOVELL,	0,	"Novell" },
   { PART_MICROPORT,	0,	"Microport" },
   { PART_GNUHURD, 	0,	"GNU Hurd", },
   { PART_NOVELL2,	0,	"Novell" },
   { PART_NOVELL3,	0,	"Novell" },
   { PART_PCIX,		0,	"PC/IX" },
   { PART_OLDMINUX,	0,	"Old Minux" },
   { PART_LINUXMINUX,   0,	"Linux - Minux" },
   { PART_LINUXSWAP,	0,	"Linux - Swap" },
   { PART_LINUXNATIVE,	0,	"Linux - Native" },
   { PART_AMOEBA,	0,	"Amoeba" },
   { PART_AMOEBA_BBT,	0,	"Amoeba - BBT" },
   { PART_BSD386,	0,	"BSD/386" },
   { PART_BSDIFS,	0,	"BSD/IFS" },
   { PART_BSDISWAP,	0,	"BSDI - Swap" },
   { PART_SYRINX,	0,	"Syrinx" },
   { PART_CPM,		0,	"CPM" },
   { PART_DOSACCESS,	0,	"DOS - Access" },
   { PART_DOSRO,	0,	"DOS - Readonly" },
   { PART_SECOND,	0,	"Secondary" },
   { PART_BBT,		0,	"BBT" },
 };

/***************************************************************************/
/* 		The Code						   */
/***************************************************************************/

int DetectDisk( int Drive )
/* This should detect a physical disk. NON-PORTABLE */
{
 int tmp;
 asm mov dx, Drive;
 asm mov ax, 0x1500;
 geninterrupt( 0x13 );
 tmp=_AX;
 return( tmp==0x300 );
}

void ResetDrive( int Drive )
/* Resets disks in case of error. NON-PORTABLE */
{
 asm mov dx, Drive;
 asm xor ax, ax
 geninterrupt( 0x13 );
}

int GetDiskinfo( struct disk_info **di, int Drive )
/* Get's disk info using Int 13/08 and Int 13/48. NON-PORTABLE */
{
 unsigned repeat, tmp, mCX, mDH;
 LUINT oldsize, newsize;
 struct int13ext_diskinfo edi;
 *di=NULL;
 if (Drive<0x80 || Drive>(0x80+FDISK_MAX_PHYS_DISKS)) return( ERROR_WRONGPARAM );
 repeat=0;
 do
  {
   asm mov dx, Drive;
   asm mov ax, 0x0800;
   geninterrupt( 0x13 );
   tmp=_FLAGS;
   mCX=_CX;
   mDH=_DH;
   if (tmp&0x01) /* Carry flag */
    ResetDrive( Drive );
  } while (++repeat<10 && tmp&0x01);
 if (tmp&0x01) return( ERROR_DISKNOPARAM );
 *di=(struct disk_info *)calloc( sizeof(struct disk_info), 1 );
 (*di)->max_track=CX2TRACK( mCX ) + 1;
 (*di)->max_head=mDH+1;
 (*di)->max_sector=CX2SECTOR( mCX );
 (*di)->Drive=Drive;
 (*di)->part=NULL;
 (*di)->num_partitions=0;
 edi.size=sizeof(edi);
 mCX=FP_SEG( &edi  );
 mDH=FP_OFF( &edi );
 asm push ds;
 asm mov dx, Drive;
 asm mov bx, mCX;
 asm mov ds, bx;
 asm mov si, mDH
 asm mov ax, 0x4800;
 geninterrupt( 0x13 );
 asm pop ds;
 tmp=_FLAGS;
 if (!(tmp&0x01))
  { /* This worked ? */
   oldsize=(long)(*di)->max_track*(long)(*di)->max_head*(long)(*di)->max_sector;
   newsize=edi.total_sectors[0]; /* Ok - were in trouble with REALLY big disks */
   if ((newsize*4L)>(oldsize*5L) && edi.info&0x02)
/*   if (edi.cylinders!=(*di)->max_track || edi.heads!=(*di)->max_head || edi.sectors!=(*di)->max_sector) */
     {
#ifdef __INT13_48_DEBUGGING__
      printf( "%ld (%ld) %ld (%ld) %ld (%ld)\n",(*di)->max_sector, edi.sectors, (*di)->max_head, edi.heads, (*di)->max_track, edi.cylinders );
#endif
      (*di)->max_sector=edi.sectors;
      (*di)->max_head=edi.heads;
      (*di)->max_track=edi.cylinders;
     }
  }
 return( ERROR_OK );
}

/* Reads a sector from a physical disk with some sanity checking */
int ReadSector( struct disk_info di, char *buf, int buf_size, int track, int head, int sector )
{
 char temp[512];
 int repeat, dtmp, ctmp, ftmp, etmp, btmp;
 if (di.Drive<0x80 || di.Drive>(0x80+FDISK_MAX_PHYS_DISKS)) return( ERROR_WRONGPARAM );
 repeat=0;
 if (track>di.max_track || head>di.max_head || sector>di.max_sector) return( ERROR_BADRANGE );
 if (buf_size<512) return( ERROR_NOSPACE );
 dtmp=di.Drive | (head<<8 );
 ctmp=sector | ((track&0x00FF)<<8) | ((track&0x0300)>>2);
 etmp=FP_SEG( temp );
 btmp=FP_OFF( temp );
 do
  {
   asm mov dx, etmp;
   asm mov es, dx
   asm mov bx, btmp;
   asm mov dx, dtmp;
   asm mov cx, ctmp;
   asm mov ax, 0x0201;
   geninterrupt( 0x13 );
   ftmp=_FLAGS;
   if (ftmp&0x01) ResetDrive(di.Drive);
  } while (++repeat<10 && ftmp&0x01);
 if (ftmp&0x01) return( ERROR_NOREAD );
 memcpy( buf, temp, 512 );
 return( ERROR_OK );
}

/* Goes through the list of partition types and try and determine whether it
   is an extended partition or not */
int isExtendedPartition( struct partrec pr, int who )
{
 int cnt;
 for (cnt=0;cnt<PART_COUNT;cnt++)
   if (partition_names[cnt].id==pr.part[who].type)
     {
      return( partition_names[cnt].next );
     }
 return( 0 );
}

/* The lazy persons way of recursively determining the number of partition
   structures to allocate.. I hate realloc */
int CountPartitions( struct disk_info *di, struct partrec *pr )
{
 struct partrec prl;
 int cnt, tmp, result;
 _UINT track, head, sector;
 for (cnt=0;cnt<4;cnt++)
  {
   tmp=isExtendedPartition( *pr, cnt );
   if (tmp)
    {
     di->num_partitions+=1;
     track=CX2TRACK( pr->part[cnt].beg_ts );
     sector=CX2SECTOR( pr->part[cnt].beg_ts );
     head=pr->part[cnt].beg_h;
/* Get next partition */
     result=ReadSector( *di, (char *)&prl, sizeof(prl), track, head, sector );
     if (result) return(result);
/* Check for loop - some nice viruses :-(( */
/* Do this later */
/* Call yourself to check that partition */
     result=CountPartitions( di, &prl );
     if (result) return(result);
    }
  }
 return( ERROR_OK );
}

/* Assumed CountPartitions called first. This will physically allocate all
   the structures and fill them. Recursive program. */
int AddPartitions( struct disk_info *di, struct partrec *pr, int index )
{
 LUINT size;
 int cnt, tmp, result;
 struct partrec prl;
 _UINT track, head, sector;
 for (cnt=0;cnt<4;cnt++)
  {
   tmp=isExtendedPartition( *pr, cnt );
   if (tmp)
    {
     track=CX2TRACK( pr->part[cnt].beg_ts );
     sector=CX2SECTOR( pr->part[cnt].beg_ts );
     head=pr->part[cnt].beg_h;
/* Get next partition */
     result=ReadSector( *di, (char *)&prl, sizeof(prl), track, head, sector );
     if (result) return(result);
/* Check for loop - some nice viruses :-(( */
/* Do this later */
/* Add partiton */
     di->part[index]=(struct partition *)calloc( sizeof( struct partition), 1 );
     if (!di->part[index]) return( ERROR_NOMEM );
     di->part[index]->sector=sector;
     di->part[index]->track=track;
     di->part[index]->head=head;
     di->part[index]->part=(struct partrec *)calloc( sizeof(struct partrec ), 1 );
     memcpy( di->part[index]->part, &prl, sizeof(prl) );
#ifdef __REAL_PARTITION_INFO__
     for (cnt=0;cnt<4;cnt++)
       {
	di->part[index]->mpart[cnt].boot=prl.part[cnt].boot;
	di->part[index]->mpart[cnt].beg_h=prl.part[cnt].beg_h;
	di->part[index]->mpart[cnt].beg_track=CX2TRACK(prl.part[cnt].beg_ts);
	di->part[index]->mpart[cnt].beg_sector=CX2SECTOR(prl.part[cnt].beg_ts);
	di->part[index]->mpart[cnt].type=prl.part[cnt].type;
	di->part[index]->mpart[cnt].end_h=prl.part[cnt].end_h;
	di->part[index]->mpart[cnt].end_track=CX2TRACK(prl.part[cnt].end_ts);
	di->part[index]->mpart[cnt].end_sector=CX2SECTOR(prl.part[cnt].end_ts);
	di->part[index]->mpart[cnt].relsec=prl.part[cnt].relsec;
	di->part[index]->mpart[cnt].size=prl.part[cnt].size;
	size=di->part[index]->mpart[cnt].beg_track*di->max_sector*di->max_head+
	     di->part[index]->mpart[cnt].beg_h*di->max_sector+
	     di->part[index]->mpart[cnt].beg_sector;
    printf( "%ld %ld\n", size, di->part[0]->mpart[cnt].relsec );
    printf( "--- %ld, %ld, %ld\n", di->part[index]->mpart[cnt].beg_track, di->part[index]->mpart[cnt].beg_h, di->part[index]->mpart[cnt].beg_sector );
    getch();
	if (di->part[0]->mpart[cnt].type!=PART_EMPTY && (size)<di->part[index]->mpart[cnt].relsec)
	     {
	      if (di->part[index]->mpart[cnt].beg_h==0 && di->part[index]->mpart[cnt].beg_sector==1)
	       {
		di->part[index]->mpart[cnt].beg_track=di->part[index]->mpart[cnt].relsec/(di->max_sector*di->max_head);
	       }
	      else
	       {
		di->part[index]->mpart[cnt].beg_track=(di->part[index]->mpart[cnt].relsec-di->part[index]->mpart[cnt].beg_sector-di->part[index]->mpart[cnt].beg_h*di->max_sector)/(di->max_sector*di->max_head);
	       }
	     }
	}
#endif
/* Call yourself to check that partition */
     result=AddPartitions( di, &prl, ++index );
     if (result) return(result);
    }
  }
 return( ERROR_OK );
}

/* Get partition info. This calls countpartition and addpartition */
int GetPartInfo( struct disk_info *di )
{
 int result, cnt;
 LUINT size;
 struct partrec pr;
 di->part=NULL;
 di->num_partitions=0;
 result=ReadSector( *di, (char *)&pr, sizeof(pr), 0, 0, 1 );
 if (result) return(result);
 di->num_partitions+=1;
 result=CountPartitions( di, &pr );
 if (result) return(result);
 di->part=(struct partition **)calloc( sizeof(struct partition *), di->num_partitions );
 if (!di->part) return( ERROR_NOMEM );
 di->part[0]=(struct partition *)calloc( sizeof( struct partition), 1 );
 if (!di->part[0]) return( ERROR_NOMEM );
 di->part[0]->sector=1;
 di->part[0]->track=0;
 di->part[0]->head=0;
 di->part[0]->part=(struct partrec *)calloc( sizeof(struct partrec ), 1 );
 memcpy( di->part[0]->part, &pr, sizeof(pr) );
#ifdef __REAL_PARTITION_INFO__
 for (cnt=0;cnt<4;cnt++)
   {
    di->part[0]->mpart[cnt].boot=pr.part[cnt].boot;
    di->part[0]->mpart[cnt].beg_h=pr.part[cnt].beg_h;
    di->part[0]->mpart[cnt].beg_track=CX2TRACK(pr.part[cnt].beg_ts);
    di->part[0]->mpart[cnt].beg_sector=CX2SECTOR(pr.part[cnt].beg_ts);
    di->part[0]->mpart[cnt].type=pr.part[cnt].type;
    di->part[0]->mpart[cnt].end_h=pr.part[cnt].end_h;
    di->part[0]->mpart[cnt].end_track=CX2TRACK(pr.part[cnt].end_ts);
    di->part[0]->mpart[cnt].end_sector=CX2SECTOR(pr.part[cnt].end_ts);
    di->part[0]->mpart[cnt].relsec=pr.part[cnt].relsec;
    di->part[0]->mpart[cnt].size=pr.part[cnt].size;
    size=di->part[0]->mpart[cnt].beg_track*di->max_sector*di->max_head+
	 di->part[0]->mpart[cnt].beg_h*di->max_sector+
	 di->part[0]->mpart[cnt].beg_sector-1;
    printf( "%ld %ld\n", size, di->part[0]->mpart[cnt].relsec );
    getch();
    if (di->part[0]->mpart[cnt].type!=PART_EMPTY && (size)<di->part[0]->mpart[cnt].relsec)
	 {
	  if (di->part[0]->mpart[cnt].beg_h==0 && di->part[0]->mpart[cnt].beg_sector==1)
	   {
	    di->part[0]->mpart[cnt].beg_track=di->part[0]->mpart[cnt].relsec/(di->max_sector*di->max_head);
	   }
	  else
	   {
	    di->part[0]->mpart[cnt].beg_track=(di->part[0]->mpart[cnt].relsec-di->part[0]->mpart[cnt].beg_sector-di->part[0]->mpart[cnt].beg_h*di->max_sector)/(di->max_sector*di->max_head);
	   }
	 }
   }
#endif
 result=AddPartitions( di, &pr, 1 );
 return( result );
}

/* The function that gathers all the information. */
int InitFdisk( struct fdisk *fd )
{
 int cnt, result;
 char buf[512];
 struct disk_info *di;
 fd->num_disks=0;
 fd->di=NULL;
 cnt=0x80;
 while (cnt<(0x80+FDISK_MAX_PHYS_DISKS) && DetectDisk( cnt ))
  {
   result=GetDiskinfo( &di, cnt++ );
   if (result) continue;
   result=ReadSector( *di, buf, 512, 0, 0, 1 );
   free( di );
   if (result) continue;
  }
 fd->num_disks=cnt-0x80;
 fd->di=(struct disk_info **)calloc( sizeof(struct disk_info *), fd->num_disks );
 if (!fd->di) return( ERROR_NOMEM );
 for (cnt=0;cnt<fd->num_disks;cnt++)
  {
   result=GetDiskinfo( &(fd->di[cnt]), cnt+0x80 );
   if (result) return(result);
   result=GetPartInfo( fd->di[cnt] );
   if (result) return(result);
  }
 return( ERROR_OK );
}

/* Print name from global constant array to stdio */
void print_parttype( int type )
{
 int cnt;
 for (cnt=0;cnt<PART_COUNT;cnt++)
   if (partition_names[cnt].id==type)
     {
      printf( " %s\n", partition_names[cnt].name );
      return;
     }
 printf( " Unknown\n" );
}

/* Calculates and prints partition size */
void print_partsize( struct partspec part )
{
 printf( "                 Size: %8.3f MB\n", (float)part.size/2048.0 );
}

/* Print a single partition's information */
void print_partinfo( struct partition pr )
{
 int cnt;
 _UINT btrack, bsector, etrack, esector;
 printf( "   - Located at Track %ld, Side %ld, Sector %ld\n", pr.track, pr.head, pr.sector );
 for (cnt=0;cnt<4;cnt++)
  if (pr.part->part[cnt].type!=PART_EMPTY)
   {
    btrack=CX2TRACK( pr.part->part[cnt].beg_ts );
    bsector=CX2SECTOR( pr.part->part[cnt].beg_ts );
    etrack=CX2TRACK( pr.part->part[cnt].end_ts );
    esector=CX2SECTOR( pr.part->part[cnt].end_ts );
    printf( "     Bootable %1d, Beginning: track %d, head %d, sector %d\n",
      pr.part->part[cnt].boot==0x80,
      btrack, pr.part->part[cnt].beg_h, bsector );
#ifdef __REAL_PARTITION_INFO__
    if (pr.mpart[cnt].beg_track!=btrack)
      printf( "                Real Beginning: track %ld, head %ld, sector %ld\n",
	 pr.mpart[cnt].beg_track, pr.mpart[cnt].beg_h, pr.mpart[cnt].beg_sector );
#endif
    printf( "                 End: track %d, head %d, sector %d",
      etrack, pr.part->part[cnt].end_h, esector );
#ifdef __REAL_PARTITION_INFO__
    if (pr.mpart[cnt].end_track!=etrack)
      printf( "                Real End: track %ld, head %ld, sector %ld\n",
	 pr.mpart[cnt].beg_track, pr.mpart[cnt].beg_h, pr.mpart[cnt].beg_sector );
#endif
    print_parttype( pr.part->part[cnt].type );
    print_partsize( pr.part->part[cnt] );
   }
}

/* Print info an a physical disk */
void print_diskinfo( struct fdisk fd )
{
 int cnt, cnt2;
 for (cnt=0;cnt<fd.num_disks;cnt++)
  {
   printf( "[Disk %d] Tracks - %ld, Sides - %ld, Sectors per Track %ld\n", cnt, fd.di[cnt]->max_track, fd.di[cnt]->max_head, fd.di[cnt]->max_sector );
   for (cnt2=0;cnt2<fd.di[cnt]->num_partitions;cnt2++)
     print_partinfo( *(fd.di[cnt]->part[cnt2]) );
  }
}

/* Guess ? */
#pragma argsused
int main( int argc, char **argv )
{
 int result;
 struct fdisk fd;
 clrscr();
 result=InitFdisk( &fd );
 if (result)
   {
    fprintf( stderr, "Unable to initialize program.\n" );
    return(result);
   }
 print_diskinfo( fd );
 return( 0 );
}