/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Module:  PCOMPUTE.C
// Module Description:  Partition Computation and Modification Functions
// Version:  1.0 RC 2
// Copyright:  1998-2001 under the terms of the GNU GPL, Version 2
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define PCOMPUTE

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <stdio.h>
#include <string.h>

#include "cmd.h"
#include "main.h"
#include "fdiskio.h"
#include "pcompute.h"
#include "pdiskio.h"
#include "userint1.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

extern char **environ;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

long Number_Of_Cylinders(unsigned long size);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Clear the Active Partition */
void Clear_Active_Partition()
{
  int index=0;

  do
    {
    part_table[(flags.drive_number-128)].active_status[index]=0x00;
    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Clear the Extended Partition Table Buffers */
void Clear_Extended_Partition_Table(int drive)
{
  int index;

  part_table[drive].ext_part_exists=FALSE;
  part_table[drive].ext_part_size_in_MB=0;
  part_table[drive].ext_part_num_sect=0;
  part_table[drive].ext_part_largest_free_space=0;

  part_table[drive].log_drive_free_space_start_cyl=0;
  part_table[drive].log_drive_free_space_end_cyl=0;

  part_table[drive].log_drive_largest_free_space_location=0;
  part_table[drive].num_of_ext_part=UNUSED;
  part_table[drive].num_of_log_drives=0;
  part_table[drive].num_of_non_dos_log_drives=0;

  index=0;
  do
    {
    part_table[drive].log_drive_num_type[index]=0;

    part_table[drive].log_drive_start_cyl[index]=0;
    part_table[drive].log_drive_start_head[index]=0;
    part_table[drive].log_drive_start_sect[index]=0;

    part_table[drive].log_drive_end_cyl[index]=0;
    part_table[drive].log_drive_end_head[index]=0;
    part_table[drive].log_drive_end_sect[index]=0;

    part_table[drive].log_drive_rel_sect[index]=0;
    part_table[drive].log_drive_num_sect[index]=0;

    part_table[drive].log_drive_size_in_MB[index]=0;
    part_table[drive].log_drive_created[index]=FALSE;

    part_table[drive].next_ext_exists[index]=FALSE;

    part_table[drive].next_ext_num_type[index]=0;

    part_table[drive].next_ext_start_cyl[index]=0;
    part_table[drive].next_ext_start_head[index]=0;
    part_table[drive].next_ext_start_sect[index]=0;

    part_table[drive].next_ext_end_cyl[index]=0;
    part_table[drive].next_ext_end_head[index]=0;
    part_table[drive].next_ext_end_sect[index]=0;

    part_table[drive].next_ext_rel_sect[index]=0;
    part_table[drive].next_ext_num_sect[index]=0;

    index++;
    }while(index<24);
}

/* Create Logical Drive */
/* Returns a 0 if successful and a 1 if unsuccessful */
int Create_Logical_Drive(int numeric_type, long size_in_MB)
{
  int index;
  int offset;

  long computed_ending_cylinder;
  long maximum_size_in_cylinders
   =part_table[(flags.drive_number-128)].ext_part_largest_free_space;
  long requested_size_in_cylinders=Number_Of_Cylinders(size_in_MB*2048);

  unsigned long computed_partition_size;
  unsigned long cylinder_size=(part_table[(flags.drive_number-0x80)]
   .total_head+1)*(part_table[(flags.drive_number-0x80)].total_sect);
  unsigned long size_in_logical_sectors;

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(requested_size_in_cylinders>maximum_size_in_cylinders)
   requested_size_in_cylinders=maximum_size_in_cylinders;

  /* Adjust the partition type, if necessary. */
  numeric_type=Partition_Type_To_Create(
   ( ( (requested_size_in_cylinders+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect) ) / 2048 ),
   numeric_type);


  /* Compute the size of the partition. */
  computed_partition_size=(requested_size_in_cylinders+1)*cylinder_size;

  /* Make space in the part_table structure, if necessary. */
  if( (part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location
   <=part_table[(flags.drive_number-128)].num_of_log_drives)
   && (part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location>0) )
    {
    index=part_table[(flags.drive_number-128)].num_of_log_drives+1;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=part_table[(flags.drive_number-128)].log_drive_num_type[(index-1)];
      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],part_table[(flags.drive_number-128)].log_drive_vol_label[(index-1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=part_table[(flags.drive_number-128)].log_drive_start_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=part_table[(flags.drive_number-128)].log_drive_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=part_table[(flags.drive_number-128)].log_drive_end_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=part_table[(flags.drive_number-128)].log_drive_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=part_table[(flags.drive_number-128)].log_drive_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=part_table[(flags.drive_number-128)].log_drive_num_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_exists[index]=part_table[(flags.drive_number-128)].next_ext_exists[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=part_table[(flags.drive_number-128)].next_ext_num_type[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=part_table[(flags.drive_number-128)].next_ext_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=part_table[(flags.drive_number-128)].next_ext_start_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=part_table[(flags.drive_number-128)].next_ext_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=part_table[(flags.drive_number-128)].next_ext_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=part_table[(flags.drive_number-128)].next_ext_end_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=part_table[(flags.drive_number-128)].next_ext_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=part_table[(flags.drive_number-128)].next_ext_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=part_table[(flags.drive_number-128)].next_ext_num_sect[(index-1)];

      index--;
      }while(index>=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    }

  /* Add the logical drive entry. */
  part_table[(flags.drive_number-128)]
   .log_drive_num_type[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]=numeric_type;
  strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location],"");

  part_table[(flags.drive_number-128)].log_drive_start_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl;
  part_table[(flags.drive_number-128)].log_drive_start_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;
  part_table[(flags.drive_number-128)].log_drive_start_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;

  /* Compute the ending cylinder */
  computed_ending_cylinder
   =part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl
   +requested_size_in_cylinders;

  part_table[(flags.drive_number-128)].log_drive_end_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)].log_drive_end_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)].log_drive_end_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
   part_table[(flags.drive_number-128)]
   .log_drive_num_type[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]
   =LBA_Partition_Type_To_Create(numeric_type);

  part_table[(flags.drive_number-128)]
   .log_drive_rel_sect[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]
   =part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)]
   .log_drive_num_sect[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]
   =computed_partition_size
   -part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)]
   .log_drive_size_in_MB[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].num_of_log_drives++;
  part_table[(flags.drive_number-128)]
   .log_drive_created[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]=TRUE;

  /* Add the linkage entry. */

  /* Add the linkage entry if there is a logical drive after this one. */
  if(part_table[(flags.drive_number-128)]
   .log_drive_num_type[(part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location+1)]>0)
    {
    part_table[(flags.drive_number-128)]
     .next_ext_exists[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]=TRUE;

    part_table[(flags.drive_number-128)]
     .next_ext_num_type[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]=5;

    part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =part_table[(flags.drive_number-128)]
     .log_drive_start_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)]
     .next_ext_start_head[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]=0;
    part_table[(flags.drive_number-128)]
     .next_ext_start_sect[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]=1;

    part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =part_table[(flags.drive_number-128)]
     .log_drive_end_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)]
     .next_ext_end_head[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)]
     .next_ext_end_sect[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =(part_table[(flags.drive_number-128)]
     .log_drive_start_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location+1)]
     -part_table[(flags.drive_number-128)]
     .pri_part_start_cyl[part_table[(flags.drive_number-128)]
     .num_of_ext_part])*(part_table[(flags.drive_number-128)]
     .total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    part_table[(flags.drive_number-128)]
     .next_ext_num_sect[part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location]
     =part_table[(flags.drive_number-128)]
     .log_drive_num_sect[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location+1)]
     +part_table[(flags.drive_number-128)].total_sect;
    }

  /* Add the linkage entry if there is a logical drive before this one. */
  if(part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location>0)
    {
    part_table[(flags.drive_number-128)]
     .next_ext_exists[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=TRUE;

    part_table[(flags.drive_number-128)]
     .next_ext_num_type[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=5;

    part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]
     =part_table[(flags.drive_number-128)]
     .log_drive_free_space_start_cyl;
    part_table[(flags.drive_number-128)]
     .next_ext_start_head[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=0;
    part_table[(flags.drive_number-128)]
     .next_ext_start_sect[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=1;

    part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]
     =computed_ending_cylinder;
    part_table[(flags.drive_number-128)]
     .next_ext_end_head[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]
     =part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)]
     .next_ext_end_sect[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]
     =part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]
     =( (part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)])
     - (part_table[(flags.drive_number-128)]
     .pri_part_start_cyl[part_table[(flags.drive_number-128)]
     .num_of_ext_part]) )*(part_table[(flags.drive_number-128)]
     .total_head+1)*part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_num_sect
     [(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=computed_partition_size;
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

#ifdef DEBUG
  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Logical_Drive(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("Number of partition that was created:  %d",part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    Position_Cursor(4,7);
    printf("Brief logical drive table:");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=9;

    Position_Cursor(4,8);
    printf(" #  NT     SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    do
      {
      if( (index>=0) && (index<24) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);
	Position_Cursor(7,offset);
	printf("%3d",part_table[(flags.drive_number-128)].log_drive_num_type[index]);
	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_sect[index]);

	Position_Cursor(58,offset);
	printf("%d",part_table[(flags.drive_number-128)].log_drive_rel_sect[index]);

	Position_Cursor(72,offset);
	printf("%5d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]);
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=11);

    Position_Cursor(4,15);
    printf("Next extended location table:");

    Position_Cursor(4,16);
    printf(" #         SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=17;

    do
      {
      if( (index>=0) && (index<24) && (part_table[(flags.drive_number-128)].next_ext_exists[index]==TRUE) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);

	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_sect[index]);

	/*
	Temporarily removed because the size of the relative sector field
	exceeds that handled by the printf statement.  As a result,
	incorrect information is returned.

	Position_Cursor(58,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_rel_sect[index]);
	*/

	Position_Cursor(72,offset);
	printf("%4d",((part_table[(flags.drive_number-128)].next_ext_num_sect[index])/2048));
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=19);

    Pause();
    }
#endif

  return(0);
}

/* Create a Primary Partition */
/* Returns partition number if successful and a 99 if unsuccessful */
int Create_Primary_Partition(int numeric_type,long size_in_MB)
{
  int index=0;
  int empty_partition_number=99;

  long computed_ending_cylinder;
  long maximum_size_in_cylinders
   =part_table[(flags.drive_number-128)].pri_part_largest_free_space;
  long requested_size_in_cylinders=Number_Of_Cylinders(size_in_MB*2048);

  unsigned long computed_partition_size;
  unsigned long cylinder_size=(part_table[(flags.drive_number-0x80)]
   .total_head+1)*(part_table[(flags.drive_number-0x80)].total_sect);

  unsigned long size_in_logical_sectors;

  /* Ensure that an empty primary partition exists. */
  do
    {
    if( (empty_partition_number==99)
     && (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0) )
      {
      empty_partition_number=index;
      }
    index++;
    }while(index<4);

  /* If all primary partitions are full, report failure. */
  if(empty_partition_number==99) return(99);

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(requested_size_in_cylinders>maximum_size_in_cylinders)
   requested_size_in_cylinders=maximum_size_in_cylinders;

  /* Make sure the starting cylinder of an extended partition is at least  */
  /* 1.  If the cylinder number is 0, increment it to 1.                   */
  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0)
      {
      part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl=1;
      requested_size_in_cylinders--;
      }
    }

  /* Re-obtain a partition type, if applicable. */
  if( (numeric_type!=5) && (numeric_type!=0x0f) )
   numeric_type=Partition_Type_To_Create(
   ( ( (requested_size_in_cylinders+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect) ) / 2048 ),
   numeric_type);

  /* Compute the ending cylinder of the partition */
  computed_ending_cylinder
   =part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl
   +requested_size_in_cylinders;

  /* Compute the size of the partition. */
  computed_partition_size=(requested_size_in_cylinders+1)*cylinder_size;

  part_table[(flags.drive_number-128)]
   .active_status[empty_partition_number]=0;
  part_table[(flags.drive_number-128)]
   .pri_part_num_type[empty_partition_number]=numeric_type;

  part_table[(flags.drive_number-128)]
   .pri_part_start_cyl[empty_partition_number]
   =part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl;
  /* If the starting cylinder is 0, then the starting head is 1...otherwise */
  /* the starting head is 1.                                                */
  if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0)
   part_table[(flags.drive_number-128)]
   .pri_part_start_head[empty_partition_number]=1;
  else part_table[(flags.drive_number-128)]
   .pri_part_start_head[empty_partition_number]=0;
  part_table[(flags.drive_number-128)]
   .pri_part_start_sect[empty_partition_number]=1;

  part_table[(flags.drive_number-128)]
   .pri_part_end_cyl[empty_partition_number]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)]
   .pri_part_end_head[empty_partition_number]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)]
   .pri_part_end_sect[empty_partition_number]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .pri_part_end_cyl[empty_partition_number]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
    {
    numeric_type=LBA_Partition_Type_To_Create(numeric_type);
    part_table[(flags.drive_number-128)]
     .pri_part_num_type[empty_partition_number]=numeric_type;
    }

  if(part_table[(flags.drive_number-128)]
   .pri_part_start_cyl[empty_partition_number]>0)
    {
    part_table[(flags.drive_number-128)]
     .pri_part_rel_sect[empty_partition_number]
     =(part_table[(flags.drive_number-128)]
     .pri_part_start_cyl[empty_partition_number])
     *(part_table[(flags.drive_number-128)]
     .total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    }
  else part_table[(flags.drive_number-128)]
   .pri_part_rel_sect[empty_partition_number]
   =part_table[(flags.drive_number-128)].total_sect;

  if(part_table[(flags.drive_number-128)]
   .pp_largest_free_space_start_cyl==0)
   computed_partition_size=computed_partition_size-part_table
   [(flags.drive_number-128)].total_sect;

  part_table[(flags.drive_number-128)]
   .pri_part_num_sect[empty_partition_number]
   =computed_partition_size;

  part_table[(flags.drive_number-128)]
   .pri_part_size_in_MB[empty_partition_number]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  part_table[(flags.drive_number-128)]
   .pri_part_created[empty_partition_number]=TRUE;

  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=TRUE;
    part_table[(flags.drive_number-128)].num_of_ext_part=empty_partition_number;
    part_table[(flags.drive_number-128)].ext_part_num_sect=computed_partition_size;
    part_table[(flags.drive_number-128)].ext_part_size_in_MB=size_in_MB;
    }

#ifdef DEBUG
  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Primary_Partition(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("empty_partition_number=%d",empty_partition_number);

    Position_Cursor(4,8);
    printf("New Partition Information:");
    Position_Cursor(4,10);
    printf("Starting Cylinder:  %d",part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]);
    Position_Cursor(4,11);
    printf("Starting Head:      %d",part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]);
    Position_Cursor(4,12);
    printf("Starting Sector:    %d",part_table[(flags.drive_number-128)].pri_part_start_sect[empty_partition_number]);

    Position_Cursor(40,10);
    printf("Ending Cylinder:    %d",part_table[(flags.drive_number-128)].pri_part_end_cyl[empty_partition_number]);
    Position_Cursor(40,11);
    printf("Ending Head:        %d",part_table[(flags.drive_number-128)].pri_part_end_head[empty_partition_number]);
    Position_Cursor(40,12);
    printf("Ending Sector:      %d",part_table[(flags.drive_number-128)].pri_part_end_sect[empty_partition_number]);

    Position_Cursor(4,14);
    printf("Relative Sectors:   %d",part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]);

    Position_Cursor(40,14);
    printf("Size of partition in MB:    %d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[empty_partition_number]);

    Pause();
    }
#endif

  return(empty_partition_number);
}

/* Delete Logical DOS Drive */
void Delete_Logical_Drive(int logical_drive_number)
{
  int index;

  /* Zero out the partition table entry for the logical drive. */
  part_table[(flags.drive_number-128)]
   .log_drive_num_type[logical_drive_number]=0;

  strcpy(part_table[(flags.drive_number-128)]
   .log_drive_vol_label[logical_drive_number],"           ");

  part_table[(flags.drive_number-128)]
   .log_drive_start_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_rel_sect[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_num_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_size_in_MB[logical_drive_number]=0;

  /* If the logical drive to be deleted is not the first logical drive.     */
  /* Assume that there are extended partition tables after this one.  If    */
  /* there are not any more extended partition tables, nothing will be      */
  /* harmed by the shift.                                                   */
  if(logical_drive_number>0)
    {
    /* Move the extended partition information from this table to the       */
    /* previous table.                                                      */
    part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_num_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_num_sect[logical_drive_number];

    /* Shift all the following extended partition tables left by 1.         */
    index=logical_drive_number;

    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]
       =part_table[(flags.drive_number-128)].log_drive_num_type[(index+1)];

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index]
       ,part_table[(flags.drive_number-128)].log_drive_vol_label[(index+1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]
       =part_table[(flags.drive_number-128)].log_drive_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]
       =part_table[(flags.drive_number-128)].log_drive_start_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]
       =part_table[(flags.drive_number-128)].log_drive_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]
       =part_table[(flags.drive_number-128)].log_drive_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]
       =part_table[(flags.drive_number-128)].log_drive_end_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]
       =part_table[(flags.drive_number-128)].log_drive_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]
       =part_table[(flags.drive_number-128)].log_drive_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]
       =part_table[(flags.drive_number-128)].log_drive_num_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]
       =part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]
       =part_table[(flags.drive_number-128)].next_ext_num_type[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]
       =part_table[(flags.drive_number-128)].next_ext_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]
       =part_table[(flags.drive_number-128)].next_ext_start_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]
       =part_table[(flags.drive_number-128)].next_ext_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]
       =part_table[(flags.drive_number-128)].next_ext_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]
       =part_table[(flags.drive_number-128)].next_ext_end_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]
       =part_table[(flags.drive_number-128)].next_ext_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]
       =part_table[(flags.drive_number-128)].next_ext_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]
       =part_table[(flags.drive_number-128)].next_ext_num_sect[(index+1)];

      if(part_table[(flags.drive_number-128)].log_drive_num_type[index]>0)
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=TRUE;
	}
      else
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=FALSE;
	}

      index++;
      }while(index<22);
    }
  part_table[(flags.drive_number-128)].num_of_log_drives--;

  /* If there aren't any more logical drives, clear the extended        */
  /* partition table to prevent lockups by any other partition utils.   */
  if(part_table[(flags.drive_number-128)].num_of_log_drives==0)
    {
    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;
      part_table[(flags.drive_number-128)].log_drive_created[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<24);
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Delete Primary Partition */
void Delete_Primary_Partition(int partition_number)
{
  int index;

  /* If partition_number is an extended partition, first delete the */
  /* extended partition.                                            */
  if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==5)
    || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=FALSE;
    part_table[(flags.drive_number-128)].num_of_ext_part=0;

    part_table[(flags.drive_number-128)].ext_part_size_in_MB=0;
    part_table[(flags.drive_number-128)].ext_part_num_sect=0;
    part_table[(flags.drive_number-128)].ext_part_largest_free_space=0;

    part_table[(flags.drive_number-128)].log_drive_largest_free_space_location=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_end_cyl=0;

    part_table[(flags.drive_number-128)].num_of_log_drives=0;

    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],"            ");;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=0;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<23);
    }

  part_table[(flags.drive_number-128)].active_status[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_rel_sect[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]=0;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  /* Delete volume label from buffer */
  strcpy(part_table[(flags.drive_number-128)].pri_part_vol_label[partition_number],"           ");
}

/* Determine the locations of free space in the partition table */
void Determine_Free_Space()
{
  int first_used_partition=UNUSED;
  int last_used_partition=UNUSED;
  int index;
  int sub_index;
  int swap;

  int drive=flags.drive_number-0x80;

  unsigned long free_space_after_last_used_partition=0;
  unsigned long free_space_before_first_used_partition=0;
  unsigned long free_space_between_partitions_0_and_1=0;
  unsigned long free_space_between_partitions_1_and_2=0;
  unsigned long free_space_between_partitions_2_and_3=0;

#ifdef DEBUG
  unsigned long cylinder_size=(part_table[drive].total_head+1)*(part_table[drive].total_sect);
#endif

  /* Reset the physical order to default */
  index=0;
  do
    {
    part_table[drive].pri_part_physical_order[index]=index;

    index++;
    }while(index<4);

  /* Determine the location and size of the largest free space in the */
  /* primary partition.                                               */

  /* 1.  Sort the primary partitions based upon starting cylinder and their*/
  /*     contents...or lack of use.                                        */

  /* Place all empty partitions at the end of the table. */
  index=0;
  do
    {

    sub_index=0;
    do
      {
      if(part_table[drive].pri_part_num_type[part_table[drive].pri_part_physical_order[sub_index]]==0)
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);

    index++;
    }while(index<4);

#ifdef DEBUG
  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 1",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);

    printf("\nContents after initial sorting of unused partitions to end:\n\n");

    index=0;
    do
      {
      printf("Partition %1d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    Position_Cursor(4,20);
    Pause();
    }
#endif

  /* Order the partitions based upon starting cylinder */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      if( (part_table[drive].pri_part_num_type
       [part_table[drive].pri_part_physical_order
       [sub_index]]!=0) && (part_table[drive]
       .pri_part_num_type[part_table[drive]
       .pri_part_physical_order[(sub_index+1)]]!=0)
       && (part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[sub_index]]
       >part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[(sub_index+1)]]) )
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);
    index++;
    }while(index<4);

#ifdef DEBUG
  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 2",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);
    printf("\nContents after sorting partitions by starting cylinder:\n\n");

    index=0;
    do
      {
      printf("Partition %d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    }
#endif

  /* 2.  Is there any free space before the first partition? */

  /* Find the first used partition and the last used partition. */
  index=0;
  do
    {
    if( (first_used_partition==UNUSED)
     && (part_table[drive].pri_part_num_type[part_table[drive]
     .pri_part_physical_order[index]]>0) )
      {
      first_used_partition=index;
      }

    if(part_table[drive].pri_part_num_type
     [part_table[drive].pri_part_physical_order[index]]>0)
      {
      last_used_partition=index;
      }

    index++;
    }while(index<4);

  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]>0)
      {
      free_space_before_first_used_partition
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[first_used_partition]]);
      }
    else free_space_before_first_used_partition=0;
    }

  /* 3.  Is there any free space after the last used partition? */
  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[last_used_partition]]<=part_table[drive].total_cyl)
      {
/* */
      free_space_after_last_used_partition
       =(part_table[drive].total_cyl-part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]);
      }
    }

  /* 4.  Is there any free space between partitions?                    */
  /*                                                                    */
  if( (first_used_partition!=UNUSED) && (last_used_partition>=1) )
    {
    if( (part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[0]]+1)
     <(part_table[drive].pri_part_start_cyl[part_table[drive]
     .pri_part_physical_order[1]]) )
      {
      free_space_between_partitions_0_and_1
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[1]]
       -part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[0]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition>=2) )
    {
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]) )
      {
      free_space_between_partitions_1_and_2=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition==3) )
    {
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]) )
      {
      free_space_between_partitions_2_and_3=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]);
      }
    }

  /* Locate the largest free space */
  if(first_used_partition!=UNUSED)
    {
    /* */
    part_table[drive].pp_largest_free_space_start_head=0;
    part_table[drive].pp_largest_free_space_start_sect=1;

    /* Default the largest free space to before the first used partition */
    part_table[drive].pri_part_largest_free_space=free_space_before_first_used_partition;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl
     =part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]-1;

    /* If the largest free space is not before the first used partition  */
    /* make the correct adjustments.                                     */
    if(free_space_after_last_used_partition>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_after_last_used_partition;
      part_table[drive].pp_largest_free_space_start_cyl
       =part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl;
      }

    if(free_space_between_partitions_0_and_1>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_0_and_1;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[0]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[1]]-1;
      }

    if(free_space_between_partitions_1_and_2>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_1_and_2;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-1;
      }

    if(free_space_between_partitions_2_and_3>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_2_and_3;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-1;
      }
    }
  else
    {
    part_table[drive].pri_part_largest_free_space=part_table[drive].total_cyl;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl-1;
    }

  /* Make final adjustments to the computed free space size. */
  part_table[drive].pri_part_largest_free_space
   =part_table[drive].pri_part_largest_free_space-2;

  if(part_table[drive].pri_part_largest_free_space<=0)
   part_table[drive].pri_part_largest_free_space=0;

  if(part_table[drive].pp_largest_free_space_start_cyl==0)
   part_table[drive].pri_part_largest_free_space++;

#ifdef DEBUG
  if(debug.determine_free_space==TRUE)
    {
    printf("\n\nFree space (in cylinders) in primary partition table:\n");
    printf("\nFree space before first used partition:  %d",free_space_before_first_used_partition);
/* */
    printf("\nFree space after last used partition:  %d",free_space_after_last_used_partition);
    printf("\nFree space between partitions 0 and 1:  %d",free_space_between_partitions_0_and_1);
    printf("\nFree space between partitions 1 and 2:  %d",free_space_between_partitions_1_and_2);
/* */
    printf("\nFree space between partitions 2 and 3:  %d",free_space_between_partitions_2_and_3);
    printf("\n\nLargest free space in primary partition table:  %d",part_table[drive].pri_part_largest_free_space);
    printf("\nStarting cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_start_cyl);
    printf("\nEnding cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_end_cyl);
    Pause();
    }
#endif

  /* Determine the location and size of the largest free space in the */
  /* extended partition, if it exists.                                */
  if(part_table[drive].ext_part_exists==TRUE)
    {
    part_table[drive].ext_part_largest_free_space=0;
    part_table[drive].log_drive_free_space_start_cyl=0;
    part_table[drive].log_drive_free_space_end_cyl=0;
    part_table[drive].log_drive_largest_free_space_location=0;

#ifdef DEBUG
    if(debug.determine_free_space==TRUE)
      {
      Clear_Screen(NULL);
      Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 3",BOLD);
      printf("\n\n");
      }
#endif

    if(part_table[drive].num_of_log_drives>0)
      {
      /* If there are logical drives in the extended partition first find  */
      /* the largest free space between the logical drives...if it exists. */
      last_used_partition=UNUSED;
      index=0;

      /* Check to see if the first possible entry in the extended partition */
      /* is unused.  If it is unused and there is a logical drive after it  */
      /* then skip checking for free space between entry 0 and 1.           */
      if(part_table[drive].log_drive_num_type[0]==0) index=1;

      do
	{
	if(part_table[drive].log_drive_num_type[index]>0)
	  {
	  last_used_partition=index;
	  }

	if( (part_table[drive].log_drive_start_cyl[(index+1)]-part_table[drive].log_drive_end_cyl[index]) >1 )
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[(index+1)]-1)-(part_table[drive].log_drive_end_cyl[index]+1);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].log_drive_end_cyl[index]+1;
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[(index+1)]-1;
	  part_table[drive].log_drive_largest_free_space_location=index+1;
	  }

#ifdef DEBUG
	if(debug.determine_free_space==TRUE)
	  {
	  if(index==12)
	    {
	    printf("\n");
	    Pause();
	    }

	  printf("\nLogical Drive #: %2d    ",index);
	  printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[index]);
	  printf("EC: %4d    ",part_table[drive].log_drive_end_cyl[index]);
	  }
#endif

	index++;
	}while(index<22);

#ifdef DEBUG
      if(debug.determine_free_space==TRUE)
	{
	printf("\nLogical Drive #: %2d    ",index);
	printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[22]);
	printf("EC: %4d    \n",part_table[drive].log_drive_end_cyl[22]);
	Pause();

	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 4",BOLD);
	printf("\n\nNote:  All values are in cylinders.\n\n");
	printf("Results of free space calculations after computing spaces between\n  logical drives:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
#endif

      /* Determine if there is any free space before the first logical      */
      /* drive in the extended partition.                                   */
      if(part_table[drive].log_drive_num_type[0]!=0)
	{
	if( (part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[0]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}
      else
	{
	if( (part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[1]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}

#ifdef DEBUG
      if(debug.determine_free_space==TRUE)
	{
	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 5",BOLD);
	printf("\n\nResults of free space calculations after computing space before\n  the first logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	}
#endif

      /* Determine if there is any free space after the last logical drive  */
      /* in the extended partition.                                         */
      if( (last_used_partition<23)
       && (part_table[drive].log_drive_end_cyl[last_used_partition]
       <part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]) )
	{
	if( (part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]
	 -part_table[drive].log_drive_end_cyl[last_used_partition])
	 >(part_table[drive].ext_part_largest_free_space) )
	  {
	  part_table[drive].ext_part_largest_free_space
	   =(part_table[drive].pri_part_end_cyl[part_table[drive]
	   .num_of_ext_part]
	   -part_table[drive].log_drive_end_cyl[last_used_partition]-1);
	  part_table[drive].log_drive_free_space_start_cyl
	   =part_table[drive].log_drive_end_cyl[last_used_partition]+1;
	  part_table[drive].log_drive_free_space_end_cyl
	   =part_table[drive].pri_part_end_cyl[part_table[drive]
	   .num_of_ext_part]; //-1;
/* */
	  part_table[drive].log_drive_largest_free_space_location=last_used_partition+1;
	  }
	}

#ifdef DEBUG
      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nResults of free space calculations after computing space after\n  the last logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
#endif
      }
    else
      {
      /* If the extended partition is empty. */
      part_table[drive].ext_part_largest_free_space=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]-1;

      part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
      part_table[drive].log_drive_free_space_end_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part];

#ifdef DEBUG
      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nThere are not any Logical DOS Drives in the Extended DOS Partition\n\n");
	printf("Location of largest free space:     0\n");
	printf("Size of largest free space:  %d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
#endif
      }
    }
}

/* Convert the standard_partition_type to an LBA partition type */
int LBA_Partition_Type_To_Create(int standard_partition_type)
{
  int lba_partition_type;

  /* Extended int 0x13 FAT 32 */
  if(standard_partition_type==0x0b) lba_partition_type=0x0c;

  /* Extended int 0x13 FAT 16 */
  if( (standard_partition_type==1) || (standard_partition_type==4)
   || (standard_partition_type==6) ) lba_partition_type=0x0e;

  /* Extended int 0x13 Extended Partition */
  if(standard_partition_type==0x05) lba_partition_type=0x0f;

  return(lba_partition_type);
}

/* Modify Partition Type */
void Modify_Partition_Type(int partition_number,int type_number)
{
  if(partition_number<4)
    {
    part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]
     =type_number;
    }
  else
    {
    part_table[(flags.drive_number-128)].log_drive_num_type[(partition_number-4)]
     =type_number;
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Calculate number of cylinders */
long Number_Of_Cylinders(unsigned long size)
{
  /* unsigned long size has to be in sectors @ 512 bytes/sector */

  long number_of_cylinders;

  unsigned long cylinder_size=(part_table[(flags.drive_number-0x80)]
   .total_head+1)*(part_table[(flags.drive_number-0x80)].total_sect);

  number_of_cylinders=size/cylinder_size;

  return(number_of_cylinders);
}

/* Transfer partition information from one slot to another */
void Primary_Partition_Slot_Transfer(int transfer_type,int source,int dest)
{
   int index;

   int active_status;

   int pri_part_num_type;
   char pri_part_vol_label[13];

   long pri_part_start_cyl;
   long pri_part_start_head;
   long pri_part_start_sect;

   long pri_part_end_cyl;
   long pri_part_end_head;
   long pri_part_end_sect;

   unsigned long pri_part_rel_sect;
   unsigned long pri_part_num_sect;

   long pri_part_size_in_MB;

   source--;
   dest--;

   /* Store destination record for later transfer to source location */
   if(transfer_type==SWAP)
     {
     active_status
      =part_table[(flags.drive_number-128)].active_status[dest];

     pri_part_num_type
      =part_table[(flags.drive_number-128)].pri_part_num_type[dest];

     index=0;
     do
       {
       pri_part_vol_label[index]
	=part_table[(flags.drive_number-128)].pri_part_vol_label[dest] [index];

       index++;
       }while(index<13);

     pri_part_start_cyl
      =part_table[(flags.drive_number-128)].pri_part_start_cyl[dest];

     pri_part_start_head
      =part_table[(flags.drive_number-128)].pri_part_start_head[dest];

     pri_part_start_sect
      =part_table[(flags.drive_number-128)].pri_part_start_sect[dest];

     pri_part_end_cyl
      =part_table[(flags.drive_number-128)].pri_part_end_cyl[dest];

     pri_part_end_head
      =part_table[(flags.drive_number-128)].pri_part_end_head[dest];

     pri_part_end_sect
      =part_table[(flags.drive_number-128)].pri_part_end_sect[dest];

     pri_part_rel_sect
      =part_table[(flags.drive_number-128)].pri_part_rel_sect[dest];

     pri_part_num_sect
      =part_table[(flags.drive_number-128)].pri_part_num_sect[dest];

     pri_part_size_in_MB
      =part_table[(flags.drive_number-128)].pri_part_size_in_MB[dest];
     }

   /* Move source record to destination location */
   part_table[(flags.drive_number-128)].active_status[dest]
    =part_table[(flags.drive_number-128)].active_status[source];

   part_table[(flags.drive_number-128)].pri_part_num_type[dest]
    =part_table[(flags.drive_number-128)].pri_part_num_type[source];

   index=0;
   do
     {
     part_table[(flags.drive_number-128)].pri_part_vol_label[dest] [index]
      =part_table[(flags.drive_number-128)].pri_part_vol_label[source] [index];

     index++;
     }while(index<13);

   part_table[(flags.drive_number-128)].pri_part_start_cyl[dest]
    =part_table[(flags.drive_number-128)].pri_part_start_cyl[source];

   part_table[(flags.drive_number-128)].pri_part_start_head[dest]
    =part_table[(flags.drive_number-128)].pri_part_start_head[source];

   part_table[(flags.drive_number-128)].pri_part_start_sect[dest]
    =part_table[(flags.drive_number-128)].pri_part_start_sect[source];

   part_table[(flags.drive_number-128)].pri_part_end_cyl[dest]
    =part_table[(flags.drive_number-128)].pri_part_end_cyl[source];

   part_table[(flags.drive_number-128)].pri_part_end_head[dest]
    =part_table[(flags.drive_number-128)].pri_part_end_head[source];

   part_table[(flags.drive_number-128)].pri_part_end_sect[dest]
    =part_table[(flags.drive_number-128)].pri_part_end_sect[source];

   part_table[(flags.drive_number-128)].pri_part_rel_sect[dest]
    =part_table[(flags.drive_number-128)].pri_part_rel_sect[source];

   part_table[(flags.drive_number-128)].pri_part_num_sect[dest]
    =part_table[(flags.drive_number-128)].pri_part_num_sect[source];

   part_table[(flags.drive_number-128)].pri_part_size_in_MB[dest]
    =part_table[(flags.drive_number-128)].pri_part_size_in_MB[source];

   /* Delete source record */
   part_table[(flags.drive_number-128)].active_status[source]=0;
   part_table[(flags.drive_number-128)].pri_part_num_type[source]=0;
   strcpy(part_table[(flags.drive_number-128)].pri_part_vol_label[source]
    ,"");
   part_table[(flags.drive_number-128)].pri_part_start_cyl[source]=0;
   part_table[(flags.drive_number-128)].pri_part_start_head[source]=0;
   part_table[(flags.drive_number-128)].pri_part_start_sect[source]=0;
   part_table[(flags.drive_number-128)].pri_part_end_cyl[source]=0;
   part_table[(flags.drive_number-128)].pri_part_end_head[source]=0;
   part_table[(flags.drive_number-128)].pri_part_end_sect[source]=0;
   part_table[(flags.drive_number-128)].pri_part_rel_sect[source]=0;
   part_table[(flags.drive_number-128)].pri_part_num_sect[source]=0;
   part_table[(flags.drive_number-128)].pri_part_size_in_MB[source]=0;

   /* Store original destination record in source location */
   if(transfer_type==SWAP)
     {
     part_table[(flags.drive_number-128)].active_status[source]
      =active_status;

     part_table[(flags.drive_number-128)].pri_part_num_type[source]
      =pri_part_num_type;

     index=0;
     do
       {
       part_table[(flags.drive_number-128)].pri_part_vol_label[source] [index]
	=pri_part_vol_label[index];

       index++;
       }while(index<13);

     part_table[(flags.drive_number-128)].pri_part_start_cyl[source]
      =pri_part_start_cyl;

     part_table[(flags.drive_number-128)].pri_part_start_head[source]
      =pri_part_start_head;

     part_table[(flags.drive_number-128)].pri_part_start_sect[source]
      =pri_part_start_sect;

     part_table[(flags.drive_number-128)].pri_part_end_cyl[source]
      =pri_part_end_cyl;

     part_table[(flags.drive_number-128)].pri_part_end_head[source]
      =pri_part_end_head;

     part_table[(flags.drive_number-128)].pri_part_end_sect[source]
      =pri_part_end_sect;

     part_table[(flags.drive_number-128)].pri_part_rel_sect[source]
      =pri_part_rel_sect;

     part_table[(flags.drive_number-128)].pri_part_num_sect[source]
      =pri_part_num_sect;

     part_table[(flags.drive_number-128)].pri_part_size_in_MB[source]
      =pri_part_size_in_MB;
     }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Compute the partition type to create. */
int Partition_Type_To_Create(unsigned long size_in_mb,
 int requested_partition_type)
{
  /* Note:  Using 0 for requested_partition_type results in a FAT partition
	    value being returned for any partition type.                   */

  int numeric_type;

  if( (requested_partition_type==2)
   || (requested_partition_type==3)
   || (requested_partition_type>=7 && requested_partition_type<=10)
   || (requested_partition_type==13)
   || (requested_partition_type>16) ) return(requested_partition_type);

  /* FAT 12 */
  if(size_in_mb<=16) numeric_type=1;

  /* Small FAT 16 */
  if( (size_in_mb>16) && (size_in_mb<=32) ) numeric_type=4;

  /* Large FAT 16 */
  if(size_in_mb>32) numeric_type=6;

  /* FAT 32 */
  if( (size_in_mb>128) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==TRUE) && (flags.fprmt==TRUE) ) numeric_type=0x0b;

  if( (size_in_mb>512) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==TRUE) ) numeric_type=0x0b;

  return(numeric_type);
}

/* Set Active Partition */
void Set_Active_Partition(int partition_number)
{
  int index=0;

  do
    {
    if(index==partition_number) part_table[(flags.drive_number-128)].active_status[index]=0x80;
    else part_table[(flags.drive_number-128)].active_status[index]=0x00;

    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}
