/*
	emm386c.c
	
	Copyright (c) by tom ehlert 2001-2005 - all rights reserved

	Licensed under the Artistic License version
	
	please see LICENSE.TXT for details

	modified for >64M and VCPI support, Michael Devore

	Documented EMS 4.0 only supports up to 32M (0x800 pages), but some EMS
	drivers allow more than 32M.  The EMS 4.0 spec could be extended to 1G-32K
	(map value of 0xFFFF means unmap) without breaking the spec API,
    but unfortunately breaking unsophisticated programs which get confused.
	I have arbitrarily decided to allow up to 32M of EMS allocations,
	leaving the high bit of page allocation alone since specific drivers
	may make use of high bit values other than 0xFFFF for their own purposes,
	or may only check high bit for unmap -- as FreeDOS EMM386 does.

	Michael Devore's changes are not copyrighted and are released
	to the public domain.
    This does not affect copyright on the rest of the code.

	this code is based on ct0890s (copyright 1990 by harry albrecht, c't)
	this was taken, hacked, patched, tweaked, to make a real EMM386.
*/

#define PROGRAM "JEMM386"

#include "useful.h"

/* globals, also defined in JEMM386.INC or JEMM32.ASM */

#define MAXK_EMS_ALLOWED (512L*1024L-32L)
#define MAXMEM16K_DEFAULT 0x1E00
#define MAX_EMS_PAGES_ALLOWED 0x800
#define DMABUFFDEFAULT 64
#define	UMB_MAX_BLOCKS	8

/* compiler switches, requires switches in JEMM386.INC to be set as well */

#define SBSUPP
#define VMESUPP
#define LOADSUPP
#define A20SUPP
#define EMXSUPP
#define PGESUPP
#define UNLOADSUPP

/* equates, must match the ones in JEMM32.ASM */

#define V86F_SB       1    /* SB compat switch set */
#define V86F_NOCHECK  2    /* NOCHECK switch set */
#define V86F_EMX      4    /* EMX compat switch set */

#define CHECK00FFONLY 0    /* I=TEST checks pages for 0x00 and 0xFF values only */

/* structure for v86 monitor initialization */
/* this structure is also defined in JEMM386.INC */

typedef struct tagJEMMINIT {
    ulong MonitorStart;     /* XMS block physical start */
    ulong MonitorEnd;       /* XMS block physical end */
    ulong TotalMemory;      /* XMS highest physical address */
    ulong MaxMem16k;        /* MAX=xxxx (default 120M), in 16kB units */
    ulong XMSHandleTable;   /* far32 address XMS handle table */
    ushort MaxEMSPages;     /* Min(32M, MaxMem16k), in 16 kB units */
    ushort XMSControlHandle;/* XMS handle for memory block */
    ushort DMABufferSize;   /* DMA buffer size in kB */
    ushort Frame;
    uchar NoEMS;
    uchar NoFrame;
    uchar NoPool;
    uchar AltBoot;
    uchar NoVME;
    uchar NoVDS;
    uchar NoPGE;
    uchar NoA20;
    uchar NoVCPI;
    uchar NoInvlPg;
    uchar V86Flags;
} JEMMINIT;

JEMMINIT jemmini = {0,0,0,-1,0,0,0,
        DMABUFFDEFAULT,0xE000,0,0,0,0,0,0,1,0,0,-1,0};

/* external ASM functions used */

ushort IsProtectedMode(void);
int emmcall(uchar);
int xmscall(uchar);
int xmscall32(uchar);
int XMSinit(void);
int IsDPMI(void);
int VMwareDetect(void);
int EmmStatus(void);
int EmmUpdate(void);
/* int AddIfContiguousWithDosMem(int,int); */
/* int InstallXMSHandler(int); */
int TestForSystemRAM(void *, int, unsigned int *);

extern char szHelp[];

/* structures */

/* UMB block 'array' ; this structure is also defined in JEMM386.INC */
struct {
	ushort segment;
	ushort size;
} UMBsegments[UMB_MAX_BLOCKS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

struct streg32
{
	ulong eax;
	ulong ebx;
	ulong ecx;
	ulong edx;
};

struct streg16
{
	ushort ax;
	ushort bx;
	ushort cx;
	ushort dx;
};

/* public globals */

uchar startup_verbose = 0;
uchar NoHigh = 0;

unsigned char emmfunction;
struct streg16 emmreg16;

void (far *XMSdriverAddress)(void);
struct streg16 reg16;
struct streg32 reg32;

unsigned char bLoad = 0;
static unsigned char NoRAM = 0;

/* local globals */

static ulong  PotentialEmsVcpiMemory;

static ushort wFRAMEwanted = 0;		/* should be set from commandline */
static unsigned long dwMinOption = 0L;	/* MIN= amount for EMS/VCPI */

static uchar MinRequest = 0;	/* MIN= has been set */

static ulong  XmsLinearAdress;			/* address for our allocated memory */
static ulong  XmsAllocatedBytes;		/* memory reserved for EMM          */
static ulong  XmsHighestMemoryByte;		/* simply end of ALL memory         */

static ushort xmsspec3 = 0;
static unsigned long xmslargest = 0;
static unsigned long xmstotal   = 0;
/* static unsigned long xmshighest = 0; */

static uchar SystemMemory[256];	/* 256*4K pagesize = 1MB */
static char ExcludeTest = 0;
static char IncludeTest = 0;
static char szError[] = {"ERROR"};
static char szWarning[] = {"WARNING"};

/*****************************************************************************/

#if 0
static int xmscall(uchar function)
{
	asm mov dx,reg16.dx
    asm mov bx,reg16.bx
	asm mov ah,function

	XMSdriverAddress();

	asm mov reg16.dx,dx
	asm mov reg16.bx,bx
	asm mov reg16.ax,ax

	return reg16.ax;
}      
#endif

#if 0
/* various machinations required since TCC doesn't support 32-bit inline asm */
static int xmscall32(uchar function)
{
	asm db 0x66
	asm mov dx,reg32.edx_low
	asm db 0x66
	asm mov bx,reg32.ebx_low
	asm mov ah,function

	XMSdriverAddress();

	asm	db 0x66
	asm mov	reg32.edx_low,dx
	asm	db 0x66
	asm mov	reg32.ecx_low,cx
	asm	db 0x66
	asm mov	reg32.ebx_low,bx
	asm	db 0x66
	asm mov	reg32.eax_low,ax

	return reg32.eax_low;
}
#endif

#if 0
static int emmcall(uchar function)
{
	emmfunction = function;
	
	emmreg16.ax = (emmreg16.ax & 0xff ) | (function << 8);

/*	printf("%25s in  %04x %04x %04x %04x -", "",emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
*/
	asm mov dx,emmreg16.dx
	asm mov cx,emmreg16.cx
	asm mov bx,emmreg16.bx
	asm mov ax,emmreg16.ax

	asm int 0x67

	asm mov emmreg16.dx,dx
	asm mov emmreg16.cx,cx
	asm mov emmreg16.bx,bx
	asm mov emmreg16.ax,ax

/*	printf("%1s out %04x %04x %04x %04x \n", "",emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
*/
	return emmreg16.ax >>8;
	
}
#endif

#if 0
static int XMSinit(void)
{   
	{
   asm     mov ax, 4300h;
   asm     int 2fh;                 /*  XMS installation check */

   asm     cmp al, 80h;
   asm     jne not_detected;

   asm     mov ax, 4310h;           /*  XMS get driver address */
   asm     int 2fh;
        
   asm     mov word ptr XMSdriverAddress+0, bx;
   asm     mov word ptr XMSdriverAddress+2, es;

   asm     mov ax, 4309h;           /*  XMS get xms handle table */
   asm     int 2fh;

   asm     cmp al,43h;
   asm     jne no_table;

   asm     mov word ptr XMS_Handle_Table+0, bx;
   asm     mov word ptr XMS_Handle_Table+2, es;

	}
no_table:
   return 1;
not_detected:
   return 0;
}
#endif

/* try to detect VMware */

#if 0
static int VMwareDetect()
{
	asm	db 0x66			/* mov eax,564d5856h */
	asm	mov ax,0x5868
	asm	dw 0x564d
	asm	db 0x66			/* mov ecx,0ah */
	asm mov cx,0x0a
	asm	dw 0x0000
	asm db 0x66			/* mov ebx,ecx */
	asm	mov bx,cx
	asm	db 0x66			/* mov edx,5658h */
	asm	mov dx,0x5658
	asm	dw 0x0000
	asm	db 0x66			/* in eax,dx */
	asm	in ax,dx
	asm	db 0x66			/* cmp ebx,564d5868h */
	asm cmp bx,0x5868
	asm	dw 0x564d
	asm	jne failed
	return 1;

failed:
	return 0;
}
#endif

#if 0
void Pause()
{
	printf("Any key please");
	asm mov ax,0;
	asm int 0x16;
	
	printf("\n");
}	
#endif

/* end of assembler functions */


/* set memory type , but honour "EXCLUDE=" and "INCLUDE=" types */

static void pascal SetMemoryType(ushort addr, uchar type)
{
	uchar *mem = &SystemMemory[addr >>8];
	
	if (*mem == 'I' && type != 'X')
		;
	else
		*mem = type;
}	

/*
	'R' = RAM
    'E' = EPROM
    'S' = Shadow-RAM activated by UMBPCI
    'G' = GRAPHICS
	'V' = VMWARE allocated, but possibly re-usable via I= (0e800-0ebffh)
	
	'U' = possible UMB space, because nothing else found
	'P' = PAGEFRAME
	
	'I' = INCLUDE = forced from commandline
	'X' = EXCLUDE = forbidden from commandline
	
	
*/	

/* 
	search memory for ROMS, adapters, graphics,...
	
    builds SystemMemory map

    the "checks" which are done are:

    - memory range 0000-9FFF is regarded as "system memory" - dont touch
    - memory range A000-BFFF is regarded as "video memory" -dont touch
    - memory range C000-EFFF is scanned for ROMS,
      if one is found, it is checked if there are pages filled with
      0x00 or 0xFF.
    - memory range C000-EFFF is also checked for RAM. if found pages are
      regarded as "reserved" (must be explicitely included with I= or S=)
    - with option "X=TEST", memory range C000-EFFF is also tested if
      content is 0x00 or 0xFF, anything else excludes page

*/	

static void ScanSystemMemory(void)
{
	uint mem,i;
    uchar far * pmem;
    uchar uc;
	uchar reuse;

	
	for (mem = 0; mem < 0xa0; mem++) /* system memory - reserved */
		SetMemoryType(mem << 8,'R');
		
	for (mem = 0xf0; mem < 0x100; mem++)	/* system EPROM F000..FFFF */
		SetMemoryType(mem << 8,'E');

/* feel free to optimize graphics memory */

	for (mem = 0xa0; mem < 0xc0; mem++) 	/* graphics */
		SetMemoryType(mem << 8,'G');

	if (VMwareDetect()) {
		/* exclude E800-EFFF for VMware, if present */
		/* E800-EBFF range may be recoverable via I=, so
		use 'V' instead of absolute 'X' */
		for (mem = 0xe8; mem < 0xec; mem++)	{
			SetMemoryType(mem << 8,'V');
		}
		for (mem = 0xec; mem <= 0xf0; mem++) {
			SetMemoryType(mem << 8,'X');
		}
	}

    /* scan for ROMS */

	for (mem = 0xc000; mem < 0xf000;) {
		uint romsize;
		pmem = (uchar far *)MK_FP(mem,0);
		
		if ( pmem[0] != 0x55u || pmem[1] != 0xaau) {
								/* no adapter EPROM */
			mem += 2048/16;		/* advance by 2K    */   
			continue;
		}

		/* ensure valid EPROM signature allocates minimum 2K for 0 size */
		/* keep romsize aligned to 2K boundary, rounding up */
        romsize = (((uint)pmem[2] * 2 + 3) / 4);

		if (startup_verbose)	
			printf("EPROM at %X, size %u kB\n", mem, romsize);
								/* romsize given in 512 byte*/

		for ( i = 0; i < romsize; i+=2)	{
            if (SystemMemory[mem>>8] == 'X')
                ;
            else
			if (!IncludeTest || (mem & 0xff)) {
				SetMemoryType(mem,'E');
			} else {
				pmem = (uchar far *)MK_FP(mem, 0);
                reuse = 1;
#if CHECK00FFONLY
		/* it the whole 4K block filled with 00 or FF? */
				for (i = 0; i < 4096; i++) {
                    if (*(pmem+i) != 0 && *(pmem+i) != 0xff) {
#else
        /* it the whole 4K block filled with one value? */
        /* and do NOT check the last byte in the page! */
				for (i = 0, uc = *pmem; i < 4095; i++) {
                    if (*(pmem+i) != uc) {
#endif
						SetMemoryType(mem,'E');
						reuse = 0;
						break;
					}
				}
				if (reuse) {
					SetMemoryType(mem,'I');
					mem += 2048/16;	/* advanced by 2K to make 4K total */
				}
			}
			mem += 2048/16;		/* advance by 2K	*/
		}
	}

#if 1
    /* test if RAM is in the non-excluded regions. */
    i = 0;
    do {
        if (i = TestForSystemRAM(&SystemMemory, i, &mem)) {
            printf("System memory found at %X-%X, region might be in use\n", i, i+mem-1);
            i = (i + mem) >> 8;
        }
    } while (i);
#endif

	/* if ExcludeTest is set, we need to scan all 'U' memory and ensure
     it is all 0 or FF values */

	if (ExcludeTest) {
		for (mem = 0xa0; mem < 0xf0; mem++) {
			if (SystemMemory[mem] == 'U') {
				/* this would be an upper memory block by default */
				/* don't check final byte as this can be garbage */
				pmem = (uchar far *)MK_FP(mem << 8, 0);
                for (i = 0; i < 4095; i++, pmem++) {
                    uc = *pmem;
					if ((uc != 0) && (uc != 0xff)) {
						/* found a nonzero, non-FF value in memory block */
						/* mark the whole block as excluded */
						SystemMemory[mem] = 'X';
						break;
					}
				}
			}
		}
	}

	/* for (i = 0xa0; i < 0xf8; i++)
			printf("%x00 : %c\n",i,SystemMemory[i]); */
}

/* 
	find a contiguous area of 64 KB 
	should handle commandline option like "FRAME=D000"
*/	
static ushort LocatePageFrame(void)
{   
    int page,i;
	ushort frame = 0;
    uchar bSearching = 0;
    uchar bWarning = 0;
    uchar bHardWanted = 0;

    if (wFRAMEwanted)
        bHardWanted = 1;
    else
        wFRAMEwanted = 0xE000;

	page = wFRAMEwanted >> 8;

    /* if a FRAME is explicitely set, ignore RAM/Video pages above A000 */

	for (i = 0; i < 16; i++) {
		if (bHardWanted && (page >= 0x80) && ((SystemMemory[page+i] == 'R') || (SystemMemory[page+i] == 'G')))
            bWarning = 1;
        else
			if (SystemMemory[page+i] != 'U')
				break;
	}
	if (i == 16) {
		frame = page;
		goto frameset;
    }
    if (bHardWanted)
        printf("Selected page frame %04x not accepted, scanning for a valid one...\n", wFRAMEwanted);

	bSearching = 1;

	for (page = 0xa0; page <= 0xE8; page++)	{
		for (i = 0; i < 16; i++) {
			if (SystemMemory[page+i] != 'U')
				break;
		}
		if (i == 16) {
			frame = page;
		}
	}

	if (frame == 0)	{
/*		printf("no suitable page frame found, which we can't handle\n");	*/
        printf("%s: no suitable page frame found, EMS functions limited.\n", szWarning);
		jemmini.NoFrame = 1;
		return 0;
	}


frameset:
	if (startup_verbose || bSearching)
		printf("Using page frame %02x00\n", frame);
	if (bWarning && (!bSearching))
        printf("%s: page frame %02x00 might not work reliably\n", szWarning, frame);

	memset(SystemMemory+frame,'P',16);
		
	return frame << 8;
}

/* old:check if there is 16 KB contiguos memory here to be used as UMB */ 
/* new:check if there is 4K KB contiguous memory here to be used as UMB */ 

static int pascal IsUMBMemory(ushort page)
{
    if (NoRAM && (page != 0xff))
        return FALSE;
	if ((SystemMemory[page] == 'U') || (SystemMemory[page] == 'I') ||
	( (jemmini.NoEMS || jemmini.NoFrame) && SystemMemory[page] == 'P') )
        return TRUE;
	else
		return FALSE;
}

/* display EMS function result if failed */

static void emmerror(char *s)
{
	printf("EMM failed: %s\n",s);
	
	printf("func %02x, out %04x %04x %04x %04x\n",emmfunction, emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
}

/* test if page is (shadow) RAM */

static int isShadowRAM(ushort base)
{
    uchar far * pmem;
    int rc = FALSE;
    uchar c;

    if (SystemMemory[base] == 'S') {
        pmem = (uchar far *)MK_FP(base,0);
        c = *pmem;
        *pmem = 0x55;
        if (*pmem == 0x55) {
            *pmem = 0xAA;
            if (*pmem == 0xAA)
                rc = TRUE;
        }
        *pmem = c;
    }
	return rc;
}

/*
	return number of pages, we need to do UMB mapping ,
	return value is in 4K pages
	does not count mapping at FF00
*/
static int UMBpageswanted(void)
{
    int wanted = 0;
    int i;
	for (i = 0xa0; i < 0xf8; i++)
		if (IsUMBMemory(i)) {
			wanted++;
		}

	return wanted;
}	

/* get total/free XMS memory */

static int GetXMSMemoryStatus(ushort usev3)
{
	int badstatus = 0;
	if (usev3) {
        if (xmscall32(0x88)) {  /* query free extended memory */
            xmslargest = reg32.eax;
            xmstotal   = reg32.edx;
            goto done;
        }
        badstatus = reg32.ebx & 0xff;
	}
	reg16.bx = 0;
    if (xmscall(8)) {  /* query free extended memory */
        xmslargest = reg16.ax;
        xmstotal   = reg16.dx;
        badstatus  = 0;
    } else
        badstatus = reg16.bx & 0xff;
done:
    if (badstatus)
        return 0;
    return 1;
}

/* if XMS manager doesnt export handle table, calc a reasonable
 amount of fixed memory to allocate
*/

static unsigned int GetReasonableFixedEMSAmount(void)
{
    unsigned int wWanted;

    if (!GetXMSMemoryStatus(xmsspec3))
        return 0;
    if (xmslargest >= (64 * 1024UL))
        wWanted = 32 * 1024;
    else
        wWanted = (unsigned int)(xmslargest / 2);
    return wWanted;
}

/* alloc XMS memory block for EMM
*/

static ushort AllocXMSMemory(int usev3, ulong kbneeded)
{
    ushort xmshandle = 0;
    ulong kbtotal;

    for ( ; !xmshandle; dwMinOption /= 2) {
        kbtotal = kbneeded + dwMinOption;
        if (kbtotal <= 0xFFFFUL) {
            reg16.dx = kbtotal;
            if (xmscall(9)) {
                xmshandle = reg16.dx;
                break;
            }
            if (usev3) goto usev3ver;
            /* try largest free block based allocation */
            if ((xmslargest > kbneeded + dwMinOption / 2) &&
                (xmslargest < kbtotal)) {
                reg16.dx = xmslargest;
                if (xmscall(9)) {
                    dwMinOption = xmslargest - kbneeded;
                    xmshandle = reg16.dx;
                    break;
                }
            }
        } else {
        usev3ver:
            reg32.edx = kbtotal;
            if (xmscall32(0x89)) {
                xmshandle = reg32.edx & 0xFFFF;
                break;
            }
            /* try largest free block based allocation */
            if ((xmslargest <= kbneeded + dwMinOption / 2) ||
                (xmslargest >= kbtotal)) {
                /* outside range, try next loop */
                break;
            }
            reg32.edx = xmslargest;
            if (xmscall32(0x89)) {
                dwMinOption = xmslargest - kbneeded;
                xmshandle = reg32.edx & 0xFFFF;
                break;
            }
        }
        if (startup_verbose)
            printf("Allocated %lu kB (%lu + %lu) from XMS\n", kbtotal, kbneeded, dwMinOption);

    }
    return xmshandle;
}

/*
	allocate memory from XMS
	find highest memory address
    determine monitor load address (must currently be <= 15MB
    dwMinOption = kB explicitely wanted for EMS with MIN=xxxx
    + memory required for UMBs
*/

static int XMSallocAndInitMem(unsigned long kbneeded)
{   
    unsigned long ulcalc;
    unsigned u;
    unsigned long dwEMSoriginal = dwMinOption;

    if (!GetXMSMemoryStatus(xmsspec3)) {
        printf("%s: can't get XMS memory status\n", szError);
		return 0;
	}


	if (startup_verbose)	
		printf("XMS largest block %lu kB, XMS total mem %lu kB\n",
				xmslargest, xmstotal);

/* reality check to throttle requests far beyond available XMS, later actual
	adjustments	are small and need not be compensated for here */
	if ((dwMinOption + kbneeded > xmstotal) && (kbneeded < xmstotal)) {
		dwMinOption = xmstotal - kbneeded;
    }

/* leave a little extended memory, if possible, for programs that want some XMS */
	if ((xmslargest > kbneeded + 384UL) && (xmslargest < kbneeded + dwMinOption + 384UL)) {
        dwMinOption = xmslargest - kbneeded - 384UL;
	}

/* kbwanted is memory in EMS pages, must be 16 kB aligned */

    dwMinOption = (dwMinOption + 15) & 0xfffffff0l;
    if (dwMinOption > MAXK_EMS_ALLOWED) {
        dwMinOption = MAXK_EMS_ALLOWED;
    }

    /* default is: all memory */
    PotentialEmsVcpiMemory = xmstotal * 1024UL;

    if (jemmini.NoPool)	{ /* Pool sharing off? */
        if (dwMinOption < xmstotal)
            PotentialEmsVcpiMemory = dwMinOption * 1024UL;
	}

    if (startup_verbose)
        printf("Potential EMS/VCPI memory: %lu kB\n", PotentialEmsVcpiMemory / 1024UL);

    /* if MIN= is set, increase MAXEMSPAGES and MAXMEM16 if needed */
    /* here dwMinOption is always below 512 MB */
    if (dwMinOption > jemmini.MaxEMSPages * 16UL) {
        jemmini.MaxEMSPages = dwMinOption / 16;
        if (jemmini.MaxEMSPages > jemmini.MaxMem16k)
            jemmini.MaxMem16k = jemmini.MaxEMSPages;
/*      printf("MAXMEM16K=%lu, MAXEMSPAGES=%u\n", jemmini.MaxMem16k, jemmini.MaxEMSPages); */
    } else
        if (jemmini.NoEMS) {
            jemmini.MaxEMSPages = dwMinOption / 16;
  /* even with NOEMS alloc space for EMS memory management for 8192 kb */
  /* else some DOS extenders (among them DOS4G) won't work (wastes 2,5 kB) */
#if 1
            if (jemmini.MaxEMSPages < 512)
                jemmini.MaxEMSPages = 512;
#endif
        }

    /* jemmini.MaxMem16k may have been set by MAX=, and above the limit */
    if (jemmini.MaxMem16k > (PotentialEmsVcpiMemory / (1024UL * 16)))
        jemmini.MaxMem16k = PotentialEmsVcpiMemory / (1024UL * 16);

    /* jemmini.MaxMem16k may have been set by MAX=, and below 32 MB! */
    /* this is valid, but then adjust max EMS pages as well */
    if (jemmini.MaxMem16k < jemmini.MaxEMSPages) {
        jemmini.MaxEMSPages = jemmini.MaxMem16k;
    }

/*  the memory pooling need ((XMS total / 1.5M) + 1) * 64 bytes
    for pool allocation	table entries
    1.5M is pool allocation maximum memory control,
    64 is pool block size,
    if dynamic XMS allocation is on, 128 more items are needed,
    which represent the maximum number of XMS handles
*/

	ulcalc = jemmini.MaxMem16k * 16 / 1536 + 2; /* no of pool items */
	if (!jemmini.NoPool)
		ulcalc = ulcalc + 128;
	ulcalc = ulcalc << 6;  /*  * 64 = pool mem to alloc */

	/* 4+1 bytes for each EMS page needed */
	/* 256*8 bytes for status table of EMS handles */
	ulcalc = ulcalc + 5 * jemmini.MaxEMSPages + 256*8;
    /* 256*8 bytes for EMS names [8 bytes if NOEMS] */
#if 0 /* it is valid to alloc EMS even with NOEMS! */
	if (jemmini.NoEMS)
		ulcalc = ulcalc + 8;
    else
#endif
		ulcalc = ulcalc + 256*8;
	ulcalc = (ulcalc + 1023) / 1024UL;	/* convert bytes back to K */
    ulcalc = (ulcalc + 3) & 0xfffffffc; /* 4k page align */
	if (startup_verbose)
		printf("%lu kB needed for VCPI and EMS handling\n", ulcalc);
    kbneeded += ulcalc;
#if 1
    /*
     the DMA buffer must be 64kb aligned. Since EMS pages *must* be
     allocated from memory physically "behind" the DMA buffer, there may
     be some space wasted, max 60-16=44 kB
    */
//    if (dwMinOption && (jemmini.DMABufferSize > 4)) {
    if ((dwMinOption || jemmini.NoPool) && (jemmini.DMABufferSize > 4)) {
        u = jemmini.DMABufferSize >= 64 ? 32 : jemmini.DMABufferSize/2;
        kbneeded = kbneeded + u;
        if (startup_verbose)
            printf("%u kB added to account for DMA buffer 64 kB alignment\n", u);
    }
#endif

	/* allocate memory from XMS */
    jemmini.XMSControlHandle = AllocXMSMemory(xmsspec3, kbneeded);

	if (!jemmini.XMSControlHandle) {
        printf("%s: can't allocate enough XMS memory(%lu kB)\n", szError, kbneeded);
		return 0;
	}

	/* lock handle to make a linear adress */
	
	reg16.dx = jemmini.XMSControlHandle;
	if (!xmscall(0x0c))	{
        printf("%s: can't lock XMS memory\n", szError);
        xmscall(0x0A);/* free the block */
		return 0;
    }
    if (dwMinOption < dwEMSoriginal)
        printf("%s: EMS memory has been reduced to %lu kB\n", szWarning, dwMinOption);

	XmsAllocatedBytes =  ((ulong)(dwMinOption + kbneeded) * 1024);
	
    XmsLinearAdress	     = 	((ulong)reg16.dx << 16) | reg16.bx;

	XmsHighestMemoryByte =  (ulong)xmstotal * 1024L + XmsLinearAdress;


/*	printf("xms locked memory at %lx, top of mem %lx(%luMB) alloc bytes %lx(%lu kB)\n",
				XmsLinearAdress,
				XmsHighestMemoryByte,XmsHighestMemoryByte/(1024*1024l),
				XmsAllocatedBytes,XmsAllocatedBytes/(1024));
*/

	return 1;	
	
}   

/* prepare UMB segment table */

static int PrepareUMBSegments(void)
{

	ushort mem,size,index;

				/* now prepare UMB segments to be used later */
	index = 0;
	for (mem = 0xa0; mem < 0xf8; )	/* allow umbs in f000-f7ff */
		if (((!isShadowRAM(mem)) && (!IsUMBMemory(mem))))
			mem++;	/* allow 4K UMB's	*/
		else {
            for (size = 1; ; size++)	/* 4K UMB's */
				if (((!isShadowRAM(mem+size)) && (!IsUMBMemory(mem+size))))
					break;

#if 0
            if (mem == 0xa0) {
                if (AddIfContiguousWithDosMem(mem << 8, size)) {
                    mem += size;
                    continue;
                }
            }
#endif

			UMBsegments[index].segment = mem  << 8;
			UMBsegments[index].size    = size << 8;			


			if (startup_verbose) 
                printf("UMB #%d at %x0-%xf (%u kB)\n",
					index,
					UMBsegments[index].segment,
					UMBsegments[index].segment + UMBsegments[index].size - 1,
					UMBsegments[index].size / (1024/16)
                      );

			index++;
			if (index >= UMB_MAX_BLOCKS)
				break;

			mem += size;				
		}

    if (startup_verbose && (UMBsegments[0].segment == 0)) {
		printf("No UMBs provided\n");
		return 0;
	}

    return 1;
}


static void PrintStartup()
{
    printf( PROGRAM " v5.45 [" __DATE__ "]\n");
    return;
}

static void PrintCopyright()
{
    printf(PROGRAM " is derived from FD Emm386 (c) tom ehlert 2001-2006 c't/H. Albrecht 1990\n");
    return;
}

/* conditional printf only if startup_verbose */

static void cprintf(char * pszText)
{
    if (startup_verbose)
        printf(pszText);
}

/* called on startup.
	handle commandline "I=B800-BFFF X=E000-EFFF" ...
	search for EPROMS+adapters (network cards)
	determine frame address
	...
	mode = 0 if called as driver
	mode = 1 if called as EXE
	
	return: 0         - everything fine
	        errorcode - exit code/abort driver
*/ 

int TheRealMain(int mode, char *commandline)
{
    char *found;
    int i,j;
    int bHelp = 0;
    int bOptionSet = 0;

    if (mode != EXECMODE_EXE) {
        PrintStartup();
        bLoad = 1;
    } else {
#ifdef UNLOADSUPP
        if (FindCommand(commandline, "UNLOAD", 0) ) {
            XMSinit();
            TryUnload();
            return 1;
        }
#endif
#ifdef LOADSUPP
        if (FindCommand(commandline, "LOAD", 0) )
            bLoad = 1;
        else {
            jemmini.NoVCPI = 0xFF;
#ifdef A20SUPP
            jemmini.NoA20  = 0xFF;
#endif
#ifdef VMESUPP
            jemmini.NoVME  = 0xFF;
#endif
#ifdef PGESUPP
            jemmini.NoPGE  = 0xFF;
#endif
        }
#endif
        if (FindCommand(commandline, "/?", 0) ||
            FindCommand(commandline, "-?", 0) ||
            FindCommand(commandline, "/H", 0) ||
            FindCommand(commandline, "-H", 0) ) {
            bHelp = 1;
        }
    }

/*  printf("'real' commandline is '%s'\n",commandline); */

	if (FindCommand(commandline, "NOVCPI", 0) ) {
		jemmini.NoVCPI = TRUE;
        bOptionSet = 1;
	}
	if (FindCommand(commandline, "VCPI", 0) ) {
		jemmini.NoVCPI = FALSE;
        bOptionSet = 1;
    }
#ifdef A20SUPP
    if (FindCommand(commandline, "NOA20", 0) ) {
        jemmini.NoA20 = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "A20", 0) ) {
        jemmini.NoA20 = 0;
        bOptionSet = 1;
    }
#endif
#ifdef VMESUPP
    if (FindCommand(commandline, "NOVME", 0) ) {
        jemmini.NoVME = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "VME", 0) ) {
        jemmini.NoVME = 0;
        bOptionSet = 1;
    }
#endif
#ifdef PGESUPP
    if (FindCommand(commandline, "NOPGE", 0) ) {
        jemmini.NoPGE = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "PGE", 0) ) {
        jemmini.NoPGE = 0;
        bOptionSet = 1;
	}
#endif
    if (bLoad) {
	memset(SystemMemory,'U',sizeof(SystemMemory));

	/******* commandline handling **********/

	if (FindCommand(commandline, "/VERBOSE", 0) ||
	    FindCommand(commandline, "/V",       0) ) {
		startup_verbose = 1;
	}

    if (FindCommand(commandline, "NODYN", 0)) {
		jemmini.NoPool = 1; /* pool sharing off */
	}

    if (FindCommand(commandline, "NOINVLPG", 0)) {
		jemmini.NoInvlPg = 1; /* dont use INVLPG opcode */
	}

    if (FindCommand(commandline, "MIN=", &found)) {
		dwMinOption = GetValue(found,10,TRUE);
		if (startup_verbose)	
            printf("Wanted preallocated EMS/VCPI memory: %lu kB\n", dwMinOption);
		MinRequest = 1;
	}

	if (FindCommand(commandline, "NOEMS", 0) )	{
		cprintf("NOEMS: EMS disabled (mostly :-)\n");
		jemmini.NoEMS = TRUE;
	}

	if (FindCommand(commandline, "NOVDS", 0) )	{
		jemmini.NoVDS = TRUE;
	}

	if (FindCommand(commandline, "VDS", 0) ) {
		jemmini.NoVDS = FALSE;
	}

    if (FindCommand(commandline, "FRAME=NONE", 0)) {
        jemmini.NoFrame = TRUE;
    } else {
        if (FindCommand(commandline, "FRAME=", &found) ||
            FindCommand(commandline, "/P",     &found) ) {
			wFRAMEwanted = GetValue(found,16,FALSE);
            if (startup_verbose)
                printf("Wanted page frame=%X\n", wFRAMEwanted);
		}
	}

	if (FindCommand(commandline, "X=TEST", 0) ) {
		ExcludeTest = 1;
	}

	if (FindCommand(commandline, "I=TEST", 0) ) {
		IncludeTest = 1;
	}

#ifdef SBSUPP
	if (FindCommand(commandline, "SB", 0) ) {
		jemmini.V86Flags = jemmini.V86Flags | V86F_SB;
    }
#endif
#ifdef EMXSUPP
	if (FindCommand(commandline, "EMX", 0) ) {
		jemmini.V86Flags = jemmini.V86Flags | V86F_EMX;
	}
#endif
#if 0
    if (FindCommand(commandline, "MEMCHECK", 0) ) {
		MEMCHECK = 1;
	}
#endif

	if (FindCommand(commandline, "NOCHECK", 0) ) {
		jemmini.V86Flags = jemmini.V86Flags | V86F_NOCHECK;
	}

	if (FindCommand(commandline, "MAX=", &found) ) {
		jemmini.MaxMem16k = GetValue(found,10,TRUE);
	}

/*
	if (FindCommand(commandline, "ENDALLOC", &found) )
		{ 
		ENDALLOC = 1;
		}
*/

	if (FindCommand(commandline, "ALTBOOT", 0) ) {
		jemmini.AltBoot = 1;
	}

    if (FindCommand(commandline, "NOHI", 0) ) {
		NoHigh = 1;
	}

	if (FindCommand(commandline, "NOMOVEXBDA", 0) ) {
		/* NOMOVEXBDA is a no-op, but helps MS EMM386 switch compatibility */
	}

	if (FindCommand(commandline, "NORAM", 0) ) {
		NoRAM = 1;
	}
	if (FindCommand(commandline, "RAM", 0) ) {
		NoRAM = 0;
	}


    if (FindCommand(commandline, "D=", &found)) {
        ushort dmasize =  GetValue(found,10,FALSE);
        if (dmasize <= 128)
            jemmini.DMABufferSize = (dmasize+3) & -4;
        else {
            jemmini.DMABufferSize = 128;
            printf("%s: wanted DMA buffer size too large, set to 128 kB\n", szWarning);
        }
    }
					/* "I=a000-afff"  "X=d000-dfff" */
	for (;;) {
		ushort rangestart,rangestop;
		char memtype;
		
		memtype = 'I';
		
        if (!FindCommand(commandline, "I=", &found) ) {
            memtype = 'S';
            if (!FindCommand(commandline, "S=", &found) ) {
                memtype = 'X';
                if (!FindCommand(commandline, "X=", &found) ) {
                    break;
                }
            }
		}

		rangestart =  GetValue(found,16,FALSE);

		if (*found == '-') {
			memcpy( found, found+1,
								strlen(found+1) +1);

		
			rangestop  =  GetValue(found,16,FALSE);
		
            if (startup_verbose)
                printf("Accepted %c=%x..%x\n",memtype, rangestart,rangestop);
			
			if (rangestart && rangestop && (rangestart <= rangestop))
				for ( ; rangestart < rangestop; rangestart++)
					SetMemoryType(rangestart,memtype);
				
		}
		
    }
    } /* endif bLoad */

	/******* commandline handling done, are there remainders **********/

	commandline = skipWhite(commandline);
    if (*commandline >= '0' && *commandline <= '9') {
        jemmini.MaxMem16k = GetValue(commandline,10,TRUE);
        commandline = skipWhite(commandline);
    }

    if (*commandline) {
        printf("* ignored commandline: '%s'\n", commandline);
        if (mode == EXECMODE_EXE)
            return 1;
    }

    /* shall the monitor be loaded? If no, do either:
		display usage info    or
		status report
	*/
	if (!bLoad) {
        PrintStartup();
        if (bHelp) {
            PrintCopyright();
            printf(szHelp);
        } else {
            if (bOptionSet) {
                if (EmmUpdate())  /* update Jemm386 status */
                    printf("option(s) passed to installed instance of " PROGRAM "\n");
            } else {
                EmmStatus();  /* display Jemm386 status */
            }
        }
		return 1;
	}

    /******* options set, now process **********/

	if (IsProtectedMode()) {
		IsEmmInstalled();
		return 1;
	}

	if (!XMSinit())	{
		printf("%s: no XMS handler found, required\n", szError);
		return 1;
    }
#if 1
    /* its not the responsibility of the EMM to ensure an installed DPMI
     host won't get confused by the EMM to uninstall. But the other way
     is to avoid: install an EMM while a DPMI host is installed.
    */
	if (IsDPMI())	{
        printf("%s: DPMI host detected\n", szError);
		return 1;
    }
#endif
    if (xmscall(0) && (reg16.ax >= 0x300))
        xmsspec3 = 1;

    if (!xmscall(5)) {  	/* enable A20 local */
		printf("%s: enable A20 failed\n", szError);
		return 1;
    }

/* system state ok for loading driver */

    if (jemmini.MaxMem16k == -1)
        jemmini.MaxMem16k = MAXMEM16K_DEFAULT;
    else
		jemmini.MaxMem16k /= 16UL;

    if (jemmini.NoVCPI)
        cprintf("VCPI disabled\n");
    if (jemmini.NoVDS)
		cprintf("VDS disabled\n");
#if 0
    /* auto disable pooling is no longer useful */
    /* if wanted it can be disabled with NODYN */
    if (jemmini.NoVCPI && jemmini.NoEMS)
        jemmini.NoPool = 1;
#endif
    /* if no int 2fh, function 4309h support, disable pool sharing */

    if ((!jemmini.XMSHandleTable) && (!jemmini.NoPool)) {
        jemmini.NoPool = 1;
        printf("%s: XMS host doesn't provide handle array, dynamic memory allocation off!\n", szWarning);
	}
    if ((jemmini.NoPool) && (!MinRequest)) {
        dwMinOption = GetReasonableFixedEMSAmount();
    }

	ScanSystemMemory();			/* build up system memory map */

    if (jemmini.NoEMS)
        jemmini.NoFrame = 1;

	if (!jemmini.NoFrame)
		jemmini.Frame = LocatePageFrame();	/* find a contiguos area of 64 KB */


	/*
			allocate from XMS the memory we need
    		this is memory for UMBs, including FF00 if ALTBOOT=0

			+ 16kB for the monitor code, GDT, IDT, stack
			+ 12kB for page tables
			+ 12kB for TSS + IO-Bitmap (includes 3 kB reserve for rounding)
			+ 64kB +-X for DMA buffering
			+ room for other control structures made inside function
								
			+ what the user wants for EMS
	*/

    i = (UMBpageswanted() + (jemmini.AltBoot ? 0 : 1)) * 4;

    jemmini.MaxEMSPages = MAX_EMS_PAGES_ALLOWED + (i + 15) / 16;

    j = 16 + 24 + jemmini.DMABufferSize;

    if (startup_verbose)
        printf("Needed: %u kB for UMBs, %u kB for DMA buffer, %u kB for monitor\n",
               i, jemmini.DMABufferSize, j - jemmini.DMABufferSize);
    dwMinOption = dwMinOption + i;

	if (!XMSallocAndInitMem(j)) {
        xmscall(6); /* local disable A20 */
		return 1;
	}

	jemmini.MonitorStart = XmsLinearAdress;
	jemmini.MonitorEnd   = XmsLinearAdress + XmsAllocatedBytes;
	jemmini.TotalMemory  = XmsHighestMemoryByte;

	if (startup_verbose)	
        printf("XMS memory block for monitor: %lx-%lx, XMS total=%lx\n",
			jemmini.MonitorStart, jemmini.MonitorEnd, jemmini.TotalMemory);

	return 0;				/* OK so far , continue */
}


/*  
	post processing. runs in v86 mode already.
	
	here some magic happens:
	we use (mostly) standard EMS functionality to map some memory
	at the UMB locations
*/

int TheV86Main(ulong ulFirstPage)
{   
	int i;
	ushort pageswanted; 
    ushort logicalpage;
    int    bInstallUMB = 1;

    if (startup_verbose) {
        printf("Physical start address of EMS pages: %lX\n", ulFirstPage);
        if (!emmcall(0x42))	{ /* kickstart dynamic count of available EMS pages */
            printf("Total/available EMS pages: %d/%d (= %lu(%lu) kB)\n",
               emmreg16.dx, emmreg16.bx, (ulong)emmreg16.dx*16, (ulong)emmreg16.bx*16);
        }
	}


/* check install state */
/*	
	emmcall(0x46);
	printf("version %x\n",emmreg16.ax);

	emmcall(0x41);
	printf("page frame at %x\n",emmreg16.bx);
	
	emmcall(0x42);
	printf("total pages %x(%dMB), free %x(%dMB)\n",emmreg16.dx,emmreg16.dx/(1024/16),emmreg16.bx,emmreg16.bx/(1024/16));

	emmcall(0x4b);
	printf("emm handles %x\n",emmreg16.bx);
*/
/*
	for (mem =0xa0; mem < 0xf8; mem+=4)
		printf("mem %x %c - %d\n",mem,SystemMemory[mem], IsUMBMemory(mem));
*/

    pageswanted = UMBpageswanted();

    /* if ALTBOOT not set, add 1 for ROM FF00h shadowing */
    /* then map page at FFxxx so we can trap jumps to FFFF:0 for rebooting */
    /* the 0xff physical page flags EMM386 to copy over ROM image */
    /* with ALTBOOT, a keyboard hook is installed instead */

    if (!jemmini.AltBoot) {
        pageswanted++;
        SetMemoryType(0xff00, 'U');
    }

	if (!pageswanted) {
        bInstallUMB = 0;
		goto mapping_done;
	}

	/* allocate a handle + some memory */

	if (startup_verbose)
		printf("Allocating %u 4K pages = %u kB for UMBs\n",
									pageswanted,pageswanted*4);

/* UMB allocations done through system handle via reallocation, since
 system handle is already present with zero assigned pages */
    emmreg16.dx = 0;					/* use system handle 0 */
	emmreg16.bx = (pageswanted+3)/4;	/* round up */
	if (emmcall(0x51) != 0)	{
		printf("Allocating %d kB for UMBs:",pageswanted*4);
		emmerror("");
        bInstallUMB = 0;
		goto mapping_done;
	}

	/*  here comes the funny part:
		during initialization phase, calling EMM_MAP_PAGE (0x44)
		with pysical page (AL) > 3 is possible.
		meaning:
		AL = highest 8 bits of logical adress, AL=E4 --> address E4000
		initialization phase is terminated with AL=9F
	*/

								/* map pages in to the UMB area */

	cprintf("Mapping UMBs (4K each) at:");

	logicalpage = 0;
	
	for (i = 0xa0; i < 0x100; i++)	/* allow 4K UMB's */
		if (IsUMBMemory(i))	{
			if (startup_verbose)	
				printf("%x00 ",i);     
		
			emmreg16.ax = i;
			emmreg16.bx = logicalpage;
            emmreg16.dx = 0;		/* use system handle 0 */
		
            if (emmcall(0x44)) {
                emmerror("mapping UMB page !!");
                bInstallUMB = 0;
				goto mapping_done;
            }

			logicalpage++;
		}
			
	cprintf("\n");

mapping_done:

    if (bInstallUMB)
        bInstallUMB = PrepareUMBSegments();

    return bInstallUMB;

}
/* EOF */
