;
; XDMA32.ASM  - a JLM driver for UltraDMA hard-disks
; based on XDMA v3.3 by Jack R. Ellis
; released under the GNU GPL license v2 (see GNU_GPL.TXT for details)
;
; This source is best viewed with tab size 4.
; It is to be assembled with MASM v6 or above!
;
; XDMA32 switch options are as follows:
;
; /L   Limits DMA to "low memory" below 640K. /L is REQUIRED to use
;	   UMBPCI or any similar driver whose upper-memory areas do not
;	   allow DMA.	 /L causes I-O requests past 640K to go through
;	   the driver's buffer.
; /Q   Quiet mode.
;
; On exit from successful I-O requests, the AH-register is zero and the
; carry flag is reset.	 If an error occurs, the carry flag is SET, and
; the AH-register has one of the following codes:
;
;	Code 08h - DMA timed out.
;		 0Fh - DMA error.
;		 20h - Controller busy before I-O.
;		 21h - Controller busy after I-O.
;		 AAh - Disk not ready before I-O.
;		 ABh - Disk not ready after I-O.
;		 CCh - Disk FAULT before I-O.
;		 CDh - Disk FAULT after I-O.
;		 E0h - Hard error at I-O end.
;		 FFh - XMS memory error.
;
; General Program Equations.
;
VER		equ <' V1.0, 11-25-2007'>
RDYTO	equ	08h			;389-msec minimum I-O timeout.
BIOSTMR equ	046Ch		;BIOS "tick" timer address.
HDISKS	equ	0475h		;BIOS hard-disk count address.
HDIOFS	equ 048Eh		;BIOS flag for HD IRQ
NUMDSK	equ 4			;max HDs supported (min is 4)
RESSIZE	equ 20h			;resident size DOS part of driver
SAVESTAT equ 1			;save/restore client state on init
SCATTER	equ 1			;support DMA scatter/gather to overcome 
						;64 kb boundaries
IRQWND	equ 1			;allow interrupts during memory copy                        

CR		equ	00Dh		;ASCII carriage-return.
LF		equ	00Ah		;ASCII line-feed.

@byte	equ <byte ptr>
@word	equ <word ptr>
@dword	equ <dword ptr>
;
; Driver Return Codes.
;
DMATIMO equ	0E8h		;DMA timeout code, 008h at exit.
DMAERR	equ	0EFh		;DMA error	 code, 00Fh at exit.
CTLRERR equ	000h		;Ctlr. busy  code, 020h/021h at exit.
DISKERR equ	08Ah		;Disk-busy	 code, 0AAh/0ABh at exit.
FAULTED	equ	0ACh		;Disk FAULT  code, 0CCh/0CDh at exit.
HARDERR equ	0BFh		;Hard-error  code, 0E0H at exit.
XMSERR	equ	0FFh		;XMS memory-error code.
;
; IDE Controller Register Definitions.
;
CDATA	equ	0h			;offset Data port.
CSECCT	equ	CDATA+2		;offset I-O sector count.
CDSEL	equ	CDATA+6		;offset Disk-select and upper LBA.
CCMD	equ	CDATA+7		;offset Command register.

CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY		equ	080h		;IDE controller is busy.
RDY		equ	040h		;IDE disk is "ready".
FLT		equ	020h		;IDE disk has a "fault".
DRQ		equ	008h		;IDE data request.
ERR		equ	001h		;IDE general error flag.
DMI		equ	004h		;DMA interrupt occured.
DME		equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
						;	 LBA48 commands are 025h/035h).
;
; LBA "Device Address Packet" Layout.
;
DAP struc
DapPL	db	?		;Packet length.
		db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
		db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (segment:offset).
DapLBA	dw	?		;48-bit logical block address (LBA).
DapLBA1	dd	?
DAP ends
;
; DPTE "Device Parameter Table Extension" Layout
;
DPTE struc
wIDEBase	dw ?	;IDE port base
wIDEAlt		dw ?	;alternate control port
bFlags		db ?	;drive flags (bit 4=1 -> drive is slave)
			db ?	;proprietary info
bIRQ		db ?	;IRQ for drive
DPTE ends


;
; DOS "Request Packet" Layout.
;
RP struc
RPHLen	db ?		;Header byte count.
RPSubU	db ?		;Subunit number.
RPOp	db ?		;Opcode.
RPStat	dw ?		;Status word.
		db 8 dup (?);(Unused by us).
RPUnit	db ?		;Number of units found.
RPSize	dd ?		;Resident driver size.
RPCL	dd ?		;Command-line data pointer.
RP ends

RPERR	equ	08003h		;Packet "error" flags.
RPDONE  equ	00100h		;Packet "done" flag.

		.386
		.model flat

		include jlm.inc
	
		.code

startcode label byte

dwBase		dd 0	;linear address driver base
dwCmdLine	dd 0
dwBufferLin	dd 0	;linear address XMS buffer
dwBufferPhy	dd 0	;physical address XMS buffer (64 kb aligned)
PrvI13		dd 0	;old real-mode int 13 vector
XMSEntry 	dd 0	;XMS "entry" address, set by Init
XMSHdl		dd 0	;XMS memory "handle", set by Init
wBaseSeg	dd 0

Flags	db	0		;I-O and IDE channel "busy" flags:
					;  080h = I-O request or timer busy.
					;  040h = Secondary channel is busy.
					;  020h = Secondary channel overlap.
					;  010h = Primary	channel is busy.
					;  008h = Primary	channel overlap.
					;  If overlap disabled, 0FFh = "busy".
ReqBF	db	0		;I-O request channel "busy" flag.
BiosHD	db	0		;(Number of BIOS disks, during Init).
HDCount	db	0		;(BIOS hard-disk count, during Init).
bOption	db	0

FO_Q	equ 1		;/Q option set
if SCATTER
FO_UF	equ 2		;/UF option
endif

		align 4

PRDAd	dd	offset IOAdr - offset startcode		;PRD 32-bit command addr. (Init set).

;--- the following is written to the IDE port 
;--- and has to be kept in this order!

SecCt2	db	0		;IDE "upper" sector count.
LBA2447 db	0,0,0	;IDE "upper" LBA bits 24-47.
SecCt	db	0		;IDE "lower" sector count.
LBA0023	db	0,0,0	;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.

		align 4

VDSLn	dd	0			;buffer length
VDSOf	dd	0			;linear address of buffer
IOAdr	dd	0			;DMA physical address
IOLen	dd	080000000h	;DMA byte count and "end" flag.
if SCATTER
IOAdr2	dd	0	
IOLen2	dd	80000000h
endif        

Units	db	NUMDSK dup (-1)	;IDE "active units" table (Init set).
CHSSec	db	NUMDSK dup (0)	;CHS sectors/head	table (Init set).
CHSHd	db	NUMDSK dup (0)	;CHS heads/cylinder table (Init set).
IDEAd	dw	1F0h, 1F0h, 170h, 170h
		dw	NUMDSK-4 dup (-1);IDE port base
IDEAlt	dw	3F6h, 3F6h, 376h, 376h
		dw	NUMDSK-4 dup (-1);IDE port base
DMAAd	dw	NUMDSK dup (-1)	;DMA port base
SelCmd	db	0E0h,0F0h,0E0h,0F0h
		db	NUMDSK-4 dup (0)

;--- table for PCI controller programming interfaces to search for.
;--- is Plug&Play IDE programming interface, found in table F0087 in
;--- RBIL FARCALL.LST.
; 7	bus mastering (read-only)
; 6-4	reserved (read-only)
; 3	secondary IDE mode bit is writable (read-only)
; 2	secondary IDE mode (0 = legacy, 1 = native)
; 1	primary IDE mode bit is writable (read-only)
; 0	primary IDE mode (0 = legacy, 1 = native)

iftab label byte
		db 08Ah, 080h	;legacy, BM
		db 08Fh, 085h	;native, BM
		db 0FAh, 0F0h
		db 0BAh, 0B0h
endiftab label byte

		align 4
;
; Driver Entry Routine.  For CHS requests, the registers contain:
;
;	AH		Request code.  We handle 002h read and 003h write.
;	AL		I-O sector count.
;	CH		Lower 8 bits of starting cylinder.
;	CL		Starting sector and upper 2 bits of cylinder.
;	DH		Starting head.
;	DL		BIOS unit number.  We handle 080h and up (hard-disks).
;	ES:BX	I-O buffer address.
;
; For LBA requests, the registers contain:
;
;	AH		Request code.  We handle 042h read and 043h write.
;	DL		BIOS unit number.  We handle 080h and up (hard-disks).
;	DS:SI	Pointer to Device Address Packet ("DAP"), described above.
;
;--- for JLMs all registers are in a client structure and EBP will
;--- point to it.

Entry:

ifdef _DEBUG
	mov	esi,offset DbgMsg7
	call DsplyM
	mov ax,@word [ebp].Client_Reg_Struc.Client_EAX
	call DsplyW
	cmp @byte [ebp].Client_Reg_Struc.Client_EAX+1, 42h
	jnz @F
	mov	esi,offset DbgMsg7a
	call DsplyM
	movzx esi,@word [ebp].Client_Reg_Struc.Client_DS
	movzx ecx,@word [ebp].Client_Reg_Struc.Client_ESI
	shl esi, 4
	add esi, ecx
	mov ax,@word [esi+6]
	call DsplyW
	mov ax,@word [esi+4]
	call DsplyW
@@: 
	call DsplyEOL
endif

	mov edx,[ebp].Client_Reg_Struc.Client_EDX
	mov	edi,1		;Reset active-units table index.
@LastU	equ	$-4		;(Last-unit index, set by Init).
NextU:
	dec	edi			;Any more active units to check?
	js	QuickX		;No, request NOT for us -- exit quick!
	cmp	dl,[edi+Units] ;Does request unit match our table?
	jne	NextU		;No, see if more table entries remain.
	mov	eax,edi		;Get driver & request "busy" flags.
	or	al,009h
	inc	eax
	shl	al,4
	test [Flags],al	;Is driver or desired channel "busy"?
	jnz	IsBusy
	mov	[ReqBF],al	;Set this request's "busy" flags.
	mov eax,[ebp].Client_Reg_Struc.Client_EAX
	cld				;Ensure FORWARD "string" commands.
	mov	dl,0BEh		;Mask out LBA and write request bits.
	and	dl,ah
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	jne	Pass		;No, let BIOS handle this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns	ValCHS		;No, go validate CHS sector count.
	
	movzx ecx,@word [ebp].Client_Reg_Struc.Client_DS
	movzx esi,@word [ebp].Client_Reg_Struc.Client_ESI
	shl ecx, 4
	add esi, ecx
	
	movzx ebx, @word [esi].DAP.DapBuf+2
	movzx edx, @word [esi].DAP.DapBuf+0
	shl ebx, 4
	add ebx, edx
	mov edx,[esi.DAP.DapLBA1]		;Get DAP LBA bits 16-47
	mov	al,[esi.DAP.DapSC]			;Get "DAP" I-O sector count.
	
	cmp	@dword [esi.DAP.DapBuf],-1	;64-bit buffer address?
	mov	si,[esi.DAP.DapLBA]			;(Get "DAP" LBA bits 0-15).
	jne	ValSC						;No, go validate "DAP" sector count.
Pass:
QuickX:
	mov eax,[PrvI13]
	mov @word [ebp].Client_Reg_Struc.Client_Eip, ax
	shr eax, 16
	mov @word [ebp].Client_Reg_Struc.Client_CS, ax
	ret
IsBusy:
	mov @byte [ebp].Client_Reg_Struc.Client_EAX+1, 1
	stc
	jmp	GetOut
	align 4
	
ValCHS:
	movzx ecx,@word [ebp].Client_Reg_Struc.Client_ES
	movzx ebx,@word [ebp].Client_Reg_Struc.Client_EBX
	shl ecx, 4
	add ebx, ecx

ValSC:
	dec	al			;Is sector count zero or over 128?
	js	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax			;Restore sector count -- LBA request?
	js	SaveSeg		;Yes, go

	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	xchg eax,ecx	;CHS -- save request code and sector count in CX
    
	movzx edx, dx	;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg al,ah
	xchg eax,edx	;Swap cylinder & head values.
	mov	al,[edi+CHSSec];Get disk CHS sectors/head value.
	or	al,al		;Were disk CHS values legitimate?
	jz	Pass		;No?  Let BIOS do this request!
	push eax		;Save CHS sectors/head value.
	mul	ah			;Convert head to sectors.
	add	si,ax		;Add result to starting sector.
	pop	eax			;Reload CHS sectors/head value.
	mul	[edi+CHSHd]	;Convert cylinder to sectors.
	mul	dx
	add	si,ax		;Add to head/sector value.
	adc	edx,0
    
	xchg eax,ecx	;restore request code and sector count in AX

;--- here LBA bits in EDX:SI, linear address buffer in EBX
;--- sector count in AL, cmd in AH
	
SaveSeg:
	mov	@word [LBA0023],si	;Save LBA bits 0-15 and 24-47.
	mov	[LBA0023+2],dl		;Save LBA bits 16-23, to free DL-reg.
	mov	dl,[ReqBF]			;Set driver & channel "busy" flags
	or	[Flags],dl			;  so other IDE drivers will WAIT!
	mov	[LBA2447],dh
	mov esi, edx
	shr esi, 16
	mov	@word [LBA2447+1],si

	or @byte [ebp].Client_Reg_Struc.Client_EFlags+1,2	;set client IF
	
	mov	[VDSOf],ebx		;Save user buffer linear address.
	
	shr	dx,12			;Shift out LBA bits 16-27.
	or	si,dx			;Anything in LBA bits 28-47?
	jnz	LBA48			;Yes, use LBA48 read/write command.
	xchg dh,[LBA2447]	;LBA28 -- reload & reset bits 24-27.
	or	ah,(DRCMD+1)	;Get LBA28 read/write command + 5.
	jmp	GetAdr			;Go get IDE and LBA address bytes.
LBA48:
	shl	ah,3			;LBA48 -- get command as 020h/030h.
GetAdr:
	mov dl,[edi+SelCmd]
	or	dl,dh			;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h			;Get final IDE command byte.
	xor	dh,ah			;(LBA28 = C8h/CAh, LBA48 = 25h/35h).
	mov	@word [DSCmd],dx;Set IDE command bytes.
	mov	[SecCt],al		;Set I-O sector count.
    movzx eax,al
	shl	eax,9
	mov	[VDSLn],eax		;Set buffer lengths.
	mov ecx,eax
    bts eax,31			;Set DMA list "end" flag.
	mov	[IOLen],eax
	mov	esi,[VDSOf]
if SCATTER
    test [bOption],FO_UF;if /UF not set, check 64 kB boundary
    setz dl
else
	mov dl,1			;check for 64 kB boundary crossing
endif
	VxDCall VDMAD_Lock_DMA_Region
	jc	BufIO			;Error -- do buffered I-O.
	mov [IOAdr], edx
	test dl,3h			;Is user I-O buffer 32-bit aligned?
	jnz	BufIO			;No, use buffered I-O routines below.
	
	cmp	@word [IOAdr+2],-1	;DMA I-O above our limit?
@DMALmt	equ	$-1			;(If 640K limit, set to 009h by Init).
	ja	BufIO			;Yes, use buffered I-O routines below.
if SCATTER
    test [bOption],FO_UF			;/UF option set?
    jz  @F
    mov eax, edx
	mov	ecx,[VDSLn]					;Get lower ending DMA address.
	dec	ecx							;(IOLen - 1 + IOAdr).
	add	ax,cx						;Would input cross a 64K boundary?
	jnc	@F							;No, set DMA flag & do transfer.
	inc	ax							;Get bytes above 64K boundary.
	cmp	ax,64						;Is this at least 64 bytes?
	jb	BufIO						;No, use buffer
	inc	cx							;Get bytes below 64K boundary.
	sub	cx,ax
	cmp	cx,64						;Is this at least 64 bytes?
	jb	BufIO						;No, use buffer
	mov	word ptr [IOLen2],ax		;Set 2nd command-list byte count.
	movzx eax,cx					;Set 1st command-list byte count.
	mov	[IOLen],eax
	add	eax,[IOAdr]					;Set 2nd command-list address.
	mov	[IOAdr2],eax
@@:    
endif
	call DoDMA			;Do direct DMA I-O with user's buffer.
Done:
	mov @byte [ebp].Client_Reg_Struc.Client_EAX+1, al	;Set error code in exiting AH-reg.
	pushfd				;Save error flag (carry bit).
	mov	al,[ReqBF]		;Reset driver & channel "busy" flags
	not	al				;  so other IDE drivers may proceed!
	and	[Flags],al
	popfd
GetOut:
	pushfd
	VMMCall Simulate_Iret
	pop eax
	and @byte [ebp].Client_Reg_Struc.Client_EFlags,not 1
	and al,1
	or	@byte [ebp].Client_Reg_Struc.Client_EFlags,al
	ret
	align 4

if IRQWND

;--- copy to/from buffer. If DMA scatter/gather method is activated
;--- the buffered io is used only if the target address is not dword-aligned.
;--- And then the copy is rather slow. Therefore it is done in 4 kB blocks.

moveblock proc
	mov ebx,ecx
nextblock:
	mov ecx, ebx
    cmp ecx, 1000h/4
    jb @F
    mov ecx, 1000h/4
@@:
	sub ebx, ecx
    rep movsd
    VMMCall Yield
    and ebx,ebx
    jnz nextblock
	ret
moveblock endp
	align 4
endif

;--- buffered write: copy conv. memory to DMA buffer
;--- then write

BufOut:
	mov ecx, [VDSLn]
	mov edx, edi	;dont destroy EDI!
	mov esi, [VDSOf]
	mov edi, [dwBufferLin]
	shr ecx, 2
if IRQWND
	call moveblock
else
	rep movsd
endif
	mov edi, edx
	call BufDMA		;Output all data from XMS buffer.
	jmp	Done		;Done -- post any return code & exit.
	align 4
	
BufIO:
	test @byte [IOCmd],012h ;Is this a write request?
	jnz	BufOut				;Yes, use output routine above.
	
;--- buffered read: read into DMA buffer, then copy to conv. memory
	
	call BufDMA		;Input all data to driver XMS buffer.
	jc	Done		;If error, post return code & exit!
	mov ecx, [VDSLn]
	mov edi, [VDSOf]
	mov esi, [dwBufferLin]
	shr ecx, 2
if IRQWND
	call moveblock
else
	rep movsd
endif
	clc
	jmp	Done		;Done -- post any return code & exit.
;
VDExit:
	ret
    align 4
;
; Subroutine to execute read and write commands.
; EDI=drive
;
BufDMA:
	mov eax, [dwBufferPhy]
	mov	[IOAdr],eax			;Buffered -- set physical buffer addr.
DoDMA:
	mov	dx,[edi*2+DMAAd]	;Ensure any previous DMA is stopped!
	in	al,dx				;(On some older chipsets, if DMA is
	and	al,0FEh				;  running, reading an IDE register
	out	dx,al				;  causes the chipset to "HANG"!!).
	mov	al,[edi+SelCmd]		;Select our desired disk.
	mov	dx,[edi*2+IDEAd]
	add edx,CDSEL
	out	dx,al
	mov	cx,(RDYTO*256)+(FLT*2)	;Get timeout & "fault" mask.
	call Await				;Await controller- and disk-ready.
	jc	VDExit				;If any errors, exit above!

	mov @byte ds:[HDIOFS],0	;reset BIOS flag for HD IRQ

	mov	esi,offset PRDAd	;Point to parameters we will output.
	test @byte [esi+ (offset IOCmd - offset PRDAd)],012h  ;Is this a write request?
	jnz	SetDMA				;Yes, reset DMA command register.
	mov	al,008h				;Get "DMA read" command bit.
SetDMA:
	mov	dx,[edi*2+DMAAd]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	edx			;Point to DMA status register.
	inc	edx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	edx			;Set PRD pointer to our DMA address.
	inc	edx
	outsd
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxtPar:
	mov dx,[edi*2+IDEAd]
	add edx,1		;pos at sector count
IDEPar:
	inc	edx			;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc	IDEPar		;Yes, loop back and output next one.
	jnz	NxtPar		;If first 4 done, go output last 6.
    mov esi, BIOSTMR
if 0	
	mov dx,[edi*2+IDEAlt]
ChkDRQ:
	cmp	ch,[esi]	;Too long without 1st data-request?
	je	DMAEnd		;Yes?  Return carry and DMA timeout!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz	ChkDRQ		;No, loop back and check again.
endif	 
	mov	dx,[edi*2+DMAAd]
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	inc	ax
	out	dx,al
	jmp	ChkDMA		;go await input end.
	align 4
;
; Subroutine to await end of I-O.
;
Await:
	mov	esi,BIOSTMR
	add	ch,[esi]	;Set timeout limit in CH-reg.
	shr	cl,1		;Are we only checking drive "ready"?
	jnc	ChkRdy		;Yes, use logic below.
ChkDMA:
	VMMCall Yield
	inc	edx			;Point to DMA status register.
	inc	edx
	in	al,dx		;Read DMA controller status.
	dec	edx			;Point back to DMA command register.
	dec	edx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,[esi]	;Has our DMA transfer timed out?
	je	HltDMA		;No, loop back and check again.
	test @byte ds:[HDIOFS],80h
	jz ChkDMA
	mov al,DMI
HltDMA:
	push eax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	eax			;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne	ErrDMA		;No?  Go see what went wrong.
	inc	edx			;Reread DMA controller status.
	inc	edx
	in	al,dx
	test al,DME		;Any "late" DMA error after DMA end?
	jnz	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx			;Check "fault" and hard-error at end.
ChkRdy:
	mov	dx,[edi*2+IDEAd]  ;Read IDE primary status.
	add edx,7
	in	al,dx
	test al,BSY+RDY	;Controller or disk still busy?
	jg	ChkErr		;No, go check for "fault" or error.
	cmp	ch,[esi]	;Too long without becoming ready?
	jne	ChkRdy		;No, loop back and check again.
	test al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp	WhichE		;Go see which error code to return.
ChkErr:
	and	al,cl		;Disk "fault" or hard-error?
	jz	ChkExit		;No, all is well -- go exit below.
	test al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:
	jz	EndErr		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
EndErr:
	add	al,cl		;Add 1 if error occurred at I-O end.
Kaput:
	stc				;Set carry flag to denote "error"!
ChkExit:
	ret
ErrDMA:
	test al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd:
	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp	WhichE		;Go see which error code to return.
	align 4

;--- end of "resident" part
;--- (for a JLM, this is irrelevant)

;
; Subroutines to display a DWORD/WORD/BYTE in EAX/AX/AL
;
DsplyD:
	push eax
    shr eax,16
    call DsplyW
    pop eax
DsplyW:
	push eax
    mov al,ah
    call DsplyB
    pop eax
DsplyB:
	push eax
    shr al,4
    call DsplyN
    pop eax
DsplyN:
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	@F			;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
@@:
	add	al,'0'		;Convert digit to ASCII.
	jmp DsplyC		;Display digit

DsplyMQ:
	test [bOption],FO_Q
    jz DsplyM
    ret
DsplyEOL:
	mov esi, offset CRMsg
;
; Subroutine to display an error-message "text" element.
;	At entry, the "text" pointer is in the ESI-register.
;
DsplyM:
DspNxt:
	lodsb			;Get next output byte.
	or	al,al		;Are we at the terminating "null"?
	jz	DspEx		;No, loop back & display next byte.
	call DsplyC		;Display character using the BIOS.
    jmp DspNxt
DspEx:    
	ret
;
; Subroutine to display a character using a BIOS "Int 010h" call.
;
DsplyC:
ifdef _DEBUG		;to allow debug messages in Int13 handler
	push edi		;the client state has to be saved/restored
	push esi
	sub esp, sizeof Client_Reg_Struc
	mov edi, esp
	VMMCall Save_Client_State
endif	 
	push	eax		;Save output byte.
	VMMCall Begin_Nest_Exec

	mov @byte [ebp].Client_Reg_Struc.Client_EAX+1, 0Fh
	mov eax, 10h
	VMMCall Exec_Int

	pop	eax			;Reload output byte.

	mov ah,0Eh
	mov @word [ebp].Client_Reg_Struc.Client_EAX, ax
	mov eax, 10h
	VMMCall Exec_Int
	
	VMMCall End_Nest_Exec

ifdef _DEBUG
	mov esi, esp
	VMMCall Restore_Client_State
	add esp, sizeof Client_Reg_Struc
	pop esi
	pop edi
endif
	
	ret
;
; Subroutine to "validate" an UltraDMA hard-disk.
; EDI=drive
;
I_ValD proc
	mov	dx,[edi*2+IDEAd]
	mov al,[edi+SelCmd]
	add edx,CDSEL
    out dx,al
    inc edx
	mov	al,0ECh				;Issue "Identify Device" command.
	out	dx,al
	mov	ebx,BIOSTMR			;Point to low-memory BIOS timer.
	mov	cl,RDYTO			;Set I-O timeout limit in CL-reg.
	add	cl,[ebx]
	mov	esi,offset IEMsg	;Point to "Identify error" message.
I_IDDly:
	cmp	cl,[ebx]			;Has our command timed out?
	je	I_SErr				;Yes, set CPU carry flag & exit.
	in	al,dx				;Get IDE controller status.
	test al,BSY+RDY			;Controller or disk still busy?
	jle I_IDDly				;Yes, loop back and check again.
	test al,ERR				;Did command cause any errors?
	jz	I_PIO				;No, read I.D. data using PIO mode.
I_SErr:
	stc						;Set carry flag (error!) and exit.
I_Exit:    
	ret
I_PIO:
	sub	edx,(CCMD-CDATA)	;Point to PIO data register.
	in	ax,dx				;Read I.D. bytes 0 and 1.
	shl	eax,1				;Save "ATA/ATAPI" flag in carry bit.
	mov	ecx,27				;Skip I.D. bytes 2-53 and
I_Skp1:
	in	ax,dx				;  read I.D. bytes 54-55.
	loop I_Skp1
	mov	edi,offset DName
	mov	cl,26				;Read & swap disk name into message.
I_RdNm:
	xchg ah,al				;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw					;  also read but are ignored.	Bytes
	in	ax,dx				;  106-107 are left in the AX-reg.).
	loop I_RdNm
	xchg eax,ebx			;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35				;Skip I.D. bytes 108-175 &
I_Skp2:
	in	ax,dx				;  read I.D. bytes 176-177.
	loop I_Skp2
	mov	bh,ah				;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167				;Skip remaining I.D. data.
I_Skp3:
	in	ax,dx
	loop I_Skp3
    
    test [bOption],FO_Q
    jnz I_Exit
	
	mov	esi,offset UEMsg	;Point to "is not UltraDMA" message.
	rcr	bl,1				;Shift "ATA/ATAPI" flag into BL-reg.
	and	bl,082h				;ATAPI disk, or UltraDMA bits invalid?
	jle	I_SErr				;Yes?  Exit & display error message.
	mov	edi,offset Modes	;Point to UltraDMA mode table.
	or	bh,bh				;Will disk do UltraDMA mode 0?
	jz	I_SErr				;No?  Exit & display message!
I_NxtM:
	cmp	bh,bl				;Will disk do next UltraDMA mode?
	jb	I_GotM				;No, use current mode.
	inc	edi					;Point to next mode table value.
	inc	edi
	inc	ecx					;Get next UltraDMA mode number.
	shl	bl,1				;More UltraDMA modes to check?
	jnz	I_NxtM				;Yes, loop back.
I_GotM:
	push @dword [edi]		;Save disk mode value.
	mov	ebx,offset DName+SIZENAME	;Point to end of disk name.
I_NxtN:
	cmp	ebx,offset DName	;Are we at the disk-name start?
	je	I_Name				;Yes, disk name is all spaces!
	dec	ebx					;Decrement disk name pointer.
	cmp	@byte [ebx],' '		;Is this name byte a space?
	je	I_NxtN				;No, continue scan for non-space.
	inc	ebx					;Skip non-space character.
	mov	@word [ebx]," ,"	;End disk name with comma/space.
	inc	ebx					;Skip comma and space.
	inc	ebx
I_Name:
	mov	@dword [ebx],"-ATA"	;Set "ATA-" and null after name.
	mov	@byte [ebx+4],0
	mov	esi,offset DNMsg	;Display "name" of this disk.
	call DsplyM
	pop	eax					;Get disk "mode" value.
    and ah,ah				;mode value needs 3 digits?
    jz @F
    push eax
    mov al,ah
	call DsplyN				;Display highest digit
    pop eax
@@:    
	call DsplyB				;Display lower 2 digits
	jmp	DsplyEOL
I_ValD endp

;--- get Ultra-DMA port for EDD 2.0
;--- EDI=drive

;--- PCI command register
;--- 0001: I/O access enabled
;--- 0002: memory access enabled
;--- 0004: BM access enabled
;--- 0008:

I_GetUDMAC proc
	mov	esi,offset iftab	;Point to interface byte table.
I_Next:
	mov	ecx,000010100h		;We want class 1 storage, subclass 1 IDE.
	lodsb					;Get next "valid" PCI interface byte.
	mov	cl,al
	mov [ebp].Client_Reg_Struc.Client_ECX, ecx
	mov @word [ebp].Client_Reg_Struc.Client_ESI, 0	;controller index
	mov	al,003h				;Returns bus/device/function in BX.
	call I_Int1A
	jc	I_Continue				;Succeeded?
	mov ebx, [ebp].Client_Reg_Struc.Client_EBX
	mov ax,4				;Get PCI command + status register
	call I_PCID
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	not	cl					;Mask Bus-Master and I-O Space bits.
	and	cl,005h				;Is this how our controller is set up?
	jnz	I_Continue			;No?  Continue scan!

;--- for a "native" controller, IDE port bases are in registers 0-3
;--- 0+1=primary base+alternate, 2+3=secondary base+alternate

	test byte ptr [esi-1],1	;native?
    jz @F
	mov ax,16+0*4			;Get primary IDE base address
	call I_PCID
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
    and cl,0FCh
    cmp cx,[edi*2+IDEAd]
    jz @F
	mov ax,16+2*4			;Get secondary IDE base address
	call I_PCID
    and cl,0FCh
    cmp cx,[edi*2+IDEAd]
    jnz I_Continue
    call getDMA
    add ecx,8
    clc
    ret
@@:
	call getDMA
	clc
	ret
I_Continue:    
	cmp	esi,offset endiftab ;More interface bytes to try?
	jb	I_Next				;Yes, go try next one.
	stc
	ret
getDMA:    
	mov ax,16+4*4			;Get PCI DMA base address (register 4).
	call I_PCID
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
    and cl,0FCh
    ret
I_GetUDMAC endp
;
; initialization
;
I_Init:
	cld
ifdef _DEBUG
	mov esi,offset DbgMsg0
    call DsplyM
endif
	mov	esi,[dwCmdLine]
I_NxtC:
	lodsb
	cmp	al,0
	je	I_Term
	cmp	al,LF
	je	I_Term
	cmp	al,CR
	je	I_Term
	cmp	al,'-'
	je	I_NxtS
	cmp	al,'/'
	jne	I_NxtC		;No, check next command-line byte.
I_NxtS:
	mov ax,[esi]
	and	ax,0DF0DFh	;Mask out lower-case bit
	cmp	al,'L'		;Is this byte an "L" or "l"?
	jne	I_ChkQ		;No, go see if byte is "Q" or "q".
	mov	@byte [@DMALmt],009h ;Set 640K "DMA limit" above.
	inc	esi			;Point to next command-line byte.
I_ChkQ:
	cmp al,'Q'		;/Q?
    jnz @F
    or [bOption],FO_Q
    inc esi
@@: 
if SCATTER
	cmp ax,'FU'		;/UF?
    jnz @F
    or [bOption],FO_UF
    inc esi
    inc esi
@@:    
endif
	jmp	I_NxtC		;Continue scanning for a terminator.
	
I_Term:
	mov	esi,offset XDMsg	;Display driver "title" message.
	call DsplyMQ
	mov	al,001h
	mov [ebp].Client_Reg_Struc.Client_EDI, 0	;Get PCI BIOS "I.D." code.
	call I_Int1A
	mov edx, [ebp].Client_Reg_Struc.Client_EDX
	mov	esi,offset PEMsg	;Get "Invalid PCI BIOS" message ptr.
	cmp	edx," ICP"			;Is PCI BIOS V2.0C or newer?
	jne	I_EOut				;Go display error message and exit.


	mov	ax,03513h			;Get and save current Int 13h vector.
	call I_Int21
	mov bx, @word [ebp].Client_Reg_Struc.Client_ES
	shl ebx, 16
	mov bx, @word [ebp].Client_Reg_Struc.Client_EBX
	mov	[PrvI13], ebx
	
	mov	ax,04300h			;Inquire about an XMS manager.
	call I_Int2F
	mov eax, [ebp].Client_Reg_Struc.Client_EAX
	
	mov	esi,offset NXMsg	;Point to "No XMS manager" message.
	cmp	al,080h				;Is an XMS manager installed?
	jne	I_XErr				;No, display message & disable XMS.
	mov	ax,04310h			;Get XMS manager "entry" address.
	call I_Int2F
	mov bx, @word [ebp].Client_Reg_Struc.Client_ES
	shl ebx, 16
	mov bx, @word [ebp].Client_Reg_Struc.Client_EBX
	mov	[XMSEntry], ebx
	
	mov	ah,009h				;Ask XMS manager for 128K of memory.
	mov	dx,128
	call I_XMS
	jnz	I_XMErr				;If error, display msg. & disable XMS.
	mov edx,[ebp].Client_Reg_Struc.Client_EDX
	mov	[XMSHdl],edx		;Save XMS buffer handle number.
	
	mov	ah,00Ch				;"Lock" our XMS memory.
	call I_XMS
	jnz	I_RidX				;If error, display msg. & disable XMS.
	
	mov eax,[ebp].Client_Reg_Struc.Client_EDX
	shl	eax,16				;Get unaligned XMS buffer address.
	mov ax,@word [ebp].Client_Reg_Struc.Client_EBX
	add	eax,65536-1			;Find 1st 64K boundary after start.
	xor	ax,ax
	mov	[dwBufferPhy],eax	;Set physical buffer address

ifdef _DEBUG
	mov	esi,offset DbgMsg1
	call DsplyM
endif

	push 0
	push 16
	push PR_SYSTEM
	VMMCall _PageReserve	;allocate a 64 kB block of address space
	add esp,3*4
	cmp eax,-1
	jz I_RidX
	mov dwBufferLin, eax
	shr eax, 12				;convert linear address to page number

ifdef _DEBUG
	push eax
	mov	esi,offset DbgMsg1a
	call DsplyM
	pop eax
endif
	push PC_INCR or PC_WRITEABLE
	mov edx, [dwBufferPhy]
	shr edx, 12
	push edx
	push 16
	push eax
	VMMCall _PageCommitPhys	;backup address space with XMS memory
	add esp,4*4

ifdef _DEBUG
	mov	esi,offset DbgMsg2
	call DsplyM
endif
	
	jmp	I_Stop
I_RidX:
	mov	edx,[XMSHdl]		;Error!  Get rid of our XMS buffer.
	call I_ReleaseXMS
I_XMErr:
	mov	esi,offset XEMsg	;Point to "XMS init error" message.
I_XErr:
	jmp	I_EOut				;Go display XMS error message & exit!
I_Stop:
	xor	eax,eax				;Zero EAX-reg. for 20-bit addressing.
	mov	[SecCt2],al			;Reset IDE upper sector count.
	or	al,ds:[HDISKS]		;Did BIOS find any hard-disks?
	jz	I_None				;No?  Display "No disk" and exit!
	mov	[BiosHD],al			;Save BIOS hard-disk count.
	shl	eax,4				;Set 20-bit driver virtual address.
	mov	[IOAdr],eax

	mov ecx,offset Entry - offset startcode
	mov esi,offset startcode
	xor edx, edx
	VxDCall VDMAD_Lock_DMA_Region
	add	[PRDAd],edx			;Set relocated 32-bit PRD address.

ifdef _DEBUG
	mov	esi,offset DbgMsg3
	call DsplyM
endif
	
	movsx ax,@byte [@DMALmt];Get DMA memory limit (if any).
	inc	ax					;Has /L limited DMA to low-memory?
	jz	I_Scan				;No, scan for UltraDMA disks to use.
	cmp	ax,@word [IOAdr+2]	;Are we loaded in upper-memory?
	ja	I_Scan				;No, scan for UltraDMA disks to use.
	mov	esi,offset PEMsg1	;Set up "/L Invalid" error message.
	mov	@word [esi],"L/"
	jmp	I_Err				;Go display error message and exit!
I_RScn:
	dec	@byte [EDDFlag]		;Were we scanning v.s. EDD BIOS data?
	jz	I_Scan				;Yes, try a "hardware only" disk scan.
I_None:
	mov	esi,offset NDMsg	;BAAAD News!  Point to "No disk" msg.
I_Err:
	mov	edx,[XMSHdl]		;Get XMS memory "handle".
	or	dx,dx				;Did we reserve XMS memory?
	jz	@F					;No, go display error message.
	call I_ReleaseXMS		;Get rid of our XMS buffer.
@@:
I_EOut:
	call DsplyM				;Display desired error message.
	mov	esi,offset Suffix	;Display error-message suffix.
	call DsplyM
	mov	ax,RPDONE+RPERR
	jmp	I_Exit				;Go post "init" packet results & exit.
I_Scan:
ifdef _DEBUG
	mov	esi,offset DbgMsg4
	call DsplyM
endif
	mov	ax,00080h			;Reset hard-disk unit number & index.
	mov	@word [HDUnit],ax
	mov	al,[BiosHD]			;Reset remaining hard-disk count.
	mov	[HDCount],al
	cmp	ah,[EDDFlag]		;Will disk scan use the EDD BIOS?
	jne	I_Next				;Yes, go start with BIOS unit 80h.
	mov	esi,offset HOMsg	;Display "Hardware-only disk scan" msg.
	call DsplyMQ
I_Next:
	movzx ebx,[HDIndex]		;Get disk unit-number index.
	cmp	bh,[EDDFlag]		;Scanning for disks using the EDD BIOS?
	je	I_NoEx				;No, check disk at "fixed" addresses.
	mov	ah,041h				;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call I_Int13
	jc	I_NoEx				;If none, ignore disk & check for more.
	mov ebx,[ebp].Client_Reg_Struc.Client_EBX
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	cmp	bx,0AA55h			;Did BIOS "reverse" our entry code?
	jne	I_NoEx				;No, ignore this disk & check for more.
	test cl,004h			;Does this disk have "EDD" extensions?
	jz	I_NoEx				;No, ignore this disk & check for more.
	
	mov eax,[wBaseSeg]
	mov @word [ebp].Client_Reg_Struc.Client_DS, ax
	mov ax,RESSIZE
	mov @word [ebp].Client_Reg_Struc.Client_ESI, ax
	mov esi,[dwBase]
	add esi,RESSIZE
	mov @dword [esi],42h	;set the FULL dword value
	mov	ah,048h				;Get this disk's "EDD" parameters.
	call I_Int13
	jc	I_ErED				;Error?  Display msg. & ignore!
	cmp	@word [esi],30		;From David Muller:  30+ bytes?
	jb	I_NoEx				;No, ignore disk & check more.
	cmp	@dword [esi+26],-1	;Valid "DPTE" pointer?
	je	I_NoEx				;No, ignore disk & check more.
	movzx ecx, @word [esi+26+0]
	movzx edx, @word [esi+26+2]
	shl edx, 4
	add edx, ecx			;Get this disk's "DPTE" pointer.
	mov	ebx,15				;Calculate "DPTE" checksum.
	xor	ecx,ecx
I_CkSm:
	add	cl,[ebx+edx]
	dec	ebx
	jns	I_CkSm
	jecxz I_EDOK			;If checksum O.K., use parameters.
I_ErED:
	mov	esi,offset EBMsg	;Display "EDD error" and ignore unit!
	jmp	I_ErrD
I_NoEx:
	jmp	I_More				;No EDD:  ignore disk & check for more.
I_EDOK:
	movzx edi, [HDIndex]
	mov	bl,[edx].DPTE.bFlags
	and bl,50h				;use the LBA + "slave" flags
	or bl,0A0h
	mov [edi+SelCmd],bl
	
	mov	ax,[edx].DPTE.wIDEBase	;Get disk's IDE base address.
	mov	cx,[edx].DPTE.wIDEAlt	;Get disk's IDE base address.
	mov [edi*2+IDEAd],ax
	mov [edi*2+IDEAlt],cx

	cmp @word [esi],42h
	jnc I_IsEDD30
    
	call I_GetUDMAC
	jc I_NoEx
    cmp [edi*2+IDEAd],1F0h	;primary or secondary?
    jz I_EDDdone
    add ecx,8
	jmp I_EDDdone
I_IsEDD30:
	mov eax, [esi+24h]
    and eax, 0FFFFFFh
	cmp eax ,"ASI"			;ISA host bus?
    jz I_NoEx				;then ignore drive
	mov eax, [esi+28h]
	cmp eax ,"ATA"			;ATA interface?
    jz @F
	cmp eax ," ATA"
    jnz I_NoEx
@@:    
	mov bh, [esi+30h]		;get interface path (Bus)
	mov bl, [esi+31h]		;get interface path (Device)
	mov al, [esi+32h]		;get interface path (Function)
	shl bl,3
	and al,7
	or bl,al
	mov ax,32				;Get PCI base address (register 4).
	call I_PCID
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
I_EDDdone:
	and cl,0FCh
	mov [edi*2+DMAAd],cx
	test [bOption],FO_Q
    jnz I_NoDskDisp
	mov	esi,offset DskMsg	;Point to "master" disk name.
	call DsplyM
	mov al,[HDUnit]
	call DsplyB
	mov	esi,offset MstMsg	;Point to "master" disk name.
	test [edi+SelCmd],10h	;Is this disk the master?
	jz	@F					;Yes, display "master" name.
	mov	esi,offset SlvMsg	;Point to "slave" disk name.
@@:
	call DsplyM				;Display disk's master/slave name.

	mov	esi,offset PortMsg
	call DsplyM				;Display controller-data trailer.
	xor	ebx,ebx				;Display DMA controller address.
	mov ax,[edi*2+IDEAd]
	call DsplyW
	mov	esi,offset SlashMsg
	call DsplyM				;Display controller-data trailer.
	mov ax,[edi*2+DMAAd]
	call DsplyW
I_NoDskDisp:	
	mov	ah,008h				;Get BIOS CHS values for this disk.
	call I_Int13
	jc	I_CHSE				;If BIOS error, zero sectors/head.
	mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	mov edx,[ebp].Client_Reg_Struc.Client_EDX
	and	cx,03Fh				;Get sectors/head value (low 6 bits).
	inc	dh					;Get heads/cylinder (BIOS value + 1).
	jnz	I_SetC				;If non-zero, save disk's CHS values.
I_CHSE:
	xor	cl,cl				;CHS error!  Zero disk's sectors/head.
I_SetC:
	mov	[edi+CHSHd],dh		;Save disk's CHS values in our tables.
	mov	[edi+CHSSec],cl
	mov	al,[HDUnit]			;Activate this disk in main driver.
	mov	[edi+Units],al
	push edi
	call I_ValD
	pop edi
	jc	I_DelD				;If any errors, DELETE this disk!
	cmp	[edi+CHSSec],bh		;Were disk's CHS values legitimate?
	jne	I_More				;Yes, check for more disks to use.
	mov	esi,offset CHMsg	;Display "BIOS must do CHS" msg.
	jmp	I_ErrD
I_DelD:
	mov	[edi+Units],-1		;DELETE disk in main driver!
I_ErrD:
	call DsplyM				;Display error for this disk.
	call DsplyEOL
I_More:
	add	@word [HDUnit],00101h	;Bump BIOS unit & disk index.
	cmp	@word [EDDFlag],08000h+NUMDSK*100h;No EDD and all NUMDSK units tested?
	je	I_AnyD					;Yes, see if we found any disks.
	dec	@byte [HDCount]			;More BIOS disks to check?
	jnz	I_Next					;Yes, loop back and do next one.
I_AnyD:
	mov	edi,NUMDSK			;Set up to scan for last unit.
I_ChkU:
	dec	edi					;Any more active units to check?
	js	I_RScn				;No, see if we should do a rescan.
	cmp	[edi+Units],-1		;Is this unit active?
	je	I_ChkU				;No, loop back and check next unit.

ifdef _DEBUG
	mov	esi,offset DbgMsg5
	call DsplyM
endif
	
	add	@dword [@LastU],edi	;Post last-unit index in main driver.

	mov esi, offset Entry
	xor edx, edx
	VMMCall Allocate_V86_Call_Back
	mov	esi,offset NoCBMsg
    jc I_EOut
	mov @word [ebp].Client_Reg_Struc.Client_EDX, ax
    shr eax, 16
	mov @word [ebp].Client_Reg_Struc.Client_DS, ax
	mov	ax,02513h
	call I_Int21			;"Hook" this driver into Int 13h.
	mov	ax,RPDONE			;Get initialization "success" code.
I_Exit:
ifdef _DEBUG
	mov edi, eax
	mov	esi,offset DbgMsg6
	call DsplyM
    mov eax, edi
endif
	ret
    
;
; Subroutines to issue initialization "external" calls.    These and
;	the "clear-stack" logic above must appear AFTER the local-stack!
;
I_ReleaseXMS:
	mov	ah,00Dh		;Error -- unlock & free XMS buffer.
	push edx
	call I_XMS
	mov	ah,00Ah
	pop	edx
I_XMS:
	mov [ebp].Client_Reg_Struc.Client_EAX, eax
	mov [ebp].Client_Reg_Struc.Client_EDX, edx
	VMMCall Begin_Nest_Exec
	movzx edx, @word [XMSEntry+0]
	mov cx, @word [XMSEntry+2]
	VMMCall Simulate_Far_Call
	VMMCall Resume_Exec
	VMMCall End_Nest_Exec
	mov eax,[ebp].Client_Reg_Struc.Client_EAX
	dec	ax				;Zero AX-reg. if success, -1 if error.
	ret

;--- call int 13h,
;--- ah=08h
;--- ah=41h
;--- ah=48h, DS:SI->EDP

I_Int13:
	mov [ebp].Client_Reg_Struc.Client_EAX, eax
	mov [ebp].Client_Reg_Struc.Client_EBX, ebx
	mov	al,[HDUnit]	;Set BIOS unit in DL-reg.
	mov @byte [ebp].Client_Reg_Struc.Client_EDX, al
	VMMCall Begin_Nest_Exec
	mov eax, 13h
	VMMCall Exec_Int
	VMMCall End_Nest_Exec
	mov ah,@byte [ebp].Client_Reg_Struc.Client_EFlags
	sahf
	ret

;--- call int 1Ah, ax=B10Ah, B103h, B101h

I_PCID:
	mov [ebp].Client_Reg_Struc.Client_EBX, ebx
	mov [ebp].Client_Reg_Struc.Client_EDI, eax
	mov	al,00Ah		;Set "PCI doubleword" request code.
I_Int1A:
	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	mov [ebp].Client_Reg_Struc.Client_EAX, eax
	VMMCall Begin_Nest_Exec
	mov eax, 1Ah
	VMMCall Exec_Int
	VMMCall End_Nest_Exec
	mov ah,@byte [ebp].Client_Reg_Struc.Client_EFlags
	sahf
	ret

;--- int 21h calls (ah=25h/35h)    
	
I_Int21:
	mov @word [ebp].Client_Reg_Struc.Client_EAX, ax
	VMMCall Begin_Nest_Exec
	mov eax, 21h
	VMMCall Exec_Int
	VMMCall End_Nest_Exec
	ret
	
;--- int 2Fh calls (ax=4300h, 4310h)
	
I_Int2F:
	mov @word [ebp].Client_Reg_Struc.Client_EAX, ax
	VMMCall Begin_Nest_Exec
	mov eax, 2Fh
	VMMCall Exec_Int
	VMMCall End_Nest_Exec
	ret
;
; Initialization Tables And Variables.
;
	align	4
Modes label word
	dw	0016h		;Mode 0, ATA-16  UDMA mode table (digit
	dw	0025h		;Mode 1, ATA-25    count in low 4 bits).
	dw	0033h		;Mode 2, ATA-33.
	dw	0044h		;Mode 3, ATA-44  (Unusual but possible).
	dw	0066h		;Mode 4, ATA-66.
	dw	0100h		;Mode 5, ATA-100.
	dw	0133h		;Mode 6, ATA-133.
	dw	0166h		;Mode 7, ATA-166.

EDDFlag db	1		;"EDD BIOS in use" flag.
HDUnit	db	0		;Current BIOS unit number.
HDIndex db	0		;IDE "index" number.

SIZENAME	equ 40

XDMsg	db	'XDMA32',VER, CR,LF,0
PEMsg	db	'PCI BI'
PEMsg1	db	'OS Invalid',0
NXMsg	db	'No XMS manager',0
XEMsg	db	'XMS init error',0
DNMsg	db	', '
DName	db 26*2 dup (0)
CRMsg	db	CR,LF,0
NDMsg	db	'No disk to use',0
EBMsg	db	'EDD BIOS error!  Unit ignored',0
HOMsg	db	'Hardware-only disk scan:',CR,LF,0
NoCBMsg	db	'No callbacks available anymore',CR,LF,0
DskMsg	db	'HD ',0
MstMsg	db	', Master',0
SlvMsg	db	', Slave',0
PortMsg	db	', ATA/DMA ports ',0
SlashMsg db	'/',0
IEMsg	db	'Identify ERROR',0
UEMsg	db	' is not UltraDMA',0
Suffix	db	'; XDMA32 not loaded!',CR,LF,0
CHMsg	db	"** BIOS must do above disk's CHS I-O",0
ifdef _DEBUG
DbgMsg0	db	'Init entry',CR,LF,0
DbgMsg1	db	'XMS allocated',CR,LF,0
DbgMsg1a db	'Buffer Address Space allocated',CR,LF,0
DbgMsg2	db	'Buffer committed',CR,LF,0
DbgMsg3	db	'Driver locked',CR,LF,0
DbgMsg4	db	'Scanning HDs...',CR,LF,0
DbgMsg5	db	'Scanning done',CR,LF,0
DbgMsg6	db	'Init exit',CR,LF,0
DbgMsg6a db	'exit DllMain',CR,LF,0
DbgMsg7	db	'Entry Int 13 PM, ax=',0
DbgMsg7a db	' DAP buffer=',0
endif

DllMain proc stdcall hModule:dword, dwReason:dword, dwRes:dword

	.if (dwReason == 1)

		mov esi, dwRes
		movzx ecx,[esi].JLCOMM.wLdrCS
		mov wBaseSeg,ecx
		shl ecx, 4
		mov dwBase, ecx
		mov eax,[esi].JLCOMM.lpCmdLine
		mov dwCmdLine, eax

;--- set EBP to the client pointer before calling I_Init

		push ebp
		VMMCall Get_Cur_VM_Handle	;get VM handle in EBX
		mov ebp,[ebx].cb_s.CB_Client_Pointer
if SAVESTAT
		sub esp, sizeof Client_Reg_Struc
		mov edi, esp
		VMMCall Save_Client_State
endif
        push esi
		call I_Init
        pop esi
		test [esi].JLCOMM.wFlags, JLF_DRIVER	;loaded as DOS device driver?
		jz @F
		mov ebx,[esi].JLCOMM.lpRequest
		mov	[ebx].RP.RPStat,ax
		mov ecx, [wBaseSeg]
		mov	@word [ebx].RP.RPSize+2,cx
        xor ecx, ecx
		mov	@word [ebx].RP.RPSize+0,cx
@@:
        cmp ax, RPDONE
        setz al
        movzx edi, al
ifdef _DEBUG
		mov	esi,offset DbgMsg6a
		call DsplyM
endif
if SAVESTAT
		mov esi, esp
		VMMCall Restore_Client_State
		add esp, sizeof Client_Reg_Struc
endif
		pop ebp
		mov eax, edi
	.endif
	ret

DllMain endp

	end DllMain
