{ ***************************************************************************
  `upxdump.pas' -- Main source code.

  This file is part of UPXDUMP.

  Copyright (C) 1997-2017  Robert Riebisch

  UPXDUMP is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  UPXDUMP is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file `licence.txt'.
  ***************************************************************************

  Build instructions for Turbo Pascal (dos16):
    1) tpc upxdump.pas
    2) apack -x -t upxdump.exe upxdump.exe

  Build instructions for Free Pascal (dos32):
    1) fpc -O3 -Os -XX upxdump.pas
    2) upx --best --lzma --all-filters upxdump.exe

  Build instructions for Free Pascal (win32):
    1) fpc -O3 -Os -XX upxdump.pas
    2) strip --strip-all upxdump.exe
    3) upx --best --lzma --all-filters upxdump.exe
  *************************************************************************** }

{ include compiler settings }
{$I SETTINGS.INC}

{ set memory sizes }
{$M 2048,0,65536}

program UpxDump;

uses Dos;

label sigfound;

const
  { size of read buffer }
  BUF_SIZE = 65528;  { value from Turbo Pascal Help for GetMem }

  { include UPX constants }
  {$I UPXCONST.INC}

  { 32RTM binary? }
  is32rtm: Boolean = FALSE;

  { exit codes }
  ERR_OPTIONS   = 1;
{$IFDEF MSDOS}
  ERR_DOS_VER   = 2;
{$ENDIF}
  ERR_NO_FILE   = 3;
  ERR_NO_UPX    = 4;
  ERR_FILE_SIZE = 5;
{$IFNDEF FPC}
  ERR_NO_MEM    = 6;
{$ENDIF}

var
  { input file }
  f: file;

  { first command-line parameter, file name, helper }
  opt1, fname, s: string;

  { pointers into read buffer }
  buf, p: Pointer;

  { UPX header variables }
  version, format, method, level: Byte;
  u_len, c_len, u_file_size: Longint;
  filter, filter_cto, chksum: Byte;

  { BlockRead() return value, loop helper }
  bytes_read, i: Word;

  { computed header checksum }
  chksum2: Byte;

  { filter byte offset within header, header size }
  off_filter, size: Byte;

{ print help and exit }
procedure helpx;
begin
  WriteLn('UPXDUMP Version 19-NOV-2017'^M^J +
          'Copyright (C) 1997-2017 Robert Riebisch'^M^J +
          '[Under GNU General Public License]'^M^J +
          ^M^J +
          'Usage: upxdump <input file>');
  Halt;
end;

{ print error message and exit with return code }
procedure errx(eval: integer; fmt: string);
begin
  WriteLn('UPXDUMP Error: ', fmt);
  Halt(eval);
end;

{ correct endianness (Thanks, Michal!) }
{$IFNDEF FPC}
function SwapEndian(li: Longint): Longint;
inline(
  $5A/      { pop dx }
  $58/      { pop ax }
  $86/$D6/  { xchg dl, dh }
  $86/$C4   { xchg al, ah }
);
{$ENDIF}

function GetPackHeaderChecksum(buf: Pointer; len: Word): Byte;
var w: Word;
begin
  Inc(Longint(buf), 4);  { 4 }
  Dec(len, 4);
  w := 0;
  while len > 0 do
  begin
    Inc(w, Byte(buf^));
    Inc(Longint(buf));
    Dec(len);
  end;
  GetPackHeaderChecksum := w mod 251;
end;

function GetPackHeaderSize: Byte;
begin
  GetPackHeaderSize := 0;
  if version <= 3 then
    GetPackHeaderSize := 24
  else
    if version <= 9 then
    begin
      if (format = UPX_F_DOS_COM) or (format = UPX_F_DOS_SYS) then
        GetPackHeaderSize := 20
      else
        if (format = UPX_F_DOS_EXE) or (format = UPX_F_DOS_EXEH) then
          GetPackHeaderSize := 25
        else
          GetPackHeaderSize := 28;
    end
    else
      begin
        if (format = UPX_F_DOS_COM) or (format = UPX_F_DOS_SYS) then
          GetPackHeaderSize := 22
        else
          if (format = UPX_F_DOS_EXE) or (format = UPX_F_DOS_EXEH) then
            GetPackHeaderSize := 27
          else
            GetPackHeaderSize := 32;
      end;
(*
  if size < 20 then
    errx(ERR_UNK_HDR, 'unknown header version');
*)
end;

begin
  { check parameters }
  if ParamCount = 0
    then helpx
  else
    begin
      opt1 := ParamStr(1);
      if (opt1 = '/?') or (opt1 = '-?') or
         (opt1 = '/h') or (opt1 = '-h') or
         (opt1 = '/H') or (opt1 = '-H') then
        helpx;
    end;
  if ParamCount > 1 then
    errx(ERR_OPTIONS, 'Too many options.');

{$IFDEF MSDOS}
  { check DOS version needed for FExpand() }
  if Lo(DosVersion) < 3 then
    errx(ERR_DOS_VER, 'MS-DOS 3.0 (or compatible) needed.');
{$ENDIF}
  fname := FExpand(opt1);

  { try to allocate buffer }
{$IFNDEF FPC}
  if MaxAvail < BUF_SIZE then
    errx(ERR_NO_MEM, 'Not enough memory.');
{$ENDIF}
  GetMem(buf, BUF_SIZE);

  WriteLn('Analyzing `', fname, '''...'^M^J);

  { read beginning of file }
  Assign(f, fname);
  FileMode := 0;  { read-only is enough }
  Reset(f, 1);
  if IOResult <> 0 then
    errx(ERR_NO_FILE, 'Input file not found.');
  WriteLn('File size:'^I^I^I, FileSize(f): 10, ' bytes');
  BlockRead(f, buf^, BUF_SIZE, bytes_read);
  Close(f);

  { firstly check file size }
  if bytes_read < 22 then
    errx(ERR_FILE_SIZE, 'Input file too small.');

  { then find 32RTM signature }
  p := buf;
  Inc(Longint(p), 512);
  if Longint(p^) = RTM32_MAGIC_LE32 then
  begin
    Inc(Longint(p), SizeOf(Longint));
    if Word(p^) = RTM32_MAGIC2_LE16 then
      is32rtm := TRUE;
  end;

  { now find UPX signature }
  p := buf;
  for i := 0 to bytes_read - SizeOf(Longint) do
  begin
    if Longint(p^) = UPX_MAGIC_LE32 then
    { don't look for `UPX_MAGIC2_LE32' for maximum
        compatibility with UPX versions prior 2001 }
      goto sigfound;
    Inc(Longint(p));
  end;
  errx(ERR_NO_UPX, 'Not packed by UPX.');
sigfound:
  buf := p;

  { Pascal translation of UPX' PackHeader::fillPackHeader() }
  p := buf;
  Inc(Longint(p), 4);  { 4 }
  version := Byte(p^);
  Inc(Longint(p));     { 5 }
  format := Byte(p^);
  Inc(Longint(p));     { 6 }
  method := Byte(p^);
  Inc(Longint(p));     { 7 }
  level := Byte(p^);

  size := GetPackHeaderSize;

  if format < 128 then
  begin
    { 16-bit little endian }
    if (format = UPX_F_DOS_COM) or (format = UPX_F_DOS_SYS) then
    begin
      p := buf;
      Inc(Longint(p), 16);        { 16 }
      u_len := Word(p^);

      Inc(Longint(p), 16 div 8);  { 18 }
      c_len := Word(p^);

      u_file_size := u_len;

      off_filter := 20;
    end
    else
      { 24-bit little endian }
      if (format = UPX_F_DOS_EXE) or (format = UPX_F_DOS_EXEH) then
      begin
        p := buf;
        Inc(Longint(p), 16);        { 16 }
        u_len := Longint(p^) and not $ff000000;

        Inc(Longint(p), 24 div 8);  { 19 }
        c_len := Longint(p^) and not $ff000000;

        Inc(Longint(p), 24 div 8);  { 22 }
        u_file_size := Longint(p^) and not $ff000000;

        off_filter := 25;
      end
      else
        { 32-bit little endian }
        begin
          p := buf;
          Inc(Longint(p), 16);            { 16 }
          u_len := Longint(p^);

          Inc(Longint(p), 32 div 8);      { 20 }
          c_len := Longint(p^);

          Inc(Longint(p), 32 div 8);      { 24 }
          u_file_size := Longint(p^);

          off_filter := 28;

          Inc(Longint(p), 32 div 8 + 1);  { 29 }
          filter_cto := Byte(p^);
        end;
    end
  else
    { 32-bit big endian }
    begin
      p := buf;
      Inc(Longint(p), 8);             { 8 }
      u_len := Longint(p^);
      u_len := SwapEndian(u_len);

      Inc(Longint(p), 32 div 8);      { 12 }
      c_len := Longint(p^);
      c_len := SwapEndian(c_len);

      Inc(Longint(p), 3 * 32 div 8);  { 24 }
      u_file_size := Longint(p^);
      u_file_size := SwapEndian(u_file_size);

      off_filter := 28;

      Inc(Longint(p), 32 div 8 + 1);  { 29 }
      filter_cto := Byte(p^);
    end;

  { compute filter & compression level }
  if version >= 10 then
  begin
    p := buf;
    Inc(Longint(p), off_filter);
    filter := Byte(p^)
  end
  else
    if (level and 128) = 0 then
      filter := 0
    else
      begin
        { convert old flags to new filter ID }
        level := level and 127;
        if (format = UPX_F_DOS_COM) or (format = UPX_F_DOS_SYS) then
          filter := $06
        else
          filter := $26;
      end;
  level := level and 15;
  { end of translation }

  { print all results }
  WriteLn('UPX header size:'^I^I, size: 10, ' bytes');
  { check header checksum }
  if version > 9 then
  begin
    p := buf;
    chksum2 := GetPackHeaderChecksum(p, size - 1);
    Inc(Longint(p), size - 1);
    chksum := Byte(p^);
    Write('UPX header checksum:'^I^I, chksum: 10);
    if chksum <> chksum2 then
      WriteLn(' (Warning! Checksum mismatch. Expected: ', chksum2, ')')
    else
      WriteLn;
  end;
  WriteLn('UPX data format version:'^I, version: 10);
  Write('Executable format:'^I^I, format: 10);
  case format of
    (* look for `virtual int getFormat() const { return ' + constant name
       in UPX' `src\p_*.h' to grab the human-readable names *)

    { little endian formats }
    UPX_F_DOS_COM:            s := 'dos/com';
    UPX_F_DOS_SYS:            s := 'dos/sys';
    UPX_F_DOS_EXE:            s := 'dos/exe';
    UPX_F_DJGPP2_COFF:        s := 'djgpp2/coff';
    UPX_F_WATCOM_LE:          s := 'watcom/le';
  { UPX_F_VXD_LE:             s := 'NOT IMPLEMENTED'; }
    UPX_F_DOS_EXEH:           s := 'dos/exe#obsolete';
    UPX_F_TMT_ADAM:           s := 'tmt/adam';
    UPX_F_WIN32_PE:           if is32rtm then
                                s := 'rtm32/pe'
                              else
                                s := 'win32/pe';
    UPX_F_LINUX_i386:         s := 'linux.exec/i386';
  { UPX_F_WIN16_NE:           s := 'NOT IMPLEMENTED'; }
    UPX_F_LINUX_ELF_i386:     s := 'linux/i386';
  { UPX_F_LINUX_SEP_i386:     s := 'NOT IMPLEMENTED'; }
    UPX_F_LINUX_SH_i386:      s := 'linux.sh/i386';
    UPX_F_VMLINUZ_i386:       s := 'vmlinuz/i386';
    UPX_F_BVMLINUZ_i386:      s := 'bvmlinuz/i386';
  { UPX_F_ELKS_8086:          s := 'NOT IMPLEMENTED'; }
    UPX_F_PS1_EXE:            s := 'ps1/exe';
    UPX_F_VMLINUX_i386:       s := 'vmlinux/i386';
    UPX_F_LINUX_ELFI_i386:    s := 'linux/elfi386';
    UPX_F_WINCE_ARM_PE:       s := 'arm/pe';
    UPX_F_LINUX_ELF64_AMD:    s := 'linux/amd64';
    UPX_F_LINUX_ELF32_ARMEL:  s := 'linux/arm';
    UPX_F_BSD_i386:           s := 'bsd.exec/i386';
    UPX_F_BSD_ELF_i386:       s := 'freebsd/i386';
  { UPX_F_BSD_SH_i386:        s := 'NOT IMPLEMENTED?'; }

    UPX_F_VMLINUX_AMD64:      s := 'vmlinux/amd64';
    UPX_F_VMLINUX_ARMEL:      s := 'vmlinux/arm';
    UPX_F_MACH_i386:          s := 'macho/i386';
    UPX_F_LINUX_ELF32_MIPSEL: s := 'linux/mipsel';
    UPX_F_VMLINUZ_ARMEL:      s := 'vmlinuz/arm';
    UPX_F_MACH_ARMEL:         s := 'macho/arm';

    UPX_F_DYLIB_i386:         s := 'dylib/i386';
    UPX_F_MACH_AMD64:         s := 'macho/amd64';
    UPX_F_DYLIB_AMD64:        s := 'dylib/amd64';

    UPX_F_WIN64_PEP:          s := 'win64/pe';

    UPX_F_MACH_ARM64EL:       s := 'macho/arm64';

    UPX_F_MACH_PPC64LE:       s := 'macho/ppc64le';
    UPX_F_LINUX_ELFPPC64LE:   s := 'linux/ppc64le';
    UPX_F_VMLINUX_PPC64LE:    s := 'vmlinux/ppc64le';
    UPX_F_DYLIB_PPC64LE:      s := 'dylib/ppc64le';

    UPX_F_LINUX_ELF64_ARM:    s := 'linux/arm64';

    { big endian formats }
    UPX_F_ATARI_TOS:          s := 'atari/tos';
  { UPX_F_SOLARIS_SPARC:      s := 'NOT IMPLEMENTED'; }
    UPX_F_MACH_PPC32:         s := 'macho/ppc32';
    UPX_F_LINUX_ELFPPC32:     s := 'linux/ppc32';
    UPX_F_LINUX_ELF32_ARMEB:  s := 'linux/armeb';
    UPX_F_MACH_FAT:           s := 'macho/fat';
    UPX_F_VMLINUX_ARMEB:      s := 'vmlinux/armeb';
    UPX_F_VMLINUX_PPC32:      s := 'vmlinux/ppc32';
    UPX_F_LINUX_ELF32_MIPSEB: s := 'linux/mips';
    UPX_F_DYLIB_PPC32:        s := 'dylib/ppc32';
  else
    s := '?';
  end;
  WriteLn(' ("', s, '")');

  Write('Compression method:'^I^I, method: 10);
  case method of
    M_NRV2B_LE32: s := 'nrv2b/le32';
    M_NRV2B_8:    s := 'nrv2b/8';
    M_NRV2B_LE16: s := 'nrv2b/le16';
    M_NRV2D_LE32: s := 'nrv2d/le32';
    M_NRV2D_8:    s := 'nrv2d/8';
    M_NRV2D_LE16: s := 'nrv2d/le16';
    M_NRV2E_LE32: s := 'nrv2e/le32';
    M_NRV2E_8:    s := 'nrv2e/8';
    M_NRV2E_LE16: s := 'nrv2b/le16';
    { M_CL1B_LE32:  s := ''; }
    { M_CL1B_8:     s := ''; }
    { M_CL1B_LE16:  s := ''; }
    M_LZMA:       s := 'lzma';
    { M_DEFLATE:    s := ''; }
  else
    s := '?';
  end;
  WriteLn(' ("', s, '")');

  WriteLn('Compression level:'^I^I, level: 10);
  WriteLn('Uncompressed data length:'^I, u_len: 10, ' bytes');
  WriteLn('Compressed data length:'^I^I, c_len: 10, ' bytes');
  WriteLn('Original file size:'^I^I, u_file_size: 10, ' bytes');
  WriteLn('Filter:'^I^I^I^I, filter: 10, '/', filter_cto);
end.

{ EOF }
