/* 
CCIDE
Copyright 2001-2011 David Lindauer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

You may contact the author at:
	mailto::camille@bluegrass.net
 */
#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <ctype.h>
#include <richedit.h>
#include "header.h"
#include <dir.h>
#include "cvinfo.h"

extern HWND hwndSourceTab;
extern HWND hwndFrame, hwndProject, hwndRegister, hwndClient, hwndTab;
extern HINSTANCE hInstance;
extern THREAD *StoppedThread;
extern PROCESS DebugProcess;
extern unsigned bitmask[];
extern HWND lastActiveEditWindow;

POINT menupos;

HWND hwndWatch;
static HWND hwndTree, hwndCtrl;
static char szWatchClassName[] = "xccWatchClass";
static char szWatchTitle[] = "Watch Window";

static HBITMAP valueBitmap, itemBitmap;

static LOGFONT fontdata = 
{
    14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, ANSI_CHARSET,
        OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, FF_DONTCARE, 
        "Helvetica"
};
static WATCHINFO *watchinfo_list;
static int watchinfo_max;
static int watchinfo_count;

void AddTypeInfoToName(char *typetab, VARINFO *v)
{
    char buf[256],  *p;
    strcpy(v->screenname, v->membername);
    p = v->screenname + strlen(v->screenname);
    sprintf(p, "(%s)", SymTypeName(buf, typetab, v));
}

//-------------------------------------------------------------------------

WATCHINFO *MatchItem(VARINFO *var)
{
    int i;
    for (i = 0; i < watchinfo_count; i++)
        if (!strcmp(watchinfo_list[i].info->membername, var->membername))
            if (watchinfo_list[i].info->typetab == var->typetab)
                return  &watchinfo_list[i];
    return 0;
}

//-------------------------------------------------------------------------

void FreeTree(VARINFO *info)
{
    while (info)
    {
        FreeTree(info->subtype);
        if (info->hTreeItem)
            TreeView_DeleteItem(hwndTree, info->hTreeItem);
        info = info->link;
    }
}

//-------------------------------------------------------------------------

void RefreshAddresses(VARINFO *var, int address, THREAD *thread, int noscope)
{
    while (var)
    {
        int unscope = noscope;
        int val;
        if (noscope)
            var->outofscope = TRUE;
        else
        {
            var->outofscope = FALSE;
			if (thread)
				var->thread = thread;
            val = var->address = address + var->offset;
            if (var->pointer)
			{
                unscope = (val = var->derefaddress) == -1 && !ReadValue(var->address, &val, 4, var->inreg ? &var
                    ->thread->regs: 0) || !val;
			}
        }
        RefreshAddresses(var->subtype, val, thread, unscope);
        var = var->link;
    }
}

//-------------------------------------------------------------------------

void WatchValue(char *typetab, char *buf, VARINFO *info, int onevalue)
{
    int i;
    if (info->outofscope || info->outofscopereg)
        sprintf(buf, "out of scope");
	else if (info->constant)
		sprintf(buf, "POINTER: %p", info->address);
    else if (info->structure)
    {
        sprintf(buf, "STRUCTURE: %p", info->address);
    }
    else if (info->unionx)
    {
        sprintf(buf, "UNION: %p", info->address);
    }
    else if (info->pointer)
    {
        int val;
        if ((val = info->derefaddress) != -1 || ReadValue(info->address, &val, 4, info->inreg ? &info->thread->regs
            : 0))
        {
            info->editable = TRUE;
            if (onevalue)
                sprintf(buf, "0x%p ", val);
            else
                sprintf(buf, "POINTER: %p ", val);
            GetStringValue(info, buf + strlen(buf), 32, val);
        }
        else
            sprintf(buf, "POINTER: <UNKNOWN>");
    }
    else if (info->enumx)
    {
        info->editable = TRUE;
        HintEnum(typetab, info, buf, FALSE, onevalue);
    }
    else if (info->bitfield)
    {
        int signedtype;
        int v = HintBf(info, &signedtype);
        if (onevalue)
            if (signedtype)
                sprintf(buf, "0x%x", v);
            else
                sprintf(buf, "0x%x", v);
            else
                if (signedtype)
                    sprintf(buf, "%d(0x%x)", v, v);
                else
                    sprintf(buf, "%u(0x%x)", v, v);
        info->editable = TRUE;
    }
    else if (info->array)
    {
        sprintf(buf, "ARRAY: %p ", info->address);
        GetStringValue(info, buf + strlen(buf), 32, info->address);
    }
    else
    {
        int signedtype;
        char buf1[20];
        LLONG_TYPE v;
        info->editable = TRUE;
        switch (HintBasicType(info, &signedtype, buf1))
        {
            case T_INT8:
                #ifndef BORLANDC
                    v = *(LLONG_TYPE*)buf1;
                    if (onevalue)
                        if (signedtype)
                            sprintf(buf, "0x%llx", v);
                        else
                            sprintf(buf, "0x%llx", v);
                        else
                            if (signedtype)
                                sprintf(buf, "%lld(0x%llx)", v, v);
                            else
                                sprintf(buf, "%lld(0x%llx)", v, v);
                    break;
                #endif 
            default:
                sprintf(buf, "unknown type");
                break;

            case T_INT4:
                v = *(int*)buf1;
                if (onevalue)
                    if (signedtype)
                        sprintf(buf, "0x%x", (int)v);
                    else
                        sprintf(buf, "0x%x", (int)v);
                    else
                        if (signedtype)
                            sprintf(buf, "%d(0x%x)", (int)v, (int)v);
                        else
                            sprintf(buf, "%u(0x%x)", (int)v, (int)v);
                break;
            case T_BOOL08:
                if (buf1[0])
                    sprintf(buf, "True");
                else
                    sprintf(buf, "False");
                break;
            case T_REAL32:
            case T_IMAGINARY32:
                sprintf(buf, "%f", (double)*(float*)buf1);
                break;
            case T_REAL80:
            case T_IMAGINARY80:
                *(double*)buf1 = *(long double*)buf1;
            case T_REAL64:
            case T_IMAGINARY64:
                sprintf(buf, "%f", *(double*)buf1);
                break;
            case T_CPLX32:
                sprintf(buf, "%f + %f * I", (double)*(float*)buf1, (double)*(float *)(buf1 + 4));
                break;
            case T_CPLX64:
                sprintf(buf, "%f + %f * I", *(double *)buf1, *(double *)(buf1 + 8));
                break;
            case T_CPLX80:
                sprintf(buf, "%f + %f * I", (double)*(long double *)buf1, (double)*(long double *)(buf1 + 10));
                break;
        }
    }
}

//-------------------------------------------------------------------------

void RefreshData(char *typetab, VARINFO *var)
{
    while (var)
    {
        WatchValue(typetab, var->value, var, FALSE);
        RefreshData(typetab, var->subtype);
        var = var->link;
    }
}

//-------------------------------------------------------------------------

void RefreshItem(WATCHINFO *var, int address, THREAD *thread)
{
    RefreshAddresses(var->info, address, thread, var->info->outofscope);
    RefreshData(var->typetab, var->info);
}

//-------------------------------------------------------------------------

HTREEITEM InsertItem(HTREEITEM hParent, HTREEITEM after, VARINFO *var)
{
    HTREEITEM rv;
    TV_INSERTSTRUCT t;
    memset(&t, 0, sizeof(t));
    t.hParent = hParent;
    t.hInsertAfter = after;
    #if !defined( _WIN32_IE) && !defined(__CCDL__)
        t.item.mask = 0;
        t.item.lParam = (int)var;
    #else 
        t.u.item.mask = 0;
        t.u.item.lParam = (int)var;
    #endif 
    rv = TreeView_InsertItem(hwndTree, &t);
    return rv;
}

//-------------------------------------------------------------------------

void InsertSubTree(HTREEITEM parent, HTREEITEM after, VARINFO *var, int index)
{
    while (var)
    {
        AddTypeInfoToName(watchinfo_list[index].typetab, var);
        var->hTreeItem = InsertItem(parent, after, var);
        var->watchindex = index;
        var->watchhead.col1Text = &var->screenname;
        var->watchhead.col2Text = &var->value;
        if (var->pointer && !var->subtype)
        {
            var->hTreeHolder = InsertItem(var->hTreeItem, TVI_LAST, var);
            TreeView_Expand(hwndTree, var->hTreeItem, TVE_COLLAPSE);
        }
        else
            InsertSubTree(var->hTreeItem, 0, var->subtype, index);
        after = var->hTreeItem;
        var = var->link;
    }
}

//-------------------------------------------------------------------------

void AddItem(char *typetab, DEBUG_INFO *dbg, char *symtab, VARINFO *var, 
			 int cursoreip)
{
    WATCHINFO *x = MatchItem(var);
    if (x)
    {
        FreeVarInfo(var);
    }
    else
    {
        HTREEITEM previous = 0;
        if (watchinfo_count >= watchinfo_max)
        {
            if (watchinfo_max >= 128)
            {
                ExtendedMessageBox("Watch Error", MB_SETFOREGROUND |
                    MB_SYSTEMMODAL, 
                    "There are too many items in the watch window\nNot adding the current selection");
                return ;
            }
            watchinfo_max += 64;
            watchinfo_list = realloc(watchinfo_list, watchinfo_max *sizeof
                (WATCHINFO));
        }
        if (watchinfo_count)
            previous = watchinfo_list[watchinfo_count - 1].info->hTreeItem;
        memset(&watchinfo_list[watchinfo_count], 0, sizeof(watchinfo_list[0]));
        watchinfo_list[watchinfo_count].info = var;
        watchinfo_list[watchinfo_count].dbg_info = dbg;
        watchinfo_list[watchinfo_count].symtab = symtab;
        watchinfo_list[watchinfo_count].cursoreip = cursoreip;
        watchinfo_list[watchinfo_count++].typetab = typetab;
        RefreshItem(&watchinfo_list[watchinfo_count - 1], var->address, NULL);
        InsertSubTree(0, previous, var, watchinfo_count - 1);
    }
}

//-------------------------------------------------------------------------

void RenumberDeleteItems(VARINFO *v)
{
    while (v)
    {
        v->watchindex--;
        if (v->subtype)
            RenumberDeleteItems(v->subtype);
        v = v->link;
    }
}

//-------------------------------------------------------------------------

void DeleteItem(POINT *pt)
{
    int i;
    WATCHINFO *x;
    TV_HITTESTINFO t;
    HTREEITEM titem;
    TV_ITEM item;
    VARINFO *v;

    ScreenToClient(hwndTree, pt);
    t.pt =  *pt;
    titem = TreeView_HitTest(hwndTree, &t);
    if (titem)
    {
        int c;
        item.mask = TVIF_PARAM;
        item.hItem = titem;
        TreeView_GetItem(hwndTree, &item);
        v = (VARINFO*)item.lParam;
        x = &watchinfo_list[c = v->watchindex];
        FreeTree(x->info);
        FreeVarInfo(x->info);
        for (i = c; i < watchinfo_count - 1; i++)
        {
            watchinfo_list[i] = watchinfo_list[i + 1];
            RenumberDeleteItems(watchinfo_list[i].info);
        }
        watchinfo_count--;
    }
}

//-------------------------------------------------------------------------

void DeleteAllItems(void)
{
    int i;
    TreeView_DeleteAllItems(hwndTree);
    for (i = 0; i < watchinfo_count; i++)
        FreeVarInfo(watchinfo_list[i].info);
    watchinfo_count = 0;
}

//-------------------------------------------------------------------------

void Unscope(WATCHINFO *wi)
{
    if (!wi->info->outofscope)
    {
        wi->info->outofscope = TRUE;
        //      FreeTree(wi->info->subtype) ;
    }
}

//-------------------------------------------------------------------------

void Rescope(WATCHINFO *wi, int index)
{
    if (wi->info->outofscope)
    {
        wi->info->outofscope = FALSE;
        //      InsertSubTree(wi->info->hTreeItem,0,wi->info->subtype, index) ;
    }
}

//-------------------------------------------------------------------------

void RefreshItems(void)
{
    int i;
    char *types,  *syms;
    int offset, offset2;
    DEBUG_INFO *dbg;
	THREAD *thread = NULL;
    for (i = 0; i < watchinfo_count; i++)
    {
        WATCHINFO *wi = &watchinfo_list[i];
		VARINFO *var;
        int level;
//        offset = wi->cursoreip;
		offset2 = offset = EIPFromCursor(0);
        var = EvalExpr(&wi->typetab, &wi->symtab, &wi->dbg_info, offset, 
							   &offset, (char*)wi->info->membername, FALSE);
		if (!var)
		{
//        if (!FindSymbol(&wi->dbg_info, &wi->typetab, &wi->symtab, &offset,
//            offset1, offset, (char*)wi->info->membername))
            Unscope(wi);
            offset = 0;
        }
        else
        {
//            VARINFO *x = GetVarInfo(wi->dbg_info, wi->typetab, wi->symtab,
//                offset, wi->info->membername, wi->ebp, wi->info->thread);
            wi->info->outofscopereg = var->outofscopereg;
            if (var->outofscopereg)
            {
                Unscope(wi);
                offset = 0;
            }
            else
            {
				int ebp, level;
                Rescope(wi, i);
                offset = var->address;
				wi->cursoreip = offset2;
		        findStackedFunction(offset2, &ebp, &level, &thread);
				if (var->pointer)
				{
					wi->info->derefaddress = var->derefaddress;
				}
            }
            FreeVarInfo(var);
        }
        RefreshItem(wi, offset, thread);
    }
    InvalidateRect(hwndTree, 0, 0);
}

//-------------------------------------------------------------------------

void ExpandPointer(VARINFO *v, int code)
{
    if (v->pointer)
    {
        if (code == TVE_EXPAND)
        {
            int val;
            int outofscope = (val = v->derefaddress) == -1 && !ReadValue(v->address, &val, 4, v->inreg ? &v
                ->thread->regs: 0) || !val;
            if (!v->subtype && watchinfo_list[v->watchindex].typetab)
            {
                TreeView_DeleteItem(hwndTree, v->hTreeHolder);
                GetPointerInfo(watchinfo_list[v->watchindex].typetab, v);
				if (v->subtype->structure)
				{
            	    InsertSubTree(v->hTreeItem, 0, v->subtype->subtype, v->watchindex);
				}
				else
	    	        InsertSubTree(v->hTreeItem, 0, v->subtype, v->watchindex);
    	        //			TreeView_Expand(hwndTree, v->hTreeItem, TVE_EXPAND) ;
            }
            RefreshAddresses(v->subtype, val, NULL, outofscope);
            RefreshData(watchinfo_list[v->watchindex].typetab, v);
        }
        else if (code == TVE_COLLAPSE)
        {
            if (v->subtype)
            {
                FreeTree(v->subtype);
                FreeVarInfo(v->subtype);
                v->subtype = 0;
                v->hTreeHolder = InsertItem(v->hTreeItem, TVI_LAST, v);
            }
        }
    }
}

//-------------------------------------------------------------------------

int WriteValue(int address, void *value, int size, CONTEXT *regs)
{
    int len;
    if (address < 0x1000)
    {
        // register
        int val = *(int*)value;
        switch (address)
        {
            case CV_REG_AL:
                regs->Eax &= 0xffffff00;
                regs->Eax |= (unsigned char)val;
                return 1;
            case CV_REG_CL:
                regs->Ecx &= 0xffffff00;
                regs->Ecx |= (unsigned char)val;
                return 1;
            case CV_REG_DL:
                regs->Edx &= 0xffffff00;
                regs->Edx |= (unsigned char)val;
                return 1;
            case CV_REG_BL:
                regs->Ebx &= 0xffffff00;
                regs->Ebx |= (unsigned char)val;
                return 1;
            case CV_REG_AH:
                regs->Eax &= 0xffff00ff;
                regs->Eax |= ((unsigned char)val) << 8;
                return 1;
            case CV_REG_CH:
                regs->Ecx &= 0xffff00ff;
                regs->Ecx |= ((unsigned char)val) << 8;
                return 1;
            case CV_REG_DH:
                regs->Edx &= 0xffff00ff;
                regs->Edx |= ((unsigned char)val) << 8;
                return 1;
            case CV_REG_BH:
                regs->Ebx &= 0xffff00ff;
                regs->Ebx |= ((unsigned char)val) << 8;
                return 1;
            case CV_REG_AX:
                regs->Eax &= 0xffff0000;
                regs->Eax |= (unsigned short)val;
                return 2;
            case CV_REG_CX:
                regs->Ecx &= 0xffff0000;
                regs->Ecx |= (unsigned short)val;
                return 2;
            case CV_REG_DX:
                regs->Edx &= 0xffff0000;
                regs->Edx |= (unsigned short)val;
                return 2;
            case CV_REG_BX:
                regs->Ebx &= 0xffff0000;
                regs->Ebx |= (unsigned short)val;
                return 2;
            case CV_REG_SP:
                regs->Esp &= 0xffff0000;
                regs->Esp |= (unsigned short)val;
                return 2;
            case CV_REG_BP:
                regs->Ebp &= 0xffff0000;
                regs->Ebp |= (unsigned short)val;
                return 2;
            case CV_REG_SI:
                regs->Esi &= 0xffff0000;
                regs->Esi |= (unsigned short)val;
                return 2;
            case CV_REG_DI:
                regs->Edi &= 0xffff0000;
                regs->Edi |= (unsigned short)val;
                return 2;
            case CV_REG_EAX:
                regs->Eax = val;
                return 4;
            case CV_REG_ECX:
                regs->Ecx = val;
                return 4;
            case CV_REG_EDX:
                regs->Edx = val;
                return 4;
            case CV_REG_EBX:
                regs->Ebx = val;
                return 4;
            case CV_REG_ESP:
                regs->Esp = val;
                return 4;
            case CV_REG_EBP:
                regs->Ebp = val;
                return 4;
            case CV_REG_ESI:
                regs->Esi = val;
                return 4;
            case CV_REG_EDI:
                regs->Edi = val;
                return 4;
            case CV_REG_ST0:
                // not supported
            case CV_REG_ST1:
            case CV_REG_ST2:
            case CV_REG_ST3:
            case CV_REG_ST4:
            case CV_REG_ST5:
            case CV_REG_ST6:
            case CV_REG_ST7:
                return 0;
        }
    }
    else
    {
        WriteProcessMemory(DebugProcess.hProcess, (LPVOID)address, (LPVOID)
            value, size, &len);
        return len;
    }
}

//-------------------------------------------------------------------------

int InputEnum(char *typetab, VARINFO *info, char *text)
{
    int signedtype;
    int v;
    short *typeptr = info->typetab;
    while (*(typeptr + 1) == LF_FIELDLIST)
    {
        int done = FALSE;
        int len =  *typeptr - 2;
        typeptr += 2;
        while (len > 0 && !done)
        {
            int xlen;
            int rv;
            char *nmptr;
            switch (*typeptr)
            {
                case LF_ENUMERATE:
                    xlen = sizeof(lfEnumerate) - 1;
                    if (v < LF_NUMERIC)
                        xlen += 2;
                    else
                        xlen += 6;
                    nmptr = ((char*)typeptr) + xlen;
                    rv = GetNumericLeaf(&((lfEnumerate*)typeptr)->value);
                    {
                        char buf1[256];
                        memset(buf1, 0, 256);
                        strncpy(buf1, nmptr + 1,  *nmptr);
                        if (!strcmp(buf1, text))
                            return rv;
                    }
                    xlen +=  *nmptr + 1;
                    if ((*((char*)typeptr + xlen) &0xf0) == 0xf0)
                        xlen += *((char*)typeptr + xlen) &15;
                    (char*)typeptr += xlen;
                    len -= xlen;
                    break;
                case LF_INDEX:
                    typeptr = LookupType(typetab, ((lfIndex*)typeptr)->index);
                    done = TRUE;
                    break;
                default:
                    return 0;
            }
        }
    }
    return 0;
}

//-------------------------------------------------------------------------

void ChangeData(VARINFO *info, char *text)
{
    if (CV_TYP_IS_REAL(info->type) || CV_TYP_IS_IMAGINARY(info->type))
    {
        float v;
        double v2;
        char data[10];
        sscanf(text, "%f", &v);
        switch (info->type)
        {
            case T_REAL32:
            case T_IMAGINARY32:
                WriteValue(info->address, &v, 4, &info->thread->regs);
                break;
            case T_REAL64:
            case T_IMAGINARY64:
                v2 = v;
                WriteValue(info->address, &v2, 8, &info->thread->regs);
                break;
            case T_REAL80:
            case T_IMAGINARY80:
                *(long double*)data = v;
                WriteValue(info->address, data, 10, &info->thread->regs);
                break;
            default:
                break;
        }
    }
    else if (CV_TYP_IS_COMPLEX(info->type)) 
    {
        float v[2];
        double v2[2];
        long double v4[2];
        sscanf(text, "%f + %f * I", &v[0],&v[1]);
        switch (info->type)
        {
            case T_CPLX32:
                WriteValue(info->address, &v[0], 8, &info->thread->regs);
                break ;
            case T_CPLX64:
                v2[0] = v[0];
                v2[1] = v[1];
                WriteValue(info->address, &v2[0], 16, &info->thread->regs);
                break ;
            case T_CPLX80:
                v4[0] = v[0];
                v4[1] = v[1];
                WriteValue(info->address, &v4[0], 20, &info->thread->regs);
                break ;
        }        
    }
    else
    {
        LLONG_TYPE value;
        int size;
        if (info->enumx && !isdigit(text[0]))
        {
            value = InputEnum(watchinfo_list[info->watchindex].typetab, info,
                text);
            size = 4;
        }
        else
        {
            #ifdef BORLANDC
                if (text[0] == '0' && text[1] == 'x')
                    sscanf(text + 2, "%x", &value);
                else if ((text[strlen(text) - 1] &0xDF) == 'H')
                    sscanf(text, "%x", &value);
                else
                    sscanf(text, "%d", &value);
            #else 
                if (text[0] == '0' && text[1] == 'x')
                    sscanf(text + 2, "%llx", &value);
                else if (text[strlen(text) - 1] &0xDF == 'H')
                    sscanf(text, "%llx", &value);
                else
                    sscanf(text, "%lld", &value);
            #endif 
            switch (CV_TYPE(info->type))
            {
                case CV_INT:
                    size = 1 << (CV_SUBT(info->type) / 2);
                    break;
                case CV_SIGNED:
                case CV_UNSIGNED:
                case CV_BOOLEAN:
                    size = 1 << CV_SUBT(info->type);
                    break;
            }
        }
        if (info->bitfield)
        {
            char data[10];
            int signedx;
            int v;
            HintBasicType(info, &signedx, &data);
            v = *(int*)data;
            v &= ~(bitmask[info->bitlength - 1] << info->bitstart);
            value = v | ((value &bitmask[info->bitlength - 1]) << info
                ->bitstart);
        }
        WriteValue(info->address, &value, size, &info->thread->regs);
        if (info->pointer)
        {
            if (!value && info->subtype)
            {
                VARINFO *inf = info->subtype;
                info->subtype = 0;
                FreeTree(inf);
                FreeVarInfo(inf);
            }
            else
                RefreshAddresses(info->subtype, value, NULL, info->outofscope);
        }
    }
    RefreshData(watchinfo_list[info->watchindex].typetab, info);
}

//-------------------------------------------------------------------------

int EIPFromCursor(HWND win)
{
    charinfo charrange;
    int offset = 0;
    int lineno;
    char *filename;
    if (!win)
    {
        win = lastActiveEditWindow;
        if (!IsWindow(win))
            return 0;
    }
    filename = (char*)SendMessage(win, WM_FILENAME, 0, 0);
    SendMessage(GetDlgItem(win, ID_EDITCHILD), EM_EXGETSEL, (WPARAM)0, (LPARAM)
        &charrange);
    lineno = SendMessage(GetDlgItem(win, ID_EDITCHILD), EM_EXLINEFROMCHAR, 0, 
        (LPARAM)charrange.min) + 1;
    lineno = TagOldLine(filename, lineno);
    offset = GetBreakpointAddress(filename, &lineno, TRUE);
    return offset;
}

//-------------------------------------------------------------------------

LRESULT CALLBACK _export WatchWndProc(HWND hwnd, UINT iMessage, WPARAM wParam,
    LPARAM lParam)
{
    static int selected;
    static char buf[256];
    RECT r,  *pr;
    WINDOWPOS wp;
    HD_ITEM hie;
    HD_LAYOUT hdl;
    NMHDR *h;
    DRAWITEMSTRUCT *dr;
    HBITMAP hbmp;
    HDC hMemDC;
    TCHeader tch;
    TV_ITEM item;
    TV_INSERTSTRUCT t;
    static int sizingbottom;
    int offset;
    char *types,  *syms;
    DEBUG_INFO *dbg;
    NM_TREEVIEW *nmt;
    VARINFO *var;
    HWND win;
    int level;
    int offset1;
	int doit;
    charinfo charrange;
    switch (iMessage)
    {
        case WM_MDIACTIVATE:
            SendMessage(hwndCtrl, LCF_SETACTIVE, 0, lParam == (LPARAM)hwnd);
            if (lParam == (LPARAM)hwnd)
            {
                SendMessage(hwndSourceTab, WM_SETACTIVETAB, 0, (LPARAM)hwnd);
                SetFocus(hwndTree);
            }
            return TRUE;
        case WM_SYSCOMMAND:
            if (wParam == SC_CLOSE)
            {
                PostMessage(hwnd, WM_CLOSE, 0, 0);
                return 0;
            }
            break;
        case WM_NOTIFY:
            h = (NMHDR*)lParam;
            switch (h->code)
            {
            case NM_RCLICK:
                {
                    HMENU menu = LoadMenu(hInstance, "WATCHMENU");
                    HMENU popup = GetSubMenu(menu, 0);
					InsertBitmapsInMenu(popup);
                    GetCursorPos(&menupos);
                    TrackPopupMenuEx(popup, TPM_BOTTOMALIGN | TPM_LEFTBUTTON,
                        menupos.x, menupos.y, hwndFrame, NULL);
                    DestroyMenu(menu);
	                SetFocus(hwndTree);
                }
                return 0;
            case TVN_ITEMEXPANDING:
                nmt = h;
                if (nmt->action)
                {
                    ExpandPointer((VARINFO*)nmt->itemNew.lParam, nmt->action);
                }
                return 0;
            case TCN_EDITQUERY:
                nmt = h;
                item.mask = TVIF_PARAM;
                item.hItem = (HTREEITEM)nmt->itemNew.hItem;
                TreeView_GetItem(hwndTree, &item);
                var = (VARINFO*)item.lParam;
                if (var->editable)
                {
                    WatchValue(watchinfo_list[var->watchindex].typetab, buf,
                        var, TRUE);
                    return buf;
                }
                return 0;
            case TCN_EDITDONE:
                nmt = h;
                item.mask = TVIF_PARAM;
                item.hItem = (HTREEITEM)nmt->itemNew.hItem;
                TreeView_GetItem(hwndTree, &item);
                var = (VARINFO*)item.lParam;
                ChangeData(var, nmt->itemNew.pszText);
				RefreshItems();
                return 0;
            }
            break;
        case WM_CREATE:
            hwndWatch = hwnd;
            GetClientRect(hwnd, &r);
            hwndCtrl = CreateControlWindow(DID_WATCHWND, hwnd, &r, FALSE,
                szWatchTitle);
            GetClientRect(hwnd, &r);
            SendMessage(hwndCtrl, LCF_ADJUSTRECT, 0, (LPARAM) &r);
            valueBitmap = LoadImage(hInstance, "ID_VALUEBMP", IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
            itemBitmap = LoadImage(hInstance, "ID_ITEMBMP", IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
            tch.colText1 = "Item";
            tch.colText2 = "Value";
            tch.colBmp1 = itemBitmap;
            tch.colBmp2 = valueBitmap;
            hwndTree = CreateextTreeWindow(hwndCtrl, WS_DLGFRAME | TCS_LINE, &r,
                &tch);
            //         PostMessage(hwnd,WM_CLOSE,0 , 0 ) ;
            //         PostMessage(hwndFrame, WM_REDOSIZING, 0, 0) ;
            return 0;
        case WM_ADDWATCHINDIRECT:
            win = (HWND)wParam;
			doit = FALSE;
		    SendMessage(win, EM_EXGETSEL, (WPARAM)0, (LPARAM) &charrange);
			if (charrange.min == charrange.max)
			{
            	doit = SendMessage(win, WM_WORDUNDERCURSOR, (WPARAM)&menupos, (LPARAM)buf);
				if (!doit)
				{
					SendMessage(hwndFrame, IDM_ADDWATCH, 0, 0);
					break;
				}
			}
			else
			{
				if (charrange.max - charrange.min < sizeof(buf))
				{
					SendMessage(win, EM_GETSELTEXT, 0, (LPARAM)buf);
					doit = TRUE ;
				}
			}
			if (doit)
            {
                int offset2 = offset = StoppedThread->regs.Eip;
                var = EvalExpr(&types, &syms, &dbg, offset, &offset, (char*)
                    buf, TRUE);
                if (var)
                {
                    AddItem(types, dbg, syms, var, offset2);
                    dmgrHideWindow(DID_WATCHWND, FALSE);
					break ;
                }
			}
            ExtendedMessageBox("Error", MB_SETFOREGROUND |
                MB_SYSTEMMODAL, "Symbol does not exist in this scope");
            break;
        case WM_ADDWATCH:
            offset = wParam;
            if (!offset)
            {
                DeleteAllItems();
            }
            else
            {
                int offset2 = offset = StoppedThread->regs.Eip;
                var = EvalExpr(&types, &syms, &dbg, offset, &offset, (char*)
                    lParam, TRUE);
                if (var)
                {
                    AddItem(types, dbg, syms, var, offset2);
                    dmgrHideWindow(DID_WATCHWND, FALSE);
                }
            }
            break;
        case WM_COMMAND:
            switch (wParam)
            {
            case ID_SETADDRESS:
                RefreshItems();
                InvalidateRect(hwndTree, 0, 0);
                break;
            case IDM_DELETEWATCH:
                DeleteItem(&menupos);
                break;
			case IDM_DELETEALLWATCH:
				DeleteAllItems();
				break;
            }
            break;
        case WM_DESTROY:
            TreeView_DeleteAllItems(hwndTree);
            DestroyWindow(hwndTree);
            DeleteObject(valueBitmap);
            DeleteObject(itemBitmap);
            DestroyWindow(hwndCtrl);
            hwndWatch = 0;
            break;

        case WM_SIZE:
            r.left = r.top = 0;
            r.right = LOWORD(lParam);
            r.bottom = HIWORD(lParam);
            MoveWindow(hwndCtrl, r.left, r.top, r.right - r.left, r.bottom -
                r.top, TRUE);
            SendMessage(hwndCtrl, LCF_ADJUSTRECT, 0, (LPARAM) &r);
            MoveWindow(hwndTree, r.left, r.top, r.right - r.left, r.bottom -
                r.top, TRUE);
            return 0;
        case WM_CLOSE:
            dmgrHideWindow(DID_WATCHWND, TRUE);
            return 0;
    }
    return DefMDIChildProc(hwnd, iMessage, wParam, lParam);
}

//-------------------------------------------------------------------------

void RegisterWatchWindow(void)
{
    WNDCLASS wc;
    memset(&wc, 0, sizeof(wc));
    wc.style = 0;
    wc.lpfnWndProc = &WatchWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(0, IDI_APPLICATION);
    wc.hCursor = LoadCursor(0, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName = 0;
    wc.lpszClassName = szWatchClassName;
    RegisterClass(&wc);
}

//-------------------------------------------------------------------------

void CreateWatchWindow(void)
{
    CreateMDIWindow(szWatchClassName, szWatchTitle, WS_CHILD | WS_CLIPSIBLINGS 
        | WS_DLGFRAME, CW_USEDEFAULT, CW_USEDEFAULT, 200, 200, hwndClient,
        hInstance, (LPARAM)0);
}
