% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/progress.R
\name{progress_bar}
\alias{progress_bar}
\title{Progress bar in the terminal}
\description{
Progress bars are configurable, may include percentage, elapsed time,
and/or the estimated completion time. They work in the command line,
in Emacs and in R Studio. The progress package was heavily influenced by
https://github.com/tj/node-progress
}
\section{Creating the progress bar}{

A progress bar is an R6 object, that can be created with
\code{progress_bar$new()}. It has the following arguments:
\describe{
  \item{format}{The format of the progress bar. A number of
    tokens can be used here, see them below. It defaults to
    \code{"[:bar] :percent"}, which means that the progress
    bar is within brackets on the left, and the percentage
    is printed on the right.}
  \item{total}{Total number of ticks to complete. If it is unknown,
     use \code{NA} here. Defaults to 100.}
  \item{width}{Width of the progress bar. Default is the current
    terminal width (see \code{options()} and \code{width}) minus two.}
  \item{stream}{This argument is deprecated, and \code{message()} is
    used to print the progress bar.}
  \item{complete}{Completion character, defaults to \code{=}.}
  \item{incomplete}{Incomplete character, defaults to \code{-}.}
  \item{current}{Current character, defaults to \code{>}.}
  \item{callback}{Callback function to call when the progress
    bar finishes. The progress bar object itself is passed to it
    as the single parameter.}
  \item{clear}{Whether to clear the progress bar on completion.
    Defaults to \code{TRUE}.}
  \item{show_after}{Amount of time in seconds, after which the progress
    bar is shown on the screen. For very short processes,
    it is probably not worth showing it at all. Defaults to two
    tenth of a second.}
  \item{force}{Whether to force showing the progress bar,
    even if the given (or default) stream does not seem to support it.}
}
}

\section{Using the progress bar}{

Three functions can update a progress bar. \code{progress_bar$tick()}
increases the number of ticks by one (or another specified value).
\code{progress_bar$update()} sets a given ratio and
\code{progress_bar$terminate()} removes the progress bar.
\code{progress_bar$finished} can be used to see if the progress bar has
finished.

Note that the progress bar is not shown immediately, but only after
\code{show_after} seconds. (Set this to zero, and call `tick(0)` to
force showing the progress bar.)

\code{progress_bar$message()} prints a message above the progress bar.
It fails if the progress bar has already finished.
}

\section{Tokens}{

They can be used in the \code{format} argument when creating the
progress bar.
\describe{
  \item{:bar}{The progress bar itself.}
  \item{:current}{Current tick number.}
  \item{:total}{Total ticks.}
  \item{:elapsed}{Elapsed time in seconds.}
  \item{:elapsedfull}{Elapsed time in hh:mm:ss format.}
  \item{:eta}{Estimated completion time in seconds.}
  \item{:percent}{Completion percentage.}
  \item{:rate}{Download rate, bytes per second. See example below.}
  \item{:tick_rate}{Similar to \code{:rate}, but we don't assume that
     the units are bytes, we just print the raw number of ticks per
     second.}
  \item{:bytes}{Shows :current, formatted as bytes. Useful
     for downloads or file reads if you don't know the size of the
     file in advance. See example below.}
  \item{:spin}{Shows a spinner that updates even when progress is
     advanced by zero.}
}

Custom tokens are also supported, and you need to pass their
values to \code{progress_bar$tick()} or \code{progress_bar$update()},
in a named list. See example below.
}

\section{Options}{

The `progress_enabled` option can be set to `FALSE` to turn off the
progress bar. This works for the C++ progress bar as well.
}

\examples{

## We don't run the examples on CRAN, because they takes >10s
## altogether. Unfortunately it is hard to create a set of
## meaningful progress bar examples that also run quickly.
\dontrun{

## Basic
pb <- progress_bar$new(total = 100)
for (i in 1:100) {
  pb$tick()
  Sys.sleep(1 / 100)
}

## ETA
pb <- progress_bar$new(
  format = "  downloading [:bar] :percent eta: :eta",
  total = 100, clear = FALSE, width= 60)
for (i in 1:100) {
  pb$tick()
  Sys.sleep(1 / 100)
}

## Elapsed time
pb <- progress_bar$new(
  format = "  downloading [:bar] :percent in :elapsed",
  total = 100, clear = FALSE, width= 60)
for (i in 1:100) {
  pb$tick()
  Sys.sleep(1 / 100)
}

## Spinner
pb <- progress_bar$new(
  format = "(:spin) [:bar] :percent",
  total = 30, clear = FALSE, width = 60)
for (i in 1:30) {
  pb$tick()
  Sys.sleep(3 / 100)
}

## Custom tokens
pb <- progress_bar$new(
  format = "  downloading :what [:bar] :percent eta: :eta",
  clear = FALSE, total = 200, width = 60)
f <- function() {
  for (i in 1:100) {
    pb$tick(tokens = list(what = "foo   "))
    Sys.sleep(2 / 100)
  }
  for (i in 1:100) {
    pb$tick(tokens = list(what = "foobar"))
    Sys.sleep(2 / 100)
  }
}
f()

## Download (or other) rates
pb <- progress_bar$new(
  format = "  downloading foobar at :rate, got :bytes in :elapsed",
  clear = FALSE, total = NA, width = 60)
f <- function() {
  for (i in 1:100) {
    pb$tick(sample(1:100 * 1000, 1))
    Sys.sleep(2/100)
  }
  pb$tick(1e7)
  invisible()
}
f()

pb <- progress_bar$new(
  format = "  got :current rows at :tick_rate/sec",
  clear = FALSE, total = NA, width = 60)
f <- function() {
  for (i in 1:100) {
    pb$tick(sample(1:100, 1))
    Sys.sleep(2/100)
  }
  pb$terminate()
  invisible()
}
f()

}

}
