/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "Comparator (CMP)"
 */

#define DEBUG	0

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	int state_vin[2];

	int state_in[7];
#if CONFIG_WINDOW || CONFIG_SAMPLE
	int state_window_sample;
#endif

	int state_daco;
	int state_cmpo;
	int state_invo;
	int state_couta;
	int state_cout_old;
	int state_cout;

	uint8_t clk_count;

	/* Filter */
	uint16_t filter_val;
	uint8_t filter_count;

	/* CMP Control Register 0 (CMPx_CR0) */
	uint8_t filter_cnt;
	uint8_t hystctr;

	/* CMP Control Register 1 (CMPx_CR1) */
#if CONFIG_SAMPLE
	uint8_t se;
#endif
#if CONFIG_WINDOW
	uint8_t we;
#endif
	uint8_t trigm;
	uint8_t pmode;
	uint8_t inv;
	uint8_t cos;
	uint8_t ope;
	uint8_t en;
	
	/* CMP Filter Period Register (CMPx_FPR) */
	uint8_t filt_per;

	/* CMP Status and Control Register (CMPx_SCR) */
	uint8_t dmaen;
	uint8_t ier;
	uint8_t ief;
	uint8_t cfr;
	uint8_t cff;

	/* DAC Control Register (CMPx_DACCR) */
	uint8_t dacen;
	uint8_t vrsel;
	uint8_t vosel;

	/* MUX Control Register (CMPx_MUXCR) */
	uint8_t pstm;
	uint8_t psel;
	uint8_t msel;
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(in_inN_set)(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
NAME_(clk)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	unsigned int irq;

	irq = (cpssp->NAME.cfr & cpssp->NAME.ier)
		| (cpssp->NAME.cff & cpssp->NAME.ief);

	NAME_(irq_set)(cpssp, irq);
}

static void
NAME_(out_cout_set)(struct cpssp *cpssp)
{
	unsigned int val;

	/* Interrupt control. */
	if (cpssp->NAME.state_cout_old == 0
	 && cpssp->NAME.state_cout == 1) {
		/* Rising edge. */
		cpssp->NAME.cfr = 1;
	} else if (cpssp->NAME.state_cout_old == 1
		&& cpssp->NAME.state_cout == 0) {
		/* Falling edge. */
		cpssp->NAME.cff = 1;
	}
	NAME_(irq_update)(cpssp);

	/* cout to other system functions. */
	NAME_(cout_set)(cpssp, cpssp->NAME.state_cout);

	/* cmp0 to PAD */
	if (cpssp->NAME.ope) {
		switch (cpssp->NAME.cos) {
		case 0: val = cpssp->NAME.state_cout; break;
		case 1: val = cpssp->NAME.state_couta; break;
		default: assert(0); /* Cannot happen. */
		}
		val = val ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0;
	} else {
		val = SIG_STD_LOGIC_Z;
	}
	NAME_(out_out_set)(cpssp, val);
}

static void
NAME_(filter_couta_set)(struct cpssp *cpssp)
{
	if (cpssp->NAME.filter_cnt == 0) {
		cpssp->NAME.state_cout = cpssp->NAME.state_couta;
		NAME_(out_cout_set)(cpssp);
	} else {
		/* Wait for filter clock... */
	}
}

static void
NAME_(filter_clk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.filter_cnt != 0) {
		unsigned int val;

		if (cpssp->NAME.filter_val != cpssp->NAME.state_couta) {
			cpssp->NAME.filter_count = cpssp->NAME.filter_cnt;
			cpssp->NAME.filter_val = cpssp->NAME.state_couta;
		}
		if (cpssp->NAME.filter_count) {
			cpssp->NAME.filter_count--;
		}
		if (cpssp->NAME.filter_count == 0) {
			val = cpssp->NAME.filter_val;
			if (cpssp->NAME.state_cout != val) {
				cpssp->NAME.state_cout = val;
				NAME_(out_cout_set)(cpssp);
			}
		}
	}
}

static void
NAME_(window_invo_set)(struct cpssp *cpssp)
{
#if CONFIG_WINDOW
	if (! cpssp->NAME.we) {
#endif
		cpssp->NAME.state_couta = cpssp->NAME.state_invo;
		NAME_(filter_couta_set)(cpssp);
#if CONFIG_WINDOW
	} else {
		/* Don't pass through... */
	}
#endif
}

#if CONFIG_WINDOW || CONFIG_SAMPLE
static void
NAME_(window_sample_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_window_sample = val;

#if CONFIG_SAMPLE
	if (val
	 && cpssp->NAME.se) {
		NAME_(filter_clk)(cpssp);
	}
#endif
}
#endif

#if CONFIG_WINDOW
static void
NAME_(window_clk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.we) {
		cpssp->NAME.state_couta = cpssp->NAME.state_invo;
		NAME_(filter_couta_set)(cpssp);
	}
}
#endif

static void
NAME_(inv_cmpo_set)(struct cpssp *cpssp)
{
	unsigned int val;

	val = cpssp->NAME.state_cmpo ^ cpssp->NAME.inv;

	cpssp->NAME.state_invo = val;
	NAME_(window_invo_set)(cpssp);
}

static void
NAME_(cmp_set)(struct cpssp *cpssp)
{
	unsigned int pval;
	unsigned int mval;
	unsigned int val;

	switch (cpssp->NAME.psel) {
	case 0: pval = cpssp->NAME.state_in[0]; break;
	case 1: pval = cpssp->NAME.state_in[1]; break;
	case 2: pval = cpssp->NAME.state_in[2]; break;
	case 3: pval = cpssp->NAME.state_in[3]; break;
	case 4: pval = cpssp->NAME.state_in[4]; break;
	case 5: pval = cpssp->NAME.state_in[5]; break;
	case 6: pval = cpssp->NAME.state_in[6]; break;
	case 7: pval = cpssp->NAME.state_daco; break;
	default: assert(0); /* Cannot happen. */
	}
	switch (cpssp->NAME.msel) {
	case 0: mval = cpssp->NAME.state_in[0]; break;
	case 1: mval = cpssp->NAME.state_in[1]; break;
	case 2: mval = cpssp->NAME.state_in[2]; break;
	case 3: mval = cpssp->NAME.state_in[3]; break;
	case 4: mval = cpssp->NAME.state_in[4]; break;
	case 5: mval = cpssp->NAME.state_in[5]; break;
	case 6: mval = cpssp->NAME.state_in[6]; break;
	case 7: mval = cpssp->NAME.state_daco; break;
	default: assert(0); /* Cannot happen. */
	}

	if (cpssp->NAME.en
	 && cpssp->NAME.psel != cpssp->NAME.msel) {
		val = mval < pval;
	} else {
		val = 0;
	}

	cpssp->NAME.state_cmpo = val;
	NAME_(inv_cmpo_set)(cpssp);
}

static void
NAME_(dac_set)(struct cpssp *cpssp)
{
	unsigned int val;

	val = cpssp->NAME.state_vin[cpssp->NAME.vrsel];

	val = val * (cpssp->NAME.vosel + 1) / 64;

	cpssp->NAME.state_daco = val;
	NAME_(cmp_set)(cpssp);
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
#if CONFIG_WINDOW
	NAME_(window_clk)(cpssp);
#endif

	if (cpssp->NAME.clk_count == 0) {
		cpssp->NAME.clk_count = cpssp->NAME.filt_per;
#if CONFIG_SAMPLE
		if (cpssp->NAME.se == 0) {
#endif
			NAME_(filter_clk)(cpssp);
#if CONFIG_SAMPLE
		}
#endif
	} else {
		cpssp->NAME.clk_count--;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 3) & 1) {
			/* CMP Status and Control Register (CMPx_SCR) */
			/* 7: reserved */
			*valp |= cpssp->NAME.dmaen << (6+24);
			/* 5: reserved */
			*valp |= cpssp->NAME.ier << (4+24);
			*valp |= cpssp->NAME.ief << (3+24);
			*valp |= cpssp->NAME.cfr << (2+24);
			*valp |= cpssp->NAME.cff << (1+24);
			*valp |= cpssp->NAME.state_cout << (0+24);
		}
		if ((bs >> 2) & 1) {
			/* CMP Filter Period Register 0 (CMPx_FPR) */
			*valp |= cpssp->NAME.filt_per << (0+16);
		}
		if ((bs >> 1) & 1) {
			/* CMP Control Register 1 (CMPx_CR1) */
#if ! CONFIG_SAMPLE
			/* 7: reserved */
#else
			*valp |= cpssp->NAME.se << (7+8);
#endif
#if ! CONFIG_WINDOW
			/* 6: reserved */
#else
			*valp |= cpssp->NAME.we << (6+8);
#endif
			*valp |= cpssp->NAME.trigm << (5+8);
			*valp |= cpssp->NAME.pmode << (4+8);
			*valp |= cpssp->NAME.inv << (3+8);
			*valp |= cpssp->NAME.cos << (2+8);
			*valp |= cpssp->NAME.ope << (1+8);
			*valp |= cpssp->NAME.en << (0+8);
		}
		if ((bs >> 0) & 1) {
			/* CMP Control Register 0 (CMPx_CR0) */
			/* 7: reserved */
			*valp |= cpssp->NAME.filter_cnt << 4;
			/* 3-2: reserved */
			*valp |= cpssp->NAME.hystctr << 0;
		}
		break;

	case 0x004:
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 1) & 1) {
			/* MUX Control Register (CMPx_MUXCR) */
			*valp |= cpssp->NAME.pstm << (7+8);
			/* 6: reserved */
			*valp |= cpssp->NAME.psel << (3+8);
			*valp |= cpssp->NAME.msel << (0+8);
		}
		if ((bs >> 0) & 1) {
			/* DAC Control Register (CMPx_DACCR) */
			*valp |= cpssp->NAME.dacen << (7+0);
			*valp |= cpssp->NAME.vrsel << (6+0);
			*valp |= cpssp->NAME.vosel << (0+0);
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}


static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		if ((bs >> 3) & 1) {
			/* CMP Status and Control Register (CMPx_SCR) */
			/* 7: reserved */
			cpssp->NAME.dmaen = (val >> (6+24)) & 1;
			/* 5: reserved */
			cpssp->NAME.ier = (val >> (4+24)) & 1;
			cpssp->NAME.ief = (val >> (3+24)) & 1;
			cpssp->NAME.cfr &= ~((val >> (2+24)) & 1);
			cpssp->NAME.cff &= ~((val >> (1+24)) & 1);
			/* 0: Read-only */
		}
		if ((bs >> 2) & 1) {
			/* CMP Filter Period Register 0 (CMPx_FPR) */
			cpssp->NAME.filt_per = (val >> (0+16)) & 0xff;
		}
		if ((bs >> 1) & 1) {
			/* CMP Control Register 1 (CMPx_CR1) */
#if ! CONFIG_SAMPLE
			/* 7: reserved */
#else
			cpssp->NAME.se = (val >> (7+8)) & 1;
#endif
#if ! CONFIG_WINDOW
			/* 6: reserved */
#else
			cpssp->NAME.we = (val >> (6+8)) & 1;
#endif
			cpssp->NAME.trigm = (val >> (5+8)) & 1;
			assert(! cpssp->NAME.trigm); /* FIXME */
			cpssp->NAME.pmode = (val >> (4+8)) & 1;
			cpssp->NAME.inv = (val >> (3+8)) & 1;
			cpssp->NAME.cos = (val >> (2+8)) & 1;
			cpssp->NAME.ope = (val >> (1+8)) & 1;
			cpssp->NAME.en = (val >> (0+8)) & 1;

			power_consume(cpssp, POWER_cmp_set);
		}
		if ((bs >> 0) & 1) {
			/* CMP Control Register 0 (CMPx_CR0) */
			/* 7: reserved */
			cpssp->NAME.filter_cnt = (val >> 4) & 0b111;
			/* 3-2: reserved */
			cpssp->NAME.hystctr = (val >> 0) & 0b11;
		}
		NAME_(cmp_set)(cpssp);
		break;

	case 0x004:
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 1) & 1) {
			/* MUX Control Register (CMPx_MUXCR) */
			cpssp->NAME.pstm = (val >> (7+8)) & 0b1;
			/* 6: reserved */
			cpssp->NAME.psel = (val >> (3+8)) & 0b111;
			cpssp->NAME.msel = (val >> (0+8)) & 0b111;
			NAME_(cmp_set)(cpssp);
		}
		if ((bs >> 0) & 1) {
			/* DAC Control Register (CMPx_DACCR) */
			cpssp->NAME.dacen = (val >> (7+0)) & 1;
			cpssp->NAME.vrsel = (val >> (6+0)) & 1;
			cpssp->NAME.vosel = (val >> (0+0)) & 0b111111;
		}
		NAME_(dac_set)(cpssp);
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(vinN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n <= 1);

	cpssp->NAME.state_vin[n] = SIG_mV(val);
	NAME_(dac_set)(cpssp);
}

static void
NAME_(in_inN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n <= 6);

	cpssp->NAME.state_in[n] = SIG_mV(val);
	NAME_(cmp_set)(cpssp);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.state_daco = 0;
	cpssp->NAME.state_cmpo = 0;
	cpssp->NAME.state_invo = 0;
	cpssp->NAME.state_couta = 0;
	cpssp->NAME.state_cout = 0;

	cpssp->NAME.clk_count = 0;

	cpssp->NAME.filter_val = 0;
	cpssp->NAME.filter_count = 0;

	/* CMP Control Register 0 (CMPx_CR0) */
	cpssp->NAME.filter_cnt = 0;
	cpssp->NAME.hystctr = 0;

	/* CMP Control Register 1 (CMPx_CR1) */
#if CONFIG_SAMPLE
	cpssp->NAME.se = 0;
#endif
#if CONFIG_WINDOW
	cpssp->NAME.we = 0;
#endif
	cpssp->NAME.trigm = 0;
	cpssp->NAME.pmode = 0;
	cpssp->NAME.inv = 0;
	cpssp->NAME.cos = 0;
	cpssp->NAME.ope = 0;
	cpssp->NAME.en = 0;

	/* CMP Filter Period Register (CMPx_FPR) */
	cpssp->NAME.filt_per = 0x00;

	/* CMP Status and Control Register (CMPx_SCR) */
	cpssp->NAME.dmaen = 0;
	cpssp->NAME.ier = 0;
	cpssp->NAME.ief = 0;
	cpssp->NAME.cfr = 0;
	cpssp->NAME.cff = 0;

	/* DAC Control Register (CMPx_DACCR) */
	cpssp->NAME.dacen = 0;
	cpssp->NAME.vrsel = 0;
	cpssp->NAME.vosel = 0;

	/* MUX Control Register (CMPx_MUXCR) */
	cpssp->NAME.pstm = 0;
	cpssp->NAME.psel = 0;
	cpssp->NAME.msel = 0;

	NAME_(dac_set)(cpssp);
	NAME_(irq_update)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
